/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.operations.cmdbf;

import java.util.Map;

import org.eclipse.cosmos.dc.internal.cmdbf.services.CMDBfInternalUtility;
import org.eclipse.cosmos.dc.internal.cmdbf.services.CMDBfMessages;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServiceException;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.IRelationshipTemplateHandler;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.impl.AbstractQueryHandler;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IRelationshipTemplate;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.IEdges;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.IQueryResult;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.QueryOutputArtifactFactory;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IItem;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IRelationship;
import org.eclipse.cosmos.rm.internal.repository.RepositoryMessages;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLInstanceDocument;

/**
 * This is the relationship record type handler for the SML
 * repository.
 * 
 * @author Ali Mehregani
 */
public class RelationshipTemplateHandler extends AbstractQueryHandler implements IRelationshipTemplateHandler
{
	@Override
	public void initialize(Map<String, Object> init) throws CMDBfServiceException
	{
		// Make sure the SML repository is included in the initialization data 
		if (!(init.get(CMDBfConstants.DATA_PROVIDER) instanceof ISMLRepository))
		{
			throw new CMDBfServiceException(
					CMDBfServiceException.RECIEVER,
					CMDBfServiceException.QUERY_ERROR,
					CMDBfMessages.faultsQueryError,
					CMDBfInternalUtility.createTextNode(RepositoryMessages.errorBadInitData));
		}
		super.initialize(init);
	}

	/**
	 * @see org.eclipse.cosmos.dc.provisional.cmdbf.services.query.service.IRelationshipTemplateHandler#execute(org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IRelationshipTemplate)
	 */
	public IEdges execute(IQueryResult result, IRelationshipTemplate relationshipTemplate, IItem source, IItem target) throws CMDBfServiceException
	{
		IEdges edges = QueryOutputArtifactFactory.getInstance().createEdges(relationshipTemplate);

		// Retrieve the source and target SML documents		
		try
		{			
			ISMLRepository repository = (ISMLRepository)getValue(CMDBfConstants.DATA_PROVIDER);
			ISMLInstanceDocument sourceDocument = CMDBfUtil.retrieveDocument(repository, source);
			ISMLInstanceDocument targetDocument = CMDBfUtil.retrieveDocument(repository, target);
			if (sourceDocument != null && targetDocument != null)
			{
				// Is targetDocument referenced in sourceDocument?
				ISMLDocument[] references = sourceDocument.retrieveReferences(targetDocument.getMetadata());
				if (references != null && references.length > 0)
				{
					IRelationship relationship = QueryOutputArtifactFactory.getInstance().createRelationship();			
					String sourceId = sourceDocument.getMetadata().getId();
					String targetId = targetDocument.getMetadata().getId();
					String localId = getLastSegment(sourceId) + "/" + getLastSegment(targetId);
					
					relationship.addInstanceId(QueryOutputArtifactFactory.getInstance().createInstanceId(CMDBfConstants.REPOSITORY_MDRID, localId));
					relationship.setSourceId(QueryOutputArtifactFactory.getInstance().createInstanceId(CMDBfConstants.REPOSITORY_MDRID, sourceId));
					relationship.setTargetId(QueryOutputArtifactFactory.getInstance().createInstanceId(CMDBfConstants.REPOSITORY_MDRID, targetId));
					edges.addRelationship(relationship);
				}
			}
		}
		catch (RepositoryOperationException e)
		{			
			throw new CMDBfServiceException(
					CMDBfServiceException.RECIEVER,
					CMDBfServiceException.QUERY_ERROR,
					CMDBfMessages.faultsQueryError,
					CMDBfInternalUtility.createTextNode(e.getMessage()));
		}
		
		return edges;
	}

	private String getLastSegment(String id)
	{
		int inx = id.lastIndexOf('/');
		if (inx >= 0)
		{
			return id.substring(inx + 1);
		}
		
		return id;
	}
}
