/***********************************************************************
 * Copyright (c) 2009 CA, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     CA, Inc. - Initial implementation
 ***********************************************************************/
package org.eclipse.cosmos.me.sdd.op.listener.event.impl;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Properties;

import org.eclipse.cosmos.me.sdd.op.listener.EntryData;
import org.eclipse.cosmos.me.sdd.op.listener.InstallListener;
import org.eclipse.cosmos.me.sdd.op.listener.InstallationCompletionListener;
import org.eclipse.cosmos.me.sdd.op.listener.InstallationProgressListener;
import org.eclipse.cosmos.me.sdd.op.listener.InstallationSetupListener;
import org.eclipse.cosmos.me.sdd.op.listener.InstallationRollbackListener;
import org.eclipse.cosmos.me.sdd.op.listener.event.IRegistry;
import org.eclipse.cosmos.me.sdd.op.listener.event.InstallationCompletion;
import org.eclipse.cosmos.me.sdd.op.listener.event.InstallationProgress;
import org.eclipse.cosmos.me.sdd.op.listener.event.InstallationRollback;
import org.eclipse.cosmos.me.sdd.op.listener.event.InstallationSetup;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventConstants;
import org.osgi.service.event.EventHandler;

public class Registry implements IRegistry {
	private BundleContext context;
	private Map<InstallListener, ServiceRegistration> registrations = new LinkedHashMap<InstallListener, ServiceRegistration>();
	
	protected void activate(ComponentContext ctxt) {
		context = ctxt.getBundleContext();
	}
	
	protected void deactivate(ComponentContext ctxt) {
		for (ServiceRegistration reg : registrations.values()) {
			reg.unregister();
		}
		registrations.clear();
		context = null;
	}
	
	public void addListener(InstallListener listener) {
		if (listener != null) {
			if (listener instanceof InstallationCompletionListener) {
				Properties props = new Properties();
				props.put(EventConstants.EVENT_TOPIC, InstallationCompletion.Topic.FILTER.value());
				ServiceRegistration reg = context.registerService(EventHandler.class.getName(), createInstallCompletionHandler((InstallationCompletionListener)listener), props);
				if (reg != null) registrations.put(listener, reg);
			}
			if (listener instanceof InstallationProgressListener) {
				Properties props = new Properties();
				props.put(EventConstants.EVENT_TOPIC, InstallationProgress.Topic.FILTER.value());
				ServiceRegistration reg = context.registerService(EventHandler.class.getName(), createInstallProgressHandler((InstallationProgressListener)listener), props);
				if (reg != null) registrations.put(listener, reg);
			}
			if (listener instanceof InstallationSetupListener) {
				Properties props = new Properties();
				props.put(EventConstants.EVENT_TOPIC, InstallationSetup.Topic.FILTER.value());
				ServiceRegistration reg = context.registerService(EventHandler.class.getName(), createInstallSetupHandler((InstallationSetupListener)listener), props);
				if (reg != null) registrations.put(listener, reg);
			}
			if (listener instanceof InstallationRollbackListener) {
				InstallationRollbackListener l = (InstallationRollbackListener) listener;
				Properties props = new Properties();
				props.put(EventConstants.EVENT_TOPIC, InstallationRollback.Topic.FILTER.value());
				ServiceRegistration reg = context.registerService(EventHandler.class.getName(), createInstallRollbackHandler(l), props);
				if (reg != null) registrations.put(listener, reg);
			}
		}
	}

	public void removeListener(InstallListener listener) {
		if (listener != null) {
			registrations.remove(listener);
		}
	}
	
	private EventHandler createInstallCompletionHandler(final InstallationCompletionListener listener) {
		return new EventHandler() {
			public void handleEvent(Event event) {
				InstallationCompletion.Topic topic = InstallationCompletion.Topic.getValue((String) event.getProperty(EventConstants.EVENT_TOPIC));

				String description = (String) event.getProperty(InstallationCompletion.DESCRIPTION);
				
				switch (topic) {
				case COMPLETE:
					listener.installationComplete(); break;
				case CANCELLED:
					listener.installationCancelled(); break;
				case FAILURE:
					listener.installationFailure(description); break;
				}
			}
		};
	}

	private EventHandler createInstallProgressHandler(final InstallationProgressListener listener) {
		return new EventHandler() {
			public void handleEvent(Event event) {
				InstallationProgress.Topic topic = InstallationProgress.Topic.getValue((String) event.getProperty(EventConstants.EVENT_TOPIC));

				String description = (String) event.getProperty(InstallationProgress.DESCRIPTION);
				String action = (String) event.getProperty(InstallationProgress.ACTION);
				EntryData metaData = (EntryData) event.getProperty(InstallationProgress.METADATA);
				Long numBytes = (Long) event.getProperty(InstallationProgress.BYTES);
				
				switch (topic) {
				case BEGINNING:
					if (action != null) {
						listener.progressBeginningAction(action, metaData);
					} else {
						listener.progressBeginningAction(description);
					}
					break;
					
				case COMPLETE:
					if (action != null) {
						listener.progressCompleteAction(action, metaData);
					} else {
						listener.progressCompleteAction(description);
					}
					break;
					
				case FAILED:
					listener.progressFailedAction(action, metaData);
					break;
					
				case BYTES_MOVED:
					listener.bytesMoved(numBytes);
					break;
				}
			}
		};
	}
	
	private EventHandler createInstallSetupHandler(final InstallationSetupListener listener) {
		return new EventHandler() {
			public void handleEvent(Event event) {
				InstallationSetup.Topic topic = InstallationSetup.Topic.getValue((String) event.getProperty(EventConstants.EVENT_TOPIC));

				String description = (String) event.getProperty(InstallationSetup.DESCRIPTION);
				String action = (String) event.getProperty(InstallationSetup.ACTION);
				EntryData metaData = (EntryData) event.getProperty(InstallationSetup.METADATA);
				
				switch (topic) {
				case SETUP:
					if (action != null) {
						listener.setupAction(action, metaData);
					} else {
						listener.setupAction(description);
					}
					break;
				}
			}
			
		};
	}

	private EventHandler createInstallRollbackHandler(final InstallationRollbackListener listener) {
		return new EventHandler() {
			public void handleEvent(Event event) {
				InstallationRollback.Topic topic = InstallationRollback.Topic.getValue((String) event.getProperty(EventConstants.EVENT_TOPIC));

				String description = (String) event.getProperty(InstallationRollback.DESCRIPTION);
				String action = (String) event.getProperty(InstallationRollback.ACTION);
				EntryData metaData = (EntryData) event.getProperty(InstallationRollback.METADATA);
				Long numBytes = (Long) event.getProperty(InstallationRollback.BYTES);
				
				switch (topic) {
				case BEGINNING:
					if (action != null) {
						listener.rollbackProgressBeginningAction(action, metaData);
					} else {
						listener.rollbackProgressBeginningAction(description);
					}
					break;
				case COMPLETE:
					if (action != null) {
						listener.rollbackProgressCompleteAction(action, metaData);
					} else {
						listener.rollbackProgressCompleteAction(description);
					}
					break;
				case BYTES_MOVED:
					listener.rollbackBytesMoved(numBytes); break;
				case FAILURE:
					listener.rollbackFailure(description); break;
				case ALL_COMPLETE:
					listener.allRollbackCompleteAction(); break;
				}
			}
		};
	}

}
