/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.provisional.repository.core;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.eclipse.cosmos.rm.internal.repository.RepositoryMessages;
import org.eclipse.cosmos.rm.provisional.repository.exception.MissingRepositoryException;
import org.eclipse.osgi.util.NLS;

/**
 * Clients are expected to use this factory class to create an instance of a repository.
 * An instance can be created by either invoking {@link #createRepository()} or passing
 * a specific {@link Properties} to {@link #createRepository(Properties)}.
 *  
 * 
 * @author Ali Mehregani
 */
public abstract class SMLRepositoryFactory
{
	/**
	 * The properties file path
	 */
	private static final String PROPERTIES_FILE_PATH = "/META-INF/sml/repository.properties";
	
	
	/**
	 * The specific property to look for
	 */
	private static final String PROPERTY_REPOSITORY_CLASS = "org.eclipse.cosmos.rm.repository";
	
	
	/**
	 * An instance of a repository will be created by first checking the system property 
	 * {@value #PROPERTY_REPOSITORY_CLASS} to determine if it's set to a class name.  
	 * If no such property is found, then the factory class will attempt to retrieve the same 
	 * property name from the properties file: /META-INF/sml/repository.properties using the 
	 * context class loader.  If no such properties file is found or the repository class file 
	 * is missing, then {@link MissingRepositoryException} is thrown.
	 * 
	 * @return An instance of a repository if one can be created
	 * @throws MissingRepositoryException If there are no repositories found
	 */
    public static ISMLRepository createRepository() throws MissingRepositoryException
    {
    	Object repository = null;
        String repositoryClass = System.getProperty(PROPERTY_REPOSITORY_CLASS);
        
        /* Check the system property */
        if (repositoryClass != null)
        {
        	repository = createClassInstance(repositoryClass);
   	
        }
        /* Check for the expected properties file */
        else
        {
        	ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        	classLoader = classLoader == null ? SMLRepositoryFactory.class.getClassLoader() : classLoader;
        	
        	InputStream propertiesInputStream = classLoader.getResourceAsStream(PROPERTIES_FILE_PATH);
        	if (propertiesInputStream != null)
        	{
        		Properties properties = new Properties();
        		try
				{
					properties.load(propertiesInputStream);
				} 
        		catch (IOException e)
				{
        			throw new MissingRepositoryException(e.getLocalizedMessage(), e);
				}
        		
        		return createRepository(properties);        		
        	}        	
        }
        
        if (repository == null)
        	throw new MissingRepositoryException(RepositoryMessages.errorMissingNoImpl);
        else if (!(repository instanceof ISMLRepository))
        	throw new MissingRepositoryException(NLS.bind(RepositoryMessages.errorMissingWrongType, new String[]{repositoryClass, ISMLRepository.class.getName()}));
        	    	
        return (ISMLRepository)repository;
    }
   

	private static Object createClassInstance(String repositoryClass) throws MissingRepositoryException
	{
    	try
		{
			return Class.forName(repositoryClass).newInstance();
		} 
    	catch (IllegalAccessException e)
		{
			throw new MissingRepositoryException(e.getLocalizedMessage(), e);
		} 
    	catch (InstantiationException e)
		{
    		throw new MissingRepositoryException(e.getLocalizedMessage(), e);
		} 
    	catch (ClassNotFoundException e)
		{
    		throw new MissingRepositoryException(e.getLocalizedMessage(), e);
		}     
	}

	
	/**
	 * Creates an instance of a repository using a specific properties file.  The
	 * property name {@value #PROPERTIES_FILE_PATH} will be used to look for a class name. 
	 *   
	 * @param properties The properties file
	 * @return An instance of a repository
	 * 
	 * @throws MissingRepositoryException If no implementation is found
	 */
	public static ISMLRepository createRepository(Properties properties) throws MissingRepositoryException
    {
		if (properties == null)
			throw new MissingRepositoryException(RepositoryMessages.errorMissingNoImpl);
		
		
		String repositoryClassName = properties.getProperty(PROPERTY_REPOSITORY_CLASS);
		if (repositoryClassName == null)
			throw new MissingRepositoryException(RepositoryMessages.errorMissingNoImpl);
		
		Object repository = createClassInstance (repositoryClassName);
		if (!(repository instanceof ISMLRepository))
        	throw new MissingRepositoryException(NLS.bind(RepositoryMessages.errorMissingWrongType, new String[]{repositoryClassName, ISMLRepository.class.getName()}));
		
		return (ISMLRepository)repository;
    }
}
