/***********************************************************************
 * Copyright (c) 2009 CA, Inc. and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     CA, Inc. - Initial implementation
 *     SAS Institute, Inc. - Bug 275669 
 *     SAS Institute, Inc. - Bug 275731
 *     SAS Institute, Inc. - Bug 287760
 ***********************************************************************/
package org.eclipse.cosmos.me.sdd.o10r.impl;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.eclipse.cosmos.me.sdd.cr.ResourceHandler;
import org.eclipse.cosmos.me.sdd.schema.AlternativeRequirementType;
import org.eclipse.cosmos.me.sdd.schema.ConsumptionConstraintType;
import org.eclipse.cosmos.me.sdd.schema.InstallableUnitType;
import org.eclipse.cosmos.me.sdd.schema.PropertyConstraintType;
import org.eclipse.cosmos.me.sdd.schema.RequirementResourceConstraintType;
import org.eclipse.cosmos.me.sdd.schema.RequirementType;
import org.eclipse.cosmos.me.sdd.schema.RequirementsType;
import org.eclipse.cosmos.me.sdd.schema.ResourceType;
import org.eclipse.cosmos.me.sdd.schema.VersionConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ext.SDDContext;

public class RequirementsResolver {
	private static final QName Q_ACTUAL = new QName("actual");
	private static final QName Q_PRIMARY = new QName("primary");
	private static final QName Q_SATISFIED = new QName("satisfied");
	private static final String FALSE = "false";
	private static final String TRUE = "true";
	private SDDContext ctxt;
	private ResourceHandler handler;
	public boolean results = false;
	RequirementResourceConstraintType unsatisfied;
	
	public RequirementsResolver(SDDContext ctxt, ResourceHandler handler) {
		this.ctxt = ctxt;
		this.handler = handler;
	}
	
	public boolean installableUnitResolver() {
		InstallableUnitType iu = ctxt.getDeploymentDescriptor().getInstallableUnit();
		
		results = resolve(iu.getRequirements());
		return results;
	}

	private boolean resolve(RequirementsType requirements) {
		boolean result = true;
		if (requirements != null) {
			for (RequirementType req : requirements.getRequirement()) {
				result = result && resolve(req);
			}
		}
		return result;
	}

	private boolean resolve(RequirementType requirement) {
		boolean result = true;
		
		// at least one must be satisfied
		if (!requirement.getAlternative().isEmpty()) {
			boolean altResult = false;
			BigInteger primary = null;
			for (AlternativeRequirementType req : requirement.getAlternative()) {			
				boolean resolved = resolve(req);	
				if(resolved) {
					// set the primary if not set.
					if(primary == null) {
						primary = req.getPriority();
					}
					BigInteger priority = req.getPriority();
					if(primary != null && priority != null) {
						if(priority.compareTo(primary) > 0 ) {
							primary = priority;
							req.getOtherAttributes().put(Q_PRIMARY, TRUE);
						}
					}
				}
				altResult = altResult || resolved;
			}
			result = result && altResult;
		}
		
		// all must be satisfied
		for (RequirementResourceConstraintType req : requirement.getResourceConstraint()) {
			result = result && resolve(req);
		}		
		return result;
	}

	private boolean resolve(AlternativeRequirementType requirement) {
		boolean result = true;
		for (RequirementResourceConstraintType req : requirement.getResourceConstraint()) {
			result = result && resolve(req);
		}
		return result;
	}

	@SuppressWarnings("unchecked")
	private boolean resolve(RequirementResourceConstraintType requirement) {
		boolean result = true;
		ResourceType rsrc = (ResourceType)requirement.getResourceRef();
		if(requirement.getResourceConstraintGroup().size() > 1) {
			result = handler.evaluate(ctxt, rsrc, requirement);
		}
		else {
			for (Object o : requirement.getResourceConstraintGroup()) {
				if (o instanceof JAXBElement) {
					Object value = ((JAXBElement<?>)o).getValue();
					if (value instanceof PropertyConstraintType) {
						result = result && resolve(rsrc, (PropertyConstraintType)value);
						continue;
					}
					if (value instanceof ConsumptionConstraintType) {
						result = result && resolve(rsrc, (ConsumptionConstraintType)value);
						continue;
					}
					if (value instanceof VersionConstraintType) {
						result = result && resolve(rsrc, (VersionConstraintType)value);
					}
				}
			}
		}
		
		if (result) {
			requirement.getOtherAttributes().put(Q_SATISFIED, TRUE);
		}
		else {
			requirement.getOtherAttributes().put(Q_SATISFIED, FALSE);
		}
		return result;
	}
	
	private boolean resolve(ResourceType rsrc, VersionConstraintType constraint) {
		boolean result = handler.evaluate(ctxt, rsrc, constraint);
		return result;
	}

	private boolean resolve(ResourceType rsrc, PropertyConstraintType constraint) {
		boolean result = handler.evaluate(ctxt, rsrc, constraint);
		return result;
	}
	
	private boolean resolve(ResourceType rsrc, ConsumptionConstraintType constraint) {
		boolean result = handler.evaluate(ctxt, rsrc, constraint);
		Long value = (Long)handler.getPropertyValue(ctxt, rsrc, constraint.getPropertyName());
		constraint.getValue().getOtherAttributes().put(Q_ACTUAL, value.toString());
		return result;
	}

	public List<RequirementResourceConstraintType> getUnsatisfiedResources() {
		List<RequirementResourceConstraintType> result = new ArrayList<RequirementResourceConstraintType>();
		InstallableUnitType iu = ctxt.getDeploymentDescriptor().getInstallableUnit();

		RequirementsType requirements = iu.getRequirements();
		if (requirements != null) {
			for (RequirementType requirement : requirements.getRequirement()) {
				for (AlternativeRequirementType req : requirement.getAlternative()) {
					for (RequirementResourceConstraintType req2 : req.getResourceConstraint()) {
						String value = req2.getOtherAttributes().get(Q_SATISFIED);
						if(FALSE.equals(value)) {
							result.add(req2);
						}
					}
				}
				
				for (RequirementResourceConstraintType req : requirement.getResourceConstraint()) {
					String value = req.getOtherAttributes().get(Q_SATISFIED);
					if(FALSE.equals(value)) {
						result.add(req);
					}
				}
			}
		}
		
		return result;
	}
}
