/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.broker.provisional;

import java.io.File;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.ResourceBundle;

import org.apache.axiom.om.OMAbstractFactory;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMFactory;
import org.apache.axiom.om.OMNamespace;
import org.apache.axis2.context.ServiceContext;
import org.apache.log4j.Logger;
import org.eclipse.cosmos.dc.broker.internal.persistence.BrokerDatabaseImpl;
import org.eclipse.cosmos.dc.broker.provisional.datamodel.BrokerConstants;
import org.eclipse.cosmos.dc.broker.provisional.datamodel.Group;
import org.eclipse.cosmos.dc.broker.provisional.datamodel.Groups;
import org.eclipse.cosmos.dc.broker.provisional.persistence.BrokerDatabase;
import org.eclipse.cosmos.dc.broker.provisional.persistence.BrokerDatabaseException;
import org.eclipse.cosmos.dc.broker.provisional.persistence.MalformedRequestException;

/**
 * A broker is an address registry that store the service EPRs of web services 
 * and information about the services.  
 *
 */
public class Broker {
	
	private BrokerDatabase database = null;
	private Logger logger = null;
	
	private String ResourceBundleName = "org.eclipse.cosmos.dc.broker.internal.persistence.brokerMessages"; 
	
	public void init(ServiceContext serviceContext) {
		// init logger
		logger = Logger.getLogger(getClass());
		logger.setResourceBundle(ResourceBundle.getBundle(ResourceBundleName));

		database = BrokerDatabaseImpl.getInstance();

		File dbdir = serviceContext.getConfigurationContext().getRealPath("database");
		logger.debug("broker database location: " + dbdir.getAbsolutePath());

		Map<String, Object> initMap = new HashMap<String, Object>();
		initMap.put("dbdir", dbdir);
		database.init(initMap);
		
	}
	
	/**
	 * Register a group of web services with the broker.
	 * 
	 * @param group
	 * @return
	 */
	public OMElement registerGroup(OMElement registerGroupRequest) throws BrokerDatabaseException {
		boolean groupAdded = false;
		OMElement groupElem = registerGroupRequest.getFirstElement();
		groupAdded = database.addGroup(Group.getGroup(groupElem));

		OMFactory fac = OMAbstractFactory.getOMFactory();
        OMNamespace omNs = fac.createOMNamespace(BrokerConstants.BROKER_NAMESPACE, "ns");
        OMElement response = fac.createOMElement("registerGroupResponse", omNs);
        if (groupAdded == true) {
        	fac.createOMText(response, BrokerConstants.REGISTRATION_SUCCESSFUL);
        } else {
        	fac.createOMText(response, BrokerConstants.REGISTRATION_DUPLICATED_SERVICE);
        }

		return response;
	}
	
	/**
	 * Deregister a group of web services from the broker.
	 * @param groupId
	 * @return
	 */
	public OMElement deregisterGroup(OMElement deregisterRequest) throws BrokerDatabaseException, MalformedRequestException {
		String host = null;
		String groupName = null;
		boolean deleted = false;
		
		Iterator<?> childIter = deregisterRequest.getChildElements();
		while (childIter.hasNext()) {
			OMElement elem = (OMElement) childIter.next();
			if (elem.getLocalName().equals("host")) {
				host = elem.getText();
			} else if (elem.getLocalName().equals("name")) {
				groupName = elem.getText();
			}
		}
		
		if (host != null && groupName != null) {
			deleted = database.deleteGroup(host, groupName);
		} else {
			throw new MalformedRequestException("Malformed request message for deregisterGroup operation.");
		}

		OMFactory fac = OMAbstractFactory.getOMFactory();
        OMNamespace omNs = fac.createOMNamespace(BrokerConstants.BROKER_NAMESPACE, "ns");
        OMElement response = fac.createOMElement("deregisterGroupResponse", omNs);
        if (deleted) {
        	fac.createOMText(response, BrokerConstants.DEREGISTRATION_SUCCESSFUL);
        } else {
        	fac.createOMText(response, BrokerConstants.DEREGISTRATION_FAILED);
        }

		return response;
	}

	
	/**
	 * Deregister a group of web services from the broker.
	 * @param groupId
	 * @return
	 */
	public OMElement deregisterGroupById(OMElement deregisterRequest) throws BrokerDatabaseException, MalformedRequestException {
		String id = null;
		boolean deleted = false;
		
		Iterator<?> childIter = deregisterRequest.getChildElements();
		while (childIter.hasNext()) {
			OMElement elem = (OMElement) childIter.next();
			if (elem.getLocalName().equals("id")) {
				id = elem.getText();
			}
		}
		
		if (id != null ) {
			deleted = database.deleteGroupById(id);
		} else {
			throw new MalformedRequestException("Malformed request message for deregisterGroupById operation.");
		}

		OMFactory fac = OMAbstractFactory.getOMFactory();
        OMNamespace omNs = fac.createOMNamespace(BrokerConstants.BROKER_NAMESPACE, "ns");
        OMElement response = fac.createOMElement("deregisterGroupResponse", omNs);
        if (deleted) {
        	fac.createOMText(response, BrokerConstants.DEREGISTRATION_SUCCESSFUL);
        } else {
        	fac.createOMText(response, BrokerConstants.DEREGISTRATION_FAILED);
        }

		return response;
	}

	
	
	/**
	 * Gets all groups of web services registered at the broker
	 * @return
	 */
	public OMElement getAllGroups(OMElement getAllGroupsRequest) throws BrokerDatabaseException {
		Groups groups = database.getAllGroups();
		return groups.toOM();
	}
	
	public OMElement getGroup(OMElement getGroupRequest) throws BrokerDatabaseException, MalformedRequestException {
		Group group = new Group();
		String host = null;
		String groupName = null;
		
		Iterator<?> childIter = getGroupRequest.getChildElements();
		while (childIter.hasNext()) {
			OMElement elem = (OMElement) childIter.next();
			if (elem.getLocalName().equals("host")) {
				host = elem.getText();
			} else if (elem.getLocalName().equals("name")) {
				groupName = elem.getText();
			}
		}
		
		if (host != null && groupName != null) {
			group = database.getGroup(host, groupName);
		} else {
			throw new MalformedRequestException("Malformed request message for getGroup operation.");
		}
		
		return group.toOM();
	}
	
	public OMElement getGroupById(OMElement getGroupRequest) throws BrokerDatabaseException, MalformedRequestException {
		Group group = new Group();
		String id = null;
		
		Iterator<?> childIter = getGroupRequest.getChildElements();
		while (childIter.hasNext()) {
			OMElement elem = (OMElement) childIter.next();
			if (elem.getLocalName().equals("id")) {
				id = elem.getText();
			} 
		}
		
		if (id != null ) {
			group = database.getGroupById(id);
		} else {
			throw new MalformedRequestException("Malformed request message for getGroupById operation.");
		}
		
		return group.toOM();
	}
	
	public OMElement getGroupsByNamespace(OMElement getGroupsByNamespaceRequest) throws BrokerDatabaseException, MalformedRequestException {
		String ns = null;
		Groups groups = new Groups();
		
		Iterator<?> childIter = getGroupsByNamespaceRequest.getChildElements();
		while (childIter.hasNext()) {
			OMElement elem = (OMElement) childIter.next();
			if (elem.getLocalName().equals("namespace")) {
				ns = elem.getText();
			}
		}
		if (ns != null) {
			groups = database.getGroupsByNamespace(ns);
		} else {
			throw new MalformedRequestException("Malformed request message for getGroupsByNamespace operation.");
		}
		return groups.toOM();
	}
	
	public OMElement getGroupsByRecordTypeNamespace(OMElement getGroupsByNamespaceRequest) throws BrokerDatabaseException, MalformedRequestException {
		String ns = null;
		Groups groups = new Groups();
		
		Iterator<?> childIter = getGroupsByNamespaceRequest.getChildElements();
		while (childIter.hasNext()) {
			OMElement elem = (OMElement) childIter.next();
			if (elem.getLocalName().equals("recordTypeNamespace")) {
				ns = elem.getText();
			}
		}
		if (ns != null) {
			groups = database.getGroupsByRecordTypeNamespace(ns);
		} else {
			throw new MalformedRequestException("Malformed request message for getGroupsByRecordTypeNamespace operation.");
		}
		return groups.toOM();
	}
}
