/**********************************************************************
 * Copyright (c) 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.artifacts;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * A cached version of all document contained in an
 * SML-IF document.  
 * 
 * @author Ali Mehregani
 */
public class SMLDocuments
{
	/**
	 * Schema elements indexed by namespace
	 */
	private Map<String, ElementSchemaModel> definitionsIndexedByNamespace;

	/**
	 * Schema elements indexed by alias
	 */
	private Map<String, ElementSchemaModel> definitionsIndexedByAlias;
	
	/**
	 * The instance documents indexed by alias
	 */
	private Map<String, ElementModel> instancesIndexedByAlias;
	
	/**
	 * Stores the instance document without an alias
	 */
	private List<ElementModel> orphanedInstances;
	
	
	/**
	 * Constructor
	 */
	public SMLDocuments()
	{
		this.definitionsIndexedByNamespace = new Hashtable<String, ElementSchemaModel>();
		this.definitionsIndexedByAlias = new Hashtable<String, ElementSchemaModel>();
		this.instancesIndexedByAlias = new Hashtable<String, ElementModel>(); 
		this.orphanedInstances = new ArrayList<ElementModel>();
	}
	
	
	/**
	 * Add an instance document to this structure
	 * 
	 * @param instance The instance document to be added
	 */
	public void addInstance(ElementModel instance)
	{
		if (instance == null)
		{
			return;
		}
		
		String[] aliases = instance.getAliases();
		if (aliases == null || aliases.length <= 0)
		{
			orphanedInstances.add(instance);
		}
		else
		{		
			for (int i = 0; i < aliases.length; i++)
			{
				instancesIndexedByAlias.put(aliases[i], instance);
			}
		}
	}

	
	/**
	 * Retrieves all instance documents
	 * 
	 * @return All instance documents
	 */
	public ElementModel[] getInstances ()
	{
		List<ElementModel> instanceSet = new ArrayList<ElementModel>();
		for (Iterator<String> aliases = instancesIndexedByAlias.keySet().iterator(); aliases.hasNext();)
		{
			String alias = aliases.next();
			ElementModel instance = instancesIndexedByAlias.get(alias);
			if (!instanceSet.contains(instance))
			{
				instanceSet.add(instance);
			}
		}
		
		instanceSet.addAll(orphanedInstances);
		return instanceSet.toArray(new ElementModel[instanceSet.size()]);
	}
	
	
	/**
	 * Retrieve an instance document based on the alias
	 * passed in.
	 * 
	 * @param alias The alias to lookup the instance document
	 * with.
	 * @return The instance document indexed by the alias
	 * passed in
	 */
	public ElementModel getInstanceByAlias(String alias)
	{
		return instancesIndexedByAlias.get(alias);
	}
	
	
	/**
	 * Add a definition document to this structure
	 * 
	 * @param definition The definition document to be added
	 */
	public void addDefinition(ElementSchemaModel definition)
	{
		if (definition == null)
		{
			return;
		}
		
		String namespace = definition.getTargetNamespace();
		if (namespace != null)
		{
			definitionsIndexedByNamespace.put(namespace, definition);
		}
		
		String[] aliases = definition.getAliases();
		for (int i = 0; i < aliases.length; i++)
		{
			definitionsIndexedByAlias.put(aliases[i], definition);
		}
	}
	

	/**
	 * Retrieves all definition documents
	 * 
	 * @return All definition documents
	 */
	public ElementSchemaModel[] getDefinitions()
	{
		List<ElementSchemaModel> definitionSet = new ArrayList<ElementSchemaModel>();
		addToList(definitionSet, definitionsIndexedByAlias);
		addToList(definitionSet, definitionsIndexedByNamespace);		
				
		return definitionSet.toArray(new ElementSchemaModel[definitionSet.size()]);		
	}	
	
	
	private void addToList(List<ElementSchemaModel> definitionSet, Map<String, ElementSchemaModel> definitions)
	{
		for (Iterator<String> keys = definitions.keySet().iterator(); keys.hasNext();)
		{
			String key = keys.next();
			ElementSchemaModel definition = definitions.get(key);
			if (!definitionSet.contains(definition))
			{
				definitionSet.add(definition);
			}
		}			
	}


	/**
	 * Retrieve the definition based on the alias passed
	 * in.
	 * 
	 * @param alias An alias
	 * @return The definition document associated with the alias
	 * passed in. 
	 */
	public ElementSchemaModel getDefinitionByAlias (String alias)
	{
		return definitionsIndexedByAlias.get(alias);
	}
	
	
	/**
	 * Retrieves the definition document based on the namespace 
	 * passed in.
	 *  
	 * @param namespace A namespace
	 * @return The definition document indexed by the namespace passed in
	 */
	public ElementSchemaModel getDefinitionByNamespace (String namespace)
	{
		return definitionsIndexedByNamespace.get(namespace);
	}
}
