/***********************************************************************
 * Copyright (c) 2009 CA, Inc. and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     CA, Inc. - Initial implementation
 *     SAS Institute, Inc. - Bug 275731
 ***********************************************************************/
package org.eclipse.cosmos.me.sdd.profile.cim.handler;

import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.eclipse.cosmos.me.sdd.cr.IResourceHandler;
import org.eclipse.cosmos.me.sdd.profile.cim.Version;
import org.eclipse.cosmos.me.sdd.schema.CapacityConstraintType;
import org.eclipse.cosmos.me.sdd.schema.CapacityValueType;
import org.eclipse.cosmos.me.sdd.schema.ConditionalPropertyConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConsumptionConstraintType;
import org.eclipse.cosmos.me.sdd.schema.PropertyConstraintType;
import org.eclipse.cosmos.me.sdd.schema.PropertyValueListType;
import org.eclipse.cosmos.me.sdd.schema.RequirementResourceConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ResourceType;
import org.eclipse.cosmos.me.sdd.schema.VersionConstraintType;
import org.eclipse.cosmos.me.sdd.schema.VersionConstraintValueType;
import org.eclipse.cosmos.me.sdd.schema.VersionRangeType;
import org.eclipse.cosmos.me.sdd.schema.VersionValueType;
import org.eclipse.cosmos.me.sdd.schema.ext.SDDContext;

public class OSHandler implements IResourceHandler {
	static final QName OS = new QName("http://docs.oasis-open.org/sdd/ns/cim-profile", "CIM_OperatingSystem");
	static final QName OS_TYPE = new QName("http://docs.oasis-open.org/sdd/ns/cim-profile", "CIM_OperatingSystem.OSType");
	static final QName OS_MEMSIZE = new QName("http://docs.oasis-open.org/sdd/ns/cim-profile", "CIM_OperatingSystem.TotalVisibleMemorySize");
	static final String Windows = "Windows";
	
	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			CapacityConstraintType constraint) {

		boolean result = true;
		Object value = getPropertyValue(ctxt, rsrc, constraint.getPropertyName());
		CapacityValueType capacityValue = constraint.getValue();
		String unit = capacityValue.getUnit();
		
		long multiplyer = 1;
		if(unit.equals("megabytes")) {
			multiplyer = 1024; 
		}
		String maxCapacity = capacityValue.getMaximum();
		if(maxCapacity != null) {
			result = result && (((Long)value).compareTo(Long.valueOf(maxCapacity)*multiplyer) <= 0);
		}
		
		String minCapacity = capacityValue.getMinimum();
		if(minCapacity != null) {
			result = result && (((Long)value).compareTo(Long.valueOf(minCapacity)*multiplyer) >= 0);
		}
		return true;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			ConsumptionConstraintType constraint) {
		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			PropertyConstraintType constraint) {
		return evaluatePropertyConstraint(ctxt, rsrc, constraint);
	}
	
	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			ConditionalPropertyConstraintType constraint) {
		return evaluatePropertyConstraint(ctxt, rsrc, constraint);
	}
	
	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			RequirementResourceConstraintType constraint) {
		boolean result = true;
		for (Object o : constraint.getResourceConstraintGroup()) {
			if (o instanceof JAXBElement<?>) {
				Object value = ((JAXBElement<?>)o).getValue();
				if (value instanceof PropertyConstraintType) {
					result = result && evaluate(ctxt, rsrc, (PropertyConstraintType)value);
					continue;
				}
				if (value instanceof ConsumptionConstraintType) {
					result = result && evaluate(ctxt, rsrc, (ConsumptionConstraintType)value);
					continue;
				}
				if (value instanceof VersionConstraintType) {
					result = result && evaluate(ctxt, rsrc, (VersionConstraintType)value);
				}
				if (value instanceof CapacityConstraintType) {
					result = result && evaluate(ctxt, rsrc, (CapacityConstraintType)value);
				}
			}
		}
		
		return result;
	}
	
	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			VersionConstraintType constraint) {
		if (OS.equals(rsrc.getType())) {
			Version version = new Version((String)getVersionValue(ctxt, rsrc));
			VersionConstraintValueType supported = constraint.getSupported();
			if (supported != null) {
				// match individual values
				for (VersionValueType value : supported.getValue()) {
					Version v = new Version(value.getVersion());
					if (version.compareTo(v) == 0) {
						return true;
					}
				}
				
				// match ranges
				for (VersionRangeType range : supported.getRange()) {
					if (range.getMinVersion() != null) {
						Version v = new Version(range.getMinVersion());
						if (version.compareTo(v) < 0) {
							continue; // version too low
						}
					}
					
					if (range.getMaxVersion() != null) {
						Version v = new Version(range.getMaxVersion().getValue());
						if (version.compareTo(v) > 0) {
							continue; // version too high
						}
					}
					return true; // passed both min and max tests
				}
			}
		}
		return false;
	}

	public Object getPropertyValue(SDDContext ctxt, ResourceType rsrc, QName qname) {
		// only handle OS resources
		if (OS.equals(rsrc.getType())) {
			if (OS_TYPE.equals(qname)) {
				return (String)System.getProperty("os.name");
			}
			if (OS_MEMSIZE.equals(qname)) {
				return Long.valueOf(Runtime.getRuntime().freeMemory());
			}
		}
		
		return null;
	}

	public Object getVersionValue(SDDContext ctxt, ResourceType rsrc) {
		// only handle OS resources
		if (OS.equals(rsrc.getType())) {
			return (String)System.getProperty("os.version");
		}
		
		return null;
	}

	private boolean evaluatePropertyConstraint(SDDContext ctxt, ResourceType rsrc, Object obj) {
		PropertyValueListType pvlt = null;
		String constraintValue = null;
		Object value = null;
		if (obj instanceof ConditionalPropertyConstraintType) {
			ConditionalPropertyConstraintType constraint = (ConditionalPropertyConstraintType) obj;
			value = getPropertyValue(ctxt, rsrc, constraint.getPropertyName());
			pvlt = constraint.getListOfValues();
			constraintValue = constraint.getValue();
		} else if (obj instanceof PropertyConstraintType) {
			PropertyConstraintType constraint = (PropertyConstraintType) obj;
			value = getPropertyValue(ctxt, rsrc, constraint.getPropertyName());
			pvlt = constraint.getListOfValues();
			constraintValue = constraint.getValue();
		}
		
		if (pvlt != null) {
			for (JAXBElement<String> e : pvlt.getValue()) {
				if (e.getValue().equals(value)) {
					return true;
				}
			}
		} else {
			if (constraintValue.equals(value)) {
				return true;
			}
		}
		return false;
	}
}
