/*******************************************************************************
 * Copyright (c) 2005 Sybase, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Sybase, Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.datatools.sqltools.result.internal;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ResourceBundle;

import org.eclipse.datatools.sqltools.result.ResultsConstants;
import org.eclipse.datatools.sqltools.result.OperationCommand;
import org.eclipse.datatools.sqltools.result.internal.core.IResultManager;
import org.eclipse.datatools.sqltools.result.internal.core.ResultManager;
import org.eclipse.datatools.sqltools.result.internal.index.HistoryIndexListener;
import org.eclipse.datatools.sqltools.result.internal.index.IResultHistoryIndex;
import org.eclipse.datatools.sqltools.result.internal.index.ResultHistoryLuceneIndex;
import org.eclipse.datatools.sqltools.result.internal.model.IResultInstance;
import org.eclipse.datatools.sqltools.result.internal.utils.ILogger;
import org.eclipse.datatools.sqltools.result.internal.utils.StatusLogger;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Color;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

/**
 * The main plugin class to be used in the desktop.
 * 
 * @author Dafan Yang
 */
public class ResultsViewPlugin extends AbstractUIPlugin
{
    // The shared instance.
    private static ResultsViewPlugin plugin;
    private String                   _tempDir;
    private ResourceBundle           _bundle           = ResourceBundle.getBundle(ResultsConstants.PLUGIN_RESOURCE_BUNDLE);
    private IResultManager           _resultManager;
    private IResultHistoryIndex      _historyIndex;
    private static final String      RESULTS_FILE_NAME = "results";
    private Color                    _disabledBakColor;
    
    /**
     * The constructor.
     */
    public ResultsViewPlugin()
    {
        plugin = this;
    }

    /**
     * This method is called upon plug-in activation
     */
    public void start(BundleContext context) throws Exception
    {
        super.start(context);
        
        if (getPreferenceStore().getBoolean(PreferenceConstants.RESULT_HISTORY_SAVE_HISTORY))
        {
            // De-serialize the result history
            String resultsStr = getDefault().getStateLocation().append(RESULTS_FILE_NAME).toOSString();
            File resultsFile = new File(resultsStr);
            if (resultsFile.exists() && resultsFile.isFile())
            {
                try
                {
                    FileInputStream fis = new FileInputStream(resultsFile);
                    ObjectInputStream ois = new ObjectInputStream(fis);
                    Object obj = ois.readObject();
                    if (obj instanceof IResultManager)
                    {
                        _resultManager = (IResultManager) obj;
                    }
                }
                catch (Exception e)
                {
                    ILogger log = new StatusLogger(getDefault().getLog(), ResultsConstants.PLUGIN_ID, _bundle);
                    log.error("ResultsViewPlugin_load_history_error", e);
                }
            }
        }

        _historyIndex = new ResultHistoryLuceneIndex();
        if(_resultManager != null)
        {
            _historyIndex.addResults(_resultManager.getAllResults());
            
            /**
             * Set status of all results whose status are "STARTED" or "RUNNING" to "FAILED" 
             */
            IResultInstance[] instances = _resultManager.getAllResults();
            for (int i = 0; i < instances.length; i++)
            {
                if(!instances[i].isFinished())
                {
                    instances[i].updateStatus(OperationCommand.STATUS_FAILED);
                }
            }
        }
        getResultManager().addResultManagerListener(new HistoryIndexListener());
    }

    /**
     * This method is called when the plug-in is stopped
     */
    public void stop(BundleContext context) throws Exception
    {
        super.stop(context);
        
        if (getPreferenceStore().getBoolean(PreferenceConstants.RESULT_HISTORY_SAVE_HISTORY))
        {
            // Serialize the result history
            if (_resultManager != null)
            {
                String resultsStr = getDefault().getStateLocation().append(RESULTS_FILE_NAME).toOSString();
                try
                {
                    File resultsFile = new File(resultsStr);
                    if (resultsFile.exists())
                    {
                        resultsFile.delete();
                    }
                    resultsFile.createNewFile();

                    FileOutputStream fos = new FileOutputStream(resultsFile);
                    ObjectOutputStream oos = new ObjectOutputStream(fos);
                    oos.writeObject(_resultManager);
                }
                catch (Exception e)
                {
                    ILogger log = new StatusLogger(getDefault().getLog(), ResultsConstants.PLUGIN_ID, _bundle);
                    log.error("ResultsViewPlugin_persist_history_error", e);
                }
            }
        }
        
        if(_disabledBakColor != null)
        {
            _disabledBakColor.dispose();
        }
        plugin = null;
    }

    /**
     * Returns the shared instance.
     */
    public static ResultsViewPlugin getDefault()
    {
        return plugin;
    }

    /**
     * Returns an image descriptor for the image file at the given plug-in relative path.
     * 
     * @param path the path
     * @return the image descriptor
     */
    public static ImageDescriptor getImageDescriptor(String path)
    {
        return AbstractUIPlugin.imageDescriptorFromPlugin(ResultsConstants.PLUGIN_ID, path);
    }

    /**
     * Returns the temporary directory for this plugin
     * 
     * @return the path of temporary directory
     */
    public String getTempDir()
    {
        if (_tempDir == null)
        {
            _tempDir = this.getStateLocation().append("temp").toOSString();
        }
        return _tempDir;
    }
    
    /**
     * Returns the result manager of SQL Results View.
     * 
     * @return the result manager
     */
    public static IResultManager getResultManager()
    {
        synchronized (getDefault())
        {
            if (getDefault()._resultManager == null)
            {
                getDefault()._resultManager = new ResultManager();
            }
            return getDefault()._resultManager;
        }
    }
    
    /**
     * Returns a logger.
     * 
     * @param bundle the bundle used in logger
     * @return a logger
     */
    public static ILogger getLogger(ResourceBundle bundle)
    {
        return new StatusLogger(getDefault().getLog(), ResultsConstants.PLUGIN_ID, bundle == null ? getDefault()._bundle
                : bundle);
    }
    
    /**
     * Returns this plugin's unique identifier
     * 
     * @retun this plugin's unique identifier
     */
    public static String getPluginId()
    {
        return getDefault().getBundle().getSymbolicName();
    }
    
    /**
     * Returns the history index
     * 
     * @return the history index
     */
    public IResultHistoryIndex getResultHistoryIndex()
    {
        return _historyIndex;
    }
    
    public synchronized Color getDisabledBakColor()
    {
        if(_disabledBakColor == null)
        {
            _disabledBakColor = new Color(getDefault().getWorkbench().getDisplay(), 238, 237, 224);
        }
        return _disabledBakColor;
    }
}
