/*******************************************************************************
 * Copyright (c) 2005 Sybase, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Sybase, Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.datatools.sqltools.result.internal.export;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.Properties;

import org.eclipse.datatools.sqltools.result.IResultSetObject;
import org.eclipse.datatools.sqltools.result.internal.model.IResultInstance;

/**
 * This class is to write the result set(s) into output stream/print writer.
 * 
 * @author Dafan Yang
 */
public abstract class AbstractOutputter
{
    /**
     * Ouputs the result set object to a given output stream.
     * 
     * @param rs the result set object
     * @param props output options
     * @param os the output stream
     * @throws IOException -- if i/o error occurs
     */
    public abstract void output(IResultSetObject rs, Properties props, OutputStream os) throws IOException;
    
    /**
     * Ouputs the result set object to a given print writer.
     * 
     * @param rs the result set object
     * @param props output options
     * @param pw the print writer
     * @throws IOException -- if i/o error occurs
     */
    public abstract void output(IResultSetObject rs, Properties props, PrintWriter pw) throws IOException;
    
    /**
     * Ouputs all the result set objects in the result instance to a given output stream.
     * 
     * @param rs the result instance
     * @param props output options
     * @param os the output stream
     * @throws IOException -- if i/o error occurs
     */
    public abstract void output(IResultInstance rs, Properties props, OutputStream os) throws IOException;
    
    /**
     * Ouputs all the result set objects in the result instance to a given print writer.
     * 
     * @param rs the result instance
     * @param props output options
     * @param pw the print writer
     * @throws IOException -- if i/o error occurs
     */
    public abstract void output(IResultInstance rs, Properties props, PrintWriter pw) throws IOException;
    
    protected void writeStringData(PrintWriter w, String s) throws IOException
    {
        if (containsNonXMLChar(s))
        {
            w.write("<![CDATA[");
            encodeCDATA(w, s.getBytes("UTF8"));
            w.write("]]>");
        }
        else
        {
            w.write(escapeForXML(s));
        }
    }

    protected void writeBinaryData(PrintWriter w, Object obj) throws IOException
    {
        if (obj instanceof byte[])
        {
            w.write("<![CDATA[");
            encodeCDATA(w, (byte[]) obj);
            w.write("]]>");
        }
    }
    
    private boolean containsNonXMLChar(String s)
    {
        boolean flag = false;
        int j = s.length();
        for (int i = 0; i < j; i++)
        {
            if (isXMLChar(s.charAt(i)))
            {
                continue;
            }
            flag = true;
            break;
        }

        return flag;
    }
    
    private boolean isXMLChar(char c)
    {
        return c == '\t' || c == '\n' || c == '\r' || ' ' <= c && c <= '\uD7FF' || '\uE000' <= c && c <= '\uFFFD';
    }
    
    protected String escapeForXML(String s)
    {
        StringBuffer stringbuffer = new StringBuffer(s.length() + 16);
        int j = s.length();
        for (int i = 0; i < j; i++)
        {
            char c = s.charAt(i);
            if (c == '&')
            {
                stringbuffer.append("&amp;");
            }
            else if (c == '<')
            {
                stringbuffer.append("&lt;");
            }
            else if (c == '>')
            {
                stringbuffer.append("&gt;");
            }
            else if (c == '\'')
            {
                stringbuffer.append("&apos;");
            }
            else if (c == '"')
            {
                stringbuffer.append("&quot;");
            }
            else
            {
                stringbuffer.append(c);
            }
        }

        return stringbuffer.toString();
    }

    private void encodeCDATA(PrintWriter w, byte[] bytes) throws IOException
    {
        int size = bytes.length;
        for (int i = 0; i < size;)
        {
            int k = (bytes[i] & 0xf0) >> 4;
            w.write(hexTable[k]);
            k = bytes[i] & 0xf;
            w.write(hexTable[k]);
            if (++i % 32 == 0)
            {
                w.println();
            }
        }
    }
    
    static final char hexTable[] =
                                 {
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
                                 };
}
