/*******************************************************************************
 * Copyright (c) 2005 Sybase, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Sybase, Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.datatools.sqltools.result.internal.ui.view;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.datatools.sqltools.result.IResultSetObject;
import org.eclipse.datatools.sqltools.result.ResultsViewUIAccessor;
import org.eclipse.datatools.sqltools.result.internal.PreferenceConstants;
import org.eclipse.datatools.sqltools.result.internal.ResultsViewPlugin;
import org.eclipse.datatools.sqltools.result.internal.model.IResultInstance;
import org.eclipse.datatools.sqltools.result.internal.model.ResultItem;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.widgets.Composite;

/**
 * <code>ResultSection</code> is an UI class, which is responsible for displaying the right part of the SQL Results
 * View. Four kinds of display modes are supported:
 * <ul>
 * <li>Grid mode in multiple windows(default)
 * <li>Grid mode in single window
 * <li>Text mode in multiple windows
 * <li>Text mode in single window
 * </ul>
 * <p>
 * So four classes are defined for each mode --- <code>MultipleTabsGridSection</code>,
 * <code>MultipleTabsTextSection</code>, <code>SingleWindowGridSection</code>, and
 * <code>SingleWindowTextSection</code>.
 * 
 * @author Dafan Yang
 */
abstract public class ResultSection
{
    /* The result instance displaying */
    protected IResultInstance  _resultInstance;
    protected IPreferenceStore _store;

    /* Preferences */
    protected boolean          _displayRowNumber;
    protected boolean          _showRowCountMsg;
    protected boolean          _showHeadings;
    protected String           _nullValue;

    /* Parent of this UI section */
    protected Composite        _parent;
    /* The results view */
    protected ResultsView      _view;
    /*
     * In some senarios, some items may be displayed for two times: showDetail and onNewItemAppended will both display
     * one item (They are invoked in two threads). So we keep the displayed item in a list.
     */
    protected List             _displayedItems;
    
    public ResultSection(IResultInstance instance, ResultsView view)
    {
        super();
        _view = view;
        _resultInstance = instance;
        _store = ResultsViewPlugin.getDefault().getPreferenceStore();
        _displayRowNumber = _store.getBoolean(PreferenceConstants.SQL_RESULTS_VIEW_SHOW_ROW_NUMBER);
        _nullValue = _store.getString(PreferenceConstants.SQL_RESULTS_VIEW_NULL_STRING);
        _showRowCountMsg = _store.getBoolean(PreferenceConstants.SQL_RESULTS_VIEW_SHOW_ROW_COUNT_MSG);
        _showHeadings = _store.getBoolean(PreferenceConstants.SQL_RESULTS_VIEW_SHOW_HEADING);
        _displayedItems = new ArrayList();
    }

    /**
     * Returns the outer control of this UI section
     * 
     * @return the outer composite
     */
    abstract public Composite getControl();

    /**
     * Shows the selected instance (when user clicks the history results item).
     * 
     * @param instance the selected result instance
     */
    public void showDetail(IResultInstance instance)
    {
        if (instance == _resultInstance)
        {
            return;
        }
        int count = instance.getItemCount();
        for (int i = 0; i < count; i++)
        {
            _displayedItems.add(instance.getItem(i));
        }
    }

    /**
     * Shows the newly-appended result item (when resultInstanceAppended event occurs) for the current result instance
     * 
     * @param item the newly-appended result item
     * @param index the index of this item
     */
    abstract public void onNewItemAppended(ResultItem item, int index);

    /**
     * Shows the parameters for the current result instance, do nothing in single window display mode, because we need
     * to display the parameters at the end in this mode
     * 
     * @param params the parameters
     */
    abstract public void onParametersShown(List params);
    
    /**
     * Does some jobs when resultInstanceFinish event occurs for the current result instance, for example, when using
     * single window display mode, we need to display the parameters at the end
     * 
     */
    abstract public void onInstanceFinished();
    
    /**
     * Refreshes UI when resultInstanceReset event occurs for the current result instance
     *
     */
    abstract public void onInstanceReseted();
    
    /**
     * Sets focus to the given tab
     * @param tabType the type of the tab to be set focus
     * @see ResultsViewUIAccessor#STATUS_TAB
     * @see ResultsViewUIAccessor#PARAM_TAB
     * @see ResultsViewUIAccessor#MESSAGE_TAB
     * @see ResultsViewUIAccessor#RESULT_TAB
     */
    public void showTab(int tabType)
    {
        // do nothing
    }
    
    /**
     * Sets focus to the given result set
     * @param result the result set
     */
    public void showResultSet(IResultSetObject result)
    {
        // do nothing
    }
    
    /**
     * Sets focus to the given message/result tab with the given number
     * 
     * @param tabType the type of the tab to be set focus
     * @see ResultsViewUIAccessor#MESSAGE_TAB
     * @see ResultsViewUIAccessor#RESULT_TAB
     * @param tabNumber the number of the tab for the given type
     */
    public void showTab(int tabType, int tabNumber)
    {
        // do nothing
    }
    
    /**
     * Returns the row count of the current displaying result set.
     * 
     * @return the row count of the current displaying result set, returns -1 if the current tab is not a result set tab
     *         or current display mode is single tab display mode
     */
    public int getRowCount()
    {
        return -1;
    }
}
