/*******************************************************************************
 * Copyright (c) 2005 Sybase, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Sybase, Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.datatools.sqltools.result.internal.ui.view;

import java.util.ArrayList;

import org.eclipse.datatools.sqltools.result.IResultSetObject;
import org.eclipse.datatools.sqltools.result.internal.model.IResultInstance;
import org.eclipse.datatools.sqltools.result.internal.model.ResultItem;
import org.eclipse.datatools.sqltools.result.internal.ui.export.actions.ExportAllResultSetsAction;
import org.eclipse.datatools.sqltools.result.internal.ui.export.actions.PrintResultSetAction;
import org.eclipse.datatools.sqltools.result.internal.ui.export.actions.SaveAllResultSetsAction;
import org.eclipse.datatools.sqltools.result.internal.ui.viewer.TextResultViewer;
import org.eclipse.datatools.sqltools.result.internal.utils.Messages;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.IDocument;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ST;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;

/**
 * This UI component displays the execution result on one tab in text mode(using text widgets to display result sets).
 * 
 * @author Dafan Yang
 */
public class SingleWindowTextSection extends SingleWindowModeSection
{
    private TextResultViewer _textViewer;

    public SingleWindowTextSection(Composite composite, ResultsView view)
    {
        super(composite, view);
    }

    public SingleWindowTextSection(Composite composite, IResultInstance instance, ResultsView view)
    {
        super(composite, instance, view);
    }

    /**
     * create viewer for the result instance, and set the text
     */
    protected void createViewerForResultInstance(IResultInstance instance)
    {
        _textViewer.getViewer().getTextWidget().dispose();

        //add actions to the context menu of text widget
        ArrayList actions = new ArrayList();
        if (_resultInstance != null)
        {
            Action saveAll = new SaveAllResultSetsAction(_composite.getShell(), _resultInstance);
            saveAll.setText(Messages.SaveAllResultSetAction_Title); 
            actions.add(saveAll);
            
            Action exportAll = new ExportAllResultSetsAction(_composite.getShell(), _resultInstance);
            exportAll.setText(Messages.ExportAllResultSetAction_Title); 
            actions.add(exportAll);
            
            actions.add(new Separator());
            
            Action printAll = new PrintResultSetAction(_resultInstance, _composite);
            printAll.setText(Messages.PrintAllResultSetAction_Title); 
            actions.add(printAll);
        }
        _textViewer = new TextResultViewer(_composite, SWT.MULTI | SWT.V_SCROLL | SWT.H_SCROLL, actions);

        GridData gd = new GridData(GridData.FILL_BOTH);
        _textViewer.getViewer().getTextWidget().setLayoutData(gd);

        _textViewer.getViewer().getTextWidget().setEditable(true);
        _textViewer.getViewer().getTextWidget().setEnabled(true);
        int count = instance.getItemCount();

        StringBuffer sb = new StringBuffer(""); //$NON-NLS-1$

        for (int i = 0; i < count; i++)
        {
            ResultItem item = instance.getItem(i);
            
            // check if some rows are cached
            if (item.getResultType() == ResultItem.RESULT_SET)
            {
                IResultSetObject result = (IResultSetObject) item.getResultObject();
                if (!result.isAllResultLoaded())
                {
                    _isResultHid = true;
                }
            }
            sb.append(ColumnAlignedResultItem.getResultItemDisplayString(item, _nullValue, _showHeadings,
                _showRowCountMsg));
        }
        IDocument document = new Document();
        document.set(sb.toString());
        _textViewer.getViewer().setDocument(document);
        onInstanceFinished();
        _composite.layout(true);
    }

    public void onNewItemAppended(ResultItem item, int index)
    {
        if (_resultInstance == null || _displayedItems.contains(item))
        {
            return;
        }
        _displayedItems.add(item);
        
        StringBuffer sb = new StringBuffer(""); //$NON-NLS-1$
        if (item.getResultType() == ResultItem.RESULT_SET)
        {
            IResultSetObject result = (IResultSetObject) item.getResultObject();
            if (!result.isAllResultLoaded())
            {
                _isResultHid = true;
            }
        }
        sb.append(_textViewer.getViewer().getTextWidget().getText()).append(
            ColumnAlignedResultItem.getResultItemDisplayString(item, _nullValue, _showHeadings, _showRowCountMsg));
        _textViewer.getViewer().getTextWidget().setVisible(false);
        _textViewer.getViewer().getTextWidget().setText(sb.toString());
        _textViewer.getViewer().getTextWidget().invokeAction(ST.TEXT_END);
        _textViewer.getViewer().getTextWidget().setVisible(true);
        
        //should reset the undo manager when writting result in the text widget
        _textViewer.resetUndoMgr();
    }

    public void createInitialControl(Composite composite)
    {
        super.createInitialControl(composite);

        _textViewer = new TextResultViewer(_composite, SWT.MULTI | SWT.V_SCROLL | SWT.H_SCROLL, null);

        GridData gd = new GridData(GridData.FILL_BOTH);
        _textViewer.getViewer().getTextWidget().setLayoutData(gd);
        _textViewer.getViewer().setDocument(new Document("")); //$NON-NLS-1$

        
        //set the result widget to be enabled when it is initialized
        _textViewer.getViewer().getTextWidget().setEditable(false);
        _textViewer.getViewer().getTextWidget().setEnabled(false);
    }

    public void onInstanceFinished()
    {
        super.onInstanceFinished();
        // always put the in/out params at the end
        
        if(_resultInstance.getParameters() == null)
        {
            return;
        }
        StringBuffer sb = new StringBuffer(_textViewer.getViewer().getTextWidget().getText());
        sb.append(Messages.SingleWindowTextSection_inout_params).append(ColumnAlignedResultItem.getLineSeparator()); 
        sb.append(ColumnAlignedResultItem.getParametersDsipalyStr(getValidParamList(_resultInstance.getParameters()),
                _nullValue));
        _textViewer.getViewer().setDocument(new Document(sb.toString()));
    }

    public void onInstanceReseted()
    {
        _textViewer.getViewer().getTextWidget().setText(""); //$NON-NLS-1$
    }
}
