/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.datatools.connectivity.internal.derby.catalog;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.eclipse.core.runtime.Status;
import org.eclipse.datatools.connectivity.internal.derby.DerbyPlugin;
import org.eclipse.datatools.connectivity.sqm.core.definition.DataModelElementFactory;
import org.eclipse.datatools.connectivity.sqm.core.definition.DatabaseDefinition;
import org.eclipse.datatools.connectivity.sqm.core.rte.ICatalogObject;
import org.eclipse.datatools.connectivity.sqm.core.rte.RefreshManager;
import org.eclipse.datatools.connectivity.sqm.internal.core.RDBCorePlugin;
import org.eclipse.datatools.modelbase.dbdefinition.PredefinedDataTypeDefinition;
import org.eclipse.datatools.modelbase.sql.datatypes.PredefinedDataType;
import org.eclipse.datatools.modelbase.sql.routines.DataAccess;
import org.eclipse.datatools.modelbase.sql.routines.ParameterMode;
import org.eclipse.datatools.modelbase.sql.routines.Routine;
import org.eclipse.datatools.modelbase.sql.routines.SQLRoutinesPackage;
import org.eclipse.datatools.modelbase.sql.routines.impl.ProcedureImpl;
import org.eclipse.datatools.modelbase.sql.schema.Database;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EStructuralFeature;

import com.ibm.icu.util.StringTokenizer;

public class DerbyCatalogProcedure extends ProcedureImpl implements ICatalogObject {
	private static final long serialVersionUID = 3691042079895859506L;
	static final String rsKeyword = "RESULT SETS"; //$NON-NLS-1$
	
	public void refresh() {
		if (this.parameterLoaded){
			this.parameters.clear();
			this.parameterLoaded = false;
		}
		
		this.loaded = false;
		RefreshManager.getInstance().referesh(this);
	}

	public boolean isSystemObject() {
		return false;
	}

	public Connection getConnection() {
		Database database = this.getCatalogDatabase();
		return ((DerbyCatalogDatabase) database).getConnection();
	}
	
	public Database getCatalogDatabase() {
		return this.getSchema().getDatabase();		
	}

	public EList getParameters() {
		if(!this.parameterLoaded) this.loadParameters();
		return this.parameters;
	}

	public String getExternalName() {
		if(!this.loaded) this.load();
		return this.externalName;
	}
	public String getLanguage() {
		if(!this.loaded) this.load();
		return this.language;
	}
	public int getMaxResultSets() {
		if(!this.loaded) this.load();
		return this.maxResultSets;
	}
	public String getParameterStyle() {
		if(!this.loaded) this.load();
		return this.parameterStyle;
	}
	public DataAccess getSqlDataAccess() {
		if(!this.loaded) this.load();
		return this.sqlDataAccess;
	}
	
	public boolean eIsSet(EStructuralFeature eFeature) {
		int id = eDerivedStructuralFeatureID(eFeature);
		if(id == SQLRoutinesPackage.PROCEDURE__PARAMETERS) {
			this.getParameters();
		}
		else if(id == SQLRoutinesPackage.PROCEDURE__EXTERNAL_NAME) {
			this.getExternalName();
		}
		else if(id == SQLRoutinesPackage.PROCEDURE__LANGUAGE) {
			this.getLanguage();
		}
		else if(id == SQLRoutinesPackage.PROCEDURE__MAX_RESULT_SETS) {
			this.getMaxResultSets();
		}
		else if(id == SQLRoutinesPackage.PROCEDURE__PARAMETER_STYLE) {
			this.getParameterStyle();
		}
		
		return super.eIsSet(eFeature);
	}
	
	private synchronized void loadParameters() {
		if(this.parameterLoaded) return;
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	
		
		try {
			DerbyCatalogProcedure.loadParameters(this.getConnection(), super.getParameters(), this);
		}
		catch(Exception e) {
			System.out.println(e.toString());
		}
		
		this.parameterLoaded = true;
		this.eSetDeliver(deliver);		
	}
	
	public static void loadParameters(Connection connection, EList parameterList, Routine routine) throws SQLException {
		final Database database = routine.getSchema().getDatabase();
		final DatabaseDefinition databaseDefinition = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(database);
		final DataModelElementFactory factory = databaseDefinition.getDataModelElementFactory();
			
		DatabaseMetaData metaData = connection.getMetaData();
		ResultSet r = metaData.getProcedureColumns(null, routine.getSchema().getName(),routine.getName(),null);
		try {
			while(r.next()) {
				
				DerbyCatalogParameter parameter = new DerbyCatalogParameter();
				final String name = r.getString(4);
				parameter.setName(name);

				final short columntype = r.getShort(5);
				if (columntype == ParameterMetaData.parameterModeInOut){
					parameter.setMode(ParameterMode.INOUT_LITERAL);
				} else if (columntype== ParameterMetaData.parameterModeOut) {
					parameter.setMode(ParameterMode.OUT_LITERAL);
				} else if (columntype== ParameterMetaData.parameterModeIn) {
					parameter.setMode(ParameterMode.IN_LITERAL);
				}


				final String typeName = r.getString(7);
				PredefinedDataTypeDefinition typeDefinition = databaseDefinition.getPredefinedDataTypeDefinition(typeName);
				if(typeDefinition != null) {
					PredefinedDataType 	type = databaseDefinition.getPredefinedDataType(typeDefinition);
					if(typeDefinition.isLengthSupported()) {
						final int length = r.getInt(9);
						EStructuralFeature feature = type.eClass().getEStructuralFeature("length");  //$NON-NLS-1$
						type.eSet(feature, new Integer(length));
					}
					else if(typeDefinition.isPrecisionSupported()) {
						final int length = r.getInt(8);
						EStructuralFeature feature = type.eClass().getEStructuralFeature("precision"); //$NON-NLS-1$
						type.eSet(feature, new Integer(length));
					}
					
					if(typeDefinition.isScaleSupported()) {
						final int length = r.getInt(10);
						EStructuralFeature feature = type.eClass().getEStructuralFeature("scale"); //$NON-NLS-1$
						type.eSet(feature, new Integer(length));
					}
					
					parameter.setContainedType(type);
				}
				else {
				}
				
				parameterList.add(parameter);
			}
			r.close();
			
		}
		catch(Exception e) {
		}
			
	}
	
	private synchronized void load() {
		if(this.loaded) return;
		this.loaded = true;

		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);
		Connection connection = this.getConnection();

		DataModelElementFactory factory = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(this.getCatalogDatabase()).getDataModelElementFactory();
		
		try {
			String sql = "SELECT ALIAS,JAVACLASSNAME,ALIASINFO" + //$NON-NLS-1$
			" FROM SYS.SYSALIASES A,SYS.SYSSCHEMAS B"+ //$NON-NLS-1$
			" WHERE A.SCHEMAID=B.SCHEMAID" + //$NON-NLS-1$
			" AND B.SCHEMANAME = ?  AND A.ALIAS = ?"; //$NON-NLS-1$
			
		PreparedStatement stmt =connection.prepareStatement(sql);
		stmt.setString(1,this.getSchema().getName());
		stmt.setString(2, this.getName());
		ResultSet r = stmt.executeQuery();

			while(r.next()) {
		
				// Cloudscape only currently supports
				// Java SPs.
				this.language = "JAVA"; //$NON-NLS-1$
				this.parameterStyle = "JAVA"; //$NON-NLS-1$

				String javaClassName = r.getString(2);
				Object aliasInfo = r.getObject(3);
				setRestOfMetaData(javaClassName, aliasInfo);
			}
	
			r.close();
			stmt.close();
		}
		catch (Exception e) {
			Status status = new Status(
					Status.ERROR,
					DerbyPlugin.getDefault().getBundle().getSymbolicName(),
					Status.ERROR,
					"###Error..org.eclipse.datatools.connectivity.internal.derby.catalog.DerbyCatalogProcedure.setRestOfMetaData", //$NON-NLS-1$
					e);
			DerbyPlugin.getDefault().getLog().log(status);
		}
		this.eSetDeliver(deliver);
	}
	protected void setRestOfMetaData(String javaClassName, Object aliasInfo) {
		try {

			// Set SQL Data Access
			String aliasInfoString = aliasInfo.toString();
			setSqlDataAccess(aliasInfoString);
			
			// It appears that the aliasInfo object is not returned in the resultset
			// if the connection is made using the DB2 Universal Driver.
			// So we will parse the string instead of executing the methods on
			// the object to obtain the information needed.
			// 
			int index = aliasInfoString.lastIndexOf(rsKeyword);
			if (index != -1){
				StringTokenizer tokenizer = new StringTokenizer(aliasInfoString.substring(index+rsKeyword.length()));
				String nextToken = tokenizer.nextToken();
				
				// If for some reason, the next token is not a number, just ignore it.
				try {
					int resultSets = Integer.parseInt(nextToken);
					this.setMaxResultSets(resultSets);
				}
				
				catch (Exception e){}
			}
			
			//	Set External name
			index = aliasInfoString.indexOf("("); //$NON-NLS-1$
			String methodName = aliasInfoString.substring(0,index);
			this.setExternalName(javaClassName+"."+methodName); //$NON-NLS-1$
			
		} catch (Exception e) {
			Status status = new Status(
					Status.ERROR,
					DerbyPlugin.getDefault().getBundle().getSymbolicName(),
					Status.ERROR,
					"###Error..org.eclipse.datatools.connectivity.internal.derby.catalog.DerbyCatalogProcedure.load", //$NON-NLS-1$
					e);
			DerbyPlugin.getDefault().getLog().log(status);
		}
	}
	

	protected void setSqlDataAccess(String aliasInfo) {
		int index = aliasInfo.indexOf("NO SQL"); //$NON-NLS-1$
		if (index != -1) {
			this.setSqlDataAccess(DataAccess.NO_SQL_LITERAL);
		} else {
			index = aliasInfo.indexOf("MODIFIES SQL DATA"); //$NON-NLS-1$
			if (index != -1) {
				this.setSqlDataAccess(DataAccess.MODIFIES_SQL_DATA_LITERAL);

			} else {
				index = aliasInfo.indexOf("CONTAINS SQL"); //$NON-NLS-1$
				if (index != -1) {
					this.setSqlDataAccess(DataAccess.CONTAINS_SQL_LITERAL);

				} else {
					index = aliasInfo.indexOf("READS SQL DATA"); //$NON-NLS-1$
					if (index != -1) {
						this.setSqlDataAccess(DataAccess.READS_SQL_DATA_LITERAL);
					}
				}
			}
		}
	}
	
	
	
	private boolean loaded = false;
	private boolean parameterLoaded = false;

}
