/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.datatools.connectivity.internal.derby.catalog;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;

import org.eclipse.datatools.connectivity.sqm.core.rte.ICatalogObject;
import org.eclipse.datatools.connectivity.sqm.core.rte.RefreshManager;
import org.eclipse.datatools.connectivity.sqm.internal.core.connection.ConnectionFilter;
import org.eclipse.datatools.connectivity.sqm.internal.core.connection.ConnectionFilterListener;
import org.eclipse.datatools.connectivity.sqm.internal.core.connection.ConnectionInfo;
import org.eclipse.datatools.connectivity.sqm.internal.core.connection.DatabaseConnectionRegistry;
import org.eclipse.datatools.modelbase.sql.routines.Routine;
import org.eclipse.datatools.modelbase.sql.routines.SQLRoutinesPackage;
import org.eclipse.datatools.modelbase.sql.schema.Database;
import org.eclipse.datatools.modelbase.sql.schema.SQLObject;
import org.eclipse.datatools.modelbase.sql.schema.SQLSchemaPackage;
import org.eclipse.datatools.modelbase.sql.schema.Schema;
import org.eclipse.datatools.modelbase.sql.schema.impl.SchemaImpl;
import org.eclipse.datatools.modelbase.sql.tables.SQLTablesPackage;
import org.eclipse.datatools.modelbase.sql.tables.Table;
import org.eclipse.emf.common.notify.NotificationChain;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EStructuralFeature;

public class DerbyCatalogSchema extends SchemaImpl implements ICatalogObject {
	private static final long serialVersionUID = 3257285846544429880L;
	
	public synchronized void refresh() {
		this.tablesLoaded = false;
		this.routineLoaded = false;
		RefreshManager.getInstance().referesh(this);
	}

	public boolean isSystemObject() {
		return false;
	}

	public Connection getConnection() {
		Database database = this.getDatabase();
		return ((DerbyCatalogDatabase) database).getConnection();
	}
	
	public Database getCatalogDatabase() {
		return this.getDatabase();
	}

	public EList getTables() {
		if(!this.tablesLoaded) this.loadTables();
		if(!this.synonymLoaded) 
		{
			this.synonymLoaded = true;
			DerbyCatalogSchema.loadSynonym(this.getConnection(),super.getTables(), this);
		}
		return this.tables;
	}
	
	public EList getRoutines() {
		if(!this.routineLoaded) this.loadRoutines();
		return this.routines;
	}
	

	public boolean eIsSet(EStructuralFeature eFeature) {
		int id = eDerivedStructuralFeatureID(eFeature);
		if(id == SQLSchemaPackage.SCHEMA__TABLES) {
			this.getTables();
		}
		else if(id == SQLSchemaPackage.SCHEMA__ROUTINES) {
			this.getRoutines();
		}
		return super.eIsSet(eFeature);
	}
	
	/**
	 * @generated NOT
	 */
	public NotificationChain basicSetDatabase(Database newDatabase,
			NotificationChain msgs) {
		if (database != null) {
			ConnectionInfo connectionInfo = DatabaseConnectionRegistry
					.getInstance().getConnectionForDatabase(database);
			connectionInfo.removeFilterListener(filterListener);
		}
		if (newDatabase != null) {
			ConnectionInfo connectionInfo = DatabaseConnectionRegistry
					.getInstance().getConnectionForDatabase(newDatabase);
			connectionInfo.addFilterListener(filterListener);
		}
		return super.basicSetDatabase(newDatabase, msgs);
	}

	private synchronized void loadTables() {
		if(this.tablesLoaded) return;
		EList tableList = super.getTables();
		Object[] oldList = tableList.toArray();
		tableList.clear();
		Connection connection = this.getConnection();
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	
		
		// TODO:  Restore filter support after new connection manager is implemented
		ConnectionInfo connectionInfo = DatabaseConnectionRegistry.getInstance().getConnectionForDatabase(getCatalogDatabase());
		ConnectionFilter tableFilter = connectionInfo.getFilter(getTableFilterKey());
		if (tableFilter == null) {  //default table filter
			tableFilter = connectionInfo.getFilter(ConnectionFilter.TABLE_FILTER);
		}
		ConnectionFilter viewFilter = connectionInfo.getFilter(getViewFilterKey());
		if (viewFilter == null) {   //default view filter
			viewFilter = connectionInfo.getFilter(ConnectionFilter.VIEW_FILTER);
		}

		String filterStr = ""; //$NON-NLS-1$
		boolean hasFilter = tableFilter != null || viewFilter != null;
		if (hasFilter) {
			if (tableFilter != null) {
				filterStr += "TABLENAME " + tableFilter.getPredicate()  + " AND TABLETYPE='T' OR TABLETYPE='S'";  //$NON-NLS-1$//$NON-NLS-2$
			}
			else
			{
				filterStr += "TABLETYPE='T' OR TABLETYPE='S'"; //$NON-NLS-1$
			}
			
			if (viewFilter != null) {
				if (filterStr.length() >0) filterStr += " OR "; //$NON-NLS-1$
				filterStr += "TABLENAME " + viewFilter.getPredicate()  + " AND TABLETYPE='V' "; //$NON-NLS-1$ //$NON-NLS-2$
			}
			else
			{
				filterStr += " OR TABLETYPE='V' "; //$NON-NLS-1$
			}
			filterStr =  " AND (" + filterStr; //$NON-NLS-1$
			filterStr += ") "; //$NON-NLS-1$
		}

		String query = "SELECT TABLENAME,TABLETYPE" + //$NON-NLS-1$
		" FROM SYS.SYSTABLES A,SYS.SYSSCHEMAS B"+ //$NON-NLS-1$
		" WHERE A.SCHEMAID=B.SCHEMAID" + //$NON-NLS-1$
		" AND B.SCHEMANAME='"+ this.getName()+"'"; //$NON-NLS-1$ //$NON-NLS-2$

		if (hasFilter) {
			query += filterStr;
		}
		try {

			Statement s = connection.createStatement();
			ResultSet r = s.executeQuery(query); 
			
			while(r.next()) {				
				String tableName = r.getString("TABLENAME"); //$NON-NLS-1$
				String type      = r.getString("TABLETYPE"); //$NON-NLS-1$
				Table table;

				EClass metaclass=null;
				if (type.equals("T") ||type.equals("S")){  //$NON-NLS-1$//$NON-NLS-2$
					metaclass = SQLTablesPackage.eINSTANCE.getPersistentTable();
				} else if (type.equals("V")){ //$NON-NLS-1$
					metaclass = SQLTablesPackage.eINSTANCE.getViewTable();
				}

				Object element = DerbyCatalogSchema.findElement(oldList,tableName,metaclass);

				if (element != null) {
					table = (Table) element;
					tableList.add(table);
					((ICatalogObject)table).refresh();
				} else {
					if(type.equals("T") ||type.equals("S") ) {  //$NON-NLS-1$//$NON-NLS-2$
						table = new DerbyCatalogTable();
					}
					else if(type.equals("V")) { //$NON-NLS-1$
						table = new DerbyCatalogView();
					}
					else continue;
					table.setName(tableName);
					tableList.add(table);
				}
				
			}
			this.tablesLoaded = true;
			r.close();
		}
		catch (Exception e) {
		}
		
		this.eSetDeliver(deliver);
	}	
	public static void loadSynonym(Connection connection, EList tableList, Schema schema) {
		String query ="SELECT TABLENAME,TABLETYPE FROM SYS.SYSTABLES A,SYS.SYSSCHEMAS B WHERE A.SCHEMAID=B.SCHEMAID AND B.SCHEMANAME='" + schema.getName() + 
		"' AND TABLETYPE='A'"; //$NON-NLS-1$
		Object[] oldList = tableList.toArray();

		ConnectionInfo connectionInfo = DatabaseConnectionRegistry.getInstance().getConnectionForDatabase(schema.getDatabase());
		ConnectionFilter synonymFilter = connectionInfo.getFilter(schema.getName() +"::"+ConnectionFilter.TABLE_FILTER); //$NON-NLS-1$
		if (synonymFilter == null) {	//default mqt filter
			synonymFilter = connectionInfo.getFilter(ConnectionFilter.ALIAS_FILTER);
		}
		
		String filterStr = ""; //$NON-NLS-1$
		if (synonymFilter != null) {
			filterStr += "A.SYNONYM_NAME " + synonymFilter.getPredicate(); //$NON-NLS-1$
			filterStr =  " AND (" + filterStr; //$NON-NLS-1$
			filterStr += ") "; //$NON-NLS-1$
			query += filterStr;
		}
		try
		{
			Statement s = connection.createStatement();
			ResultSet r = s.executeQuery(query);
			while(r.next()) {
			
				String synonymName = r.getString("TABLENAME"); //$NON-NLS-1$
	
				EClass metaclass = org.eclipse.datatools.modelbase.derby.DerbyModelPackage.eINSTANCE.getSynonym();
	
				Object element = DerbyCatalogSchema.findElement(oldList,synonymName,metaclass);
				org.eclipse.datatools.modelbase.derby.Synonym synonym;
	
				if (element != null) {
					synonym = (org.eclipse.datatools.modelbase.derby.Synonym) element;
					((ICatalogObject)synonym).refresh();
				} else {
					synonym = new DerbyCatalogSynonym();
					synonym.setName(synonymName);
				}
				
				tableList.add(synonym);
			}
			r.close();
			s.close();
		}
		catch (Exception e) {
		}
		
	}	

	private synchronized void loadRoutines() {
		if(this.routineLoaded) return;
		EList routineList = super.getRoutines();
		Object[] oldList = routineList.toArray();
		routineList.clear();
		Connection connection = this.getConnection();
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	
		
		ConnectionInfo connectionInfo = DatabaseConnectionRegistry.getInstance().getConnectionForDatabase(getCatalogDatabase());
		ConnectionFilter spFilter = connectionInfo.getFilter(getSPFilterKey());
		if (spFilter == null) {   //default sp filter
			spFilter = connectionInfo.getFilter(ConnectionFilter.STORED_PROCEDURE_FILTER);
		}

		String filterStr;
		if (spFilter == null) {
			filterStr = new String();
		}
		else {
			filterStr = " AND "+spFilter.getPredicate();
		}

		try {
			final String query = "SELECT ALIAS,ALIASTYPE" + //$NON-NLS-1$
								" FROM SYS.SYSALIASES A,SYS.SYSSCHEMAS B"+ //$NON-NLS-1$
								" WHERE A.ALIASTYPE IN ('P','F')" + //$NON-NLS-1$
								" AND A.SCHEMAID=B.SCHEMAID" + //$NON-NLS-1$
								" AND B.SCHEMANAME='"+ this.getName()+"'"+filterStr;  //$NON-NLS-1$//$NON-NLS-2$
			Statement s = connection.createStatement();
			ResultSet r = s.executeQuery(query); 
					
			while(r.next()) {
				String routineName = r.getString("ALIAS"); //$NON-NLS-1$
				String type      = r.getString("ALIASTYPE"); //$NON-NLS-1$
				Routine routine;

				EClass metaclass=null;
				if (type.equals("P")){ //$NON-NLS-1$
					metaclass = SQLRoutinesPackage.eINSTANCE.getProcedure();
				} else if (type.equals("F")){ //$NON-NLS-1$
					metaclass = SQLRoutinesPackage.eINSTANCE.getUserDefinedFunction();
				}

				Object element = DerbyCatalogSchema.findElement(oldList,routineName,metaclass);
				if (element != null) {
					routine = (Routine) element;
					((ICatalogObject)routine).refresh();
				} else {
				
					if(type.equals("P"))  //$NON-NLS-1$
						routine = new DerbyCatalogProcedure();
					else if(type.equals("F"))  //$NON-NLS-1$
						routine = new DerbyCatalogUserDefinedFunction();
					else continue;
					
					routine.setName(routineName);					
				}
				routineList.add(routine);
			}
			this.routineLoaded = true;
			r.close();
		}
		catch (Exception e) {
		}
		
		this.eSetDeliver(deliver);
	}	
	
	protected static Object findElement(Object[] list, String name,EClass metaclass){
		Object object = null;
		for (int i = 0; i < list.length; i++){
			SQLObject sqlObject = (SQLObject) list[i];
			if (sqlObject.getName().equals(name) && sqlObject.eClass() == metaclass){
				object = list[i];
				break;
			}
		}
		return object;
	}
	
	private void handleFilterChanged(String filterKey) {
		boolean refresh = false;
		ConnectionInfo conInf = DatabaseConnectionRegistry.getInstance()
				.getConnectionForDatabase(getCatalogDatabase());
		if (routineLoaded
				&& (filterKey.equals(getSPFilterKey()) || (ConnectionFilter.STORED_PROCEDURE_FILTER
						.equals(filterKey)
						&& conInf != null && conInf.getFilter(getSPFilterKey()) == null))) {
			routineLoaded = false;
			refresh = true;
		}
		if (tablesLoaded
				&& (filterKey.equals(getTableFilterKey())
						|| filterKey.equals(getViewFilterKey()) || (conInf != null
						&& (ConnectionFilter.TABLE_FILTER.equals(filterKey) && conInf
								.getFilter(getTableFilterKey()) == null) || (ConnectionFilter.VIEW_FILTER
						.equals(filterKey) && conInf
						.getFilter(getViewFilterKey()) == null)))) {
			tablesLoaded = false;
			synonymLoaded = false;
			refresh = true;
		}
		if (refresh) {
			RefreshManager.getInstance().referesh(this);
		}
	}

	private String getTableFilterKey() {
		return this.getName() + "::" + ConnectionFilter.TABLE_FILTER; //$NON-NLS-1$
	}
	
	private String getSPFilterKey() {
		return this.getName() + "::" + ConnectionFilter.STORED_PROCEDURE_FILTER; //$NON-NLS-1$
	}
	
	private String getViewFilterKey() {
		return this.getName() + "::" + ConnectionFilter.VIEW_FILTER; //$NON-NLS-1$
	}
	
	private boolean tablesLoaded = false;
	private boolean synonymLoaded = false;
	private boolean routineLoaded = false;
	private transient ConnectionFilterListener filterListener = new ConnectionFilterListener() {

		public void connectionFilterAdded(String filterKey) {
			handleFilterChanged(filterKey);
		}

		public void connectionFilterRemoved(String filterKey) {
			handleFilterChanged(filterKey);
		}

	};
}
