/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.datatools.connectivity.internal.derby.catalog;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.Statement;

import org.eclipse.datatools.connectivity.internal.derby.ddl.DerbyDdlParser;
import org.eclipse.datatools.connectivity.sqm.core.definition.DataModelElementFactory;
import org.eclipse.datatools.connectivity.sqm.core.definition.DatabaseDefinition;
import org.eclipse.datatools.connectivity.sqm.core.rte.ICatalogObject;
import org.eclipse.datatools.connectivity.sqm.core.rte.RefreshManager;
import org.eclipse.datatools.connectivity.sqm.internal.core.RDBCorePlugin;
import org.eclipse.datatools.modelbase.dbdefinition.PredefinedDataTypeDefinition;
import org.eclipse.datatools.modelbase.sql.datatypes.PredefinedDataType;
import org.eclipse.datatools.modelbase.sql.expressions.QueryExpression;
import org.eclipse.datatools.modelbase.sql.expressions.SQLExpressionsPackage;
import org.eclipse.datatools.modelbase.sql.expressions.ValueExpression;
import org.eclipse.datatools.modelbase.sql.expressions.ValueExpressionDefault;
import org.eclipse.datatools.modelbase.sql.schema.Database;
import org.eclipse.datatools.modelbase.sql.tables.CheckType;
import org.eclipse.datatools.modelbase.sql.tables.Column;
import org.eclipse.datatools.modelbase.sql.tables.SQLTablesPackage;
import org.eclipse.datatools.modelbase.sql.tables.impl.ViewTableImpl;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EStructuralFeature;
;

public class DerbyCatalogView extends ViewTableImpl implements ICatalogObject {
	private static final long serialVersionUID = 3258125856181269553L;
	
	public void refresh() {
		this.columnsLoaded = false;
		this.viewLoaded = false;
		RefreshManager.getInstance().referesh(this);
	}

	public boolean isSystemObject() {
		return false;
	}

	public Connection getConnection() {
		Database database = this.getCatalogDatabase();
		return ((DerbyCatalogDatabase) database).getConnection();
	}
	
	public Database getCatalogDatabase() {
		return this.getSchema().getDatabase();		
	}
	
	public EList getColumns() {
		if(!this.columnsLoaded) this.loadColumns();
		return this.columns;
	}
	
	public QueryExpression getQueryExpression() {
		if(!this.viewLoaded) this.loadView();
		return this.queryExpression;
	}
	
	public CheckType getCheckType() {
		if(!this.viewLoaded) this.loadView();
		return this.checkType;
	}
	
	public boolean eIsSet(EStructuralFeature eFeature) {
		int id = eDerivedStructuralFeatureID(eFeature);
		if(id == SQLTablesPackage.VIEW_TABLE__COLUMNS) {
			this.getColumns();
		}
		if(id == SQLTablesPackage.VIEW_TABLE__CHECK_TYPE) {
			this.getCheckType();
		}
		if(id == SQLTablesPackage.VIEW_TABLE__QUERY_EXPRESSION) {
			this.getQueryExpression();
		}
		return super.eIsSet(eFeature);
	}
	
	
	private synchronized void loadColumns() {
		if(this.columnsLoaded) return;
		EList columnList = super.getColumns();
		columnList.clear();
		Connection connection = this.getConnection();
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	
		
		try {
			DatabaseMetaData metaData = this.getConnection().getMetaData();
			String catalogName = null;
			if(metaData.supportsCatalogsInTableDefinitions()) {
				catalogName = connection.getCatalog();
			}
			final DatabaseDefinition databaseDefinition = this.getDatabaseDefinition();
			final DataModelElementFactory factory = databaseDefinition.getDataModelElementFactory();
			ResultSet r = metaData.getColumns(catalogName, this.getSchema().getName(), this.getName(), null);
			while(r.next()) {
				Column column = new DerbyCatalogColumn();
				
				final String columnName = r.getString(4);
				column.setName(columnName);
				
				final String remarks = r.getString(12);
				column.setDescription(remarks);

				String typeName = r.getString(6);
				
				PredefinedDataTypeDefinition typeDefinition = databaseDefinition.getPredefinedDataTypeDefinition(typeName);
				if(typeDefinition != null) {
					PredefinedDataType type = databaseDefinition.getPredefinedDataType(typeDefinition);
					if(typeDefinition.isLengthSupported()) {
						EStructuralFeature feature = type.eClass().getEStructuralFeature("length");  //$NON-NLS-1$
						type.eSet(feature, new Integer(r.getInt(7)));
					}
					else if(typeDefinition.isPrecisionSupported()) {
						EStructuralFeature feature = type.eClass().getEStructuralFeature("precision"); //$NON-NLS-1$
						type.eSet(feature, new Integer(r.getInt(10)));
					}
					
					if(typeDefinition.isScaleSupported()) {
						EStructuralFeature feature = type.eClass().getEStructuralFeature("scale"); //$NON-NLS-1$
						type.eSet(feature, new Integer(r.getInt(9)));
					}
					column.setContainedType(type);
				}
				else {
					ValueExpression expr = (ValueExpression) factory.create(SQLExpressionsPackage.eINSTANCE.getValueExpressionDefault());
					column.setGenerateExpression(expr);
					((ValueExpressionDefault)expr).setSQL(typeName);
				}
				
				columnList.add(column);
			}
			this.columnsLoaded = true;
			r.close();
		}
		catch (Exception e) {
		}
		
		this.eSetDeliver(deliver);
	}

	private synchronized void loadView() {
		if(this.viewLoaded) return;
		Connection connection = this.getConnection();
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	
		
		try {
			Statement s = connection.createStatement();
			String query=" SELECT CHECKOPTION, VIEWDEFINITION" + //$NON-NLS-1$
						" FROM SYS.SYSVIEWS A, SYS.SYSTABLES B, SYS.SYSSCHEMAS C" + //$NON-NLS-1$
						" WHERE A.TABLEID=B.TABLEID"+ //$NON-NLS-1$
						" AND B.TABLENAME='"+ this.getName() +"'"+ //$NON-NLS-1$ //$NON-NLS-2$
						" AND B.SCHEMAID=C.SCHEMAID"+ //$NON-NLS-1$
						" AND C.SCHEMANAME='"+this.getSchema().getName()+"'";  //$NON-NLS-1$//$NON-NLS-2$
			ResultSet r = s.executeQuery(query);
			while (r.next()){
				final String viewCheck = r.getString("CHECKOPTION"); //$NON-NLS-1$
				if(viewCheck.equals("N")) this.setCheckType(CheckType.NONE_LITERAL); //$NON-NLS-1$
				else if(viewCheck.equals("L")) this.setCheckType(CheckType.LOCAL_LITERAL); //$NON-NLS-1$
				else if(viewCheck.equals("C")) this.setCheckType(CheckType.CASCADED_LITERAL); //$NON-NLS-1$
	
				final String text = r.getString("VIEWDEFINITION"); //$NON-NLS-1$
				DerbyDdlParser ddlParser = new DerbyDdlParser(this.getDatabaseDefinition());
				ddlParser.parseView(this,text);
				
				
			}
			this.viewLoaded = true;
			r.close();
			s.close();
		}
		catch (Exception e) {
			System.out.println(e.toString());
		}
		
		this.eSetDeliver(deliver);
	}
	
	
	
	private DatabaseDefinition getDatabaseDefinition() {
		Database d = this.getSchema().getDatabase();
		return RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(d);
	}
	
	private boolean columnsLoaded=false;
	private boolean viewLoaded= false;
}
