package org.eclipse.datatools.sqltools.debugger.core;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.datatools.connectivity.ConnectEvent;
import org.eclipse.datatools.connectivity.IConnectionProfile;
import org.eclipse.datatools.sqltools.core.DatabaseIdentifier;
import org.eclipse.datatools.sqltools.core.EditorCorePlugin;
import org.eclipse.datatools.sqltools.core.IResourceDisposeListener;
import org.eclipse.datatools.sqltools.core.SQLDevToolsConfiguration;
import org.eclipse.datatools.sqltools.core.SQLToolsFacade;
import org.eclipse.datatools.sqltools.core.ServerIdentifier;
import org.eclipse.datatools.sqltools.core.profile.ConnectProfile;
import org.eclipse.datatools.sqltools.core.profile.ISQLToolsProfileListener;
import org.eclipse.datatools.sqltools.core.profile.ProfileUtil;
import org.eclipse.datatools.sqltools.core.profile.SQLToolsProfileListenersManager;
import org.eclipse.datatools.sqltools.editor.core.connection.ISQLToolsConnectListener;
import org.eclipse.datatools.sqltools.editor.core.connection.SQLToolsConnectListenersManager;

/**
 * Used to manage all debug handlers: creation, destroy
 * 
 * @author Idull
 */
public class DebugHandlerManager implements IDebugHandlerManager, ISQLToolsProfileListener, ISQLToolsConnectListener
{
    /**
     * Holds one debug handler for each profile
     */
    Map _debugHandlers = new Hashtable();

    public DebugHandlerManager()
    {
        SQLToolsProfileListenersManager pManager = SQLToolsProfileListenersManager.getInstance();
        pManager.addProfileListener(this);
        SQLToolsConnectListenersManager cManager = SQLToolsConnectListenersManager.getInstance();
        cManager.addConnectListener(this);
    }

    public synchronized IDebugHandler getOrCreateDebugHandler(String profileName)
    {
        IDebugHandler debugHandler = (IDebugHandler) _debugHandlers.get(profileName);
        if (debugHandler == null)
        {
        	SQLDevToolsConfiguration conf = SQLToolsFacade.getConfigurationByProfileName(profileName);

            // the factory may be null for generic JDBC profile
            if (conf != null && conf instanceof SQLDebuggerConfiguration)
            {
            	SQLDebuggerConfiguration factory = (SQLDebuggerConfiguration)conf;
                debugHandler = factory.createDebugHandler(profileName);
                _debugHandlers.put(profileName, debugHandler);
                EditorCorePlugin.getControlConnectionManager().addResourceDisposeListener(profileName,
                    new DebugHandlerDisposeListener(debugHandler));
                debugHandler.init();
            }
        }
        return debugHandler;
    }

    public synchronized void dispose()
    {
        SQLToolsProfileListenersManager pManager = SQLToolsProfileListenersManager.getInstance();
        pManager.removeProfileListener(this);
        SQLToolsConnectListenersManager cManager = SQLToolsConnectListenersManager.getInstance();
        cManager.removeConnectListener(this);

        Iterator iter = _debugHandlers.values().iterator();
        while (iter.hasNext())
        {
            IDebugHandler handler = (IDebugHandler) iter.next();
            handler.dispose();
        }
        _debugHandlers.clear();
    }

    public boolean okToClose(ConnectEvent event)
    {
        return true;
    }

    public void aboutToClose(ConnectEvent event)
    {

    }

    public void profileConnected(ConnectEvent event)
    {

    }

    public void closeConnection(ConnectEvent event)
    {
        IDebugHandler handler = (IDebugHandler) _debugHandlers.get(event.getConnectionProfile().getName());
        if (handler != null)
        {
            handler.dispose();
            _debugHandlers.remove(event.getConnectionProfile().getName());
        }
    }

    public void profileChanged(IConnectionProfile profile, String oldName, String oldDesc, Boolean oldAutoConnect, boolean onlyNameChanged, ConnectProfile oldProfile)
    {
        IDebugHandler handler = (IDebugHandler) _debugHandlers.get(oldName);
        if (handler != null)
        {
            _debugHandlers.remove(oldName);
            _debugHandlers.put(profile.getName(), handler);
            EditorCorePlugin.getControlConnectionManager().removeResourceDisposeListener(oldName);
            EditorCorePlugin.getControlConnectionManager().addResourceDisposeListener(profile.getName(),
                new DebugHandlerDisposeListener(handler));
        }
    }

    public void profileAdded(IConnectionProfile profile)
    {
        if (!ProfileUtil.isSupportedProfile(profile))
        {
            return;
        }
    }

    public void profileDeleted(IConnectionProfile profile)
    {
        IDebugHandler handler = (IDebugHandler) _debugHandlers.get(profile.getName());
        if (handler != null)
        {
            handler.dispose();
            _debugHandlers.remove(profile.getName());
            EditorCorePlugin.getControlConnectionManager().removeResourceDisposeListener(profile.getName());

        }
    }

    public void profileChanged(IConnectionProfile profile)
    {
        // do nothing
    }

    class DebugHandlerDisposeListener implements IResourceDisposeListener
    {
        IDebugHandler _debugHandler;

        public DebugHandlerDisposeListener(IDebugHandler debugHandler)
        {
            _debugHandler = debugHandler;
        }
        public void dispose()
        {
            _debugHandler.dispose();
            Iterator iter = _debugHandlers.keySet().iterator();
            String shouldRemoveProfile = null;
            while(iter.hasNext())
            {
                String profile = (String)iter.next();
                if(_debugHandlers.get(profile) == _debugHandler)
                {
                    shouldRemoveProfile = profile;
                    break;
                }
            }

            if(shouldRemoveProfile != null)
            {
                _debugHandlers.remove(shouldRemoveProfile);
            }
        }
    }

    public IDebugHandler[] getDebugHandler(ServerIdentifier serverIdentifier)
    {
        ArrayList handlers = new ArrayList();
        Iterator iter = _debugHandlers.keySet().iterator();
        while(iter.hasNext())
        {
            String profile = (String)iter.next();
            ServerIdentifier serverId = ProfileUtil.getServerIdentifier(new DatabaseIdentifier(profile));
            if(serverId.equals(serverIdentifier))
            {
                handlers.add(_debugHandlers.get(profile));
            }
        }
        return (IDebugHandler[]) handlers.toArray(new IDebugHandler[handlers.size()]);
    }

    public IDebugHandler[] getDebugHandlers()
    {
        return (IDebugHandler[]) _debugHandlers.values().toArray(new IDebugHandler[_debugHandlers.values().size()]);
    }

    public IDebugHandler getDebugHandler(String profileName)
    {
        return (IDebugHandler) _debugHandlers.get(profileName);
    }

    public ServerIdentifier[] getServerIdentifiers()
    {
        ArrayList servers = new ArrayList();
        Iterator iter = _debugHandlers.keySet().iterator();
        while(iter.hasNext())
        {
            ServerIdentifier serverIdentifier = ProfileUtil.getServerIdentifier(new DatabaseIdentifier((String) iter
                .next()));
            if(serverIdentifier != null)
            {
                if(!servers.contains(serverIdentifier))
                {
                    servers.add(serverIdentifier);
                }
            }
        }
        return (ServerIdentifier[]) servers.toArray(new ServerIdentifier[servers.size()]);
    }
}
