/*******************************************************************************
 * Copyright (c) 2001, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.datatools.connectivity.sqm.server.internal.ui.explorer.actions.popup;

import java.util.Properties;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.datatools.connectivity.IConnectionProfile;
import org.eclipse.datatools.connectivity.IPropertySetChangeEvent;
import org.eclipse.datatools.connectivity.IPropertySetListener;
import org.eclipse.datatools.connectivity.ProfileRule;
import org.eclipse.datatools.connectivity.sqm.core.internal.ui.explorer.popup.AbstractAction;
import org.eclipse.datatools.connectivity.sqm.internal.core.connection.ConnectionInfo;
import org.eclipse.datatools.connectivity.sqm.internal.core.connection.ConnectionInfoImpl;
import org.eclipse.datatools.connectivity.sqm.server.internal.ui.util.ServerToolsUIConstants;
import org.eclipse.datatools.connectivity.sqm.server.internal.ui.util.resources.ResourceLoader;
import org.eclipse.datatools.modelbase.sql.schema.Database;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.progress.UIJob;

public class WorkOfflineAction extends AbstractAction
{
    private static final String TITLE = ResourceLoader.INSTANCE.queryString("DATATOOLS.SERVER.UI.CONNECTION.OFFLINE.TITLE"); //$NON-NLS-1$
    private static final String MESSAGE = ResourceLoader.INSTANCE.queryString("DATATOOLS.SERVER.UI.CONNECTION.OFFLINE.MESSAGE"); //$NON-NLS-1$
    private static final String OFFLINE_NAME = ResourceLoader.INSTANCE.queryString("DATATOOLS.SERVER.UI.EXPLORER.WORKOFFLINE"); //$NON-NLS-1$
    private static final String OFFLINE_JOB_NAME = ResourceLoader.INSTANCE.queryString("DATATOOLS.SERVER.UI.EXPLORER.WORKOFFLINE_JOB"); //$NON-NLS-1$

    private static Properties connectedStateProperty = new Properties();
    private static Properties disconnectedProperty = new Properties();

    public void selectionChanged(SelectionChangedEvent event)
    {
        super.selectionChanged(event);
        IConnectionProfile connection = (IConnectionProfile) ((IStructuredSelection) this.event.getSelection()).getFirstElement();
        setEnabled(!connection.isConnected());
    }

    public void initialize()
    {
        initializeAction(null, null, OFFLINE_NAME, OFFLINE_NAME);
    }

    public void run()
    {
        IConnectionProfile connection = (IConnectionProfile) ((IStructuredSelection) this.event.getSelection()).getFirstElement();
        Job j = new ConnectionJob(connection);
        j.setUser(true);
        j.schedule();
    }

    private class ConnectionJob extends UIJob
    {
        private IConnectionProfile connection;

        public ConnectionJob(IConnectionProfile connection)
        {
            super(WorkOfflineAction.OFFLINE_JOB_NAME);
            this.connection = connection;
            setRule(new ProfileRule(this.connection));
        }

        public boolean belongsTo(Object family)
        {
            if (family == WorkOfflineAction.OFFLINE_JOB_NAME)
            {
                return true;
            }
            return false;
        }

        public IStatus runInUIThread(final IProgressMonitor monitor)
        {
            try
            {
                monitor.beginTask(this.getName(), 100);
                ConnectionInfo info = new ConnectionInfoImpl(connection,
                        org.eclipse.datatools.connectivity.sqm.internal.core.connection.ConnectionFactory.class, false);

                monitor.worked(40);
                final Database database = info.getCachedDatabase();
                monitor.worked(80);
                if (database != null)
                {
                    try
                    {
                        connectedStateProperty.put(IConnectionProfile.CONNECTED_PROPERTY_ID, Boolean.toString(false));
                        connection.setProperties(IConnectionProfile.CONNECTION_PROFILE_PROPERTY_SET, connectedStateProperty);
    
                        disconnectedProperty.put(ServerToolsUIConstants.CONNECTION_INFO, info);
                        connection.setProperties(ServerToolsUIConstants.OFFLINE_PROPERTY_SET, disconnectedProperty);
                        
                        info.setSharedDatabase(database);
                        connectedStateProperty.put(IConnectionProfile.CONNECTED_PROPERTY_ID, Boolean.toString(true));
                        connection.setProperties(IConnectionProfile.CONNECTION_PROFILE_PROPERTY_SET, connectedStateProperty);
                    }
                    finally
                    {
                        connection.addPropertySetListener(new IPropertySetListener()
                        {
                            public void propertySetChanged(IPropertySetChangeEvent event)
                            {
                                if (IConnectionProfile.CONNECTION_PROFILE_PROPERTY_SET.equals(event.getPropertySetType())
                                        && event.getChangedProperty(IConnectionProfile.CONNECTED_PROPERTY_ID) != null
                                        && !Boolean.valueOf(event.getChangedProperty(
                                                IConnectionProfile.CONNECTED_PROPERTY_ID).getNewValue()).booleanValue())
                                {
                                    connection.setProperties(ServerToolsUIConstants.OFFLINE_PROPERTY_SET, new Properties());
                                    disconnectedProperty.clear();
                                    connectedStateProperty.clear();
                                }
                            }
                        });
                    }
                }
                else
                {
                    MessageDialog.openError(Display.getDefault().getActiveShell(), TITLE, MESSAGE);
                }
                if (monitor.isCanceled())
                {
                    return Status.CANCEL_STATUS;
                }
                monitor.worked(100);
            }
            finally
            {
                monitor.done();
            }
            return Status.OK_STATUS;
        }
    }
}
