/*******************************************************************************
 * Copyright (c) 2004-2005 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: rcernich, shongxum - initial API and implementation
 ******************************************************************************/
package org.eclipse.datatools.connectivity.ui.dse.views;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.datatools.connectivity.ICategory;
import org.eclipse.datatools.connectivity.IConnectionProfile;
import org.eclipse.datatools.connectivity.IProfileListener;
import org.eclipse.datatools.connectivity.IPropertySetChangeEvent;
import org.eclipse.datatools.connectivity.IPropertySetListener;
import org.eclipse.datatools.connectivity.ProfileManager;
import org.eclipse.datatools.connectivity.internal.repository.IConnectionProfileRepositoryConstants;
import org.eclipse.datatools.connectivity.ui.RefreshProfileJob;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.ui.navigator.CommonViewer;

/**
 * @author rcernich, shongxum, brianf
 */
public class ConnectionProfileContentProvider implements ITreeContentProvider {

	private boolean mShowCategories = true;

	private Viewer mViewer;

	private Object mInput;

	private IProfileListener mProfileListener = new IProfileListener() {

		public void profileAdded(IConnectionProfile profile) {
			profile.addPropertySetListener(mPropertyListener);
			refreshViewer(mShowCategories ? profile.getCategory() : null, true);
		}

		public void profileDeleted(IConnectionProfile profile) {
			profile.removePropertySetListener(mPropertyListener);
			refreshViewer(mShowCategories ? profile.getCategory() : null, true);
		}

		public void profileChanged(IConnectionProfile profile) {
			refreshViewer(profile, false);
		}

	};

	private IPropertySetListener mPropertyListener = new IPropertySetListener() {

		public void propertySetChanged(IPropertySetChangeEvent event) {
			handleProfilePropertyChanged(event);
		}
	};

	/**
	 * Constructor
	 */
	public ConnectionProfileContentProvider() {
		super();
		ProfileManager.getInstance().addProfileListener(mProfileListener);
	}

	public void setShowCategories(boolean showCategories) {
		if (showCategories != mShowCategories) {
			mShowCategories = showCategories;
			refreshViewer(null, false);
		}
	}

	public boolean isShowCategories() {
		return mShowCategories;
	}

	protected void refreshViewer(final Object object,
			final boolean expand) {
		if (mViewer instanceof TreeViewer && object instanceof IConnectionProfile ) {
			RefreshProfileJob.scheduleRefreshProfileJob((IConnectionProfile)object,
					(TreeViewer) mViewer);
		}
		else {
			Runnable runner = new Runnable() {

				public void run() {
					((CommonViewer)mViewer).refresh(object);
				}
			};
			mViewer.getControl().getDisplay().syncExec(runner);
		}
	}

	protected void handleProfilePropertyChanged(IPropertySetChangeEvent event) {
		if (IConnectionProfile.CONNECTION_PROFILE_PROPERTY_SET.equals(event
				.getPropertySetType())
				&& event
						.getChangedProperty(IConnectionProfile.CONNECTED_PROPERTY_ID) != null) {
			refreshViewer(event.getConnectionProfile(), false);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
	 */
	public Object[] getChildren(Object parentElement) {
		Object[] children = new Object[0];
		if (parentElement instanceof IWorkspaceRoot) {
			return getChildren(ProfileManager.getInstance());
		}
		else if (parentElement instanceof ProfileManager) {
			if (mShowCategories) {
				// Collapse repository profiles into the root
				ProfileManager pm = (ProfileManager) parentElement;
				List kids = new ArrayList();
				kids
						.addAll(Arrays
								.asList(pm
										.getProfilesByCategory(IConnectionProfileRepositoryConstants.REPOSITORY_CATEGORY_ID)));
				kids.addAll(Arrays.asList(pm.getRootCategories()));
				kids.remove(pm.getCategory(IConnectionProfileRepositoryConstants.REPOSITORY_CATEGORY_ID));
				children = kids.toArray();
			}
			else {
				children = ((ProfileManager) parentElement).getProfiles();
			}
		}
		else if (parentElement instanceof Object[]) {
			children = (Object[]) parentElement;
		}
		else if (parentElement instanceof ICategory) {
			ICategory icat = (ICategory) parentElement;
			ArrayList arry = new ArrayList();
			if (mShowCategories) {
				arry.addAll(icat.getChildCategories());
				arry.addAll(icat.getAssociatedProfiles());
			}
			else {
				arry.addAll(icat.getAssociatedProfiles());
				recurseSubCategoryProfiles(icat, arry);
			}
			children = arry.toArray();
		}
		else if (parentElement instanceof IConnectionProfile) {
			children = new Object[0];
		}
		return children;
	}

	private void recurseSubCategoryProfiles(ICategory parent, List arry) {
		for (Iterator it = parent.getChildCategories().iterator(); it.hasNext();) {
			ICategory sub = (ICategory) it.next();
			arry.addAll(sub.getAssociatedProfiles());
			recurseSubCategoryProfiles(sub, arry);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
	 */
	public Object getParent(Object element) {
		Object parent = null;

		if (element instanceof ICategory) {
			parent = ((ICategory) element).getParent();
			if (parent == null) {
				parent = mInput;
			}
			if (parent == null) {
				parent = ProfileManager.getInstance();
			}
		}
		else if (element instanceof IConnectionProfile) {
			if (mShowCategories) {
				parent = ((IConnectionProfile) element).getCategory();
				if (parent == null
						|| IConnectionProfileRepositoryConstants.REPOSITORY_CATEGORY_ID
								.equals(((ICategory) parent).getId())) {
					parent = mInput;
				}
				if (parent == null) {
					parent = ProfileManager.getInstance();
				}
			}
			else {
				parent = mInput;
			}
		}

		return parent;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
	 */
	public boolean hasChildren(Object element) {
		return getChildren(element).length > 0;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
	 */
	public Object[] getElements(Object inputElement) {
		return getChildren(inputElement);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
	 */
	public void dispose() {
		ProfileManager.getInstance().removeProfileListener(mProfileListener);
		for (Iterator it = Arrays.asList(
				ProfileManager.getInstance().getProfiles()).iterator(); it
				.hasNext();) {
			((IConnectionProfile) it.next())
					.removePropertySetListener(mPropertyListener);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer,
	 *      java.lang.Object, java.lang.Object)
	 */
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		mInput = newInput;
		mViewer = viewer;
		if (mViewer == null) {
			// Remove listeners
			for (Iterator it = Arrays.asList(
					ProfileManager.getInstance().getProfiles()).iterator(); it
					.hasNext();) {
				((IConnectionProfile) it.next())
						.removePropertySetListener(mPropertyListener);
			}
		}
		else {
			// Add listeners
			for (Iterator it = Arrays.asList(
					ProfileManager.getInstance().getProfiles()).iterator(); it
					.hasNext();) {
				((IConnectionProfile) it.next())
						.addPropertySetListener(mPropertyListener);
			}
		}
	}
}