/********************************************************************************
 * Copyright (c) 2005 - 2007 IBM Corporation. All rights reserved.
 * This program and the accompanying materials are made available under the terms
 * of the Eclipse Public License v1.0 which accompanies this distribution, and is 
 * available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Initial Contributors:
 * The following IBM employees contributed to the Remote System Explorer
 * component that contains this file: David McKnight, Kushal Munir, 
 * Michael Berger, David Dykstal, Phil Coulthard, Don Yantzi, Eric Simpson, 
 * Emily Bruner, Mazen Faraj, Adrian Storisteanu, Li Ding, and Kent Hawley.
 * 
 * Contributors:
 * Michael Berger (IBM) - Fixing 140408 - FTP upload does not work
 * Javier Montalvo Orús (Symbian) - Migrate to jakarta commons net FTP client
 * Javier Montalvo Orus (Symbian) - Fixing 161211 - Cannot expand /pub folder as 
 *    anonymous on ftp.wacom.com
 * Javier Montalvo Orus (Symbian) - Fixing 161238 - [ftp] connections to VMS servers are not usable
 * Javier Montalvo Orus (Symbian) - Fixing 176216 - [api] FTP sould provide API to allow clients register their own FTPListingParser   
 ********************************************************************************/

package org.eclipse.rse.internal.services.files.ftp;


import java.io.File;

import org.apache.commons.net.ftp.FTPFile;
import org.eclipse.rse.services.clientserver.archiveutils.ArchiveHandlerManager;
import org.eclipse.rse.services.files.IHostFile;

public class FTPHostFile implements IHostFile
{
	
	private String _name;
	private String _parentPath;
	private boolean _isDirectory;
	private boolean _isArchive;
	private long _lastModified;
	private long _size;
	private boolean _canRead = true;
	private boolean _canWrite = true;
	private boolean _isRoot;
	private boolean _exists;
	private FTPFile _ftpFile;
		
	public FTPHostFile(String parentPath, String name, boolean isDirectory, boolean isRoot, long lastModified, long size, boolean exists)
	{
		_parentPath = parentPath;
		_name = name;
		_isDirectory = isDirectory;
		_lastModified = lastModified;
		_size = size;
		_isArchive = internalIsArchive();
		_canRead = true;
		_canWrite = false;
		_isRoot = isRoot;
		_exists = exists;
	}

	public FTPHostFile(String parentPath, FTPFile ftpFile)
	{
		_parentPath = parentPath;
		_ftpFile = ftpFile;
		
		_name = ftpFile.getName();
		
		_isDirectory = ftpFile.isDirectory();
		_lastModified = ftpFile.getTimestamp().getTimeInMillis();
		_size = ftpFile.getSize();
		_isArchive = internalIsArchive();
		
		_canRead = ftpFile.hasPermission(FTPFile.USER_ACCESS, FTPFile.READ_PERMISSION);
		_canWrite = ftpFile.hasPermission(FTPFile.USER_ACCESS, FTPFile.WRITE_PERMISSION);
		
		_isRoot = false;
		_exists = true;
	}
	
	
	public long getSize()
	{
		return _size;
	}
	
	public boolean isDirectory()
	{
		return _isDirectory;
	}
	
	public boolean isFile()
	{
		return !(_isDirectory || _isRoot);
	}
	
	public String getName()
	{
		return _name;
	}
	
	public boolean canRead() {
		return _canRead;
	}

	public boolean canWrite() {
		return _canWrite;
	}

	public boolean exists() {
		return _exists;
	}

	public String getAbsolutePath()
	{
		if (isRoot() || _parentPath==null) {
			return getName();
		} else {
			StringBuffer path = new StringBuffer(getParentPath());
			if (!_parentPath.endsWith("/") && !_parentPath.endsWith("\\"))//$NON-NLS-1$ //$NON-NLS-2$
			{
				path.append('/');
			}
			path.append(getName());
			return path.toString();
		}
		
	}

	public long getModifiedDate()
	{
		return _lastModified;
	}

	public String geParentPath() {
		return _parentPath;
	}

	public boolean isArchive() {
		return _isArchive;
	}

	public boolean isHidden() {
		return false;
	}

	public boolean isRoot() {
		return _isRoot;
		
	}

	public String getParentPath() {
		return _parentPath;
	}

	public void renameTo(String newAbsolutePath) 
	{
		int i = newAbsolutePath.lastIndexOf("/"); //$NON-NLS-1$
		if (i == -1)
		{
			_name = newAbsolutePath;
		}
		else
		{
			_parentPath = newAbsolutePath.substring(0, i);
			_name = newAbsolutePath.substring(i+1);
		}
		
		_isArchive = internalIsArchive();
	}
	
	public int getUserPermissions()
	{
		int userRead = 0;
		int userWrite = 0;
		int userExec = 0;
		
		//user
		if(_ftpFile!=null)
		{
			userRead = _ftpFile.hasPermission(FTPFile.USER_ACCESS, FTPFile.READ_PERMISSION) ? 1 : 0;
			userWrite = _ftpFile.hasPermission(FTPFile.USER_ACCESS, FTPFile.WRITE_PERMISSION) ? 1 : 0;
			userExec = _ftpFile.hasPermission(FTPFile.USER_ACCESS, FTPFile.EXECUTE_PERMISSION) ? 1 : 0;
		}
		else
		{
			userRead = _canRead ? 1 : 0;
			userWrite = _canWrite ? 1 : 0;
			userExec = 0;
			
		}
		
		return userRead << 2  | userWrite << 1  | userExec;
	}
	
	public int getGroupPermissions()
	{	
		int groupRead = 0;
		int groupWrite = 0;
		int groupExec = 0;
		
		//group
		if(_ftpFile!=null)
		{
			groupRead = _ftpFile.hasPermission(FTPFile.GROUP_ACCESS, FTPFile.READ_PERMISSION) ? 1 : 0;
			groupWrite = _ftpFile.hasPermission(FTPFile.GROUP_ACCESS, FTPFile.WRITE_PERMISSION) ? 1 : 0;
			groupExec = _ftpFile.hasPermission(FTPFile.GROUP_ACCESS, FTPFile.EXECUTE_PERMISSION) ? 1 : 0;
		}
		
		return groupRead << 2 | groupWrite << 1 | groupExec;
	}
		
	public int getOtherPermissions()
	{
		int otherRead = 0;
		int otherWrite = 0;
		int otherExec = 0;
		
		//other
		if(_ftpFile!=null)
		{
			otherRead = _ftpFile.hasPermission(FTPFile.WORLD_ACCESS, FTPFile.READ_PERMISSION) ? 1 : 0;
			otherWrite = _ftpFile.hasPermission(FTPFile.WORLD_ACCESS, FTPFile.WRITE_PERMISSION) ? 1 : 0;
			otherExec = _ftpFile.hasPermission(FTPFile.WORLD_ACCESS, FTPFile.EXECUTE_PERMISSION) ? 1 : 0;
		}
		
		return otherRead << 2 | otherWrite << 1 | otherExec;
	}
	
	
	
	
	protected boolean internalIsArchive()
	{
		return ArchiveHandlerManager.getInstance().isArchive(new File(getAbsolutePath())) 
		&& !ArchiveHandlerManager.isVirtual(getAbsolutePath());
	}
	
}