/*******************************************************************************
 * Copyright (c) 2010 Engineering Group.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Marco Cortella (Engineering Group) - initial API and implementation and/or initial documentation
 *******************************************************************************/
package org.eclipse.ebam.emf;

import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.FlushModeType;
import javax.persistence.Query;

import org.eclipse.ebam.model.entities.SbiThreshold;
import org.eclipse.ebam.model.entities.SbiThresholdValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;



public class SbiThresholdValueTransactions extends EMF {

	/**
	 * Logger.
	 */
	private static Logger logger = 
		LoggerFactory.getLogger(SbiThresholdValueTransactions.class);


	/** Get all thresholdValues contacts.
	 * 
	 * @return list of alarms
	 * @throws Exception
	 */

	public final List getSbiThresholdValueList(Integer thresId) throws Exception {
		List thresValue = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(SbiThresholdValue.class);	
			
			Query query = em.createQuery("select s from "
					+ "SbiThresholdValue s where s.sbiThreshold.thresholdId = :id");
			query.setParameter("id", thresId);
			thresValue = query.getResultList();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return thresValue;
	}

	
	/** Get all thresholdValues contacts in order by position.
	 * 
	 * @return list of alarms
	 * @throws Exception
	 */

	public final List getOrderedSbiThresholdValueList(Integer thresId) throws Exception {
		List thresValue = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(SbiThresholdValue.class);	

			Query query = em.createQuery("select s from "
					+ "SbiThresholdValue s where s.sbiThreshold.thresholdId = :id order by s.position");
			query.setParameter("id", thresId);
			thresValue = query.getResultList();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return thresValue;
	}

	
	
	public final SbiThresholdValue getSbiThresholdValueByLabel(Integer thresId, String label) throws Exception {
		SbiThresholdValue thresValue = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(SbiThresholdValue.class);	

			Query query = em.createQuery("select s from "
					+ "SbiThresholdValue s where s.sbiThreshold.thresholdId = :id AND s.label = :label");
			query.setParameter("id", thresId);
			query.setParameter("label", label);
			thresValue =(SbiThresholdValue) query.getSingleResult();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return thresValue;
	}

	public final SbiThresholdValue getSbiThresholdValueById(Integer thresValueId) throws Exception {
		SbiThresholdValue thresValue = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(SbiThresholdValue.class);	

			Query query = em.createQuery("select s from "
					+ "SbiThresholdValue s where s.idThresholdValue = :id ");
			query.setParameter("id", thresValueId);
			thresValue =(SbiThresholdValue) query.getSingleResult();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return thresValue;
	}

	
	public final SbiThresholdValue getSbiThresholdValueByPosition(Integer thresId, Integer position) throws Exception {
		SbiThresholdValue thresValue = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(SbiThresholdValue.class);	

			Query query = em.createQuery("select s from "
					+ "SbiThresholdValue s where s.sbiThreshold.thresholdId = :id AND position = :position");
			query.setParameter("id", thresId);
			query.setParameter("position", position);
			thresValue =(SbiThresholdValue) query.getSingleResult();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return thresValue;
	}

	
//	/** get AlarmContact by id
//	 * 
//	 * @param id
//	 * @return one AlarmContact
//	 * @throws Exception
//	 */
//
//	public final SbiAlarmContact getSbiAlarmContactById(Integer id) 
//	throws Exception {
//		SbiAlarmContact alarmContact = null;
//		EntityManager em = getEMF().createEntityManager();
//		try {
//			Query query = em.createQuery("select s from "
//					+ "SbiAlarmContact s where s.alarmContactId = :id");
//			query.setParameter("id", id);
//			alarmContact = (SbiAlarmContact) query.getSingleResult();        
//
//		} catch (Exception e) {
//			logger.error(e.getMessage(), e);
//		} finally {			
//			em.close();
//		}
//		return alarmContact;
//	}	
//
//

	/** merge Sbi threshold value.
	 * 
	 */
	public final boolean mergeSbiThresholdValue(final SbiThresholdValue sbiThresholdValue) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			//take all contacts

			em.merge(sbiThresholdValue);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}

	
	/**Method that persists <code>SbiThresholdValue</code> on database.
	 * @param j <code>SbiThresholdValue</code> to save
	 * @return operation result
	 * @throws Exception error on saving 
	 */
	public final boolean createNewSbiThresholdValue(final SbiThresholdValue sbiThresholdValue) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			em.persist(sbiThresholdValue);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}
	
	
	/**
	 *  delete sbiThresholdValue.
	 * @param sbiThresholdValue
	 * @return
	 * @throws Exception
	 */
	
	public final boolean deleteSbiThresholdValue(final SbiThresholdValue sbiThresholdValue) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			em.setFlushMode(FlushModeType.COMMIT);

			SbiThresholdValue toRemove = 
				em.find(SbiThresholdValue.class, sbiThresholdValue.getIdThresholdValue());
			//toRemove.setSbiThreshold(null);
			em.remove(toRemove);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}

	
}
