/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.core.tests.filesystem;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;
import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileInfo;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.filesystem.IFileSystem;
import org.eclipse.core.filesystem.provider.FileSystem;
import org.eclipse.core.internal.filesystem.Messages;
import org.eclipse.core.internal.filesystem.NullFileSystem;
import org.eclipse.core.internal.filesystem.local.LocalFile;
import org.eclipse.core.internal.filesystem.local.LocalFileSystem;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.tests.filesystem.OnCloseWritingFileStore;
import org.eclipse.core.tests.internal.localstore.LocalStoreTestUtil;
import org.eclipse.core.tests.resources.ResourceTestUtil;
import org.eclipse.core.tests.resources.util.WorkspaceResetExtension;
import org.eclipse.osgi.util.NLS;
import org.junit.Assume;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.api.io.TempDir;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;

@ExtendWith(value={WorkspaceResetExtension.class})
public class FileStoreTest {
    private static final Random RANDOM = new Random();
    @TempDir
    static Path tempDirectory;
    private static final AtomicInteger PATH_INDEX;

    static {
        PATH_INDEX = new AtomicInteger();
    }

    private IFileStore createDir(IFileStore store, boolean clear) throws CoreException {
        if (clear && store.fetchInfo().exists()) {
            store.delete(0, null);
        }
        store.mkdir(0, null);
        IFileInfo info = store.fetchInfo();
        Assertions.assertTrue((boolean)info.exists());
        Assertions.assertTrue((boolean)info.isDirectory());
        return store;
    }

    private static Path randomUniqueNotExistingPath() {
        return tempDirectory.resolve(System.currentTimeMillis() + "-" + PATH_INDEX.incrementAndGet());
    }

    private static IFileStore randomUniqueNotExistingFileStore() throws IOException {
        return ResourceTestUtil.getFileStore(FileStoreTest.randomUniqueNotExistingPath());
    }

    private void createFile(IFileStore target, String content) throws CoreException {
        this.createFile(target, content.getBytes());
    }

    private void createFile(IFileStore target, byte[] content) throws CoreException {
        target.write(content, 0, null);
    }

    @Test
    public void testBrokenFetchInfo() throws Exception {
        IFileStore broken = EFS.getStore((URI)new URI("broken://a/b/c"));
        IFileInfo info = broken.fetchInfo();
        Assertions.assertTrue((!info.exists() ? 1 : 0) != 0, (String)"file info does not exist");
        Assertions.assertThrows(CoreException.class, () -> {
            IFileInfo iFileInfo = broken.fetchInfo(0, ResourceTestUtil.createTestMonitor());
        });
    }

    private IFileStore getDirFileStore(String path) throws CoreException {
        IFileStore store = EFS.getLocalFileSystem().getStore(IPath.fromOSString((String)path));
        if (!store.toLocalFile(0, ResourceTestUtil.createTestMonitor()).exists()) {
            store.mkdir(0, null);
        }
        return store;
    }

    private IFileStore[] getFileStoresOnTwoVolumes() {
        IFileStore[] tempDirs = new IFileStore[2];
        char c = 'c';
        while (c <= 'z') {
            block4: {
                try {
                    IFileStore store = this.getDirFileStore(c + ":/temp");
                    IFileInfo info = store.fetchInfo();
                    if (!info.exists() || !info.isDirectory() || info.getAttribute(2)) break block4;
                    if (tempDirs[0] == null) {
                        tempDirs[0] = store;
                        break block4;
                    }
                    tempDirs[1] = store;
                    break;
                }
                catch (CoreException coreException) {
                    // empty catch block
                }
            }
            c = (char)(c + '\u0001');
        }
        return tempDirs;
    }

    @ParameterizedTest(name="File size {0} bytes")
    @ValueSource(ints={10, 0x10000A})
    public void testCopyAcrossVolumes(int fileSize) throws Throwable {
        IFileStore[] tempDirectories = this.getFileStoresOnTwoVolumes();
        Assume.assumeFalse((String)"only executable if at least two volumes are present", (tempDirectories == null || tempDirectories.length < 2 || tempDirectories[0] == null || tempDirectories[1] == null ? 1 : 0) != 0);
        IFileStore tempSrc = tempDirectories[0];
        IFileStore tempDest = tempDirectories[1];
        String subfolderName = "target_" + System.currentTimeMillis();
        IFileStore target = tempSrc.getChild(subfolderName);
        this.createDir(target, true);
        LocalStoreTestUtil.createTree(LocalStoreTestUtil.getTree(target), fileSize);
        IFileStore destination = tempDest.getChild(subfolderName);
        target.copy(destination, 0, null);
        Assertions.assertTrue((boolean)this.verifyTree(LocalStoreTestUtil.getTree(destination)));
        destination.delete(0, null);
        String copyOfSubfolderName = "copy of " + subfolderName;
        destination = tempDest.getChild(copyOfSubfolderName);
        target.copy(destination, 0, null);
        Assertions.assertTrue((boolean)this.verifyTree(LocalStoreTestUtil.getTree(destination)));
        destination.delete(0, null);
        destination = tempDest.getChild(subfolderName);
        String anotherContent = "nothing..................gnihton";
        this.createFile(destination, anotherContent);
        Assertions.assertTrue((!destination.fetchInfo().isDirectory() ? 1 : 0) != 0);
        IFileStore immutableDestination = destination;
        Assertions.assertThrows(CoreException.class, () -> target.copy(immutableDestination, 0, null));
        Assertions.assertTrue((!this.verifyTree(LocalStoreTestUtil.getTree(destination)) ? 1 : 0) != 0);
        destination.delete(0, null);
        destination = tempDest.getChild(subfolderName);
        this.createDir(destination, true);
        target.copy(destination, 0, null);
        Assertions.assertTrue((boolean)this.verifyTree(LocalStoreTestUtil.getTree(destination)));
        destination.delete(0, null);
    }

    @Test
    public void testCopyDirectory() throws Throwable {
        IFileStore temp = FileStoreTest.randomUniqueNotExistingFileStore();
        temp.mkdir(0, null);
        Assertions.assertTrue((boolean)temp.fetchInfo().isDirectory());
        IFileStore target = temp.getChild("target");
        target.delete(0, null);
        LocalStoreTestUtil.createTree(LocalStoreTestUtil.getTree(target));
        IFileStore copyOfTarget = temp.getChild("copy of target");
        target.copy(copyOfTarget, 0, null);
        Assertions.assertTrue((boolean)this.verifyTree(LocalStoreTestUtil.getTree(copyOfTarget)));
    }

    @Test
    public void testCopyDirectoryParentMissing() throws Throwable {
        IFileStore parent = FileStoreTest.randomUniqueNotExistingFileStore();
        IFileStore child = parent.getChild("child");
        IFileStore existing = FileStoreTest.randomUniqueNotExistingFileStore();
        this.createFile(existing, ResourceTestUtil.createRandomString());
        Assertions.assertThrows(CoreException.class, () -> existing.copy(child, 0, ResourceTestUtil.createTestMonitor()));
        Assertions.assertTrue((!child.fetchInfo().exists() ? 1 : 0) != 0);
    }

    @ParameterizedTest(name="File size {0} bytes")
    @ValueSource(ints={10, 0x10000A})
    public void testCaseInsensitive(int fileSize) throws Throwable {
        IFileStore temp = this.createDir(FileStoreTest.randomUniqueNotExistingFileStore(), true);
        boolean isCaseSensitive = temp.getFileSystem().isCaseSensitive();
        Assume.assumeFalse((String)"only relevant for platforms with case-sensitive file system", (boolean)isCaseSensitive);
        byte[] content = new byte[fileSize];
        RANDOM.nextBytes(content);
        IFileStore fileWithSmallName = temp.getChild("filename");
        fileWithSmallName.delete(0, null);
        this.createFile(fileWithSmallName, content);
        System.out.println(fileWithSmallName.fetchInfo().getName());
        Assertions.assertTrue((boolean)fileWithSmallName.fetchInfo().exists());
        Assertions.assertTrue((boolean)Arrays.equals(content, fileWithSmallName.readAllBytes(0, null)));
        IFileStore fileWithOtherName = temp.getChild("FILENAME");
        System.out.println(fileWithOtherName.fetchInfo().getName());
        Assertions.assertTrue((boolean)Arrays.equals(content, fileWithOtherName.readAllBytes(0, null)));
        fileWithSmallName.copy(fileWithOtherName, 2, null);
        Assertions.assertTrue((boolean)Arrays.equals(content, fileWithOtherName.readAllBytes(0, null)));
        Assertions.assertTrue((boolean)fileWithOtherName.fetchInfo().exists());
        Assertions.assertTrue((boolean)fileWithSmallName.fetchInfo().exists());
        fileWithOtherName.delete(0, null);
        Assertions.assertFalse((boolean)fileWithOtherName.fetchInfo().exists());
        Assertions.assertFalse((boolean)fileWithSmallName.fetchInfo().exists());
        CoreException exception = (CoreException)Assertions.assertThrows(CoreException.class, () -> fileWithSmallName.move(fileWithOtherName, 0, null));
        String message = NLS.bind((String)Messages.couldNotMove, (Object)fileWithSmallName.toString());
        Assertions.assertEquals((Object)message, (Object)exception.getMessage());
    }

    @ParameterizedTest(name="File size {0} bytes")
    @ValueSource(ints={10, 0x10000A})
    public void testCopyFile(int fileSize) throws Throwable {
        IFileStore temp = this.createDir(FileStoreTest.randomUniqueNotExistingFileStore(), true);
        byte[] content = new byte[fileSize];
        RANDOM.nextBytes(content);
        IFileStore target = temp.getChild("target");
        target.delete(0, null);
        this.createFile(target, content);
        Assertions.assertTrue((boolean)target.fetchInfo().exists());
        Assertions.assertTrue((boolean)Arrays.equals(content, target.readAllBytes(0, null)));
        IFileStore copyOfTarget = temp.getChild("copy of target");
        target.copy(copyOfTarget, 2, null);
        Assertions.assertTrue((boolean)Arrays.equals(content, copyOfTarget.readAllBytes(0, null)));
        copyOfTarget.delete(0, null);
        if (ResourceTestUtil.isReadOnlySupported()) {
            copyOfTarget = temp.getChild("copy of target");
            ResourceTestUtil.setReadOnly(target, true);
            target.copy(copyOfTarget, 2, null);
            Assertions.assertTrue((boolean)Arrays.equals(content, copyOfTarget.readAllBytes(0, null)));
            ResourceTestUtil.setReadOnly(copyOfTarget, false);
            copyOfTarget.delete(0, null);
            ResourceTestUtil.setReadOnly(target, false);
            target.delete(0, null);
        }
    }

    @ParameterizedTest(name="File size {0} bytes")
    @ValueSource(ints={10, 0x10000A})
    public void testCopyFileAcrossVolumes(int fileSize) throws Throwable {
        IFileStore[] tempDirectories = this.getFileStoresOnTwoVolumes();
        Assume.assumeFalse((String)"only executable if at least two volumes are present", (tempDirectories == null || tempDirectories.length < 2 || tempDirectories[0] == null || tempDirectories[1] == null ? 1 : 0) != 0);
        IFileStore tempSrc = tempDirectories[0];
        IFileStore tempDest = tempDirectories[1];
        byte[] content = new byte[fileSize];
        RANDOM.nextBytes(content);
        String subfolderName = "target_" + System.currentTimeMillis();
        IFileStore target = tempSrc.getChild(subfolderName);
        target.delete(0, null);
        this.createFile(target, content);
        Assertions.assertTrue((boolean)target.fetchInfo().exists());
        Assertions.assertTrue((boolean)Arrays.equals(content, target.readAllBytes(0, null)));
        IFileStore destination = tempDest.getChild(subfolderName);
        target.copy(destination, 2, null);
        Assertions.assertTrue((boolean)Arrays.equals(content, destination.readAllBytes(0, null)));
        destination.delete(0, null);
        String copyOfSubfoldername = "copy of " + subfolderName;
        destination = tempDest.getChild(copyOfSubfoldername);
        target.copy(destination, 2, null);
        Assertions.assertTrue((boolean)Arrays.equals(content, destination.readAllBytes(0, null)));
        destination.delete(0, null);
        destination = tempDest.getChild(subfolderName);
        String anotherContent = "nothing..................gnihton";
        this.createFile(destination, anotherContent);
        Assertions.assertTrue((!destination.fetchInfo().isDirectory() ? 1 : 0) != 0);
        target.copy(destination, 2, null);
        Assertions.assertTrue((boolean)Arrays.equals(content, destination.readAllBytes(0, null)));
        destination.delete(0, null);
        destination = tempDest.getChild(subfolderName);
        this.createDir(destination, true);
        Assertions.assertTrue((boolean)destination.fetchInfo().isDirectory());
        IFileStore immutableDestination = destination;
        Assertions.assertThrows(CoreException.class, () -> target.copy(immutableDestination, 0, null));
        target.delete(0, null);
        this.createFile(target, content);
        Assertions.assertTrue((boolean)destination.fetchInfo().isDirectory());
        destination.delete(0, null);
    }

    @Test
    public void testFileAttributeCopyForSmallFiles() throws CoreException, IOException {
        Path root = Files.createDirectories(FileStoreTest.randomUniqueNotExistingPath(), new FileAttribute[0]);
        Path sourceFile = root.resolve("source.txt");
        Files.writeString(sourceFile, (CharSequence)"This is a test file.", new OpenOption[0]);
        Path targetFile = root.resolve("target.txt");
        IFileStore sourceStore = EFS.getLocalFileSystem().getStore(IPath.fromPath((Path)sourceFile));
        OnCloseWritingFileStore targetStore = new OnCloseWritingFileStore(targetFile);
        sourceStore.copy((IFileStore)targetStore, 0, (IProgressMonitor)new NullProgressMonitor());
        String sourceContent = Files.readString(sourceFile);
        String targetContent = Files.readString(targetFile);
        Assertions.assertEquals((Object)sourceContent, (Object)targetContent);
    }

    @Test
    public void testGetLength() throws Exception {
        IFileStore temp = this.createDir(FileStoreTest.randomUniqueNotExistingFileStore(), true);
        IFileStore target = temp.getChild("target");
        Assertions.assertEquals((long)0L, (long)target.fetchInfo().getLength());
        target.openOutputStream(0, null).close();
        Assertions.assertEquals((long)0L, (long)target.fetchInfo().getLength());
        Throwable throwable = null;
        Object var4_5 = null;
        try (OutputStream out = target.openOutputStream(0, null);){
            out.write(5);
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
        Assertions.assertEquals((long)1L, (long)target.fetchInfo().getLength());
    }

    @Test
    public void testGetStat() throws CoreException, IOException {
        IFileStore temp = this.createDir(FileStoreTest.randomUniqueNotExistingFileStore(), true);
        IFileStore target = temp.getChild("target");
        long stat = target.fetchInfo().getLastModified();
        Assertions.assertEquals((long)0L, (long)stat);
        this.createDir(target, true);
        stat = target.fetchInfo().getLastModified();
        Assertions.assertTrue((0L != stat ? 1 : 0) != 0);
    }

    @Test
    public void testDelete() throws Throwable {
        IFileStore tempC = this.createDir(FileStoreTest.randomUniqueNotExistingFileStore(), true);
        System.out.println("Threadcount=" + Runtime.getRuntime().availableProcessors());
        IFileStore singleDirectory = tempC.getChild("tree");
        this.createDir(singleDirectory, true);
        ArrayList<IFileStore> treeItems = new ArrayList<IFileStore>();
        int i = 0;
        while (i < 2300) {
            treeItems.add(singleDirectory.getChild("file" + i));
            ++i;
        }
        LocalStoreTestUtil.createTree((IFileStore[])treeItems.toArray(IFileStore[]::new));
        Assertions.assertTrue((boolean)singleDirectory.fetchInfo().exists());
        long n0 = System.nanoTime();
        singleDirectory.delete(0, null);
        long n1 = System.nanoTime();
        System.out.println("delete singleDirectory took ms:" + (n1 - n0) / 1000000L);
        Assertions.assertFalse((boolean)singleDirectory.fetchInfo().exists());
        IFileStore binaryTree = tempC.getChild("tree");
        this.createDir(binaryTree, true);
        this.createChildDirs(binaryTree, 2, 10);
        Assertions.assertTrue((boolean)binaryTree.fetchInfo().exists());
        long n02 = System.nanoTime();
        binaryTree.delete(0, null);
        long n12 = System.nanoTime();
        System.out.println("delete binaryTree took ms:" + (n12 - n02) / 1000000L);
        Assertions.assertFalse((boolean)binaryTree.fetchInfo().exists());
    }

    private void createChildDirs(IFileStore parent, int count, int depth) throws CoreException {
        if (depth <= 0) {
            return;
        }
        int i = 0;
        while (i < count) {
            IFileStore child = parent.getChild("dir-" + depth + "-" + i);
            child.mkdir(0, null);
            this.createChildDirs(child, count, depth - 1);
            ++i;
        }
    }

    @Test
    public void testMove() throws Throwable {
        IFileStore tempC = this.createDir(FileStoreTest.randomUniqueNotExistingFileStore(), true);
        IFileStore target = tempC.getChild("target");
        String content = "just a content.....tnetnoc a tsuj";
        this.createFile(target, content);
        Assertions.assertTrue((boolean)target.fetchInfo().exists());
        IFileStore tree = tempC.getChild("tree");
        this.createDir(tree, true);
        LocalStoreTestUtil.createTree(LocalStoreTestUtil.getTree(tree));
        IFileStore destination = tempC.getChild("destination");
        target.move(destination, 0, null);
        Assertions.assertTrue((!destination.fetchInfo().isDirectory() ? 1 : 0) != 0);
        Assertions.assertTrue((!target.fetchInfo().exists() ? 1 : 0) != 0);
        destination.move(target, 0, null);
        Assertions.assertTrue((!target.fetchInfo().isDirectory() ? 1 : 0) != 0);
        Assertions.assertTrue((!destination.fetchInfo().exists() ? 1 : 0) != 0);
        String anotherContent = "another content";
        this.createFile(destination, anotherContent);
        IFileStore immutableFileDestination = destination;
        Assertions.assertThrows(CoreException.class, () -> target.move(immutableFileDestination, 0, null));
        Assertions.assertTrue((!target.fetchInfo().isDirectory() ? 1 : 0) != 0);
        destination.delete(0, null);
        Assertions.assertTrue((!destination.fetchInfo().exists() ? 1 : 0) != 0);
        this.createDir(destination, true);
        IFileStore immutableFolderDestination = destination;
        Assertions.assertThrows(CoreException.class, () -> target.move(immutableFolderDestination, 0, null));
        Assertions.assertTrue((!target.fetchInfo().isDirectory() ? 1 : 0) != 0);
        destination.delete(0, null);
        Assertions.assertTrue((!destination.fetchInfo().exists() ? 1 : 0) != 0);
        destination = tempC.getChild("destination");
        tree.move(destination, 0, null);
        Assertions.assertTrue((boolean)this.verifyTree(LocalStoreTestUtil.getTree(destination)));
        Assertions.assertTrue((!tree.fetchInfo().exists() ? 1 : 0) != 0);
        destination.move(tree, 0, null);
        Assertions.assertTrue((boolean)this.verifyTree(LocalStoreTestUtil.getTree(tree)));
        Assertions.assertTrue((!destination.fetchInfo().exists() ? 1 : 0) != 0);
    }

    @Test
    public void testMoveAcrossVolumes() throws Throwable {
        IFileStore[] tempDirectories = this.getFileStoresOnTwoVolumes();
        Assume.assumeFalse((String)"only executable if at least two volumes are present", (tempDirectories == null || tempDirectories.length < 2 || tempDirectories[0] == null || tempDirectories[1] == null ? 1 : 0) != 0);
        IFileStore tempSrc = tempDirectories[0];
        IFileStore tempDest = tempDirectories[1];
        String subfolderName = "target_" + System.currentTimeMillis();
        IFileStore target = tempSrc.getChild(subfolderName);
        String content = "just a content.....tnetnoc a tsuj";
        this.createFile(target, content);
        Assertions.assertTrue((boolean)target.fetchInfo().exists());
        IFileStore tree = tempSrc.getChild("tree");
        this.createDir(tree, true);
        LocalStoreTestUtil.createTree(LocalStoreTestUtil.getTree(tree));
        IFileStore destination = tempDest.getChild(subfolderName);
        target.move(destination, 0, null);
        Assertions.assertTrue((!destination.fetchInfo().isDirectory() ? 1 : 0) != 0);
        Assertions.assertTrue((!target.fetchInfo().exists() ? 1 : 0) != 0);
        destination.move(target, 0, null);
        Assertions.assertTrue((!target.fetchInfo().isDirectory() ? 1 : 0) != 0);
        Assertions.assertTrue((!destination.fetchInfo().exists() ? 1 : 0) != 0);
        destination = tempDest.getChild(subfolderName);
        tree.move(destination, 0, null);
        Assertions.assertTrue((boolean)this.verifyTree(LocalStoreTestUtil.getTree(destination)));
        Assertions.assertTrue((!tree.fetchInfo().exists() ? 1 : 0) != 0);
        destination.move(tree, 0, null);
        Assertions.assertTrue((boolean)this.verifyTree(LocalStoreTestUtil.getTree(tree)));
        Assertions.assertTrue((!destination.fetchInfo().exists() ? 1 : 0) != 0);
    }

    @Test
    public void testMoveDirectoryParentMissing() throws Throwable {
        IFileStore parent = FileStoreTest.randomUniqueNotExistingFileStore();
        IFileStore child = parent.getChild("child");
        IFileStore existing = FileStoreTest.randomUniqueNotExistingFileStore();
        this.createFile(existing, ResourceTestUtil.createRandomString());
        Assertions.assertThrows(CoreException.class, () -> existing.move(child, 0, ResourceTestUtil.createTestMonitor()));
        Assertions.assertTrue((!child.fetchInfo().exists() ? 1 : 0) != 0);
    }

    @Test
    public void testPutInfo() throws IOException {
        IFileStore nonExisting = FileStoreTest.randomUniqueNotExistingFileStore();
        IFileInfo info = nonExisting.fetchInfo();
        info.setLastModified(System.currentTimeMillis());
        Assertions.assertThrows(CoreException.class, () -> nonExisting.putInfo(info, 2048, ResourceTestUtil.createTestMonitor()));
        IFileInfo refetchedInfo = nonExisting.fetchInfo();
        info.setAttribute(2, false);
        Assertions.assertThrows(CoreException.class, () -> nonExisting.putInfo(refetchedInfo, 1024, ResourceTestUtil.createTestMonitor()));
    }

    @Test
    public void testReadOnly() throws Exception {
        this.testAttribute(2);
    }

    @Test
    public void testPermissionsEnabled() {
        String os = Platform.getOS();
        if ("linux".equals(os) || "macosx".equals(os)) {
            Assertions.assertTrue((boolean)ResourceTestUtil.isAttributeSupported(0x400000));
            Assertions.assertTrue((boolean)ResourceTestUtil.isAttributeSupported(0x800000));
            Assertions.assertTrue((boolean)ResourceTestUtil.isAttributeSupported(0x1000000));
            Assertions.assertTrue((boolean)ResourceTestUtil.isAttributeSupported(0x2000000));
            Assertions.assertTrue((boolean)ResourceTestUtil.isAttributeSupported(0x4000000));
            Assertions.assertTrue((boolean)ResourceTestUtil.isAttributeSupported(0x8000000));
            Assertions.assertTrue((boolean)ResourceTestUtil.isAttributeSupported(0x10000000));
            Assertions.assertTrue((boolean)ResourceTestUtil.isAttributeSupported(0x20000000));
            Assertions.assertTrue((boolean)ResourceTestUtil.isAttributeSupported(0x40000000));
        } else {
            Assertions.assertFalse((boolean)ResourceTestUtil.isAttributeSupported(0x400000));
            Assertions.assertFalse((boolean)ResourceTestUtil.isAttributeSupported(0x800000));
            Assertions.assertFalse((boolean)ResourceTestUtil.isAttributeSupported(0x1000000));
            Assertions.assertFalse((boolean)ResourceTestUtil.isAttributeSupported(0x2000000));
            Assertions.assertFalse((boolean)ResourceTestUtil.isAttributeSupported(0x4000000));
            Assertions.assertFalse((boolean)ResourceTestUtil.isAttributeSupported(0x8000000));
            Assertions.assertFalse((boolean)ResourceTestUtil.isAttributeSupported(0x10000000));
            Assertions.assertFalse((boolean)ResourceTestUtil.isAttributeSupported(0x20000000));
            Assertions.assertFalse((boolean)ResourceTestUtil.isAttributeSupported(0x40000000));
        }
    }

    @Test
    public void testPermissions() throws Exception {
        this.testAttribute(0x400000);
        this.testAttribute(0x800000);
        this.testAttribute(0x1000000);
        this.testAttribute(0x2000000);
        this.testAttribute(0x4000000);
        this.testAttribute(0x8000000);
        this.testAttribute(0x10000000);
        this.testAttribute(0x20000000);
        this.testAttribute(0x40000000);
    }

    private void testAttribute(int attribute) throws Exception {
        Assume.assumeTrue((String)("only relevant for platforms supporting attribute: " + attribute), (boolean)ResourceTestUtil.isAttributeSupported(attribute));
        IFileStore targetFolder = this.createDir(FileStoreTest.randomUniqueNotExistingFileStore(), true);
        IFileStore targetFile = targetFolder.getChild("targetFile");
        ResourceTestUtil.createInFileSystem(targetFile);
        boolean init = targetFile.fetchInfo().getAttribute(attribute);
        this.setAttribute(targetFile, attribute, !init);
        Assertions.assertTrue((boolean)(targetFile.fetchInfo().getAttribute(attribute) ^ init));
        this.setAttribute(targetFile, attribute, init);
        Assertions.assertTrue((targetFile.fetchInfo().getAttribute(attribute) == init ? 1 : 0) != 0);
        init = targetFolder.fetchInfo().getAttribute(attribute);
        this.setAttribute(targetFolder, attribute, !init);
        Assertions.assertTrue((boolean)(targetFolder.fetchInfo().getAttribute(attribute) ^ init));
        this.setAttribute(targetFolder, attribute, init);
        Assertions.assertTrue((targetFolder.fetchInfo().getAttribute(attribute) == init ? 1 : 0) != 0);
    }

    private void setAttribute(IFileStore target, int attribute, boolean value) throws CoreException {
        Assertions.assertTrue((boolean)ResourceTestUtil.isAttributeSupported(attribute));
        IFileInfo fileInfo = target.fetchInfo();
        fileInfo.setAttribute(attribute, value);
        target.putInfo(fileInfo, 1024, null);
    }

    @Test
    public void testGetFileStore() throws Exception {
        Path root = Files.createDirectories(FileStoreTest.randomUniqueNotExistingPath(), new FileAttribute[0]);
        File file = root.resolve("test.txt").toFile();
        file.createNewFile();
        Assertions.assertTrue((boolean)file.exists());
        IFileStore tempStore = this.createDir(ResourceTestUtil.getFileStore(root.resolve("temp")), true);
        this.createDir(ResourceTestUtil.getFileStore(root.resolve("temp/temp2")), true);
        file = root.resolve("temp/temp2/test.txt").toFile();
        file.createNewFile();
        Assertions.assertTrue((boolean)file.exists());
        IPath relativePath = IPath.fromOSString((String)"../test.txt");
        IFileStore relativeStore = tempStore.getFileStore(relativePath);
        Assertions.assertNotNull((Object)relativeStore);
        IFileInfo info = relativeStore.fetchInfo();
        Assertions.assertNotNull((Object)info);
        Assertions.assertTrue((boolean)info.exists());
        relativePath = IPath.fromOSString((String)".././test.txt");
        relativeStore = tempStore.getFileStore(relativePath);
        Assertions.assertNotNull((Object)relativeStore);
        info = relativeStore.fetchInfo();
        Assertions.assertNotNull((Object)info);
        Assertions.assertTrue((boolean)info.exists());
        relativePath = IPath.fromOSString((String)"temp2/test.txt");
        relativeStore = tempStore.getFileStore(relativePath);
        Assertions.assertNotNull((Object)relativeStore);
        info = relativeStore.fetchInfo();
        Assertions.assertNotNull((Object)info);
        Assertions.assertTrue((boolean)info.exists());
    }

    @Test
    public void testSortOrder() {
        IFileSystem nullfs = NullFileSystem.getInstance();
        if (nullfs == null) {
            nullfs = new NullFileSystem();
            ((FileSystem)nullfs).initialize("null");
        }
        IFileStore nabc = nullfs.getStore(IPath.fromOSString((String)"/a/b/c"));
        IFileStore nabd = nullfs.getStore(IPath.fromOSString((String)"/a/b/d"));
        Assertions.assertEquals((int)-1, (int)nabc.compareTo(nabd));
        Assertions.assertEquals((int)0, (int)nabc.compareTo(nabc));
        Assertions.assertEquals((int)1, (int)nabd.compareTo(nabc));
        IFileSystem lfs = LocalFileSystem.getInstance();
        IFileStore labc = lfs.getStore(IPath.fromOSString((String)"/a/b/c"));
        IFileStore labd = lfs.getStore(IPath.fromOSString((String)"/a/b/d"));
        Assertions.assertEquals((int)-1, (int)labc.compareTo(labd));
        Assertions.assertEquals((int)0, (int)labc.compareTo(labc));
        Assertions.assertEquals((int)1, (int)labd.compareTo(labc));
        int schemeCompare = nullfs.getScheme().compareTo(lfs.getScheme());
        Assertions.assertEquals((int)schemeCompare, (int)nabd.compareTo(labc));
        Assertions.assertEquals((int)schemeCompare, (int)nabc.compareTo(labd));
        Assertions.assertEquals((int)(-schemeCompare), (int)labd.compareTo(nabc));
        Assertions.assertEquals((int)(-schemeCompare), (int)labc.compareTo(nabd));
        Assertions.assertEquals((int)1, (int)labc.compareTo(null));
        Assertions.assertEquals((int)1, (int)nabc.compareTo(null));
    }

    @Test
    public void testSortOrderPaths() {
        IFileSystem lfs = LocalFileSystem.getInstance();
        boolean isWindows = File.separatorChar == '\\';
        String prefix = isWindows ? "/D:" : "";
        List<String> paths = List.of("/a", "/a/", "/a/b", "/a/./c", "/a/e/../c", "/a/d", "/aa", "/b").stream().map(s -> prefix + s).toList();
        List<String> pathsTrimmed = paths.stream().map(s -> s.replaceAll("/$", "").replaceAll("/[^/]+/\\.\\./", "/").replaceAll("/\\./", "/")).toList();
        paths = new ArrayList<String>(paths);
        Collections.shuffle(paths);
        Stream<IFileStore> pathStores = paths.stream().map(IPath::fromOSString).map(arg_0 -> ((IFileSystem)lfs).getStore(arg_0));
        List<String> sortedPathStores = pathStores.sorted(IFileStore::compareTo).map(IFileStore::toURI).map(URI::getPath).toList();
        Assertions.assertEquals(pathsTrimmed, sortedPathStores);
        Stream<IFileStore> localFileStores = paths.stream().map(File::new).map(LocalFile::new);
        List<String> sortedLocalFileStores = localFileStores.sorted(IFileStore::compareTo).map(IFileStore::toURI).map(URI::getPath).toList();
        Assertions.assertEquals(pathsTrimmed, sortedLocalFileStores);
    }

    private boolean verifyNode(IFileStore node) {
        char type = node.getName().charAt(0);
        return type == 'd' == node.fetchInfo().isDirectory();
    }

    private boolean verifyTree(IFileStore[] tree) {
        IFileStore[] iFileStoreArray = tree;
        int n = tree.length;
        int n2 = 0;
        while (n2 < n) {
            IFileStore t = iFileStoreArray[n2];
            if (!this.verifyNode(t)) {
                return false;
            }
            ++n2;
        }
        return true;
    }
}

