/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.core.tests.internal.builders;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import org.assertj.core.api.AbstractIntegerAssert;
import org.assertj.core.api.AbstractIterableAssert;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.IterableAssert;
import org.assertj.core.api.ListAssert;
import org.eclipse.core.resources.IBuildConfiguration;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceDescription;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobGroup;
import org.eclipse.core.tests.harness.TestBarrier2;
import org.eclipse.core.tests.internal.builders.TimerBuilder;
import org.eclipse.core.tests.resources.ResourceTestUtil;
import org.eclipse.core.tests.resources.TestUtil;
import org.eclipse.core.tests.resources.util.WorkspaceResetExtension;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;

@ExtendWith(value={WorkspaceResetExtension.class})
public class ParallelBuildChainTest {
    private static final int TIMEOUT_IN_MILLIS = 30000;

    @BeforeEach
    public void setUp() throws Exception {
        ResourceTestUtil.setAutoBuilding(false);
    }

    @AfterEach
    public void tearDown() throws Exception {
        ResourceTestUtil.waitForBuild();
        ResourcesPlugin.getWorkspace().getRoot().delete(true, true, ResourceTestUtil.createTestMonitor());
        TimerBuilder.abortCurrentBuilds();
    }

    private void setWorkspaceMaxNumberOfConcurrentBuilds(int maximumNumberOfConcurrentBuilds) throws CoreException {
        IWorkspaceDescription description = ResourcesPlugin.getWorkspace().getDescription();
        description.setMaxConcurrentBuilds(maximumNumberOfConcurrentBuilds);
        ResourcesPlugin.getWorkspace().setDescription(description);
    }

    @Test
    public void testIndividualProjectBuilds_NoConflictRule() throws Exception {
        int numberOfParallelBuilds = 3;
        this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.IMMEDIATE, TimerBuilder.RuleType.NO_CONFLICT);
        List<IProject> longRunningProjects = this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.LONG_RUNNING, TimerBuilder.RuleType.NO_CONFLICT);
        this.executeIndividualFullProjectBuilds(numberOfParallelBuilds, () -> {
            this.assertBuildsToStart(this.getAllProjects());
            this.assertMinimumNumberOfSimultaneousBuilds(longRunningProjects.size());
        });
    }

    @Test
    public void testIndividualProjectBuilds_ProjectRelaxedRule() throws Exception {
        int numberOfParallelBuilds = 3;
        this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.IMMEDIATE, TimerBuilder.RuleType.CURRENT_PROJECT_RELAXED);
        List<IProject> longRunningProjects = this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.LONG_RUNNING, TimerBuilder.RuleType.CURRENT_PROJECT_RELAXED);
        this.executeIndividualFullProjectBuilds(numberOfParallelBuilds, () -> {
            this.assertBuildsToStart(this.getAllProjects());
            this.assertMinimumNumberOfSimultaneousBuilds(longRunningProjects.size());
        });
    }

    @Test
    public void testIndividualProjectBuilds_WithManyProjects_ProjectRelaxedRule() throws Exception {
        int numberOfParallelBuilds = 15;
        List<IProject> longRunningProjects = this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.LONG_RUNNING, TimerBuilder.RuleType.CURRENT_PROJECT_RELAXED);
        this.executeIndividualFullProjectBuilds(numberOfParallelBuilds, () -> {
            this.assertBuildsToStart(this.getAllProjects());
            this.assertMinimumNumberOfSimultaneousBuilds(longRunningProjects.size());
        });
    }

    @Test
    public void testWorkspaceBuild_NoConflictRule() throws Exception {
        int numberOfParallelBuilds = 3;
        this.setWorkspaceMaxNumberOfConcurrentBuilds(numberOfParallelBuilds);
        this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.IMMEDIATE, TimerBuilder.RuleType.NO_CONFLICT);
        List<IProject> longRunningBuildProjects = this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.LONG_RUNNING, TimerBuilder.RuleType.NO_CONFLICT);
        this.executeIncrementalWorkspaceBuild(() -> {
            this.assertBuildsToStart(longRunningBuildProjects);
            this.assertMinimumNumberOfSimultaneousBuilds(longRunningBuildProjects.size());
            this.assertMaximumNumberOfConcurrentWorkspaceBuilds();
        });
    }

    @Test
    public void testWorkspaceBuild_NoConflictRule_WithBuildConfigurations() throws Exception {
        int numberOfParallelBuilds = 3;
        this.setWorkspaceMaxNumberOfConcurrentBuilds(numberOfParallelBuilds);
        this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.IMMEDIATE, TimerBuilder.RuleType.NO_CONFLICT);
        List<IProject> longRunningBuildProjects = this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.LONG_RUNNING, TimerBuilder.RuleType.NO_CONFLICT);
        IBuildConfiguration[] buildConfigurations = ParallelBuildChainTest.getBuildConfigurations(this.getAllProjects());
        this.executeIncrementalWorkspaceBuild(buildConfigurations, () -> {
            this.assertBuildsToStart(longRunningBuildProjects);
            this.assertMinimumNumberOfSimultaneousBuilds(longRunningBuildProjects.size());
            this.assertMaximumNumberOfConcurrentWorkspaceBuilds();
        });
    }

    @Test
    public void testWorkspaceBuild_ProjectRule() throws Exception {
        int numberOfParallelBuilds = 3;
        this.setWorkspaceMaxNumberOfConcurrentBuilds(numberOfParallelBuilds);
        this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.IMMEDIATE, TimerBuilder.RuleType.CURRENT_PROJECT);
        List<IProject> longRunningProjects = this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.LONG_RUNNING, TimerBuilder.RuleType.CURRENT_PROJECT);
        this.executeIncrementalWorkspaceBuild(() -> {
            this.assertBuildsToStart(longRunningProjects);
            this.assertMinimumNumberOfSimultaneousBuilds(longRunningProjects.size());
            this.assertMaximumNumberOfConcurrentWorkspaceBuilds();
        });
    }

    @Test
    public void testWorkspaceBuild_ProjectRule_WithBuildConfigurations() throws Exception {
        int numberOfParallelBuilds = 3;
        this.setWorkspaceMaxNumberOfConcurrentBuilds(numberOfParallelBuilds);
        this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.IMMEDIATE, TimerBuilder.RuleType.CURRENT_PROJECT);
        List<IProject> longRunningBuildProjects = this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.LONG_RUNNING, TimerBuilder.RuleType.CURRENT_PROJECT);
        IBuildConfiguration[] buildConfigurations = ParallelBuildChainTest.getBuildConfigurations(this.getAllProjects());
        this.executeIncrementalWorkspaceBuild(buildConfigurations, () -> {
            this.assertBuildsToStart(longRunningBuildProjects);
            this.assertMinimumNumberOfSimultaneousBuilds(longRunningBuildProjects.size());
            this.assertMaximumNumberOfConcurrentWorkspaceBuilds();
        });
    }

    @Test
    public void testWorkspaceBuild_ConflictingRule() throws Exception {
        int millisToWaitForUnexpectedParallelBuild = 3000;
        int numberOfParallelBuilds = 3;
        this.setWorkspaceMaxNumberOfConcurrentBuilds(numberOfParallelBuilds);
        List<IProject> longRunningProjects = this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.LONG_RUNNING, TimerBuilder.RuleType.WORKSPACE_ROOT);
        this.executeIncrementalWorkspaceBuild(() -> {
            TestUtil.waitForCondition(() -> TimerBuilder.getStartedProjectBuilds().size() > 1, millisToWaitForUnexpectedParallelBuild);
            ((ListAssert)Assertions.assertThat((List)longRunningProjects).withFailMessage("all build jobs have started in time although infinitely running builds with conflicting rules exist", new Object[0])).anySatisfy(longRunningProject -> {
                AbstractIterableAssert abstractIterableAssert = Assertions.assertThat(TimerBuilder.getStartedProjectBuilds()).doesNotContain((Object[])new IProject[]{longRunningProject});
            });
            this.assertMaximumNumberOfSimultaneousBuilds(1);
        });
    }

    @Test
    public void testWorkspaceBuild_DependentProjects() throws Exception {
        int numberOfParallelBuilds = 3;
        this.setWorkspaceMaxNumberOfConcurrentBuilds(numberOfParallelBuilds);
        this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.IMMEDIATE, TimerBuilder.RuleType.NO_CONFLICT);
        List<IProject> shortRunningProjects = this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.SHORT_RUNNING, TimerBuilder.RuleType.NO_CONFLICT);
        List<IProject> projectsToBuild = this.getAllProjects();
        this.makeProjectsDependOnEachOther(projectsToBuild);
        int minimumExecutionTimeInMillis = shortRunningProjects.size() * BuildDurationType.SHORT_RUNNING.getDurationInMillis();
        ExpectedExecutionTime expectedExecutionTime = ExpectedExecutionTime.captureFromCurrentTime(minimumExecutionTimeInMillis);
        this.executeIncrementalWorkspaceBuild(() -> {
            this.assertBuildsToFinish(projectsToBuild);
            expectedExecutionTime.assertMinimumExecutionTimeReached();
            this.assertMaximumNumberOfSimultaneousBuilds(1);
            this.assertSequentialBuildEventsForProjects(projectsToBuild);
        });
    }

    @Test
    public void testWorkspaceBuild_DependentProjects_ProjectSubset() throws Exception {
        int numberOfParallelBuilds = 3;
        this.setWorkspaceMaxNumberOfConcurrentBuilds(numberOfParallelBuilds);
        List<IProject> immediateBuiltProjects = this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.IMMEDIATE, TimerBuilder.RuleType.NO_CONFLICT);
        List<IProject> shortRunningProjects = this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.SHORT_RUNNING, TimerBuilder.RuleType.NO_CONFLICT);
        List<IProject> projectsToBuild = List.of(immediateBuiltProjects.get(0), immediateBuiltProjects.get(immediateBuiltProjects.size() - 1), shortRunningProjects.get(0), shortRunningProjects.get(shortRunningProjects.size() - 1));
        this.makeProjectsDependOnEachOther(projectsToBuild);
        IBuildConfiguration[] selectedBuildConfigurations = ParallelBuildChainTest.getBuildConfigurations(projectsToBuild);
        int minimumExecutionTimeInMillis = 2 * BuildDurationType.SHORT_RUNNING.getDurationInMillis();
        ExpectedExecutionTime expectedExecutionTime = ExpectedExecutionTime.captureFromCurrentTime(minimumExecutionTimeInMillis);
        this.executeIncrementalWorkspaceBuild(selectedBuildConfigurations, () -> {
            this.assertBuildsToFinish(projectsToBuild);
            expectedExecutionTime.assertMinimumExecutionTimeReached();
            this.assertMaximumNumberOfSimultaneousBuilds(1);
            this.assertSequentialBuildEventsForProjects(projectsToBuild);
        });
    }

    @Test
    public void testWorkspaceBuild_DependentProjectBuildConfigurations() throws Exception {
        int numberOfParallelBuilds = 3;
        this.setWorkspaceMaxNumberOfConcurrentBuilds(numberOfParallelBuilds);
        this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.IMMEDIATE, TimerBuilder.RuleType.NO_CONFLICT);
        List<IProject> shortRunningProjects = this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.SHORT_RUNNING, TimerBuilder.RuleType.NO_CONFLICT);
        List<IProject> projectsToBuild = this.getAllProjects();
        this.makeProjectBuildConfigurationsDependOnEachOther(projectsToBuild);
        int minimumExecutionTimeInMillis = shortRunningProjects.size() * BuildDurationType.SHORT_RUNNING.getDurationInMillis();
        ExpectedExecutionTime expectedExecutionTime = ExpectedExecutionTime.captureFromCurrentTime(minimumExecutionTimeInMillis);
        this.executeIncrementalWorkspaceBuild(() -> {
            this.assertBuildsToFinish(projectsToBuild);
            expectedExecutionTime.assertMinimumExecutionTimeReached();
            this.assertMaximumNumberOfSimultaneousBuilds(1);
            this.assertSequentialBuildEventsForProjects(projectsToBuild);
        });
    }

    @Test
    public void testWorkspaceBuild_DependentProjectBuildConfigurations_ProjectSubset() throws Exception {
        int numberOfParallelBuilds = 3;
        this.setWorkspaceMaxNumberOfConcurrentBuilds(numberOfParallelBuilds);
        List<IProject> immediateBuiltProjects = this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.IMMEDIATE, TimerBuilder.RuleType.NO_CONFLICT);
        List<IProject> shortRunningProjects = this.createMultipleTestProjects(numberOfParallelBuilds, BuildDurationType.SHORT_RUNNING, TimerBuilder.RuleType.NO_CONFLICT);
        List<IProject> projectsToBuild = List.of(immediateBuiltProjects.get(0), immediateBuiltProjects.get(immediateBuiltProjects.size() - 1), shortRunningProjects.get(0), shortRunningProjects.get(shortRunningProjects.size() - 1));
        this.makeProjectBuildConfigurationsDependOnEachOther(this.getAllProjects());
        IBuildConfiguration[] selectedBuildConfigurations = ParallelBuildChainTest.getBuildConfigurations(projectsToBuild);
        int minimumExecutionTimeInMillis = 2 * BuildDurationType.SHORT_RUNNING.getDurationInMillis();
        ExpectedExecutionTime expectedExecutionTime = ExpectedExecutionTime.captureFromCurrentTime(minimumExecutionTimeInMillis);
        this.executeIncrementalWorkspaceBuild(selectedBuildConfigurations, () -> {
            this.assertBuildsToFinish(projectsToBuild);
            expectedExecutionTime.assertMinimumExecutionTimeReached();
            this.assertMaximumNumberOfSimultaneousBuilds(1);
            this.assertSequentialBuildEventsForProjects(projectsToBuild);
        });
    }

    private List<IProject> getAllProjects() {
        return Arrays.asList(ResourcesPlugin.getWorkspace().getRoot().getProjects());
    }

    private static IBuildConfiguration[] getBuildConfigurations(List<IProject> projects) throws CoreException {
        IBuildConfiguration[] buildConfigurations = new IBuildConfiguration[projects.size()];
        int projectNumber = 0;
        while (projectNumber < projects.size()) {
            buildConfigurations[projectNumber] = projects.get(projectNumber).getActiveBuildConfig();
            ++projectNumber;
        }
        return buildConfigurations;
    }

    private List<IProject> createMultipleTestProjects(int numberOfProjects, BuildDurationType buildDurationType, TimerBuilder.RuleType ruleType) throws CoreException {
        ArrayList<IProject> result = new ArrayList<IProject>();
        int projectNumber = 0;
        while (projectNumber < numberOfProjects) {
            result.add(this.createTestProject(buildDurationType, ruleType));
            ++projectNumber;
        }
        return result;
    }

    private IProject createTestProject(BuildDurationType buildDurationType, TimerBuilder.RuleType ruleType) throws CoreException {
        String projectName = this.createUniqueProjectName(buildDurationType.toString());
        IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
        IProject project = root.getProject(projectName);
        ResourceTestUtil.createInWorkspace((IResource)project);
        this.configureTimerBuilder(project, buildDurationType.getDurationInMillis(), ruleType);
        return project;
    }

    private String createUniqueProjectName(String projectPrefix) {
        int suffix = 0;
        IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
        while (root.getProject(projectPrefix + "Project" + suffix).exists()) {
            ++suffix;
        }
        return projectPrefix + "Project" + suffix;
    }

    private void configureTimerBuilder(IProject project, int duration, TimerBuilder.RuleType ruleType) throws CoreException {
        ResourceTestUtil.updateProjectDescription(project).addingCommand("org.eclipse.core.tests.resources.timerbuilder").withAdditionalBuildArgument("duration", Integer.toString(duration)).withAdditionalBuildArgument("ruleType", ruleType.toString()).apply();
    }

    private void makeProjectsDependOnEachOther(List<IProject> projects) throws CoreException {
        int projectNumber = 1;
        while (projectNumber < projects.size()) {
            IProject project = projects.get(projectNumber);
            IProjectDescription desc = project.getDescription();
            desc.setReferencedProjects(new IProject[]{projects.get(projectNumber - 1)});
            project.setDescription(desc, ResourceTestUtil.createTestMonitor());
            ++projectNumber;
        }
    }

    private void makeProjectBuildConfigurationsDependOnEachOther(List<IProject> projects) throws CoreException {
        int projectNumber = 1;
        while (projectNumber < projects.size()) {
            IProject project = projects.get(projectNumber);
            IProjectDescription description = project.getDescription();
            description.setBuildConfigReferences(project.getActiveBuildConfig().getName(), new IBuildConfiguration[]{projects.get(projectNumber - 1).getActiveBuildConfig()});
            project.setDescription(description, ResourceTestUtil.createTestMonitor());
            ++projectNumber;
        }
    }

    private void executeIncrementalWorkspaceBuild(Runnable executeWhileRunningBuild) throws Exception {
        this.executeIncrementalWorkspaceBuild(null, executeWhileRunningBuild);
    }

    private void executeIncrementalWorkspaceBuild(final IBuildConfiguration[] buildConfigurations, Runnable executeWhileRunningBuild) throws Exception {
        int expectedNumberOfBuilds = buildConfigurations != null ? buildConfigurations.length : this.getAllProjects().size();
        TimerBuilder.setExpectedNumberOfBuilds(expectedNumberOfBuilds);
        final TestBarrier2 waitForRunningJobBarrier = new TestBarrier2();
        Job job = new Job("Workspace Build"){

            protected IStatus run(IProgressMonitor monitor) {
                try {
                    waitForRunningJobBarrier.setStatus(3);
                    if (buildConfigurations != null) {
                        ResourcesPlugin.getWorkspace().build(buildConfigurations, 10, false, ResourceTestUtil.createTestMonitor());
                    } else {
                        ResourcesPlugin.getWorkspace().build(10, ResourceTestUtil.createTestMonitor());
                    }
                    return Status.OK_STATUS;
                }
                catch (CoreException e) {
                    return new Status(4, "org.eclipse.core.tests.resources", e.getMessage(), (Throwable)e);
                }
            }
        };
        job.schedule();
        waitForRunningJobBarrier.waitForStatus(3);
        try {
            executeWhileRunningBuild.run();
        }
        finally {
            TimerBuilder.abortCurrentBuilds();
            job.cancel();
            boolean joinSuccessful = job.join(30000L, ResourceTestUtil.createTestMonitor());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)joinSuccessful, (String)"timeout occurred when waiting for job that runs the build to finish");
        }
    }

    private void executeIndividualFullProjectBuilds(int numberOfParallelBuilds, Runnable executeWhileRunningBuild) throws Exception {
        List<IProject> projects = this.getAllProjects();
        TimerBuilder.setExpectedNumberOfBuilds(projects.size());
        JobGroup jobGroup = new JobGroup("Build Group", numberOfParallelBuilds, projects.size());
        final HashMap<IProject, TestBarrier2> waitForRunningJobBarriers = new HashMap<IProject, TestBarrier2>();
        for (final IProject project : projects) {
            waitForRunningJobBarriers.put(project, new TestBarrier2());
            Job job = new Job("Building " + project.getName()){

                protected IStatus run(IProgressMonitor monitor) {
                    try {
                        ((TestBarrier2)waitForRunningJobBarriers.get(project)).setStatus(3);
                        project.build(6, ResourceTestUtil.createTestMonitor());
                        return Status.OK_STATUS;
                    }
                    catch (CoreException e) {
                        return new Status(4, "org.eclipse.core.tests.resources", e.getMessage(), (Throwable)e);
                    }
                }
            };
            job.setJobGroup(jobGroup);
            job.schedule();
        }
        for (TestBarrier2 barrier : waitForRunningJobBarriers.values()) {
            barrier.waitForStatus(3);
        }
        try {
            executeWhileRunningBuild.run();
        }
        finally {
            TimerBuilder.abortCurrentBuilds();
            jobGroup.cancel();
            boolean joinSuccessful = jobGroup.join(30000L, ResourceTestUtil.createTestMonitor());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)joinSuccessful, (String)"timeout occurred when waiting for job group that runs the builds to finish");
        }
    }

    private void assertMinimumNumberOfSimultaneousBuilds(int minimumNumberOfSimulaneousBuilds) {
        ((AbstractIntegerAssert)Assertions.assertThat((int)TimerBuilder.getMaximumNumberOfSimultaneousBuilds()).as("check maximum number of parallel builds exceeds minimum expected number", new Object[0])).isGreaterThanOrEqualTo(minimumNumberOfSimulaneousBuilds);
    }

    private void assertMaximumNumberOfSimultaneousBuilds(int maximumNumberOfSimulaneousBuilds) {
        ((AbstractIntegerAssert)Assertions.assertThat((int)TimerBuilder.getMaximumNumberOfSimultaneousBuilds()).as("check maximum number of parallel builds does not exceed maximum expected number", new Object[0])).isLessThanOrEqualTo(maximumNumberOfSimulaneousBuilds);
    }

    private void assertMaximumNumberOfConcurrentWorkspaceBuilds() {
        ((AbstractIntegerAssert)Assertions.assertThat((int)TimerBuilder.getMaximumNumberOfSimultaneousBuilds()).as("check maximum number of parallel builds does not exceed workspace limit", new Object[0])).isLessThanOrEqualTo(ResourcesPlugin.getWorkspace().getDescription().getMaxConcurrentBuilds());
    }

    private void assertBuildsToStart(List<IProject> projects) {
        TestUtil.waitForCondition(() -> TimerBuilder.getStartedProjectBuilds().containsAll(projects), 30000);
        ((ListAssert)Assertions.assertThat(TimerBuilder.getStartedProjectBuilds()).as("check all build jobs have started in time", new Object[0])).containsAll(projects);
    }

    private void assertBuildsToFinish(List<IProject> projects) {
        TestUtil.waitForCondition(() -> TimerBuilder.getFinishedProjectBuilds().containsAll(projects), 30000);
        ((ListAssert)Assertions.assertThat(TimerBuilder.getFinishedProjectBuilds()).as("check all build jobs have finished in time", new Object[0])).containsAll(projects);
    }

    private void assertSequentialBuildEventsForProjects(Iterable<IProject> projects) {
        ((IterableAssert)Assertions.assertThat(TimerBuilder.getBuildEvents()).as("order of build events", new Object[0])).isEqualTo(this.getExpectedSequentialBuildEvents(projects));
    }

    private Iterable<Object> getExpectedSequentialBuildEvents(Iterable<IProject> projects) {
        ArrayList<Object> res = new ArrayList<Object>();
        for (IProject project : projects) {
            res.add(TimerBuilder.createStartEvent(project));
            res.add(TimerBuilder.createCompleteEvent(project));
        }
        return res;
    }

    private static enum BuildDurationType {
        IMMEDIATE,
        SHORT_RUNNING,
        LONG_RUNNING;


        public int getDurationInMillis() {
            switch (this) {
                case LONG_RUNNING: {
                    return 40000;
                }
                case SHORT_RUNNING: {
                    return 300;
                }
            }
            return 0;
        }

        public String toString() {
            switch (this) {
                case IMMEDIATE: {
                    return "immediateBuild";
                }
                case LONG_RUNNING: {
                    return "longRunningBuild";
                }
                case SHORT_RUNNING: {
                    return "shortRunningBuild";
                }
            }
            throw new UnsupportedOperationException();
        }
    }

    private static class ExpectedExecutionTime {
        final long startTimeInNs = System.nanoTime();
        final long minimumExecutionTimeInMillis;

        private ExpectedExecutionTime(int minimumExecutionTimeInMillis) {
            this.minimumExecutionTimeInMillis = minimumExecutionTimeInMillis;
        }

        private long getExecutionTimeInMillis() {
            return (int)((System.nanoTime() - this.startTimeInNs) / 1000000L);
        }

        void assertMinimumExecutionTimeReached() {
            ((AbstractLongAssert)Assertions.assertThat((long)this.getExecutionTimeInMillis()).as("check build was not faster than the expected execution time (in milliseconds)", new Object[0])).isGreaterThanOrEqualTo(this.minimumExecutionTimeInMillis);
        }

        static ExpectedExecutionTime captureFromCurrentTime(int minimumExecutionTimeInMillis) {
            return new ExpectedExecutionTime(minimumExecutionTimeInMillis);
        }
    }
}

