/*******************************************************************************
 * Copyright (c) 2000, 2023 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jface.tests.viewers;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.viewers.AbstractTreeViewer;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Item;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.tests.harness.util.DisplayHelper;
import org.junit.After;
import org.junit.Test;

public abstract class AbstractTreeViewerTest extends StructuredItemViewerTest {

	AbstractTreeViewer fTreeViewer;

	@Override
	protected void assertSelectionEquals(String message, TestElement expected) {
		IStructuredSelection selection = fViewer.getStructuredSelection();
		List<TestElement> expectedList = new ArrayList<>();
		expectedList.add(expected);
		assertEquals("selectionEquals - " + message, expectedList, selection.toList());
	}

	protected abstract int getItemCount(TestElement element); // was IElement

	@Test
	public void testBulkExpand() {
		// navigate
		TestElement first = fRootElement.getFirstChild();
		TestElement first2 = first.getFirstChild();
		TestElement last = fRootElement.getLastChild();

		// expand a few nodes
		fTreeViewer.expandToLevel(first, 2);
		fTreeViewer.expandToLevel(first2, 2);
		fTreeViewer.expandToLevel(last, 2);
		// get expand state
		Object[] list1 = fTreeViewer.getExpandedElements();
		// flush viewer
		setInput();
		processEvents();

		// restore old expand state
		fTreeViewer.collapseAll();
		fTreeViewer.expandToLevel(first, 2);
		fTreeViewer.expandToLevel(first2, 2);
		fTreeViewer.expandToLevel(last, 2);

		Object[] list2 = fTreeViewer.getExpandedElements();

		assertArrayEquals("old and new expand state are the same", list1, list2);
	}

	@Test
	public void testDeleteChildExpanded() {
		TestElement first = fRootElement.getFirstChild();
		TestElement first2 = first.getFirstChild();
		fTreeViewer.expandToLevel(first2, 0);

		assertNotNull("first child is visible", fViewer.testFindItem(first2));
		first.deleteChild(first2);
		assertNull("first child is not visible", fViewer.testFindItem(first2));
	}

	@Test
	public void testDeleteChildren() {
		TestElement first = fRootElement.getFirstChild();
		first.deleteChildren();
		assertEquals("no children", 0, getItemCount(first));
	}

	@Test
	public void testDeleteChildrenExpanded() {
		TestElement first = fRootElement.getFirstChild();
		TestElement first2 = first.getFirstChild();
		fTreeViewer.expandToLevel(first2, 0);
		assertNotNull("first child is visible", fViewer.testFindItem(first2));

		first.deleteChildren();
		assertEquals("no children", 0, getItemCount(first));
	}

	@Test
	public void testExpand() {
		TestElement first = fRootElement.getFirstChild();
		TestElement first2 = first.getFirstChild();
		assertNull("first child is not visible", fViewer.testFindItem(first2));
		fTreeViewer.expandToLevel(first2, 0);
		assertNotNull("first child is visible", fViewer.testFindItem(first2));
	}

	@Test
	public void testExpandElement() {
		TestElement first = fRootElement.getFirstChild();
		TestElement first2 = first.getFirstChild();
		TestElement first3 = first2.getFirstChild();
		fTreeViewer.expandToLevel(first3, 0);
		assertNotNull("first3 is visible", fViewer.testFindItem(first3));
		assertNotNull("first2 is visible", fViewer.testFindItem(first2));
	}

	@Test
	public void testExpandElementAgain() {
		TestElement first = fRootElement.getFirstChild();
		TestElement first2 = first.getFirstChild();
		TestElement first3 = first2.getFirstChild();
		fTreeViewer.expandToLevel(first3, 0);
		assertTrue("first is expanded", fTreeViewer.getExpandedState(first));
		assertTrue("first2 is expanded", fTreeViewer.getExpandedState(first2));
		assertNotNull("first3 is visible", fViewer.testFindItem(first3));

		fTreeViewer.setExpandedState(first, false);
		fTreeViewer.expandToLevel(first3, 0);
		assertTrue("first is expanded", fTreeViewer.getExpandedState(first)); // bug 54116
		assertTrue("first2 is expanded", fTreeViewer.getExpandedState(first2));
		assertNotNull("first3 is visible", fViewer.testFindItem(first3));
	}

	@Test
	public void testExpandToLevel() {
		TestElement first = fRootElement.getFirstChild();
		TestElement first2 = first.getFirstChild();
		TestElement first3 = first2.getFirstChild();
		fTreeViewer.expandToLevel(3);

		DisplayHelper.waitAndAssertCondition(fShell.getDisplay(), () -> {
			assertNotNull(fViewer.testFindItem(first2));
			assertNotNull(fViewer.testFindItem(first3));
		});

		assertNotNull("first2 is visible", fViewer.testFindItem(first2));
		assertNotNull("first3 is visible", fViewer.testFindItem(first3));
	}

	@Test
	public void testAutoExpandOnSingleChild() {
		TestElement modelRoot = TestElement.createModel(5, 1);
		TestElement trivialPathRoot = modelRoot.getFirstChild();
		fViewer.setInput(modelRoot);

		fTreeViewer.setAutoExpandOnSingleChildLevels(2);
		fTreeViewer.setExpandedStateWithAutoExpandOnSingleChild(trivialPathRoot, true);

		assertTrue("The expanded widget child is not expanded", fTreeViewer.getExpandedState(trivialPathRoot));
		assertTrue("The first child of the trivial path was not auto-expanded",
				fTreeViewer.getExpandedState(trivialPathRoot.getFirstChild()));
		assertFalse("Trivial path is expanded further than specified depth ",
				fTreeViewer.getExpandedState(trivialPathRoot.getFirstChild().getFirstChild()));
	}

	@Test
	public void testAutoExpandOnSingleChildDeeperDownPath() {
		TestElement modelRoot = TestElement.createModel(6, 1);
		TestElement trivialPathRoot = modelRoot.getFirstChild();
		fViewer.setInput(modelRoot);

		fTreeViewer.setExpandedState(modelRoot, true); // https://github.com/eclipse-platform/eclipse.platform.ui/pull/1072#discussion_r1431558570
		fTreeViewer.setExpandedState(trivialPathRoot, true); // https://github.com/eclipse-platform/eclipse.platform.ui/pull/1072#discussion_r1431558570
		fTreeViewer.setAutoExpandOnSingleChildLevels(2);
		fTreeViewer.setExpandedStateWithAutoExpandOnSingleChild(trivialPathRoot.getFirstChild(), true);

		assertTrue("The first child of the trivial path was not auto-expanded",
				fTreeViewer.getExpandedState(trivialPathRoot.getFirstChild()));
		assertTrue("The second child of the trivial path was not auto-expanded",
				fTreeViewer.getExpandedState(trivialPathRoot.getFirstChild().getFirstChild()));
		assertFalse("Trivial path is expanded further than specified depth ",
				fTreeViewer.getExpandedState(trivialPathRoot.getFirstChild().getFirstChild().getFirstChild()));
	}

	@Test
	public void testAutoExpandOnSingleChildFromRoot() {
		TestElement modelRoot = TestElement.createModel(5, 5);
		TestElement trivialPathRoot = modelRoot;
		fViewer.setInput(modelRoot);

		fTreeViewer.setAutoExpandOnSingleChildLevels(2);
		fTreeViewer.setExpandedStateWithAutoExpandOnSingleChild(trivialPathRoot, true);

		assertFalse("The first child of the trivial path was auto-expanded",
				fTreeViewer.getExpandedState(trivialPathRoot.getFirstChild()));
	}

	@Test
	public void testAutoExpandOnSingleChildFromRootWithSensibleTrivialPath() {
		TestElement modelRoot = TestElement.createModel(5, 1);
		TestElement trivialPathRoot = modelRoot;
		fViewer.setInput(modelRoot);

		fTreeViewer.setAutoExpandOnSingleChildLevels(2);
		fTreeViewer.setExpandedStateWithAutoExpandOnSingleChild(trivialPathRoot, true);

		assertTrue("The first child of the trivial path was auto-expanded",
				fTreeViewer.getExpandedState(trivialPathRoot.getFirstChild()));
	}

	@Test
	public void testAutoExpandOnSingleChildSmallerThanAutoExpandDepth() {
		TestElement modelRoot = TestElement.createModel(2, 1);
		TestElement trivialPathRoot = modelRoot.getFirstChild();
		fViewer.setInput(modelRoot);

		fTreeViewer.setAutoExpandOnSingleChildLevels(10);
		fTreeViewer.setExpandedStateWithAutoExpandOnSingleChild(trivialPathRoot, true);

		assertTrue("The expanded widget child is not expanded", fTreeViewer.getExpandedState(trivialPathRoot));
		assertFalse("The first child of the trivial path was auto-expanded although it contains zero children",
				fTreeViewer.getExpandedState(trivialPathRoot.getFirstChild()));
	}

	@Test
	public void testSetAutoExpandOnSingleChildLevels() {
		fTreeViewer.setAutoExpandOnSingleChildLevels(AbstractTreeViewer.NO_EXPAND);
		assertEquals("Setting an auto-expansion level of NO_EXPAND works",
				fTreeViewer.getAutoExpandOnSingleChildLevels(), AbstractTreeViewer.NO_EXPAND);

		fTreeViewer.setAutoExpandOnSingleChildLevels(4);
		assertEquals("Setting a non-trivial auto-expansion level works", fTreeViewer.getAutoExpandOnSingleChildLevels(),
				4);

		fTreeViewer.setAutoExpandOnSingleChildLevels(AbstractTreeViewer.NO_EXPAND);
		assertEquals("Setting an auto-expansion level of NO_EXPAND works",
				fTreeViewer.getAutoExpandOnSingleChildLevels(), AbstractTreeViewer.NO_EXPAND);
	}

	@Test
	public void testAutoExpandOnSingleChildManualDisable() {
		// We need our own model since some default models do not generate trivial
		// paths
		TestElement modelRoot = TestElement.createModel(5, 1);
		TestElement trivialPathRoot = modelRoot.getFirstChild();
		fViewer.setInput(modelRoot);

		fTreeViewer.setAutoExpandOnSingleChildLevels(AbstractTreeViewer.NO_EXPAND);
		fTreeViewer.setExpandedStateWithAutoExpandOnSingleChild(trivialPathRoot, true);

		assertTrue("The expanded widget child is not expanded", fTreeViewer.getExpandedState(trivialPathRoot));
		assertFalse("The first child of the trivial path was auto-expanded",
				fTreeViewer.getExpandedState(trivialPathRoot.getFirstChild()));
	}

	public void testAutoExpandOnSingleChildManualEnableAndThenDisable() {
		TestElement modelRoot = TestElement.createModel(5, 1);
		TestElement trivialPathRoot = modelRoot.getFirstChild();
		fViewer.setInput(modelRoot);

		fTreeViewer.setAutoExpandOnSingleChildLevels(2);
		fTreeViewer.setAutoExpandOnSingleChildLevels(AbstractTreeViewer.NO_EXPAND);
		fTreeViewer.setExpandedStateWithAutoExpandOnSingleChild(trivialPathRoot, true);

		assertTrue("The expanded widget child is not expanded", fTreeViewer.getExpandedState(trivialPathRoot));
		assertFalse("The first child of the trivial path was auto-expanded",
				fTreeViewer.getExpandedState(trivialPathRoot.getFirstChild()));
	}

	@Test
	public void testAutoExpandOnSingleChildInfiniteExpand() {
		TestElement modelRoot = TestElement.createModel(5, 1);
		TestElement trivialPathRoot = modelRoot.getFirstChild();
		fViewer.setInput(modelRoot);

		fTreeViewer.setAutoExpandOnSingleChildLevels(AbstractTreeViewer.ALL_LEVELS);
		fTreeViewer.setExpandedStateWithAutoExpandOnSingleChild(trivialPathRoot, true);

		assertTrue("The expanded widget child is not expanded", fTreeViewer.getExpandedState(trivialPathRoot));
		TestElement child = trivialPathRoot.getFirstChild();
		for (int depth = 1; child != null; depth++) {
			assertTrue("The " + depth + ". child of the trivial path was not auto-expanded",
					fTreeViewer.getExpandedState(trivialPathRoot.getFirstChild()));
			child = child.getFirstChild();
		}
	}

	public void testFilterExpanded() {
		TestElement first = fRootElement.getFirstChild();
		TestElement first2 = first.getFirstChild();
		fTreeViewer.expandToLevel(first2, 0);

		fTreeViewer.addFilter(new TestLabelFilter());
		assertEquals("filtered count", 5, getItemCount());
	}

	@Test
	public void testInsertChildReveal() {
		TestElement first = fRootElement.getFirstChild();
		TestElement newElement = first.addChild(TestModelChange.INSERT | TestModelChange.REVEAL);
		assertNotNull("new sibling is visible", fViewer.testFindItem(newElement));
	}

	@Test
	public void testInsertChildRevealSelect() {
		TestElement last = fRootElement.getLastChild();
		TestElement newElement = last
				.addChild(TestModelChange.INSERT | TestModelChange.REVEAL | TestModelChange.SELECT);
		assertNotNull("new sibling is visible", fViewer.testFindItem(newElement));
		assertSelectionEquals("new element is selected", newElement);
	}

	@Test
	public void testInsertChildRevealSelectExpanded() {
		TestElement first = fRootElement.getFirstChild();
		TestElement newElement = first
				.addChild(TestModelChange.INSERT | TestModelChange.REVEAL | TestModelChange.SELECT);
		assertNotNull("new sibling is visible", fViewer.testFindItem(newElement));
		assertSelectionEquals("new element is selected", newElement);
	}

	/**
	 * Regression test for 1GDN0PX: ITPUI:WIN2000 - SEVERE - AssertionFailure when
	 * expanding Navigator Problem was: - before addition, parent item had no
	 * children, and was expanded - after addition, during refresh(), updatePlus()
	 * added dummy node even though parent item was expanded - in updateChildren, it
	 * wasn't handling a dummy node
	 */
	@Test
	public void testRefreshWithAddedChildren() {
		TestElement parent = fRootElement.addChild(TestModelChange.INSERT);
		TestElement child = parent.addChild(TestModelChange.INSERT);
		((AbstractTreeViewer) fViewer).setExpandedState(parent, true);
		parent.deleteChild(child);
		child = parent.addChild(TestModelChange.STRUCTURE_CHANGE);
		// On some platforms (namely GTK), removing all children causes the
		// parent to collapse (actually it's worse than that: GTK doesn't
		// allow there to be an empty expanded tree item, even if you do a
		// setExpanded(true)).
		// This behaviour makes it impossible to do this regression test.
		// See bug 40797 for more details. Because GTK 3 takes longer to
		// process, a wait statement is needed so that the assert will be done
		// correctly without failing.
		waitForJobs(300, 1000);
		processEvents();
		if (((AbstractTreeViewer) fViewer).getExpandedState(parent)) {
			assertNotNull("new child is visible", fViewer.testFindItem(child));
		}
	}

	/**
	 * Regression test for 1GBDB5A: ITPUI:WINNT - Exception in AbstractTreeViewer
	 * update. Problem was: node has child A node gets duplicate child A viewer is
	 * refreshed rather than using add for new A
	 * AbstractTreeViewer.updateChildren(...) was not properly handling it
	 */
	@Test
	public void testRefreshWithDuplicateChild() {
		TestElement first = fRootElement.getFirstChild();
		TestElement newElement = (TestElement) first.clone();
		fRootElement.addChild(newElement, new TestModelChange(TestModelChange.STRUCTURE_CHANGE, fRootElement));
		assertNotNull("new sibling is visible", fViewer.testFindItem(newElement));
	}

	/**
	 * Regression test for Bug 3840 [Viewers] free expansion of jar happening when
	 * deleting projects (1GEV2FL) Problem was: - node has children A and B - A is
	 * expanded, B is not - A gets deleted - B gets expanded because it reused A's
	 * item
	 */
	@Test
	public void testRefreshWithReusedItems() {
		// TestElement a= fRootElement.getFirstChild();
		// TestElement aa= a.getChildAt(0);
		// TestElement ab= a.getChildAt(1);
		// fTreeViewer.expandToLevel(aa, 1);
		// List expandedBefore = Arrays.asList(fTreeViewer.getExpandedElements());
		// assertTrue(expandedBefore.contains(a));
		// assertTrue(expandedBefore.contains(aa));
		// assertFalse(expandedBefore.contains(ab));
		// a.deleteChild(aa, new TestModelChange(TestModelChange.STRUCTURE_CHANGE, a));
		// List expandedAfter = Arrays.asList(fTreeViewer.getExpandedElements());
		// assertFalse(expandedAfter.contains(ab));
	}

	@Test
	public void testRenameChildElement() {
		TestElement first = fRootElement.getFirstChild();
		TestElement first2 = first.getFirstChild();
		fTreeViewer.expandToLevel(first2, 0);
		assertNotNull("first child is visible", fViewer.testFindItem(first2));

		String newLabel = first2.getLabel() + " changed";
		first2.setLabel(newLabel);
		Widget widget = fViewer.testFindItem(first2);
		assertTrue(widget instanceof Item);
		assertEquals("changed label", first2.getID() + " " + newLabel, ((Item) widget).getText());
	}

	/**
	 * Regression test for Bug 26698 [Viewers] stack overflow during debug session,
	 * causing IDE to crash Problem was: - node A has child A - setExpanded with A
	 * in the list caused an infinite recursion
	 */
	@Test
	public void testSetExpandedWithCycle() {
		TestElement first = fRootElement.getFirstChild();
		first.addChild(first, new TestModelChange(TestModelChange.INSERT, first, first));
		fTreeViewer.setExpandedElements(new Object[] { first });

	}

	/**
	 * Test for Bug 41710 - assertion that an object may not be added to a given
	 * TreeItem more than once.
	 */
	@Test
	public void testSetDuplicateChild() {
		// Widget root = fViewer.testFindItem(fRootElement);
		// assertNotNull(root);
		TestElement parent = fRootElement.addChild(TestModelChange.INSERT);
		TestElement child = parent.addChild(TestModelChange.INSERT);
		int initialCount = getItemCount(parent);
		fRootElement.addChild(child, new TestModelChange(TestModelChange.INSERT, fRootElement, child));
		int postCount = getItemCount(parent);
		assertEquals("Same element added to a parent twice.", initialCount, postCount);
	}

	/**
	 * Test for Bug 571844 - assert that an item is not added twice if the
	 * comparator returns 0 = equal for more than one tree item. Problem was that
	 * the method AbstractTreeViewer#createAddedElements only searched forward but
	 * not backwards for an equal element, if the comparator returned 0. The example
	 * below is a case where the previous implementation would fail.
	 */
	@Test
	public void testChildIsNotDuplicatedWhenCompareEquals() {
		fTreeViewer.setComparator(new TestLabelComparator());
		fRootElement.deleteChildren();

		TestElement child1 = fRootElement.addChild(TestModelChange.INSERT);
		child1.setLabel("1");
		TestElement child2 = fRootElement.addChild(TestModelChange.INSERT);
		child2.setLabel("1");
		TestElement child3 = fRootElement.addChild(TestModelChange.INSERT);
		child3.setLabel("0");

		// Every duplicated element must not be added as TreeItem.
		fRootElement.addChild(child1, new TestModelChange(TestModelChange.INSERT, fRootElement, child1));
		fRootElement.addChild(child2, new TestModelChange(TestModelChange.INSERT, fRootElement, child2));
		fRootElement.addChild(child3, new TestModelChange(TestModelChange.INSERT, fRootElement, child3));

		Tree tree = (Tree) fTreeViewer.getControl();
		assertEquals("Same element added to parent twice.", 3, tree.getItems().length);
	}

	@Test
	public void testContains() {
		// some random element.
		assertFalse("element must not be available on the viewer", fTreeViewer.contains(fRootElement, ""));

		// first child of root.
		assertTrue("element must be available on the viewer",
				fTreeViewer.contains(fRootElement, fRootElement.getFirstChild()));

		// last child of the root
		assertTrue("element must be available on the viewer",
				fTreeViewer.contains(fRootElement, fRootElement.getLastChild()));
		// child of first element is not expanded
		assertFalse("element must not be available on the viewer",
				fTreeViewer.contains(fRootElement.getFirstChild(), fRootElement.getFirstChild().getFirstChild()));
		fTreeViewer.expandAll();
		// child of first element when expanded.
		assertTrue("element must be available on the viewer",
				fTreeViewer.contains(fRootElement.getFirstChild(), fRootElement.getFirstChild().getFirstChild()));
	}

	@After
	@Override
	public void tearDown() {
		super.tearDown();
		fTreeViewer = null;
	}
}
