/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.scada.hds;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Date;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.eclipse.scada.hds.AbstractValueSource;
import org.eclipse.scada.hds.DataFileAccessor;
import org.eclipse.scada.hds.DataFilePool;
import org.eclipse.scada.hds.Quantizer;
import org.eclipse.scada.hds.ValueVisitor;
import org.eclipse.scada.utils.lang.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataStoreAccesor
extends AbstractValueSource {
    private static final Logger logger = LoggerFactory.getLogger(DataStoreAccesor.class);
    private final File basePath;
    private final long time;
    private final TimeUnit unit;
    private final Quantizer quantizer;
    private final int count;
    private final DataFilePool pool;
    private final Set<AccessorWrapper> wrappers = new HashSet<AccessorWrapper>();
    private final Lock lock = new ReentrantLock();
    private final Condition wrapperCondition = this.lock.newCondition();
    private boolean disposed;

    public DataStoreAccesor(File basePath, DataFilePool pool) throws Exception {
        this.basePath = basePath;
        this.pool = pool;
        if (!basePath.isDirectory()) {
            throw new IllegalStateException(String.format("'%s' is not a directory", basePath));
        }
        Properties p = new Properties();
        p.loadFromXML(new FileInputStream(new File(basePath, "settings.xml")));
        this.time = Long.parseLong(p.getProperty("time"));
        this.unit = TimeUnit.valueOf(p.getProperty("unit"));
        this.count = Integer.parseInt(p.getProperty("count"));
        this.quantizer = new Quantizer(this.time, this.unit, this.count);
    }

    public int getCount() {
        return this.count;
    }

    public long getTimeSlice() {
        return TimeUnit.MILLISECONDS.convert(this.time, this.unit);
    }

    protected void take(AccessorWrapper wrapper) {
        this.lock.lock();
        try {
            this.wrappers.add(wrapper);
        }
        finally {
            this.lock.unlock();
        }
    }

    protected void giveBack(AccessorWrapper wrapper) {
        logger.debug("Giving back resource: {}", (Object)wrapper);
        this.lock.lock();
        try {
            this.wrappers.remove(wrapper);
            this.wrapperCondition.signalAll();
        }
        finally {
            this.lock.unlock();
        }
    }

    protected void insertValue(double value, Date date, boolean error, boolean manual, boolean heartbeat) throws Exception {
        logger.debug("Inserting value - value: {}, timestamp: {}, error: {}, manual: {}", new Object[]{value, date, error, manual});
        AccessorWrapper file = this.createOrGetFile(date, true);
        if (file != null) {
            try {
                file.insertValue(value, date, error, manual, heartbeat);
            }
            finally {
                file.dispose();
            }
        } else {
            throw new IllegalStateException(String.format("Unable to get file for %tc", date));
        }
        if (!heartbeat && !Double.isNaN(value)) {
            logger.debug("Starting forward correction");
            Date now = new Date();
            Date current = this.quantizer.getValidStart(date);
            while (current != null && current.before(now)) {
                logger.debug("Forward correcting - starting: {}", (Object)current);
                AccessorWrapper file2 = this.createOrGetFile(current, true);
                if (file2 != null) {
                    try {
                        file2.forwardCorrect(value, date);
                    }
                    finally {
                        file2.dispose();
                    }
                }
                current = this.quantizer.getNext(current);
            }
            logger.debug("Finished forward correcting");
        }
    }

    public void insertValue(double value, Date date, boolean error, boolean manual) throws Exception {
        this.insertValue(value, date, error, manual, false);
    }

    public void insertHeartbeat(Date date) throws Exception {
        this.insertValue(Double.NaN, date, false, false, true);
    }

    @Override
    public boolean visit(ValueVisitor visitor, Date start, Date end) {
        Date next;
        logger.debug("Process visit - start: {}, end: {}", (Object)start, (Object)end);
        Date current = this.quantizer.getStart(start);
        logger.debug("Quantized start: {}", (Object)current);
        logger.trace("Searching backwards");
        boolean firstRead = false;
        do {
            Date startTimestamp = this.quantizer.getPrevious(current);
            Date endTimestamp = current;
            logger.debug("Visiting - start: {}, end: {}", (Object)startTimestamp, (Object)endTimestamp);
            AccessorWrapper file = null;
            try {
                try {
                    file = this.createOrGetFile(startTimestamp, false);
                    logger.trace("Aquire file - {} -> {}", (Object)startTimestamp, (Object)file);
                }
                catch (Exception e) {
                    logger.warn(String.format("Failed to access file for timestamp %tc -> %tc", startTimestamp, endTimestamp), (Throwable)e);
                }
                if (file == null) {
                    logger.info("Did not receive file. Skipping...");
                } else {
                    try {
                        logger.trace("Visiting file");
                        firstRead = file.visitFirstValue(visitor);
                    }
                    catch (Exception e) {
                        logger.warn("Failed to visit file", (Throwable)e);
                    }
                }
            }
            finally {
                if (file != null) {
                    file.dispose();
                }
            }
            current = startTimestamp;
            logger.debug("Current timestamp is now: {}", (Object)current);
        } while (!firstRead && this.quantizer.getValidStart(current) != null);
        logger.debug("Searching forwards");
        current = this.quantizer.getStart(start);
        logger.debug("Starting with: {}", (Object)current);
        do {
            next = this.quantizer.getNext(current);
            logger.debug("Visiting - start: {}, end: {}", (Object)current, (Object)next);
            AccessorWrapper file = null;
            try {
                try {
                    file = this.createOrGetFile(current, false);
                    logger.trace("Aquire file - {} -> {}", (Object)current, (Object)file);
                }
                catch (Exception e) {
                    logger.warn(String.format("Failed to access file for timestamp %tc -> %tc", current, next), (Throwable)e);
                }
                if (file == null) {
                    logger.info("Did not receive file. Marking invalid!");
                    if (!visitor.value(Double.NaN, current, true, false)) {
                        logger.debug("Visitor requested stop");
                        return false;
                    }
                } else {
                    logger.debug("Visiting file");
                    if (!file.visit(visitor)) {
                        logger.debug("Visitor requested stop");
                        return false;
                    }
                }
            }
            finally {
                if (file != null) {
                    file.dispose();
                }
            }
        } while ((current = next).before(end));
        logger.debug("Completed visit");
        return true;
    }

    public void purge() {
        logger.info("Purging {}", (Object)this.basePath);
        File[] fileArray = this.basePath.listFiles();
        int n = fileArray.length;
        int n2 = 0;
        while (n2 < n) {
            File file = fileArray[n2];
            logger.debug("Checking file: {}", (Object)file);
            if (!file.isFile()) {
                logger.debug("{} is not a file. Ignoring.", (Object)file);
            } else {
                try {
                    DataFileAccessor accessor = this.pool.getAccessor(file);
                    if (accessor == null) {
                        logger.warn("No accessor. Ignoring file: {}", (Object)file);
                    } else if (accessor.getEnd().before(this.quantizer.getEndOfPeriod(null))) {
                        accessor.delete();
                    } else {
                        accessor.dispose();
                    }
                }
                catch (Exception e) {
                    logger.warn(String.format("Failed to check file: %s", file), (Throwable)e);
                }
            }
            ++n2;
        }
    }

    private AccessorWrapper createOrGetFile(Date date, boolean create) throws Exception {
        Date end;
        if (this.disposed) {
            throw new IllegalStateException("Data store is disposed");
        }
        Date start = this.quantizer.getValidStart(date);
        if (start == null) {
            logger.info("{} is outside of the valid time period", (Object)date);
            return null;
        }
        File file = new File(this.basePath, String.format("%08x.hds", start.getTime()));
        Pair<DataFileAccessor, Boolean> poolResult = this.pool.getAccessor(file, start, end = this.quantizer.getEnd(date), create);
        if (poolResult == null) {
            logger.info("Pool returned no file");
            return null;
        }
        if (((Boolean)poolResult.second).booleanValue()) {
            logger.debug("Pool file was created. Notify change");
            this.notifyChange(start, end);
        }
        AccessorWrapper result = new AccessorWrapper((DataFileAccessor)poolResult.first);
        Date secondStart = this.quantizer.getValidStart(date);
        if (secondStart == null) {
            logger.info("Resource timeout out while we waited for it");
            result.dispose();
            return null;
        }
        return result;
    }

    public static DataStoreAccesor create(File basePath, long time, TimeUnit unit, int count, DataFilePool pool) throws Exception {
        if (basePath.exists()) {
            throw new IllegalArgumentException(String.format("'%s' must not exists", basePath));
        }
        if (!basePath.getParentFile().isDirectory()) {
            throw new IllegalArgumentException(String.format("Parent directory '%s' must exists and must be a directory", basePath.getParentFile()));
        }
        if (!basePath.mkdir()) {
            throw new IllegalArgumentException(String.format("Unable to create directory %s", basePath));
        }
        Properties p = new Properties();
        p.put("time", "" + time);
        p.put("unit", unit.name());
        p.put("count", "" + count);
        p.put("version", "1");
        try {
            p.storeToXML(new FileOutputStream(new File(basePath, "settings.xml")), "Eclipse SCADA HDS Settings");
        }
        catch (Exception e) {
            basePath.delete();
            throw new IllegalStateException(e);
        }
        return new DataStoreAccesor(basePath, pool);
    }

    public void dispose() {
        this.lock.lock();
        try {
            this.disposed = true;
            logger.info("Waiting for all resources to be given back");
            while (!this.wrappers.isEmpty()) {
                try {
                    this.wrapperCondition.await();
                }
                catch (InterruptedException e) {
                    logger.warn("Failed to wait for shutdown", (Throwable)e);
                    this.lock.unlock();
                    return;
                }
            }
            logger.info("Everbody home");
        }
        finally {
            this.lock.unlock();
        }
    }

    private class AccessorWrapper
    implements DataFileAccessor {
        private final DataFileAccessor accessor;

        public AccessorWrapper(DataFileAccessor accessor) {
            this.accessor = accessor;
            DataStoreAccesor.this.take(this);
        }

        @Override
        public void insertValue(double value, Date date, boolean error, boolean manual, boolean heartbeat) throws IOException {
            this.accessor.insertValue(value, date, error, manual, heartbeat);
        }

        @Override
        public boolean visit(ValueVisitor visitor) throws Exception {
            return this.accessor.visit(visitor);
        }

        @Override
        public boolean visitFirstValue(ValueVisitor visitor) throws Exception {
            return this.accessor.visitFirstValue(visitor);
        }

        @Override
        public void forwardCorrect(double value, Date date) throws Exception {
            this.accessor.forwardCorrect(value, date);
        }

        @Override
        public void delete() {
            this.accessor.delete();
            this.dispose();
        }

        @Override
        public Date getStart() {
            return this.accessor.getStart();
        }

        @Override
        public Date getEnd() {
            return this.accessor.getEnd();
        }

        @Override
        public void dispose() {
            try {
                this.accessor.dispose();
            }
            finally {
                DataStoreAccesor.this.giveBack(this);
            }
        }
    }
}

