/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.scada.hds;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.eclipse.scada.hds.DataFileAccessor;
import org.eclipse.scada.hds.DataFileAccessorImpl;
import org.eclipse.scada.hds.ValueVisitor;
import org.eclipse.scada.utils.lang.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataFilePool {
    private static final Logger logger = LoggerFactory.getLogger(DataFilePool.class);
    private static final long LOCK_TIME = 10000L;
    private final Lock lock = new ReentrantLock();
    private final Condition condition = this.lock.newCondition();
    private final long timeout;
    private final Map<File, AccessorWrapper> freePool = new HashMap<File, AccessorWrapper>();
    private final Map<File, AccessorWrapper> usedPool = new HashMap<File, AccessorWrapper>();
    private boolean disposed;
    private int instanceCountTarget;

    public DataFilePool(int instanceCountTarget) {
        this(10000L);
        this.instanceCountTarget = instanceCountTarget;
    }

    public DataFilePool(long timeout) {
        this.timeout = timeout;
    }

    public DataFileAccessor getAccessor(File file) throws Exception {
        Pair<DataFileAccessor, Boolean> result = this.getAccessor(file, null, null, false);
        if (result == null) {
            return null;
        }
        return (DataFileAccessor)result.first;
    }

    public Pair<DataFileAccessor, Boolean> getAccessor(File file, Date start, Date end, boolean create) throws Exception {
        Date deadline = new Date(System.currentTimeMillis() + this.timeout);
        logger.debug("Looking for file: {}", (Object)file);
        if (this.lock.tryLock(this.timeout, TimeUnit.MILLISECONDS)) {
            if (this.disposed) {
                throw new IllegalStateException("Pool is disposed");
            }
            try {
                if (file.exists()) {
                    logger.debug("File exists");
                    Pair pair = new Pair((Object)this.waitForFile(file, deadline), (Object)false);
                    return pair;
                }
                if (create) {
                    logger.debug("File does not exists and we are requested to create");
                    AccessorWrapper result = this.wrap(file, DataFileAccessorImpl.create(file, start, end));
                    this.usedPool.put(file, result);
                    Pair pair = new Pair((Object)result, (Object)true);
                    return pair;
                }
                logger.debug("File does not exists and no request to create");
                return null;
            }
            finally {
                this.lock.unlock();
            }
        }
        throw new IllegalStateException(String.format("Failed to acquire create lock within %s ms", this.timeout));
    }

    private AccessorWrapper waitForFile(File file, Date deadline) throws Exception {
        logger.info("Waiting until {} for {}", (Object)deadline, (Object)file);
        long startTix = System.currentTimeMillis();
        try {
            do {
                if (this.usedPool.containsKey(file)) continue;
                AccessorWrapper result = this.freePool.remove(file);
                if (result != null) {
                    logger.debug("Fetching file {} from free pool", (Object)file);
                    this.usedPool.put(file, result);
                    AccessorWrapper accessorWrapper = result;
                    return accessorWrapper;
                }
                AccessorWrapper newResult = this.wrap(file, new DataFileAccessorImpl(file));
                this.usedPool.put(file, newResult);
                logger.debug("Acquired resource {} by creating accessor", (Object)file);
                AccessorWrapper accessorWrapper = newResult;
                return accessorWrapper;
            } while (this.condition.awaitUntil(deadline));
            throw new IllegalStateException(String.format("Failed to acquire create lock within %s ms for resource %s", this.timeout, file));
        }
        finally {
            logger.info("Waiting took {} ms", (Object)(System.currentTimeMillis() - startTix));
        }
    }

    protected AccessorWrapper wrap(File file, DataFileAccessor accessor) {
        this.lock.lock();
        try {
            this.closeUnused();
            AccessorWrapper accessorWrapper = new AccessorWrapper(file, accessor);
            return accessorWrapper;
        }
        finally {
            this.lock.unlock();
        }
    }

    private void closeUnused() {
        int instances = this.usedPool.size() + this.freePool.size();
        int num = instances - this.instanceCountTarget;
        if (num <= 0) {
            return;
        }
        logger.info("Trying to reduce by {}", (Object)num);
        LinkedList<AccessorWrapper> entries = new LinkedList<AccessorWrapper>(this.freePool.values());
        Collections.sort(entries, new Comparator<AccessorWrapper>(){

            @Override
            public int compare(AccessorWrapper o1, AccessorWrapper o2) {
                return o1.getLastAccess().compareTo(o2.getLastAccess());
            }
        });
        while (!this.freePool.isEmpty() && !entries.isEmpty() && num > 0) {
            AccessorWrapper entry = entries.pollLast();
            logger.info("Removing {} from pool", (Object)entry.getFile());
            this.freePool.remove(entry.getFile());
            entry.getTarget().dispose();
            --num;
        }
    }

    public void giveBack(AccessorWrapper accessor) {
        logger.debug("Giving back: {}", (Object)accessor.getFile());
        this.lock.lock();
        try {
            if (this.usedPool.remove(accessor.getFile()) != null) {
                this.freePool.put(accessor.getFile(), accessor);
            }
            this.condition.signalAll();
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void dispose() {
        this.lock.lock();
        try {
            try {
                this.disposed = true;
                block13: while (true) {
                    if (this.usedPool.isEmpty()) {
                        return;
                    }
                    while (true) {
                        if (this.condition.await(30L, TimeUnit.SECONDS)) continue block13;
                        logger.warn("Still waiting for resources to be returned");
                    }
                    break;
                }
            }
            catch (InterruptedException e) {
                logger.warn("Failed to await end of dispose", (Throwable)e);
                try {
                    this.disposeFreePool();
                    return;
                }
                finally {
                    this.lock.unlock();
                }
            }
        }
        finally {
            try {
                this.disposeFreePool();
            }
            finally {
                this.lock.unlock();
            }
        }
    }

    private void disposeFreePool() {
        for (Map.Entry<File, AccessorWrapper> wrapper : this.freePool.entrySet()) {
            try {
                wrapper.getValue().getTarget().dispose();
            }
            catch (Exception e) {
                logger.warn(String.format("Failed to dispose %s", wrapper.getKey()), (Throwable)e);
            }
        }
        this.freePool.clear();
    }

    private class AccessorWrapper
    implements DataFileAccessor {
        private final DataFileAccessor accessor;
        private final File file;
        private long accesses;
        private Date lastAccess = new Date();

        public AccessorWrapper(File file, DataFileAccessor accessor) {
            this.file = file;
            this.accessor = accessor;
        }

        @Override
        public void insertValue(double value, Date date, boolean error, boolean manual, boolean heartbeat) throws IOException {
            this.check();
            this.accessor.insertValue(value, date, error, manual, heartbeat);
        }

        @Override
        public boolean visit(ValueVisitor visitor) throws Exception {
            this.check();
            return this.accessor.visit(visitor);
        }

        @Override
        public boolean visitFirstValue(ValueVisitor visitor) throws Exception {
            this.check();
            return this.accessor.visitFirstValue(visitor);
        }

        @Override
        public void forwardCorrect(double value, Date date) throws Exception {
            this.check();
            this.accessor.forwardCorrect(value, date);
        }

        @Override
        public Date getStart() {
            return this.accessor.getStart();
        }

        @Override
        public Date getEnd() {
            return this.accessor.getEnd();
        }

        @Override
        public void delete() {
            this.check();
            this.accessor.delete();
            this.dispose();
        }

        @Override
        public void dispose() {
            DataFilePool.this.giveBack(this);
        }

        public File getFile() {
            return this.file;
        }

        public DataFileAccessor getTarget() {
            return this.accessor;
        }

        protected void check() {
            ++this.accesses;
            this.lastAccess = new Date();
        }

        public long getAccesses() {
            return this.accesses;
        }

        public Date getLastAccess() {
            return this.lastAccess;
        }
    }
}

