/*******************************************************************************
 * Copyright © 2008, 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM Corporation - initial API and implementation
 *
 *******************************************************************************/
package org.eclipse.edt.ide.ui.internal.refactoring.changes;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileInfo;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.edt.ide.core.model.EGLModelException;
import org.eclipse.edt.ide.core.model.IEGLModelStatusConstants;
import org.eclipse.edt.ide.ui.internal.UINlsStrings;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;

import com.ibm.icu.text.MessageFormat;

public class CreateFileChange extends Change {

	private String fChangeName;

	private IPath fPath;
	private String fSource;
	private String fEncoding;
	private boolean fExplicitEncoding;
	private long fStampToRestore;

	public CreateFileChange(IPath path, String source, String encoding) {
		this(path, source, encoding, IResource.NULL_STAMP);
	}

	public CreateFileChange(IPath path, String source, String encoding, long stampToRestore) {
		Assert.isNotNull(path, "path"); //$NON-NLS-1$
		Assert.isNotNull(source, "source"); //$NON-NLS-1$
		fPath= path;
		fSource= source;
		fEncoding= encoding;
		fExplicitEncoding= fEncoding != null;
		fStampToRestore= stampToRestore;
	}

	protected void setEncoding(String encoding, boolean explicit) {
		Assert.isNotNull(encoding, "encoding"); //$NON-NLS-1$
		fEncoding= encoding;
		fExplicitEncoding= explicit;
	}

	public String getName() {
		if (fChangeName == null)
			return MessageFormat.format(UINlsStrings.createFile_Create_file, new String[] {fPath.toOSString()});
		else
			return fChangeName;
	}

	public void setName(String name) {
		fChangeName= name;
	}
	
	protected void setSource(String source) {
		fSource= source;
	}

	protected String getSource() {
		return fSource;
	}

	protected void setPath(IPath path) {
		fPath= path;
	}

	protected IPath getPath() {
		return fPath;
	}

	public Object getModifiedElement() {
		return ResourcesPlugin.getWorkspace().getRoot().getFile(fPath);
	}

	public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException {
		RefactoringStatus result= new RefactoringStatus();
		IFile file= ResourcesPlugin.getWorkspace().getRoot().getFile(fPath);
		
		URI location= file.getLocationURI();
		if (location == null) {
			result.addFatalError(MessageFormat.format(
				UINlsStrings.CreateFileChange_error_unknownLocation, 
				new String[] {file.getFullPath().toString()}));
			return result;
		}
		
		IFileInfo jFile= EFS.getStore(location).fetchInfo();
		if (jFile.exists()) {
			result.addFatalError(MessageFormat.format(
				UINlsStrings.CreateFileChange_error_exists, 
				new String[] {file.getFullPath().toString()}));
			return result;
		}
		return result;
	}

	public Change perform(IProgressMonitor pm) throws CoreException {

		InputStream is= null;
		try {
			pm.beginTask(UINlsStrings.createFile_creating_resource, 3); 

			initializeEncoding();
			IFile file= getOldFile(new SubProgressMonitor(pm, 1));
			try {
				is= new ByteArrayInputStream(fSource.getBytes(fEncoding));
				file.create(is, false, new SubProgressMonitor(pm, 1));
				if (fStampToRestore != IResource.NULL_STAMP) {
					file.revertModificationStamp(fStampToRestore);
				}
				if (fExplicitEncoding) {
					file.setCharset(fEncoding, new SubProgressMonitor(pm, 1));
				} else {
					pm.worked(1);
				}
				return new DeleteFileChange(file);
			} catch (UnsupportedEncodingException e) {
				throw new EGLModelException(e, IEGLModelStatusConstants.IO_EXCEPTION);
			}
		} finally {
			try {
				if (is != null)
					is.close();
			} catch (IOException ioe) {
				throw new EGLModelException(ioe, IEGLModelStatusConstants.IO_EXCEPTION);
			} finally {
				pm.done();
			}
		}
	}

	protected IFile getOldFile(IProgressMonitor pm) {
		pm.beginTask("", 1); //$NON-NLS-1$
		try {
			return ResourcesPlugin.getWorkspace().getRoot().getFile(fPath);
		} finally {
			pm.done();
		}
	}

	private void initializeEncoding() {
		if (fEncoding == null) {
			fExplicitEncoding= false;
			IFile file= ResourcesPlugin.getWorkspace().getRoot().getFile(fPath);
			if (file != null) {
				try {
					if (file.exists()) {
						fEncoding= file.getCharset(false);
						if (fEncoding == null) {
							fEncoding= file.getCharset(true);
						} else {
							fExplicitEncoding= true;
						}
					} else {
						fEncoding= file.getCharset(true);
					}
				} catch (CoreException e) {
					fEncoding= ResourcesPlugin.getEncoding();
					fExplicitEncoding= true;
				}
			} else {
				fEncoding= ResourcesPlugin.getEncoding();
				fExplicitEncoding= true;
			}
		}
		Assert.isNotNull(fEncoding);
	}

	public void initializeValidationData(IProgressMonitor pm) {
	}
}
