/*******************************************************************************
 * Copyright © 2011 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM Corporation - initial API and implementation
 *
 *******************************************************************************/
package org.eclipse.edt.debug.internal.core.java;

import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.core.model.IValue;
import org.eclipse.debug.core.model.IVariable;
import org.eclipse.edt.debug.core.IEGLVariable;
import org.eclipse.jdt.debug.core.IJavaValue;
import org.eclipse.jdt.debug.core.IJavaVariable;

/**
 * Wraps an IJavaVariable.
 */
public class EGLJavaVariable extends EGLJavaDebugElement implements IEGLVariable
{
	/**
	 * The underlying Java variable.
	 */
	protected final IJavaVariable javaVariable;
	
	/**
	 * The EGL-wrapped value of the variable.
	 */
	protected EGLJavaValue value;
	
	/**
	 * The variable information retrieved from the SMAP.
	 */
	protected final SMAPVariableInfo variableInfo;
	
	/**
	 * Constructor.
	 * 
	 * @param javaVariable The underlying Java variable.
	 * @param variableInfo The variable information retreived from the SMAP.
	 * @param parent The parent of this variable; this should be null if the variable is toplevel.
	 */
	public EGLJavaVariable( IDebugTarget target, IJavaVariable javaVariable, SMAPVariableInfo variableInfo )
	{
		super( target );
		this.javaVariable = javaVariable;
		this.variableInfo = variableInfo;
	}
	
	@Override
	public Object getAdapter( Class adapter )
	{
		if ( adapter == IVariable.class || adapter == EGLJavaVariable.class || adapter == IEGLVariable.class )
		{
			return this;
		}
		if ( adapter == IJavaVariable.class )
		{
			return javaVariable;
		}
		return super.getAdapter( adapter );
	}
	
	@Override
	public void setValue( String expression ) throws DebugException
	{
	}
	
	@Override
	public void setValue( IValue value ) throws DebugException
	{
	}
	
	@Override
	public boolean supportsValueModification()
	{
		return false;
	}
	
	@Override
	public boolean verifyValue( String expression ) throws DebugException
	{
		return false;
	}
	
	@Override
	public boolean verifyValue( IValue value ) throws DebugException
	{
		return false;
	}
	
	@Override
	public IValue getValue() throws DebugException
	{
		IValue javaValue = javaVariable.getValue();
		if ( javaValue instanceof IJavaValue )
		{
			if ( value == null || value.getJavaValue() != javaValue )
			{
				value = new EGLJavaValue( getDebugTarget(), (IJavaValue)javaValue, this );
			}
			return value;
		}
		return javaValue;
	}
	
	@Override
	public String getName() throws DebugException
	{
		return variableInfo.eglName;
	}
	
	@Override
	public String getReferenceTypeName() throws DebugException
	{
		return variableInfo.type;
	}
	
	@Override
	public boolean hasValueChanged() throws DebugException
	{
		return javaVariable.hasValueChanged();
	}
	
	/**
	 * @return the underlying variable.
	 */
	public IJavaVariable getJavaVariable()
	{
		return javaVariable;
	}
	
	/**
	 * @return the underlying Java element.
	 */
	@Override
	public Object getJavaElement()
	{
		return getJavaVariable();
	}
}
