/*******************************************************************************
 * Copyright © 2011 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM Corporation - initial API and implementation
 *
 *******************************************************************************/
package org.eclipse.edt.debug.internal.core.java;

import java.util.Hashtable;
import java.util.List;

import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.model.IDropToFrame;
import org.eclipse.debug.core.model.IRegisterGroup;
import org.eclipse.debug.core.model.IStackFrame;
import org.eclipse.debug.core.model.IThread;
import org.eclipse.debug.core.model.IVariable;
import org.eclipse.edt.debug.core.IEGLStackFrame;
import org.eclipse.edt.debug.core.IEGLThread;
import org.eclipse.edt.debug.core.java.IEGLJavaStackFrame;
import org.eclipse.edt.debug.core.java.IEGLJavaThread;
import org.eclipse.edt.debug.core.java.IEGLJavaValue;
import org.eclipse.edt.debug.core.java.IEGLJavaVariable;
import org.eclipse.edt.debug.core.java.SMAPFunctionInfo;
import org.eclipse.edt.debug.core.java.SMAPUtil;
import org.eclipse.edt.debug.core.java.SMAPVariableInfo;
import org.eclipse.jdt.debug.core.IJavaStackFrame;

/**
 * Wraps an IJavaStackFrame.
 */
public class EGLJavaStackFrame extends EGLJavaDebugElement implements IEGLJavaStackFrame, IDropToFrame
{
	/**
	 * The underlying Java stack frame.
	 */
	private IJavaStackFrame javaFrame;
	
	/**
	 * The EGL-wrapped thread.
	 */
	private final IEGLJavaThread eglThread;
	
	/**
	 * The EGL-wrapped variables.
	 */
	private IVariable[] variables;
	
	/**
	 * The current cached EGL variables.
	 */
	private Hashtable<String, IEGLJavaVariable> currentEGLVariables;
	
	/**
	 * The cached EGL variables from the last time we suspended.
	 */
	private Hashtable<String, IEGLJavaVariable> previousEGLVariables;
	
	/**
	 * The SMAP data from the source debug extension of the class file.
	 */
	private String smap;
	
	/**
	 * The variable information retrieved from the SMAP.
	 */
	private SMAPVariableInfo[] smapVariableInfos;
	
	/**
	 * The function information retrieved from the SMAP, for this stack frame.
	 */
	private SMAPFunctionInfo smapFunctionInfo;
	
	/**
	 * The line number this frame was suspended at before a stepInto was performed.
	 */
	private int lineBeforeStepInto;
	
	/**
	 * Constructor.
	 * 
	 * @param frame The Java stack frame.
	 * @param thread The EGL-wrapped thread.
	 */
	public EGLJavaStackFrame( IJavaStackFrame frame, EGLJavaThread thread )
	{
		super( thread.getDebugTarget() );
		this.eglThread = thread;
		bind( frame );
	}
	
	/**
	 * Binds to the given Java frame.
	 * 
	 * @param javaFrame The underlying Java stack frame.
	 */
	protected void bind( IJavaStackFrame javaFrame )
	{
		this.javaFrame = javaFrame;
		previousEGLVariables = currentEGLVariables;
		
		int hashSize = 10;
		if ( previousEGLVariables != null )
		{
			int oldSize = previousEGLVariables.size();
			if ( oldSize > hashSize )
			{
				hashSize = oldSize;
			}
		}
		currentEGLVariables = new Hashtable<String, IEGLJavaVariable>( hashSize );
		
		this.variables = null;
	}
	
	@Override
	public boolean canStepInto()
	{
		return javaFrame.canStepInto();
	}
	
	@Override
	public boolean canStepOver()
	{
		return javaFrame.canStepOver();
	}
	
	@Override
	public boolean canStepReturn()
	{
		return javaFrame.canStepReturn();
	}
	
	@Override
	public boolean isStepping()
	{
		return javaFrame.isStepping();
	}
	
	@Override
	public void stepInto() throws DebugException
	{
		javaFrame.stepInto();
	}
	
	@Override
	public void stepOver() throws DebugException
	{
		javaFrame.stepOver();
	}
	
	@Override
	public void stepReturn() throws DebugException
	{
		javaFrame.stepReturn();
	}
	
	@Override
	public boolean canResume()
	{
		return javaFrame.canResume();
	}
	
	@Override
	public boolean canSuspend()
	{
		return javaFrame.canSuspend();
	}
	
	@Override
	public boolean isSuspended()
	{
		return javaFrame.isSuspended();
	}
	
	@Override
	public void resume() throws DebugException
	{
		javaFrame.resume();
	}
	
	@Override
	public void suspend() throws DebugException
	{
		javaFrame.suspend();
	}
	
	@Override
	public boolean canTerminate()
	{
		return javaFrame.canTerminate();
	}
	
	@Override
	public boolean isTerminated()
	{
		return javaFrame.isTerminated();
	}
	
	@Override
	public void terminate() throws DebugException
	{
		javaFrame.terminate();
	}
	
	@Override
	public IThread getThread()
	{
		return eglThread;
	}
	
	@Override
	public IEGLJavaThread getEGLThread()
	{
		return eglThread;
	}
	
	@Override
	public synchronized IVariable[] getVariables() throws DebugException
	{
		if ( variables != null )
		{
			return variables;
		}
		
		if ( getSMAP().length() == 0 )
		{
			// Couldn't get the variable info from the SMAP...just return the Java variables.
			variables = javaFrame.getVariables();
		}
		else
		{
			// We could just manually add a "this" variable and a "function" variable, but this way if the language gets
			// extended by some client, e.g. concept of 'static' added to EGL, this would support that.
			List<IEGLJavaVariable> newEGLVariables = VariableUtil.filterAndWrapVariables( javaFrame.getVariables(), this,
					true, null );
			variables = new EGLJavaVariable[ newEGLVariables.size() + 1 ];
			newEGLVariables.toArray( variables );
			variables[ variables.length - 1 ] = getCorrespondingVariable( new EGLJavaFunctionVariable( this ), null );
		}
		
		return variables;
	}
	
	@Override
	public boolean hasVariables() throws DebugException
	{
		return getVariables().length != 0;
	}
	
	@Override
	public int getLineNumber() throws DebugException
	{
		return javaFrame.getLineNumber();
	}
	
	@Override
	public int getCharStart() throws DebugException
	{
		return -1;
	}
	
	@Override
	public int getCharEnd() throws DebugException
	{
		return -1;
	}
	
	@Override
	public String getName() throws DebugException
	{
		return javaFrame.getName();
	}
	
	@Override
	public IRegisterGroup[] getRegisterGroups() throws DebugException
	{
		return javaFrame.getRegisterGroups();
	}
	
	@Override
	public boolean hasRegisterGroups() throws DebugException
	{
		return javaFrame.hasRegisterGroups();
	}
	
	@Override
	public Object getAdapter( Class adapter )
	{
		if ( adapter == IStackFrame.class || adapter == EGLJavaStackFrame.class || adapter == IEGLStackFrame.class || adapter == IDropToFrame.class
				|| adapter == IEGLJavaStackFrame.class )
		{
			return this;
		}
		if ( adapter == IThread.class || adapter == EGLJavaThread.class || adapter == IEGLThread.class || adapter == IEGLJavaThread.class )
		{
			return getThread();
		}
		if ( adapter == IJavaStackFrame.class )
		{
			return javaFrame;
		}
		return super.getAdapter( adapter );
	}
	
	/**
	 * @return the underlying stack frame.
	 */
	public IJavaStackFrame getJavaStackFrame()
	{
		return javaFrame;
	}
	
	/**
	 * @return the underlying Java debug element.
	 */
	@Override
	public Object getJavaDebugElement()
	{
		return getJavaStackFrame();
	}
	
	@Override
	public boolean canDropToFrame()
	{
		return javaFrame.canDropToFrame();
	}
	
	@Override
	public void dropToFrame() throws DebugException
	{
		javaFrame.dropToFrame();
	}
	
	/**
	 * Returns the SMAP information for the stack frame. It will never be null. If there is no SMAP information, or the Java type was not a type that
	 * we recognize, then this will return blank.
	 * 
	 * @return the SMAP information.
	 * @throws DebugException
	 */
	public String getSMAP() throws DebugException
	{
		if ( smap == null )
		{
			smap = SMAPUtil.getSMAP( javaFrame.getReferenceType() );
		}
		return smap;
	}
	
	@Override
	public SMAPVariableInfo[] getSMAPVariableInfos() throws DebugException
	{
		if ( smapVariableInfos == null )
		{
			smapVariableInfos = SMAPUtil.parseVariables( getSMAP(), this );
		}
		return smapVariableInfos;
	}
	
	@Override
	public void setSMAPVariableInfos( SMAPVariableInfo[] infos )
	{
		this.smapVariableInfos = infos;
	}
	
	@Override
	public SMAPFunctionInfo getSMAPFunctionInfo() throws DebugException
	{
		getSMAPVariableInfos(); // Make sure we've parsed the SMAP data
		return smapFunctionInfo;
	}
	
	@Override
	public void setSMAPFunctionInfo( SMAPFunctionInfo info )
	{
		this.smapFunctionInfo = info;
	}
	
	public int getLineBeforeStepInto()
	{
		return this.lineBeforeStepInto;
	}
	
	public void setLineBeforeStepInto( int line )
	{
		this.lineBeforeStepInto = line;
	}
	
	@Override
	public IEGLJavaVariable getCorrespondingVariable( IEGLJavaVariable newVariable, IEGLJavaValue parent ) throws DebugException
	{
		IEGLJavaVariable eglVar = null;
		
		String qualifiedName = VariableUtil.getQualifiedName( newVariable );
		if ( previousEGLVariables != null )
		{
			eglVar = previousEGLVariables.get( qualifiedName );
		}
		
		if ( eglVar != null )
		{
			eglVar.initialize( newVariable, parent );
		}
		else
		{
			eglVar = newVariable;
		}
		
		if ( currentEGLVariables != null )
		{
			currentEGLVariables.put( qualifiedName, eglVar );
		}
		
		return eglVar;
	}
}
