/*
 * Licensed Materials - Property of IBM
 *
 * Copyright IBM Corporation 2008, 2010. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or
 * disclosure restricted by GSA DP Schedule Contract with IBM Corp.
 */
package org.eclipse.edt.ide.ui.internal.actions;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.edt.gen.egl.EglGenerator;
import org.eclipse.edt.ide.core.internal.lookup.ProjectEnvironment;
import org.eclipse.edt.ide.core.internal.lookup.ProjectEnvironmentManager;
import org.eclipse.edt.ide.core.internal.lookup.ProjectIREnvironment;
import org.eclipse.edt.ide.core.utils.EclipseUtilities;
import org.eclipse.edt.ide.generator.egl.Activator;
import org.eclipse.edt.ide.ui.internal.EclipseJava2EGL;
import org.eclipse.edt.mof.serialization.IEnvironment;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.IActionDelegate;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchSite;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.progress.IProgressService;


public class Java2EglAction implements IActionDelegate {

	protected IStructuredSelection fSelection;
	private IWorkbenchSite fSite;	
	
	public void run(IAction action) {
		IResource resource;
		for (Iterator<IResource> iter = fSelection.iterator(); iter.hasNext();) {
			try{
				resource = iter.next();
				if (resource instanceof IFile) {	
					IPath path = resource.getProjectRelativePath();
					IPath filePath = path.removeFirstSegments(1).removeFileExtension();
					ClassLoader loader = new URLClassLoader(new URL[]{resource.getProject().getLocation().append(path.segment(0)).toFile().toURI().toURL()}, null);
					ProjectEnvironment environment = ProjectEnvironmentManager.getInstance().getProjectEnvironment(resource.getProject());
					ProjectIREnvironment env = environment.getIREnvironment();
					env.initSystemEnvironment(environment.getSystemEnvironment()); 
					generateClass(filePath.toOSString().replace(File.separatorChar, '.'), resource.getProject(), loader, env);
				}
				else if(resource instanceof IFolder){
					IPath path = resource.getProjectRelativePath();
					IPath root = resource.getProject().getLocation().append(path.segment(0)); 
					IPath relativePath =  path.removeFirstSegments(1);
					createEglIR(root, resource.getProject(), relativePath);
				}
			}
			catch( Exception e ) {
				e.printStackTrace();
			}
		}	
	}

	void createEglIR(final IPath root, final IProject project, final IPath relativePath){
		final MultiStatus status = new MultiStatus(Activator.PLUGIN_ID, IStatus.OK, "completed", null);
		
		IProgressService progressService = PlatformUI.getWorkbench().getProgressService();

		try{
			progressService.run(true, true, new WorkspaceModifyOperation() {
				@Override
				protected void execute(IProgressMonitor monitor) throws CoreException, InvocationTargetException, InterruptedException {
					
					List<String> classes = ClassFiles.getClassList(root.toOSString(), relativePath.toOSString());
					monitor.beginTask("Creating EGL ir files", classes.size());
					try{
						ClassLoader classLoader = new URLClassLoader(new URL[]{root.toFile().toURI().toURL()}, null);
						IEnvironment env = ProjectEnvironmentManager.getInstance().getProjectEnvironment(project).getIREnvironment();
						for(String className : classes){
							monitor.worked(1);
							monitor.setTaskName(className + ".ir");
							try{
								Java2EglAction.generateClass(className.replace(File.separatorChar, '.'), project, classLoader, env);
							}
							catch(Exception e){
								status.add(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Error create egl file " + className, e ));
							}
						}
					}
					catch(Exception e){
						status.add(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Error creating Class loader to convert Java classes to EGL", e ));
					}
				}
			});		
		}
		catch(InterruptedException ie){
			Activator.log(ie);
		} catch (InvocationTargetException e) {
			Activator.log(e);
		}
		if(!status.isOK() && fSite != null){
			String title = "UINlsStrings.EGLFormatting";
			ErrorDialog dialog = new ErrorDialog(fSite.getShell(), title, null, status, IStatus.ERROR);
			dialog.open();
		}
	}
	static void generateClass(String className, IProject project, ClassLoader loader, IEnvironment env) throws MalformedURLException,
			Exception {
		EclipseJava2EGL gen = new EclipseJava2EGL(loader);
		gen.generate(EclipseJava2EGL.buildArgs(project, className), new EglGenerator(gen), env);
	}

	/**
	 * Notifies this action delegate that the selection in the workbench has changed.
	 * <p>
	 * Implementers can use this opportunity to change the availability of the
	 * action or to modify other presentation properties.
	 * </p>
	 *
	 * @param action the action proxy that handles presentation portion of the action
	 * @param selection the current selection in the workbench
	 */
	public void selectionChanged(IAction action, ISelection selection) {
		fSelection = (IStructuredSelection)selection;
	}
	
	protected void writeFile(String fileData, IFolder root, IPath path) throws CoreException, UnsupportedEncodingException {
		EclipseUtilities.writeFileInEclipse(root.getFolder(path.removeLastSegments(1)), path.removeFirstSegments(path.segmentCount() - 1).addFileExtension("egl"), new ByteArrayInputStream(fileData.getBytes("UTF-8")), true);
	}
	public void setActivePart(IAction action, IWorkbenchPart targetPart) {
		fSite = targetPart.getSite();
	}
	
}
