/*******************************************************************************
 * Copyright © 2000, 2011 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM Corporation - initial API and implementation
 *
 *******************************************************************************/
package org.eclipse.edt.ide.core.internal.model;

import java.util.Map;

import org.eclipse.edt.ide.core.model.EGLModelException;
import org.eclipse.edt.ide.core.model.IEGLElement;


/**
 * Destroys a working copy (remove it from its cache if it is shared)
 * and signal its removal through a delta.
 */
public class DestroyWorkingCopyOperation extends EGLModelOperation {
	
	public DestroyWorkingCopyOperation(IEGLElement workingCopy) {
		super(new IEGLElement[] {workingCopy});
	}
	/**
	 * @exception EGLModelException if setting the source
	 * 	of the original compilation unit fails
	 */
	protected void executeOperation() throws EGLModelException {
		WorkingCopy workingCopy = getWorkingCopy();
		workingCopy.close();
		
		// if original element is not on classpath flush it from the cache 
		IEGLElement originalElement = workingCopy.getOriginalElement();
		if (!workingCopy.getParent().exists()) {
			((EGLFile)originalElement).close();
		}
		
		// remove working copy from the cache if it is shared
		EGLModelManager manager = EGLModelManager.getEGLModelManager();
		
		// In order to be shared, working copies have to denote the same compilation unit 
		// AND use the same buffer factory.
		// Assuming there is a little set of buffer factories, then use a 2 level Map cache.
		Map sharedWorkingCopies = manager.sharedWorkingCopies;
		
		Map perFactoryWorkingCopies = (Map) sharedWorkingCopies.get(workingCopy.bufferFactory);
		if (perFactoryWorkingCopies != null){
			if (perFactoryWorkingCopies.remove(originalElement) != null
					&& EGLFile.SHARED_WC_VERBOSE) {
				System.out.println("Destroying shared working copy " + workingCopy.toStringWithAncestors());//$NON-NLS-1$
			}
		}
		
		// report removed egl delta
		EGLElementDelta delta = new EGLElementDelta(this.getEGLModel());
		delta.removed(workingCopy);
		addDelta(delta);
		removeReconcileDelta(workingCopy);
	}
	/**
	 * Returns the working copy this operation is working on.
	 */
	protected WorkingCopy getWorkingCopy() {
		return (WorkingCopy)getElementToProcess();
	}
	/**
	 * @see EGLModelOperation#isReadOnly
	 */
	public boolean isReadOnly() {
		return true;
	}
}
