/*******************************************************************************
 * Copyright (c) 2010, 2011 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - Bug 331203 - table model editor - initial API and implementation
 *    Nicolas Bros (Mia-Software) - Bug 332437 - NatTable : pluggable cell editors
 *    Nicolas Guyomar (Mia-Software) - Bug 331442 - To be able to add and remove lines (model elements) from the table
 *    Nicolas Bros (Mia-Software) - Bug 332226 - To be able to create or delete model element from the table
 *    Nicolas Guyomar (Mia-Software) - Bug 332226 - To be able to create or delete model element from the table
 *    Nicolas Bros (Mia-Software) - Bug 332438 - NatTable : table type
 *    Nicolas Bros (Mia-Software) - Bug 332440 - NatTable : force cell editors
 *    Nicolas Bros (Mia-Software) - Bug 331675 - NatTable : copy cells as text 
 *    Nicolas Bros (Mia-Software) - Bug 331900 - customizable NatTable
 *    Nicolas Bros (Mia-Software) - Bug 332010 - view Facet customizations on NatTable
 *    Nicolas Bros (Mia-Software) - Bug 332215 - customizable NatTable column headers
 *    Nicolas Guyomar (Mia-Software) - Bug 332215 - customizable NatTable column headers
 *    Nicolas Guyomar (Mia-Software) - Bug 332924 - To be able to save the table
 *    Nicolas Guyomar (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *    Gregoire Dupe (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *    Gregoire Dupe (Mia-Software) - Bug 333015 - To be able to hide columns
 *    Nicolas Guyomar (Mia-Software) - Bug 333015 - To be able to hide columns
 *    Nicolas Guyomar (Mia-Software) - Bug 333029 - To be able to save the size of the lines and the columns
 *    Nicolas Guyomar (Mia-Software) - Bug 333414 - The user must be able to save the column order
 *    Nicolas Guyomar (Mia-Software) - Bug 335154 - Sort Column By Type : Cannot modify resource set without a write transaction
 *    Nicolas Guyomar (Mia-Software) - Bug 335155 - Hide Empty Column : Cannot modify resource set without a write transaction
 *    Nicolas Guyomar (Mia-Software) - Bug 335156 - Only Show Common column : Cannot modify resource set without a write transaction
 *    Nicolas Guyomar (Mia-Software) - Bug 335020 - Nattable widget should use the Eclipse framework
 *    Nicolas Guyomar (Mia-Software) - Bug 337322 - [TableConfiguration] Customization declared in the file tableconfiguration is not loaded
 *    Nicolas Guyomar (Mia-Software) - Bug 338536 - Problem with the refresh of the table : the scrollbar returns to this initial position
 *    Nicolas Bros (Mia-Software) - Bug 338536 - Problem with the refresh of the table : the scrollbar returns to this initial position
 *    Nicolas Guyomar (Mia-Software) - Bug 337454 - We can't delete a query Column
 *    Nicolas Guyomar (Mia-Software) - Bug 337395 - Unused columns should be destroyed
 *    Nicolas Guyomar (Mia-Software) - Bug 339554 - org.eclipse.emf.facet.widgets.celleditors API cleaning
 *    Nicolas Guyomar (Mia-Software) - Bug 339922 - INatTableWidget method isCellSelected should use the plural
 *    Nicolas Guyomar (Mia-Software) - Bug 340681 - Facet column implementation
 *    Vincent Lorenzo (CEA-LIST) - Bug 337326 - Show/Hide Column : Sort the columns by name 
 *    Nicolas Guyomar (Mia-Software) - Bug 340940 - To be able to view facet attributes and facet references in a table
 *    Vincent Lorenzo (CEA-LIST) - Bug 337408 - Add an action to sort columns by name 
 *    Nicolas Guyomar (Mia-Software) - Bug 336482 - KeyBinding to edit element in Table : F2 
 *    Vincent Lorenzo (CEA-LIST) - Bug 341238 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Gregoire Dupe (Mia-Software) - Bug 341238 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Nicolas Guyomar (Mia-Software) - Bug 342451 - To be able to edit derived facet attributes and derived facet references in a table
 *    Vincent Lorenzo (CEA-LIST) - Bug 341238 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Nicolas Guyomar (Mia-Software) - Bug 343411 - [Table] Create new elements does not support IJavaModelQuery anymore
 *    Gregoire Dupe (Mia-Software) - Bug 343811 - EMF Facet Regression : Created elements in a table are not serialized
 *    Vincent Lorenzo (CEA-LIST) - Bug 344125 - The API should provide a method selectRows(List<EObject> elementsToSelect)
 *    Nicolas Guyomar (Mia-Software) - Bug 344274 - SWT BOT fail on Hudson
 *    Nicolas Guyomar (Mia-Software) - Bug 344475 - To be able to select a cell by EStructuralFeature in the table
 *    Nicolas Guyomar (Mia-Software) - Bug 342028 - Field can be edited even if they are marked as N/A
 *    Nicolas Guyomar (Mia-Software) - Bug 344413 - Facet Columns are never created when we begin with an empty table
 *    Gregoire Dupe (Mia-Software) - Bug 343859 - The local customizations are not applied when we reopen a table
 *    Nicolas Guyomar (Mia-Software) - Bug 344670 - Problems with the columns creation : very slow + scrollbar blinked
 *    Vincent Lorenzo (CEA LIST) - Bug 341238 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Nicolas Guyomar (Mia-Software) - Bug 344925 - Undo/Redo after the action Show Columns
 *    Gregoire Dupe (Mia-Software) - Bug 344925 - Undo/Redo after the action Show Columns - Regression fix
 *    Nicolas Guyomar (Mia-Software) - Bug 345665 - Columns are duplicated when you drop many elements in the same time
 *******************************************************************************/
package org.eclipse.emf.facet.widgets.nattable.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import net.sourceforge.nattable.coordinate.PositionCoordinate;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CompoundCommand;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.facet.infra.browser.custom.CustomView;
import org.eclipse.emf.facet.infra.browser.custom.CustomViewFeature;
import org.eclipse.emf.facet.infra.browser.custom.CustomizableFeatures;
import org.eclipse.emf.facet.infra.browser.custom.MetamodelView;
import org.eclipse.emf.facet.infra.browser.custom.StaticFeatureValue;
import org.eclipse.emf.facet.infra.browser.custom.TypeView;
import org.eclipse.emf.facet.infra.browser.custom.emf.UicustomFactory;
import org.eclipse.emf.facet.infra.browser.custom.emf.UicustomPackage;
import org.eclipse.emf.facet.infra.browser.custom.util.UicustomUtil;
import org.eclipse.emf.facet.infra.common.core.internal.utils.ModelUtils;
import org.eclipse.emf.facet.infra.facet.Facet;
import org.eclipse.emf.facet.infra.facet.FacetAttribute;
import org.eclipse.emf.facet.infra.facet.FacetReference;
import org.eclipse.emf.facet.infra.query.core.exception.ModelQueryException;
import org.eclipse.emf.facet.widgets.celleditors.ICommandFactoriesRegistry;
import org.eclipse.emf.facet.widgets.celleditors.ICommandFactory;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.AttributeColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.Column;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FacetAttributeColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FacetReferenceColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FeatureColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.ReferenceColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.Row;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.TableInstance;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.TableinstanceFactory;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.TableinstancePackage;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance2.TableInstance2;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance2.Tableinstance2Package;
import org.eclipse.jface.viewers.IStructuredSelection;

public final class TableInstanceCommandFactory {

	public static Command createLoadCustomizationsCommand(
			final List<MetamodelView> customizationsToLoad, final NatTableWidget natTableWidget) {
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		List<MetamodelView> localCustomizations = natTableWidget.getLocalCustomizations();
		if (!localCustomizations.isEmpty()
				&& !customizationsToLoad.containsAll(localCustomizations)) {
			// the locals custom must always be set here
			customizationsToLoad.addAll(0, localCustomizations);
		}
		if (!customizationsToLoad.isEmpty()) {
			return commandFactory.createSetCommand(natTableWidget.getEditingDomain(),
					natTableWidget.getTableInstance(),
					TableinstancePackage.eINSTANCE.getTableInstance_Customizations(),
					customizationsToLoad);
		}
		return null;
	}

	public static Command createDeleteSelectedColumnCommand(final NatTableWidget natTableWidget) {
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		CompoundCommand compoundCommand = new CompoundCommand();

		for (Column column : natTableWidget.getSelectedColumns()) {
			Command removeRowCommand = commandFactory.createRemoveCommand(
					natTableWidget.getEditingDomain(), natTableWidget.getTableInstance(),
					TableinstancePackage.eINSTANCE.getTableInstance_Columns(), column);
			compoundCommand.append(removeRowCommand);
		}
		return compoundCommand;
	}

	public static Command createDeleteSelectedElementCommand(final NatTableWidget natTableWidget,
			final ICommandFactory commandFactory) {
		List<Integer> toBeDeleted = null;
		CompoundCommand compoundCommand = new CompoundCommand();
		toBeDeleted = new ArrayList<Integer>();
		PositionCoordinate[] selectedCells = natTableWidget.getBodyLayer().getSelectionLayer()
				.getSelectedCells();

		for (PositionCoordinate positionCoordinate : selectedCells) {
			if (!toBeDeleted.contains(positionCoordinate.rowPosition)) {
				toBeDeleted.add(positionCoordinate.rowPosition);
			}
		}
		TableInstance tableInstance = natTableWidget.getTableInstance();
		for (int i = 0; i < tableInstance.getRows().size(); i++) {
			if (toBeDeleted.contains(i)) {
				Command deleteEltCommand = commandFactory.createDeleteCommand(
						natTableWidget.getEditingDomain(), tableInstance.getRows().get(i)
								.getElement());
				Command removeRowCommand = commandFactory.createRemoveCommand(natTableWidget
						.getEditingDomain(), tableInstance, TableinstancePackage.eINSTANCE
						.getTableInstance_Rows(), tableInstance.getRows().get(i));
				compoundCommand.append(removeRowCommand);
				compoundCommand.append(deleteEltCommand);
			}
		}
		return compoundCommand;
	}

	public static Command createRemoveLineCommand(final NatTableWidget natTableWidget) {
		CompoundCommand compoundCommand = new CompoundCommand();
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		TableInstance tableInstance = natTableWidget.getTableInstance();
		List<Integer> toBeRemoved = new ArrayList<Integer>();
		PositionCoordinate[] selectedCells = natTableWidget.getBodyLayer().getSelectionLayer()
				.getSelectedCells();

		for (PositionCoordinate positionCoordinate : selectedCells) {
			if (!toBeRemoved.contains(positionCoordinate.rowPosition)) {
				toBeRemoved.add(positionCoordinate.rowPosition);
			}
		}
		for (int i = 0; i < tableInstance.getElements().size(); i++) {
			if (toBeRemoved.contains(i)) {
				Command removeRowCommand = commandFactory.createRemoveCommand(natTableWidget
						.getEditingDomain(), tableInstance, TableinstancePackage.eINSTANCE
						.getTableInstance_Rows(), tableInstance.getRows().get(i));
				compoundCommand.append(removeRowCommand);
			}
		}
		return compoundCommand;
	}

	public static CompoundCommand createAddRowsCommand(final List<EObject> newElements,
			final NatTableWidget natTableWidget) {
		CompoundCommand compoundCommand = new CompoundCommand();
		// the EPackage for which the MetamodelView has already been created
		Set<EPackage> alreadyDone = new HashSet<EPackage>();
		TableInstance tableInstance = natTableWidget.getTableInstance();
		EditingDomain editingDomain = natTableWidget.getEditingDomain();
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		List<EStructuralFeature> processedEStructuralFeatures = new ArrayList<EStructuralFeature>();
		for (EObject eObject : newElements) {
			if (!tableInstance.getElements().contains(eObject)) {
				Row row = TableinstanceFactory.eINSTANCE.createRow();
				Command setEobjectToRowCMD = commandFactory.createSetCommand(editingDomain, row,
						TableinstancePackage.eINSTANCE.getRow_Element(), eObject);
				compoundCommand.append(setEobjectToRowCMD);
				Command addRowCMD = commandFactory.createAddCommand(editingDomain, tableInstance,
						TableinstancePackage.eINSTANCE.getTableInstance_Rows(), row);
				compoundCommand.append(addRowCMD);
				for (EStructuralFeature eStructuralFeature : eObject.eClass()
						.getEAllStructuralFeatures()) {
					if (!processedEStructuralFeatures.contains(eStructuralFeature)) {
						processedEStructuralFeatures.add(eStructuralFeature);
						if (!natTableWidget.isColumnAlreadyDeclared(eStructuralFeature)) {
							if (eStructuralFeature instanceof EReference) {
								ReferenceColumn referenceColumn = TableinstanceFactory.eINSTANCE
										.createReferenceColumn();
								referenceColumn.setReference((EReference) eStructuralFeature);
								Command cmd = commandFactory.createAddCommand(editingDomain,
										tableInstance,
										TableinstancePackage.eINSTANCE.getTableInstance_Columns(),
										referenceColumn);
								compoundCommand.append(cmd);
	
							} else if (eStructuralFeature instanceof EAttribute) {
								AttributeColumn attributeColumn = TableinstanceFactory.eINSTANCE
										.createAttributeColumn();
								attributeColumn.setAttribute((EAttribute) eStructuralFeature);
								Command cmd = commandFactory.createAddCommand(editingDomain,
										tableInstance,
										TableinstancePackage.eINSTANCE.getTableInstance_Columns(),
										attributeColumn);
								compoundCommand.append(cmd);
							}
	
							// we add the local customization file
							if (tableInstance instanceof TableInstance2) {
								List<MetamodelView> localCustoms = natTableWidget
										.getLocalCustomizations();
								EObject container = eStructuralFeature.eContainer();
								if (container != null) {
									container = container.eContainer();
									if (container instanceof EPackage) {
										if (!alreadyDone.contains(container)) {
											if (UicustomUtil.getMetamodelViewByEPackage(localCustoms,
													(EPackage) container) == null) {
												Command cmd = createMetamodelViewCommand(
														((EPackage) container).getNsURI(),
														natTableWidget);
												if (cmd.canExecute()) {
													compoundCommand.append(cmd);
												}
												alreadyDone.add((EPackage) container);
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
		return compoundCommand;
	}

	private static Command createMetamodelViewCommand(final String nsURI,
			final NatTableWidget natTableWidget) {
		CompoundCommand compoundCommand = new CompoundCommand();
		TableInstance tableInstance = natTableWidget.getTableInstance();
		EditingDomain editingDomain = natTableWidget.getEditingDomain();
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		MetamodelView newMetamodelView = UicustomFactory.eINSTANCE.createMetamodelView();
		Command createMetamodelView = commandFactory.createSetCommand(editingDomain,
				newMetamodelView, UicustomPackage.eINSTANCE.getMetamodelView_MetamodelURI(), nsURI);
		compoundCommand.append(createMetamodelView);

		if (tableInstance instanceof TableInstance2) {
			Command setMetamodelViewLocalCmd = commandFactory.createAddCommand(editingDomain,
					tableInstance,
					Tableinstance2Package.eINSTANCE.getTableInstance2_LocalCustomizations(),
					newMetamodelView);
			compoundCommand.append(setMetamodelViewLocalCmd);

			List<MetamodelView> views = new ArrayList<MetamodelView>();
			views.addAll(tableInstance.getCustomizations());
			// we look for the index of the local new custom
			List<MetamodelView> localCustom = natTableWidget.getLocalCustomizations();
			int i = 0;
			for (; i < views.size(); i++) {
				if (localCustom.contains(views.get(i))) {
					break;
				}
			}
			views.add(i, newMetamodelView);
			Command setMetamodelViewCmd = commandFactory.createSetCommand(editingDomain,
					tableInstance,
					TableinstancePackage.eINSTANCE.getTableInstance_Customizations(), views);
			compoundCommand.append(setMetamodelViewCmd);

		} else {
			// TODO should be removed when all tables will be TableInstance2
			Command setMetamodelView = commandFactory.createSetCommand(editingDomain,
					tableInstance,
					TableinstancePackage.eINSTANCE.getTableInstance_LocalCustomization(),
					newMetamodelView);
			compoundCommand.append(setMetamodelView);
		}
		return compoundCommand;
	}

	public static Command createShowHideColumnCommand(final NatTableWidget natTableWidget,
			final List<Column> columnsToShow, final List<Column> columnsToHide,
			final boolean putOnTheTop) {
		HashMap<EClass, TypeView> typeViewMap = new HashMap<EClass, TypeView>();
		CompoundCommand compoundCommand = new CompoundCommand();
		TableInstance tableInstance = natTableWidget.getTableInstance();
		EditingDomain editingDomain = natTableWidget.getEditingDomain();
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		if (tableInstance instanceof TableInstance2) {
			for (Column current : columnsToShow) {
				if (current instanceof FeatureColumn) {
					Command tmp = getShowHideColumnCommand(natTableWidget, current, false, typeViewMap);
					if (tmp != null && tmp.canExecute()) {
						compoundCommand.append(tmp);
					}
				} else {
					Command cmd = commandFactory.createSetCommand(editingDomain, current,
							TableinstancePackage.eINSTANCE.getColumn_IsHidden(), false);
					if (cmd.canExecute()) {
						compoundCommand.append(cmd);
					}
				}
			}

			for (Column current : columnsToHide) {
				if (current instanceof FeatureColumn) {
					Command tmp = getShowHideColumnCommand(natTableWidget, current, true, typeViewMap);
					if (tmp != null && tmp.canExecute()) {
						compoundCommand.append(tmp);
					}
				} else {
					Command cmd = commandFactory.createSetCommand(
							natTableWidget.getEditingDomain(), current,
							TableinstancePackage.eINSTANCE.getColumn_IsHidden(), true);
					if (cmd.canExecute()) {
						compoundCommand.append(cmd);
					}
				}
			}
		} else {
			for (Column current : columnsToShow) {
				Command cmd = commandFactory.createSetCommand(editingDomain, current,
						TableinstancePackage.eINSTANCE.getColumn_IsHidden(), false);
				if (cmd.canExecute()) {
					compoundCommand.append(cmd);
				}
			}
			for (Column current : columnsToHide) {
				Command cmd = commandFactory.createSetCommand(editingDomain, current,
						TableinstancePackage.eINSTANCE.getColumn_IsHidden(), true);
				if (cmd.canExecute()) {
					compoundCommand.append(cmd);
				}
			}
		}
		if (putOnTheTop) {
			Command cmd = createPutLocalCustomizationOnTheTopCommand(natTableWidget);
			if (cmd.canExecute()) {
				compoundCommand.append(cmd);
			}
		}
		return compoundCommand;
	}

	public static Command createPutLocalCustomizationOnTheTopCommand(
			final NatTableWidget natTableWidget) {
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		List<MetamodelView> locals = natTableWidget.getLocalCustomizations();
		List<MetamodelView> views = new ArrayList<MetamodelView>();
		views.addAll(natTableWidget.getTableInstance().getCustomizations());
		views.removeAll(locals);
		views.addAll(0, locals);
		return commandFactory.createSetCommand(natTableWidget.getEditingDomain(),
				natTableWidget.getTableInstance(),
				TableinstancePackage.eINSTANCE.getTableInstance_Customizations(), views);
	}

	public static Command createSetFacetsCommand(final Collection<Facet> facets,
			final NatTableWidget natTableWidget) throws CoreException {
		return createSetFacetsCommand(facets, null, natTableWidget);
	}

	public static Command createSetFacetsCommand(final Collection<Facet> facets,
			final List<EObject> newElements, final NatTableWidget natTableWidget)
			throws CoreException {
		CompoundCommand compoundCommand = new CompoundCommand();
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		TableInstance2 tableInstance = (TableInstance2) natTableWidget.getTableInstance();
		// Add new facets
		for (Facet facet : facets) {
			if (!tableInstance.getFacets2().contains(facet)) {
				Command command = commandFactory.createAddCommand(
						natTableWidget.getEditingDomain(), tableInstance,
						Tableinstance2Package.eINSTANCE.getTableInstance2_Facets2(), facet);
				compoundCommand.append(command);
			}
		}

		// Remove non referenced one
		for (Facet facet : tableInstance.getFacets2()) {
			if (!facets.contains(facet)) {
				Command command = commandFactory.createRemoveCommand(
						natTableWidget.getEditingDomain(), tableInstance,
						Tableinstance2Package.eINSTANCE.getTableInstance2_Facets2(), facet);
				compoundCommand.append(command);
			}
		}

		final Set<EReference> eReferences = new HashSet<EReference>();
		final Set<EAttribute> eAttributes = new HashSet<EAttribute>();
		List<IStatus> statusList = new ArrayList<IStatus>();
		List<EObject> elementsInTable = new ArrayList<EObject>();
		elementsInTable.addAll(tableInstance.getElements());
		if (newElements != null) {
			elementsInTable.addAll(newElements);
		}
		for (EObject eObject : elementsInTable) {
			try {
				eAttributes.addAll(natTableWidget.getFacetContext().getAttributes(eObject));
				eReferences.addAll(natTableWidget.getFacetContext().getReferences(eObject));
			} catch (ModelQueryException e) {
				IStatus status = new Status(
						IStatus.ERROR,
						Activator.PLUGIN_ID,
						"An exception has occured while retrieving structural features of:" + eObject, e); //$NON-NLS-1$
				statusList.add(status);
			}
		}
		if (!statusList.isEmpty()) {
			IStatus globalStatus = new MultiStatus(
					Activator.PLUGIN_ID,
					0,
					"An exception has occured while retrieving structural features of the eObjects", //$NON-NLS-1$
					new Exception());
			throw new CoreException(globalStatus);
		}

		// Remove outdated columns
		for (Column c : tableInstance.getColumns()) {
			if (c instanceof FacetReferenceColumn) {
				FacetReferenceColumn facetReferenceColumn = (FacetReferenceColumn) c;
				if (!eReferences.contains(facetReferenceColumn.getReference())) {
					// Remove the column
					Command command = commandFactory.createRemoveCommand(
							natTableWidget.getEditingDomain(), tableInstance,
							TableinstancePackage.eINSTANCE.getTableInstance_Columns(), c);
					compoundCommand.append(command);
				}
			} else if (c instanceof FacetAttributeColumn) {
				FacetAttributeColumn facetAttributeColumn = (FacetAttributeColumn) c;
				if (!eAttributes.contains(facetAttributeColumn.getAttribute())) {
					// Remove the column
					Command command = commandFactory.createRemoveCommand(
							natTableWidget.getEditingDomain(), tableInstance,
							TableinstancePackage.eINSTANCE.getTableInstance_Columns(), c);
					compoundCommand.append(command);
				}
			}
		}

		for (EAttribute eAttribute : eAttributes) {
			if (!natTableWidget.isColumnAlreadyDeclared(eAttribute)) {
				FacetAttributeColumn facetAttributeColumn = TableinstanceFactory.eINSTANCE
						.createFacetAttributeColumn();
				facetAttributeColumn.setAttribute((FacetAttribute) eAttribute);
				Command command = commandFactory.createAddCommand(
						natTableWidget.getEditingDomain(), tableInstance,
						TableinstancePackage.eINSTANCE.getTableInstance_Columns(),
						facetAttributeColumn);
				compoundCommand.append(command);
			}
		}

		for (EReference eReference : eReferences) {
			if (!natTableWidget.isColumnAlreadyDeclared(eReference)) {
				FacetReferenceColumn facetReferenceColumn = TableinstanceFactory.eINSTANCE
						.createFacetReferenceColumn();
				facetReferenceColumn.setReference((FacetReference) eReference);
				Command command = commandFactory.createAddCommand(
						natTableWidget.getEditingDomain(), tableInstance,
						TableinstancePackage.eINSTANCE.getTableInstance_Columns(),
						facetReferenceColumn);
				compoundCommand.append(command);
			}
		}
		if (!compoundCommand.isEmpty()) {
			return compoundCommand;
		}
		return null;
	}

	public static CompoundCommand createDropCommand(final NatTableWidget natTableWidget,
			final IStructuredSelection structuredSelection, final Object element,
			final EStructuralFeature feature) {
		EditingDomain editingDomain = natTableWidget.getEditingDomain();
		if (editingDomain == null) {
			return null;
		}
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		CompoundCommand compoundCommand = new CompoundCommand();
		if (feature.isMany()) {
			Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				Object object = iterator.next();
				Command addCommand = commandFactory.createAddCommand(editingDomain, element,
						feature, object);
				compoundCommand.append(addCommand);
			}
		} else {
			if (structuredSelection.toList().size() == 1) {
				Object firstElement = structuredSelection.getFirstElement();
				Command addCommand = commandFactory.createSetCommand(editingDomain, element,
						feature, firstElement);
				compoundCommand.append(addCommand);
			}
		}
		return compoundCommand;
	}

	/**
	 * 
	 * @param column
	 *            a column
	 * @param newValue
	 *            the new feature value for this column
	 * @param customizedFeatureName
	 *            the customized feature name
	 * @return
	 */
	public static Command getShowHideColumnCommand(final NatTableWidget natTableWidget,
			final Column column, final Boolean newValue, final HashMap<EClass, TypeView> typeViewMap) {
		Command cmd = null;
		if (column instanceof FeatureColumn
				&& natTableWidget.getTableInstance() instanceof TableInstance2) {
			FeatureColumn featureColumn = (FeatureColumn) column;
			String customizedFeatureName = "visible"; //$NON-NLS-1$
			CompoundCommand comCommand = new CompoundCommand();
			EditingDomain editingDomain = natTableWidget.getEditingDomain();
			ICommandFactory commandFactory = ICommandFactoriesRegistry.INSTANCE
					.getCommandFactoryFor(editingDomain);
			EStructuralFeature featureOfTheColumn = featureColumn.getFeature();
			String customViewName = featureOfTheColumn.getName();
			MetamodelView metamodelView = null;

			EObject container = featureOfTheColumn.eContainer().eContainer();
			if (container instanceof EPackage) {
				// 1 we look for the corresponding metamodel view
				List<MetamodelView> metamodels = new ArrayList<MetamodelView>();
				metamodels = ((TableInstance2) natTableWidget.getTableInstance())
						.getLocalCustomizations();
				metamodelView = UicustomUtil.getMetamodelViewByEPackage(metamodels,
						(EPackage) container);
			}
			// 2 : for each element in the table, we look for if its exists
			// already a customization
			if (metamodelView != null) {
				// should not be null, because it is created when we add an
				// object in the table we write that the column is hidden on the element which
				// provides the feature :
				EStructuralFeature feature = featureColumn.getFeature();
				EObject featureContainer = feature.eContainer();
				TypeView typeView = UicustomUtil.getTypeViewByQualifiedName(
						metamodelView.getTypes(), (EClass) featureContainer);
				CustomView customView = null;
				StaticFeatureValue featureValue = null;
				CustomViewFeature customViewFeature = null;
				CustomizableFeatures customizableFeature = null;
				// 4 We look for the customView
				if (typeView == null) {
					typeView = typeViewMap.get(featureContainer);
				}
				if (typeView != null) {
					if (featureColumn instanceof AttributeColumn
							|| featureColumn instanceof FacetAttributeColumn) {
						customView = UicustomUtil.getAttributeViewByEStructuralFeature(
								typeView.getAttributes(), featureColumn.getFeature());
						if (customView != null) {
							// 5 we look for the customViewFeature
							customViewFeature = UicustomUtil.getCustomViewFeatureByName(
									customView.getCustomizedFeatures(), customizedFeatureName);
							if (customViewFeature != null) {
								customizableFeature = customViewFeature.getCustomizedFeature();
							}
						}
					} else if (featureColumn instanceof ReferenceColumn
							|| featureColumn instanceof FacetReferenceColumn) {
						customView = UicustomUtil.getReferenceViewByEStructuralFeature(
								typeView.getReferences(), featureColumn.getFeature());
						if (customView != null) {
							// 5 we look for the customViewFeature
							customViewFeature = UicustomUtil.getCustomViewFeatureByName(
									customView.getCustomizedFeatures(), customizedFeatureName);
							if (customViewFeature != null) {
								customizableFeature = customViewFeature.getCustomizedFeature();
							}
						}
					}
				}
				// we create the elements which are null!
				if (typeView == null) {
					typeView = UicustomPackage.eINSTANCE.getUicustomFactory().createTypeView();
					typeViewMap.put((EClass) featureContainer, typeView);
					Command cmd0 = commandFactory.createAddCommand(editingDomain, metamodelView,
							UicustomPackage.eINSTANCE.getMetamodelView_Types(), typeView);
					String elementQN = ModelUtils
							.getMetaclassQualifiedName((EClass) featureContainer);
					Command cmd1 = commandFactory.createSetCommand(editingDomain, typeView,
							UicustomPackage.eINSTANCE.getTypeView_MetaclassName(), elementQN);
					comCommand.append(cmd0);
					comCommand.append(cmd1);
				}

				// we apply the behavior to the sub-instance
				Command subInstanceCommand = commandFactory.createSetCommand(editingDomain,
						typeView, UicustomPackage.eINSTANCE.getTypeView_AppliesToSubInstances(),
						new Boolean(true));
				comCommand.append(subInstanceCommand);

				if (customView == null) {
					if (featureColumn instanceof AttributeColumn
							|| featureColumn instanceof FacetAttributeColumn) {
						customView = UicustomPackage.eINSTANCE.getUicustomFactory()
								.createAttributeView();
						Command cmd2 = commandFactory.createAddCommand(editingDomain, typeView,
								UicustomPackage.eINSTANCE.getTypeView_Attributes(), customView);
						comCommand.append(cmd2);
						Command cmd3 = commandFactory.createSetCommand(editingDomain, customView,
								UicustomPackage.eINSTANCE.getAttributeView_AttributeName(),
								customViewName);
						comCommand.append(cmd3);
					} else if (featureColumn instanceof ReferenceColumn
							|| featureColumn instanceof FacetReferenceColumn) {

						customView = UicustomPackage.eINSTANCE.getUicustomFactory()
								.createReferenceView();
						Command cmd2 = commandFactory.createAddCommand(editingDomain, typeView,
								UicustomPackage.eINSTANCE.getTypeView_References(), customView);
						comCommand.append(cmd2);
						Command cmd3 = commandFactory.createSetCommand(editingDomain, customView,
								UicustomPackage.eINSTANCE.getReferenceView_ReferenceName(),
								customViewName);
						comCommand.append(cmd3);
					}
				}
				if (customViewFeature == null) {
					customViewFeature = UicustomPackage.eINSTANCE.getUicustomFactory()
							.createCustomViewFeature();
					Command cmd4 = commandFactory.createAddCommand(editingDomain, customView,
							UicustomPackage.eINSTANCE.getCustomView_CustomizedFeatures(),
							customViewFeature);
					comCommand.append(cmd4);
				}
				if (customizableFeature == null) {
					customizableFeature = CustomizableFeatures.getByName(customizedFeatureName);
					Command cmd5 = commandFactory.createSetCommand(editingDomain,
							customViewFeature,
							UicustomPackage.eINSTANCE.getCustomViewFeature_CustomizedFeature(),
							customizableFeature);
					comCommand.append(cmd5);
				}

				featureValue = UicustomPackage.eINSTANCE.getUicustomFactory()
						.createStaticFeatureValue();
				Command cmd6 = commandFactory.createSetCommand(editingDomain, featureValue,
						UicustomPackage.eINSTANCE.getStaticFeatureValue_Value(),
						Boolean.toString(!newValue));
				comCommand.append(cmd6);

				Command cmd7 = commandFactory
						.createSetCommand(editingDomain, customViewFeature,
								UicustomPackage.eINSTANCE.getCustomViewFeature_DefaultValue(),
								featureValue);
				comCommand.append(cmd7);

			}
			return comCommand;
		}
		return cmd;
	}

	private TableInstanceCommandFactory() {
		// Prevents instantiation
	}

}
