/*******************************************************************************
 * Copyright (c) 2010, 2011 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - Bug 331203 - table model editor - initial API and implementation
 *    Nicolas Bros (Mia-Software) - Bug 332437 - NatTable : pluggable cell editors
 *    Nicolas Guyomar (Mia-Software) - Bug 331442 - To be able to add and remove lines (model elements) from the table
 *    Nicolas Bros (Mia-Software) - Bug 332226 - To be able to create or delete model element from the table
 *    Nicolas Guyomar (Mia-Software) - Bug 332226 - To be able to create or delete model element from the table
 *    Nicolas Bros (Mia-Software) - Bug 332438 - NatTable : table type
 *    Nicolas Bros (Mia-Software) - Bug 332440 - NatTable : force cell editors
 *    Nicolas Bros (Mia-Software) - Bug 331675 - NatTable : copy cells as text
 *    Nicolas Bros (Mia-Software) - Bug 331900 - customizable NatTable
 *    Nicolas Bros (Mia-Software) - Bug 332010 - view Facet customizations on NatTable
 *    Nicolas Bros (Mia-Software) - Bug 332215 - customizable NatTable column headers
 *    Nicolas Guyomar (Mia-Software) - Bug 332215 - customizable NatTable column headers
 *    Nicolas Guyomar (Mia-Software) - Bug 332924 - To be able to save the table
 *    Nicolas Guyomar (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *    Gregoire Dupe (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *    Gregoire Dupe (Mia-Software) - Bug 333015 - To be able to hide columns
 *    Nicolas Guyomar (Mia-Software) - Bug 333015 - To be able to hide columns
 *    Nicolas Guyomar (Mia-Software) - Bug 333029 - To be able to save the size of the lines and the columns
 *    Nicolas Guyomar (Mia-Software) - Bug 333414 - The user must be able to save the column order
 *    Nicolas Guyomar (Mia-Software) - Bug 335154 - Sort Column By Type : Cannot modify resource set without a write transaction
 *    Nicolas Guyomar (Mia-Software) - Bug 335155 - Hide Empty Column : Cannot modify resource set without a write transaction
 *    Nicolas Guyomar (Mia-Software) - Bug 335156 - Only Show Common column : Cannot modify resource set without a write transaction
 *    Nicolas Guyomar (Mia-Software) - Bug 335020 - Nattable widget should use the Eclipse framework
 *    Nicolas Guyomar (Mia-Software) - Bug 337322 - [TableConfiguration] Customization declared in the file tableconfiguration is not loaded
 *    Nicolas Guyomar (Mia-Software) - Bug 338536 - Problem with the refresh of the table : the scrollbar returns to this initial position
 *    Nicolas Bros (Mia-Software) - Bug 338536 - Problem with the refresh of the table : the scrollbar returns to this initial position
 *    Nicolas Guyomar (Mia-Software) - Bug 337454 - We can't delete a query Column
 *    Nicolas Guyomar (Mia-Software) - Bug 337395 - Unused columns should be destroyed
 *    Nicolas Guyomar (Mia-Software) - Bug 339554 - org.eclipse.emf.facet.widgets.celleditors API cleaning
 *    Nicolas Guyomar (Mia-Software) - Bug 339922 - INatTableWidget method isCellSelected should use the plural
 *    Nicolas Guyomar (Mia-Software) - Bug 340681 - Facet column implementation
 *    Vincent Lorenzo (CEA-LIST) - Bug 337326 - Show/Hide Column : Sort the columns by name
 *    Nicolas Guyomar (Mia-Software) - Bug 340940 - To be able to view facet attributes and facet references in a table
 *    Vincent Lorenzo (CEA-LIST) - Bug 337408 - Add an action to sort columns by name
 *    Nicolas Guyomar (Mia-Software) - Bug 336482 - KeyBinding to edit element in Table : F2
 *    Vincent Lorenzo (CEA-LIST) - Bug 341238 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Gregoire Dupe (Mia-Software) - Bug 341238 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Nicolas Guyomar (Mia-Software) - Bug 342451 - To be able to edit derived facet attributes and derived facet references in a table
 *    Vincent Lorenzo (CEA-LIST) - Bug 341238 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Nicolas Guyomar (Mia-Software) - Bug 343411 - [Table] Create new elements does not support IJavaModelQuery anymore
 *    Gregoire Dupe (Mia-Software) - Bug 343811 - EMF Facet Regression : Created elements in a table are not serialized
 *    Vincent Lorenzo (CEA-LIST) - Bug 344125 - The API should provide a method selectRows(List<EObject> elementsToSelect)
 *    Nicolas Guyomar (Mia-Software) - Bug 344274 - SWT BOT fail on Hudson
 *    Nicolas Guyomar (Mia-Software) - Bug 344475 - To be able to select a cell by EStructuralFeature in the table
 *    Nicolas Guyomar (Mia-Software) - Bug 342028 - Field can be edited even if they are marked as N/A
 *    Nicolas Guyomar (Mia-Software) - Bug 344413 - Facet Columns are never created when we begin with an empty table
 *    Gregoire Dupe (Mia-Software) - Bug 343859 - The local customizations are not applied when we reopen a table
 *    Nicolas Guyomar (Mia-Software) - Bug 344670 - Problems with the columns creation : very slow + scrollbar blinked
 *    Nicolas Guyomar (Mia-Software) - Bug 345554 - SWTBot regression on build 303
 *    Nicolas Guyomar (Mia-Software) - Bug 344925 - Undo/Redo after the action Show Columns
 *    Nicolas Guyomar (Mia-Software) - Bug 345730 - Deleting an element in the model breaks the table
 *    Nicolas Guyomar (Mia-Software) - Bug 344921 - Undo/Redo just after the creation of the table
 *    Nicolas Guyomar (Mia-Software) - Bug 346465 - [EMF Facet Table] Remove line does not remove obsolete column
 *    Gregoire Dupe (Mia-Software) - Bug 345730 - Deleting an element in the model breaks the table
 *    Vincent Lorenzo (CEA-LIST) - Bug 352822 - CellEditor EEnum is not available for FacetAttribute
 *    Nicolas Bros (Mia-Software) - Bug 352822 - CellEditor EEnum is not available for FacetAttribute
 *    Vincent Lorenzo (CEA-LIST) - Bug 351931 - Use local cell editor in table
 *    Vincent Lorenzo (CEA-LIST) - Bug 352580 - Facets are loaded to late
 *    Gregoire Dupe (Mia-Software) - Bug 354224 - mutually exclusive Facets
 *******************************************************************************/

package org.eclipse.emf.facet.widgets.nattable.internal;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import net.sourceforge.nattable.NatTable;
import net.sourceforge.nattable.command.LayerCommandUtil;
import net.sourceforge.nattable.config.AbstractRegistryConfiguration;
import net.sourceforge.nattable.config.DefaultNatTableStyleConfiguration;
import net.sourceforge.nattable.config.IConfigRegistry;
import net.sourceforge.nattable.config.IEditableRule;
import net.sourceforge.nattable.coordinate.ColumnPositionCoordinate;
import net.sourceforge.nattable.coordinate.PositionCoordinate;
import net.sourceforge.nattable.coordinate.Range;
import net.sourceforge.nattable.copy.command.CopyDataToClipboardCommand;
import net.sourceforge.nattable.data.IDataProvider;
import net.sourceforge.nattable.data.validate.IDataValidator;
import net.sourceforge.nattable.edit.EditConfigAttributes;
import net.sourceforge.nattable.edit.event.InlineCellEditEvent;
import net.sourceforge.nattable.grid.command.AutoResizeColumnCommandHandler;
import net.sourceforge.nattable.grid.data.DefaultColumnHeaderDataProvider;
import net.sourceforge.nattable.grid.data.DefaultCornerDataProvider;
import net.sourceforge.nattable.grid.data.DefaultRowHeaderDataProvider;
import net.sourceforge.nattable.grid.layer.CornerLayer;
import net.sourceforge.nattable.grid.layer.GridLayer;
import net.sourceforge.nattable.hideshow.ColumnHideShowLayer;
import net.sourceforge.nattable.layer.AbstractLayerTransform;
import net.sourceforge.nattable.layer.DataLayer;
import net.sourceforge.nattable.layer.ILayerListener;
import net.sourceforge.nattable.layer.cell.ColumnOverrideLabelAccumulator;
import net.sourceforge.nattable.layer.cell.LayerCell;
import net.sourceforge.nattable.layer.event.ILayerEvent;
import net.sourceforge.nattable.reorder.ColumnReorderLayer;
import net.sourceforge.nattable.reorder.event.ColumnReorderEvent;
import net.sourceforge.nattable.resize.event.ColumnResizeEvent;
import net.sourceforge.nattable.resize.event.RowResizeEvent;
import net.sourceforge.nattable.selection.SelectionLayer;
import net.sourceforge.nattable.selection.command.SelectAllCommand;
import net.sourceforge.nattable.selection.command.SelectRowsCommand;
import net.sourceforge.nattable.selection.config.DefaultSelectionStyleConfiguration;
import net.sourceforge.nattable.selection.event.CellSelectionEvent;
import net.sourceforge.nattable.selection.event.ColumnSelectionEvent;
import net.sourceforge.nattable.selection.event.RowSelectionEvent;
import net.sourceforge.nattable.sort.config.SingleClickSortConfiguration;
import net.sourceforge.nattable.style.BorderStyle;
import net.sourceforge.nattable.style.BorderStyle.LineStyleEnum;
import net.sourceforge.nattable.style.DisplayMode;
import net.sourceforge.nattable.style.HorizontalAlignmentEnum;
import net.sourceforge.nattable.style.VerticalAlignmentEnum;
import net.sourceforge.nattable.util.GUIHelper;
import net.sourceforge.nattable.viewport.ViewportLayer;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.command.BasicCommandStack;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CompoundCommand;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.emf.edit.domain.AdapterFactoryEditingDomain;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.domain.IEditingDomainProvider;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.edit.ui.dnd.LocalTransfer;
import org.eclipse.emf.facet.infra.browser.custom.MetamodelView;
import org.eclipse.emf.facet.infra.browser.uicore.ChangeListener;
import org.eclipse.emf.facet.infra.browser.uicore.internal.customization.CustomizationEngine;
import org.eclipse.emf.facet.infra.facet.Facet;
import org.eclipse.emf.facet.infra.facet.core.FacetContext;
import org.eclipse.emf.facet.infra.facet.core.FacetSetCatalog;
import org.eclipse.emf.facet.infra.query.ModelQuery;
import org.eclipse.emf.facet.infra.query.core.AbstractModelQuery;
import org.eclipse.emf.facet.infra.query.core.AbstractModelQueryWithEditingDomain;
import org.eclipse.emf.facet.infra.query.core.ModelQuerySetCatalog;
import org.eclipse.emf.facet.infra.query.core.exception.ModelQueryException;
import org.eclipse.emf.facet.infra.query.core.java.IJavaModelQueryWithEditingDomain;
import org.eclipse.emf.facet.infra.query.runtime.ModelQueryParameterValue;
import org.eclipse.emf.facet.infra.query.runtime.RuntimeFactory;
import org.eclipse.emf.facet.infra.query.ui.dialogs.QuerySelectionDialog;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.emf.facet.util.emf.core.CommandFactory;
import org.eclipse.emf.facet.util.emf.core.command.ILockableUndoCommand;
import org.eclipse.emf.facet.util.emf.core.internal.EMFUtils;
import org.eclipse.emf.facet.widgets.celleditors.CellEditorsUtils;
import org.eclipse.emf.facet.widgets.celleditors.ICellEditorsRegistry;
import org.eclipse.emf.facet.widgets.celleditors.ICommandFactoriesRegistry;
import org.eclipse.emf.facet.widgets.celleditors.ICommandFactory;
import org.eclipse.emf.facet.widgets.celleditors.IModelCellEditor;
import org.eclipse.emf.facet.widgets.celleditors.IModelCellEditorContainer;
import org.eclipse.emf.facet.widgets.celleditors.IModelCellEditorValidator;
import org.eclipse.emf.facet.widgets.celleditors.INaryEAttributeCellEditor;
import org.eclipse.emf.facet.widgets.celleditors.INaryEReferenceCellEditor;
import org.eclipse.emf.facet.widgets.celleditors.INaryFeatureCellEditor;
import org.eclipse.emf.facet.widgets.celleditors.internal.CellEditorsRegistry;
import org.eclipse.emf.facet.widgets.celleditors.internal.ModelCellEditor;
import org.eclipse.emf.facet.widgets.celleditors.internal.ModelCellEditorContainer;
import org.eclipse.emf.facet.widgets.celleditors.internal.core.EEnumCellEditor;
import org.eclipse.emf.facet.widgets.celleditors.modelCellEditor.AbstractModelCellEditor;
import org.eclipse.emf.facet.widgets.celleditors.modelCellEditor.BasicCellEditor;
import org.eclipse.emf.facet.widgets.celleditors.modelCellEditor.NaryFeatureCellEditor;
import org.eclipse.emf.facet.widgets.celleditors.modelCellEditor.UnaryReferenceCellEditor;
import org.eclipse.emf.facet.widgets.internal.CustomizableLabelProvider;
import org.eclipse.emf.facet.widgets.nattable.INatTableWidget;
import org.eclipse.emf.facet.widgets.nattable.INatTableWidget2;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.AttributeColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.Column;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.ContextColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.DefaultLabelColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.EContainerColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FacetAttributeColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FacetReferenceColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FeatureColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.MetaClassColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.QueryColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.ReferenceColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.Row;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.TableInstance;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.TableinstanceFactory;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.TableinstancePackage;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance2.TableInstance2;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance2.Tableinstance2Factory;
import org.eclipse.emf.facet.widgets.nattable.internal.actions.LoadCustomizationsAction;
import org.eclipse.emf.facet.widgets.nattable.internal.actions.LoadFacetsAction;
import org.eclipse.emf.facet.widgets.nattable.internal.dialogs.ColumnsToHideDialog;
import org.eclipse.emf.facet.widgets.nattable.internal.exception.MultiModelQueryException;
import org.eclipse.emf.facet.widgets.nattable.internal.listeners.RowDeleteCommandStackListener;
import org.eclipse.emf.facet.widgets.nattable.internal.listeners.RowDeleteTriggerListener;
import org.eclipse.emf.facet.widgets.nattable.internal.preference.pages.NatTablePreferencePage;
import org.eclipse.emf.facet.widgets.nattable.internal.preference.pages.PreferenceConstants;
import org.eclipse.emf.facet.widgets.nattable.tableconfiguration.InstantiationMethod;
import org.eclipse.emf.facet.widgets.nattable.tableconfiguration.InstantiationMethodParameters;
import org.eclipse.emf.facet.widgets.nattable.tableconfiguration.TableConfiguration;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTarget;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionDelegate;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.ListDialog;
import org.eclipse.ui.dialogs.SaveAsDialog;
import org.eclipse.ui.part.FileEditorInput;

public class NatTableWidget extends Composite implements DisposeListener, INatTableWidget,
		INatTableWidget2, INatTableWidgetInternal {

	protected static final String DEBUG_RESOURCE_LISTENER_OPTION = "org.eclipse.emf.facet.widgets.nattable/debug/NatTableWidget/resource_listener"; //$NON-NLS-1$
	protected static final boolean DEBUG_RESOURCE_LISTENER = Activator.getDefault().isDebugging()
			&& Boolean.parseBoolean(Platform.getDebugOption(NatTableWidget.DEBUG_RESOURCE_LISTENER_OPTION));

	protected static final String DEBUG_REMOVE_USELESS_ROWS_AND_COLUMNS_OPTION = "org.eclipse.emf.facet.widgets.nattable/debug/NatTableWidget/removeUselessRowsAndColumns"; //$NON-NLS-1$
	protected static final boolean DEBUG_REMOVE_USELESS_ROWS_AND_COLUMNS = Activator.getDefault().isDebugging()
			&& Boolean.parseBoolean(Platform.getDebugOption(NatTableWidget.DEBUG_REMOVE_USELESS_ROWS_AND_COLUMNS_OPTION));

	protected static final String DEBUG_SET_FACET_OPTION = "org.eclipse.emf.facet.widgets.nattable/debug/NatTableWidget/setFacet"; //$NON-NLS-1$
	protected static final boolean DEBUG_SET_FACET = Activator.getDefault().isDebugging()
			&& Boolean.parseBoolean(Platform.getDebugOption(NatTableWidget.DEBUG_SET_FACET_OPTION));
	
	private static final int REFRESH_JOB_DELAY = 100;
	private static final int DEFAULT_COLUMN_WIDTH = 150;
	private static final int DEFAULT_COLUMN_HEIGHT = 20;
	private static final Color BG_COLOR = GUIHelper.getColor(217, 232, 251);
	private static final String ENUM_CELLEDITOR_LABEL = "enum_celleditor"; //$NON-NLS-1$
	protected static final String TMP_FILE_NAME = Activator.PLUGIN_ID + ".tmp.table"; //$NON-NLS-1$
	public static final File DEFAULT_RESOURCE_FILE = new File(Platform.getStateLocation(Activator.getDefault()
			.getBundle()).toOSString(), NatTableWidget.TMP_FILE_NAME);

	/** The NatTable widget */
	private NatTable natTable;
	/** The Input */
	private final TableInstance tableInstance;
	/** The label provider */
	private TableLabelProvider tableLabelProvider;

	/** Layers */
	private BodyDataProvider fBodyDataProvider;
	private BodyLayerStack fBodyLayer;
	private GridLayer gridLayer;
	private ColumnHeaderLayerStack fColumnHeaderLayer;
	private RowHeaderLayerStack fRowHeaderLayer;

	private final Collection<ISelectionChangedListener> fSelectionChangedListeners = new ArrayList<ISelectionChangedListener>();

	/** To remove the listeners when disposed */
	private final List<Resource> listenedResources = new ArrayList<Resource>();
	private CustomizationEngine customizationEngine;
	/** All the metaclasses of the elements */
	private HashSet<EClass> metaclasses;
	private Resource resource;
	private final MenuManager menuMgr;
	/** Whether to enable editability of the table's data */
	private boolean dataEditable;
	private final EditingDomain editingDomain;
	/** The command factory used to create every command for the widget*/
	private final ICommandFactory commandFactory;

	private boolean listenColumn = true;
	private boolean listenReorderEvent = true;

	/** Allows optimized refreshes of the table */
	private Job refreshJob = null;

	private List<InstantiationMethod> launchedInstMethHistory;

	/**
	 *
	 * @param parent
	 *            This composite parent
	 * @param editingDomainProvider
	 *            An editing provider
	 * @param tableInstanceParam
	 *            The {@link TableInstance} which contains all the needed
	 *            information to open a table
	 * @param menuMgr
	 * @throws CoreException
	 */
	public NatTableWidget(final Composite parent,
			final IEditingDomainProvider editingDomainProvider,
			final TableInstance tableInstanceParam, final MenuManager menuMgr) {
		super(parent, SWT.NONE);
		this.menuMgr = menuMgr;
		// Forces the catalogs instanciation to ensure facet resolving
		FacetSetCatalog.getSingleton(); // should be done before setInput();

		ComposedAdapterFactory adapterFactory = new ComposedAdapterFactory(
				ComposedAdapterFactory.Descriptor.Registry.INSTANCE);
		BasicCommandStack commandStack = new BasicCommandStack();
		if (editingDomainProvider == null || editingDomainProvider.getEditingDomain() == null) {
			this.editingDomain = new AdapterFactoryEditingDomain(adapterFactory, commandStack,
					new HashMap<Resource, Boolean>());
			this.dataEditable = false;
		} else {
			this.editingDomain = editingDomainProvider.getEditingDomain();
			this.dataEditable = true;
		}


		if (tableInstanceParam == null) {
			this.tableInstance = Tableinstance2Factory.eINSTANCE.createTableInstance2();
			this.resource = createDefaultResource();
			this.resource.getContents().add(this.tableInstance);
		} else {
			if (tableInstanceParam.eResource() != null) {
				this.tableInstance = tableInstanceParam;
				this.resource = tableInstanceParam.eResource();
			} else {
				this.tableInstance = tableInstanceParam;
				this.resource = createDefaultResource();
				Command setResourceContentCommand = CommandFactory.createSetResourceContentsCommand(this.resource, this.tableInstance);
				if (setResourceContentCommand instanceof ILockableUndoCommand) {
					ILockableUndoCommand lockableUndoCommand = (ILockableUndoCommand) setResourceContentCommand;
					lockableUndoCommand.enableCanUndo(false);
				}
				this.editingDomain.getCommandStack().execute(setResourceContentCommand);
				this.resource.getContents().add(this.tableInstance);
			}
		}
		initLaunchedHistory();
		this.commandFactory = ICommandFactoriesRegistry.INSTANCE
				.getCommandFactoryFor(this.editingDomain);
		init();
	}

	private Resource createDefaultResource() {
		return this.editingDomain.getResourceSet().createResource(
				URI.createFileURI(NatTableWidget.DEFAULT_RESOURCE_FILE.getPath()));
	}

	private void init() {
		this.customizationEngine = new CustomizationEngine();
		setInput();
		setLayout(new FillLayout());
		addDisposeListener(this);
		if (this.tableInstance != null) {
			this.tableInstance.eAdapters().add(this.tableInstanceAdapter);
			this.resource.setTrackingModification(true);
			this.resource.eAdapters().add(this.tableInstanceAdapter);
		}
		//Depending of the editing domain nature, we use a differente model listening strategy.
		if (this.editingDomain instanceof TransactionalEditingDomain) {
			TransactionalEditingDomain transactionalEditingDomain = (TransactionalEditingDomain) this.editingDomain;
			transactionalEditingDomain.addResourceSetListener(new RowDeleteTriggerListener(this.tableInstance));
		} else {
			this.editingDomain.getCommandStack().addCommandStackListener(new RowDeleteCommandStackListener(this.tableInstance, this.editingDomain));
		}
	}

	private void initFacets() throws CoreException {
		// Forces the catalogs instanciation to ensure facet resolving
		FacetSetCatalog.getSingleton();
		if (this.tableInstance instanceof TableInstance2) {
			TableInstance2 tableInstance2 = (TableInstance2) this.tableInstance;
			if (tableInstance2.getFacets2() != null && !tableInstance2.getFacets2().isEmpty()) {
				setFacets(tableInstance2.getFacets2());
			}
		}
	}

	/**
	 * This method should not be executed twice at the same time
	 */
	private synchronized void setInput() {
		if (NatTableWidget.DEBUG_RESOURCE_LISTENER) {
			System.out.println(this.getClass().getSimpleName() + ".setInput()"); //$NON-NLS-1$
		}
		removeUselessRowsAndColumns();
		boolean reset = false;
		int horizontalScrollbarPos = 0;
		int verticalScrollbarPos = 0;
		PositionCoordinate[] selectedCells = null;
		if (this.natTable != null && !this.natTable.isDisposed()) {
			horizontalScrollbarPos = this.natTable.getHorizontalBar().getSelection();
			verticalScrollbarPos = this.natTable.getVerticalBar().getSelection();
			SelectionLayer selectionLayer = this.fBodyLayer.getSelectionLayer();
			selectedCells = selectionLayer.getSelectedCells();
			reset = true;
		}
		Control[] children = getChildren();
		// if already built, dispose the previous control and rebuild the table
		for (Control childControl : children) {
			childControl.dispose();
		}
		updateMetaClassList();
		String[] columnNames = getColumnNames();
		if (NatTableWidget.DEBUG_SET_FACET) {
			System.out.println(this.getClass().getName() + ".setInput(): nbColumn=" //$NON-NLS-1$
					+ this.tableInstance.getColumns().size());
			System.out.println(this.getClass().getName() + ".setInput(): columnNames.length=" //$NON-NLS-1$
					+ columnNames.length);
		}
		this.tableLabelProvider = new TableLabelProvider(this.tableInstance.getColumns(),
				this.customizationEngine, this.facetContext);
		this.fBodyDataProvider = new BodyDataProvider(this.tableInstance.getRows(), this.tableInstance
				.getColumns().size(), this.tableInstance.getColumns(), getShell(),
				this.editingDomain, this.tableLabelProvider, this.facetContext);
		DefaultColumnHeaderDataProvider colHeaderDataProvider = new DefaultColumnHeaderDataProvider(
				columnNames);
		DefaultRowHeaderDataProvider rowHeaderDataProvider = new DefaultRowHeaderDataProvider(
				this.fBodyDataProvider);
		this.fBodyLayer = new BodyLayerStack(this.fBodyDataProvider);
		this.fColumnHeaderLayer = new ColumnHeaderLayerStack(colHeaderDataProvider,
				this.fBodyLayer, this.fBodyDataProvider);
		this.fRowHeaderLayer = new RowHeaderLayerStack(rowHeaderDataProvider, this.fBodyLayer);
		DefaultCornerDataProvider cornerDataProvider = new DefaultCornerDataProvider(
				colHeaderDataProvider, rowHeaderDataProvider);
		CornerLayer cornerLayer = new CornerLayer(new DataLayer(cornerDataProvider),
				this.fRowHeaderLayer, this.fColumnHeaderLayer);
		this.gridLayer = new GridLayer(this.fBodyLayer, this.fColumnHeaderLayer,
				this.fRowHeaderLayer, cornerLayer, false);
		this.gridLayer.registerCommandHandler(new AutoResizeColumnCommandHandler(this.gridLayer));
		this.gridLayer.addConfiguration(new GridLayerConfiguration(this.gridLayer, this));

		// NOTE: Register the accumulator on the body data layer.
		// This ensures that the labels are bound to the column index and are
		// unaffected by column order.
		DataLayer bodyDataLayer = this.fBodyLayer.getBodyDataLayer();
		final ColumnOverrideLabelAccumulator columnLabelAccumulator = new ColumnOverrideLabelAccumulator(
				bodyDataLayer);
		bodyDataLayer.setConfigLabelAccumulator(columnLabelAccumulator);
		this.natTable = new NatTable(this, this.gridLayer, false);
		this.natTable.addConfiguration(new SingleClickSortConfiguration());
		addCustomStyling(this.natTable);
		this.natTable.addConfiguration(new StyleConfiguration(this.fBodyLayer,
				this.tableLabelProvider, this.customizationEngine, this));
		if (isDataEditable()) {
			this.natTable.addConfiguration(editableGridConfiguration(columnLabelAccumulator,
					this.fBodyDataProvider));
		}

		this.natTable.configure();
		initializeColumnAndRowSize();
		addDragAndDropSupport();
		addDataLayerListener(this.fBodyLayer.getBodyDataLayer());
		addColumnReorderLayerListener(this.fBodyLayer.getColumnReorderLayer());
		reapplySettings();
		setupContextMenu();
		addSelectionListener(this.natTable);
		addChangeListener(this.tableInstance.getRows());
		layout();
		if (this.tableInstance instanceof TableInstance2) {
			this.facetContext.clear();
			this.facetContext.addFacets(((TableInstance2) this.tableInstance).getFacets2());
		}
		if (this.tableInstance.getCustomizations() != null) {
			loadCustomizationsInCustomEngine(this.tableInstance.getCustomizations());
		}
		if (reset && !this.natTable.isDisposed()) {
			this.natTable.getHorizontalBar().setSelection(horizontalScrollbarPos);
			this.natTable.getHorizontalBar().notifyListeners(SWT.Selection, new Event());
			this.natTable.getVerticalBar().setSelection(verticalScrollbarPos);
			this.natTable.getVerticalBar().notifyListeners(SWT.Selection, new Event());
			if (selectedCells != null) {
				this.fBodyLayer.getSelectionLayer().clear();
				for (PositionCoordinate coordinate : selectedCells) {
					this.fBodyLayer.getSelectionLayer().selectCell(coordinate.columnPosition,
							coordinate.rowPosition, false, true);
				}
			}
		}
		this.natTable.setFocus();
	}

	/**
	 * This method maintain the metaclass list up to date, in order not to
	 * evaluate its value every time we need it
	 */
	private void updateMetaClassList() {
		this.metaclasses = new HashSet<EClass>();
		for (EObject eObject : this.tableInstance.getElements()) {
			this.metaclasses.add(eObject.eClass());
		}
	}

	private void initializeColumnAndRowSize() {
		for (int i = 0; i < this.fBodyLayer.getBodyDataLayer().getColumnCount(); i++) {
			if (this.tableInstance.getColumns().get(i).getWidth() != -1) {
				this.fBodyLayer.getBodyDataLayer().setColumnWidthByPosition(i,
						this.tableInstance.getColumns().get(i).getWidth());
			} else {
				this.fBodyLayer.getBodyDataLayer().setColumnWidthByPosition(i,
						NatTableWidget.DEFAULT_COLUMN_WIDTH);
			}
		}
		for (int i = 0; i < this.fBodyLayer.getBodyDataLayer().getRowCount(); i++) {
			if (this.tableInstance.getRows().get(i).getHeight() != -1) {
				this.fBodyLayer.getBodyDataLayer().setRowHeightByPosition(i,
						this.tableInstance.getRows().get(i).getHeight());
			} else {
				this.fBodyLayer.getBodyDataLayer().setRowHeightByPosition(i,
						NatTableWidget.DEFAULT_COLUMN_HEIGHT);
			}
		}
	}

	private void initLaunchedHistory() {
		this.launchedInstMethHistory = new ArrayList<InstantiationMethod>();
		if (getTableConfiguration() != null
				&& getTableConfiguration().getInstantiationMethod() != null) {
			for (InstantiationMethod meth : getTableConfiguration().getInstantiationMethod()) {
				this.launchedInstMethHistory.add(meth);
			}
		}
	}

	/**
	 * Enable the table to receive dropped elements
	 */
	private void addDragAndDropSupport() {
		int operations = DND.DROP_MOVE | DND.DROP_COPY | DND.DROP_DEFAULT;
		DropTarget target = new DropTarget(this.natTable, operations);
		final LocalTransfer localTransfer = LocalTransfer.getInstance();
		Transfer[] types = new Transfer[] { localTransfer };
		target.setTransfer(types);
		NatTableDropListener dropListener = new NatTableDropListener(this.natTable, this);
		target.addDropListener(dropListener);
	}

	private void addColumnReorderLayerListener(final ColumnReorderLayer columnReorderLayer) {
		columnReorderLayer.addLayerListener(new ILayerListener() {

			public void handleLayerEvent(final ILayerEvent event) {
				synchronized (this) {
					if (NatTableWidget.this.listenReorderEvent) {
						synchronized (this) {
							NatTableWidget.this.listenColumn = false;
						}
						if (event instanceof ColumnReorderEvent) {
							ColumnReorderEvent columnReorderEvent = (ColumnReorderEvent) event;
							int start = -1;
							int end = columnReorderEvent.getBeforeToColumnPosition();
							for (Range range : columnReorderEvent
									.getBeforeFromColumnPositionRanges()) {
								start = range.start;
								break;
							}
							if (start != -1) {
								// This solve an index difference between moving
								// a column from right to left and moving a
								// column from left to right
								if (start >= 0 && start < end) {
									end--;
								}
								EObject element = NatTableWidget.this.tableInstance.getColumns()
										.get(start);
								if (element != null) {
									Command command = NatTableWidget.this.commandFactory.createMoveCommand(
											NatTableWidget.this.editingDomain,
											NatTableWidget.this.tableInstance,
											TableinstancePackage.eINSTANCE
													.getTableInstance_Columns(), element, end);
									NatTableWidget.this.editingDomain.getCommandStack().execute(command);
								}
							}
						}
						synchronized (this) {
							NatTableWidget.this.listenColumn = true;
						}
					}
				}
			}
		});
	}

	/**
	 * Add Layer Listener to the layer to catch ColumnResizeEvent and
	 * RowResizeEvent in order to update the model when UI changes are commited
	 *
	 * @param dataLayer
	 *            the layer to watch
	 */
	private void addDataLayerListener(final DataLayer dataLayer) {
		dataLayer.addLayerListener(new ILayerListener() {

			public void handleLayerEvent(final ILayerEvent event) {
				synchronized (this) {
					if (NatTableWidget.this.listenColumn) {
						synchronized (this) {
							NatTableWidget.this.listenColumn = false;
						}
						if (event instanceof ColumnResizeEvent) {
							ColumnResizeEvent columnResizeEvent = (ColumnResizeEvent) event;
							Collection<Range> rangeList = columnResizeEvent
									.getColumnPositionRanges();
							for (Range r : rangeList) {
								if (r.start != -1) {

									int width = NatTableWidget.this.fBodyLayer
											.getColumnReorderLayer().getColumnWidthByPosition(
													r.start);

									Column column = NatTableWidget.this.getTableInstance()
											.getColumns().get(r.start);

									Command cmd = NatTableWidget.this.commandFactory
											.createSetCommand(NatTableWidget.this.editingDomain,
													column, TableinstancePackage.eINSTANCE
															.getColumn_Width(), width);
									NatTableWidget.this.editingDomain.getCommandStack().execute(cmd);
								}
							}

						} else if (event instanceof RowResizeEvent) {
							RowResizeEvent rowResizeEvent = (RowResizeEvent) event;
							Collection<Range> rangeList = rowResizeEvent.getRowPositionRanges();
							for (Range r : rangeList) {
								if (r.start != -1) {
									int height = NatTableWidget.this.getBodyLayer()
											.getViewportLayer().getRowHeightByPosition(r.start);
									Row row = NatTableWidget.this.getTableInstance().getRows()
											.get(r.start);
									Command cmd = NatTableWidget.this.commandFactory.createSetCommand(
											NatTableWidget.this.editingDomain, row,
											TableinstancePackage.eINSTANCE.getRow_Height(), height);
									NatTableWidget.this.editingDomain.getCommandStack()
											.execute(cmd);
								}
							}
						}
						synchronized (this) {
							NatTableWidget.this.listenColumn = true;
						}
					}
				}
			}
		});
	}

	private void reapplySettings() {
		// reapply settings
		updateHiddenColumns();
	}

	private final Adapter modelChangeAdapter = new AdapterImpl() {
		@Override
		public void notifyChanged(final Notification msg) {
			int eventType = msg.getEventType();
			final boolean redrawNeeded = eventType != Notification.REMOVING_ADAPTER && eventType != Notification.RESOLVE;
			if (NatTableWidget.DEBUG_RESOURCE_LISTENER) {
				System.out.println(this.getClass().getName() + ": notification=" + msg); //$NON-NLS-1$
				System.out.println(this.getClass().getName() + ": redrawNeeded=" + redrawNeeded); //$NON-NLS-1$
			}
			if (redrawNeeded) {
				// redraw table when model changes
				if (NatTableWidget.this.natTable != null
						&& !NatTableWidget.this.natTable.isDisposed()) {
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							if (NatTableWidget.DEBUG_RESOURCE_LISTENER) {
								System.out.println(this.getClass().getName() + ": call of the refresh=" + msg); //$NON-NLS-1$
							}
							if (NatTableWidget.this.natTable != null
									&& !NatTableWidget.this.natTable.isDisposed()) {
								refreshDelayed();
								NatTableWidget.this.natTable.redraw();
							}
						}
					});
				}
			}
		}
	};

	private final Adapter tableInstanceAdapter = new AdapterImpl() {

		@Override
		public synchronized void notifyChanged(final Notification msg) {
			int eventType = msg.getEventType();
			if (eventType != Notification.REMOVING_ADAPTER && eventType != Notification.RESOLVE) {
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							if (NatTableWidget.this.tableInstance instanceof TableInstance2) {
								TableInstance2 tableInstance2 = (TableInstance2) NatTableWidget.this.tableInstance;
								NatTableWidget.this.facetContext.addFacets(tableInstance2.getFacets2());
							}
							refreshDelayed();
						}
					});
			}
		}
	};

	private void addChangeListener(final List<Row> elementsValue) {
		Set<Resource> resources = new HashSet<Resource>();
		for (Row row : elementsValue) {
			EObject eObject = row.getElement();
			if (eObject != null) {
				resources.add(eObject.eResource());
			}
		}
		for (Resource referedResource : resources) {
			if (referedResource != null) {
				if (!this.listenedResources.contains(referedResource)) {
					this.listenedResources.add(referedResource);
					referedResource.setTrackingModification(true);
					if (!referedResource.eAdapters().contains(this.modelChangeAdapter)) {
						referedResource.eAdapters().add(this.modelChangeAdapter);
					}
				}
			}
		}
	}

	public void widgetDisposed(final DisposeEvent e) {
		for (Resource resourceElt : this.listenedResources) {
			resourceElt.eAdapters().remove(this.modelChangeAdapter);
		}
	}

	private static Object getStructuralFeatureValue(final EObject eObject,
			final EStructuralFeature structuralFeature) {
		return eObject.eGet(structuralFeature);
	}

	private static boolean hasStructuralFeature(final EObject eObject,
			final EStructuralFeature structuralFeature) {
		final EClass eClass = eObject.eClass();
		return eClass.getEAllStructuralFeatures().contains(structuralFeature);
	}

	private void addSelectionListener(final NatTable natTableValue) {
		natTableValue.addLayerListener(new ILayerListener() {
			public void handleLayerEvent(final ILayerEvent event) {
				if (event instanceof CellSelectionEvent || event instanceof RowSelectionEvent
						|| event instanceof ColumnSelectionEvent) {
					tableSelectionChanged();
				}
			}
		});
	}

	protected void tableSelectionChanged() {
		for (ISelectionChangedListener listener : new ArrayList<ISelectionChangedListener>(
				this.fSelectionChangedListeners)) {
			listener.selectionChanged(new SelectionChangedEvent(this, getSelection()));
		}
	}

	/**
	 *
	 * @see org.eclipse.jface.viewers.ISelectionProvider#getSelection()
	 *
	 * @return <ul>
	 *         <li>the element represented by the row when a row is fully
	 *         selected</li>
	 *         <li>the element owned by the cell</li>
	 *         </ul>
	 */
	public ISelection getSelection() {
		List<EObject> selectedEObjects = new ArrayList<EObject>();
		if (!this.tableInstance.getRows().isEmpty() && !this.tableInstance.getColumns().isEmpty()) {
		List<Integer> selectedRows = getSelectedRows();
		// when a row is fully selected, we return the object represented by
		// this row!
		int rowsSize = this.tableInstance.getRows().size();
		for (Integer current : selectedRows) {
			if (current.intValue() < rowsSize) {
				Row row = this.tableInstance.getRows().get(current.intValue());
				if (row != null) {
					selectedEObjects.add(row.getElement());
				}
			}
		}
		if (this.fBodyLayer != null) {
			SelectionLayer selectionLayer = this.fBodyLayer.getSelectionLayer();
			PositionCoordinate[] selectedCells = selectionLayer.getSelectedCells();
			for (PositionCoordinate positionCoordinate : selectedCells) {
				if (!selectedRows.contains(positionCoordinate.getRowPosition())) {
					LayerCell cell = selectionLayer
							.getCellByPosition(positionCoordinate.getColumnPosition(),
									positionCoordinate.getRowPosition());
					if (cell != null) {
						Object dataValue = cell.getDataValue();
						if (dataValue instanceof GridElement) {
							GridElement gridElement = (GridElement) dataValue;
							EObject eObject = getEObject(gridElement);
							if (eObject != null) {
								selectedEObjects.add(eObject);
							}
						}
					}
				}
			}
		}
		}
		return new StructuredSelection(selectedEObjects);
	}

	/**
	 *
	 * @return a list of integer representing the selected rows
	 */
	private List<Integer> getSelectedRows() {
		List<Integer> rowsIndex = new ArrayList<Integer>();
		if (this.fBodyLayer != null) {
			SelectionLayer selectionLayer = this.fBodyLayer.getSelectionLayer();
			int[] selectedRows = selectionLayer.getFullySelectedRowPositions();
			for (int index : selectedRows) {
				rowsIndex.add(Integer.valueOf(index));
			}
		}
		return rowsIndex;
	}

	public List<LayerCell> getSelectedCells() {
		List<LayerCell> selectedCells = new ArrayList<LayerCell>();
		SelectionLayer selectionLayer = this.fBodyLayer.getSelectionLayer();
		PositionCoordinate[] selectedPositions = selectionLayer.getSelectedCells();
		for (PositionCoordinate positionCoordinate : selectedPositions) {
			LayerCell cell = selectionLayer.getCellByPosition(
					positionCoordinate.getColumnPosition(), positionCoordinate.getRowPosition());
			if (cell != null) {
				selectedCells.add(cell);
			}
		}
		return selectedCells;
	}

	public List<PositionCoordinate> getSelectedCellsPositions() {
		PositionCoordinate[] selectedCells = this.fBodyLayer.getSelectionLayer().getSelectedCells();
		return Arrays.asList(selectedCells);
	}

	/** Convert column position from data layer to column hide/show layer */
	public int convertColumnPositionDataToHideShow(final int column) {
		// we can only convert from upper to lower layer
		// y = f(x) : test f(x) for each x, in order to find x for y = column
		for (int i = 0; i < this.fBodyDataProvider.getColumnCount(); i++) {
			ColumnPositionCoordinate converted = LayerCommandUtil
					.convertColumnPositionToTargetContext(new ColumnPositionCoordinate(
							this.fBodyLayer.getColumnHideShowLayer(), i), this.fBodyLayer
							.getBodyDataLayer());
			if (converted != null && converted.getColumnPosition() == column) {
				return i;
			}
		}
		return -1;
	}

	/** Convert column position from data layer to column reorder layer */
	public int convertColumnPositionDataToReorder(final int column) {
		// we can only convert from upper to lower layer
		// y = f(x) : test f(x) for each x, in order to find x for y = column
		for (int i = 0; i < this.fBodyDataProvider.getColumnCount(); i++) {
			ColumnPositionCoordinate converted = LayerCommandUtil
					.convertColumnPositionToTargetContext(new ColumnPositionCoordinate(
							this.fBodyLayer.getColumnReorderLayer(), i), this.fBodyLayer
							.getBodyDataLayer());
			if (converted.getColumnPosition() == column) {
				return i;
			}
		}
		throw new IllegalStateException("column index not found"); //$NON-NLS-1$
	}

	/** Convert column position from selection layer to data layer */
	public int convertColumnPositionSelectionToData(final int column) {
		ColumnPositionCoordinate columnPositionCoordinate = LayerCommandUtil
				.convertColumnPositionToTargetContext(new ColumnPositionCoordinate(getBodyLayer()
						.getSelectionLayer(), column), getBodyLayer().getBodyDataLayer());
		return columnPositionCoordinate.getColumnPosition();
	}

	public int convertCellPositionToDataLayer(final LayerCell cell) {
		ColumnPositionCoordinate columnPositionCoordinate = LayerCommandUtil
				.convertColumnPositionToTargetContext(new ColumnPositionCoordinate(cell.getLayer(),
						cell.getColumnPosition()), getBodyLayer().getBodyDataLayer());
		return columnPositionCoordinate.getColumnPosition();
	}

	public void setSelection(final ISelection selection) {
		// The selection can not be set through the API
	}

	/**
	 * Add a {@link ISelectionChangedListener} to the widget.
	 */
	public void addSelectionChangedListener(final ISelectionChangedListener listener) {
		if (!this.fSelectionChangedListeners.contains(listener)) {
			this.fSelectionChangedListeners.add(listener);
		}
	}

	/**
	 * Remove a {@link ISelectionChangedListener} from the widget.
	 */
	public void removeSelectionChangedListener(final ISelectionChangedListener listener) {
		this.fSelectionChangedListeners.remove(listener);
	}

	private void addCustomStyling(final NatTable natTableValue) {
		// Setup NatTable default styling

		// NOTE: Getting the colors and fonts from the GUIHelper ensures that
		// they are disposed properly (required by SWT)
		DefaultNatTableStyleConfiguration natTableConfiguration = new DefaultNatTableStyleConfiguration();
		// natTableConfiguration.bgColor = GUIHelper.getColor(249, 172, 7);
		// natTableConfiguration.fgColor = GUIHelper.getColor(30, 76, 19);
		natTableConfiguration.hAlign = HorizontalAlignmentEnum.LEFT;
		natTableConfiguration.vAlign = VerticalAlignmentEnum.TOP;

		// A custom painter can be plugged in to paint the cells differently
		// natTableConfiguration.cellPainter = new PaddingDecorator(new
		// TextPainter(), 1);
		// natTableConfiguration.cellPainter = new CellImagePainter(); //new
		// CellPainterDecorator(new TextPainter(), CellEdgeEnum.LEFT, new
		// CellImagePainter());
		// natTableConfiguration.cellPainter = new TextPainter();

		// Setup even odd row colors - row colors override the NatTable default
		// colors
		// DefaultRowStyleConfiguration rowStyleConfiguration = new
		// DefaultRowStyleConfiguration();
		// rowStyleConfiguration.oddRowBgColor = GUIHelper.getColor(254, 251,
		// 243);
		// rowStyleConfiguration.evenRowBgColor = GUIHelper.COLOR_WHITE;

		// Setup selection styling
		DefaultSelectionStyleConfiguration selectionStyle = new DefaultSelectionStyleConfiguration();
		// selectionStyle.selectionFont =
		// this.tableEditorInput.getBrowserConfiguration()
		// .getAppearanceConfiguration().getCustomFont();
		selectionStyle.selectionBgColor = NatTableWidget.BG_COLOR;
		selectionStyle.selectionFgColor = GUIHelper.COLOR_BLACK;
		selectionStyle.anchorBorderStyle = new BorderStyle(1, GUIHelper.COLOR_DARK_GRAY,
				LineStyleEnum.SOLID);
		// was (65, 113, 43);
		selectionStyle.anchorBgColor = NatTableWidget.BG_COLOR;
		selectionStyle.anchorFgColor = GUIHelper.COLOR_BLACK;
		selectionStyle.selectedHeaderFgColor = GUIHelper.COLOR_BLACK;

		// Add all style configurations to NatTable
		natTableValue.addConfiguration(natTableConfiguration);
		// natTable.addConfiguration(rowStyleConfiguration);
		natTableValue.addConfiguration(selectionStyle);

		// Column/Row header style and custom painters
		natTableValue.addConfiguration(new StyledRowHeaderConfiguration());
		natTableValue.addConfiguration(new StyledColumnHeaderConfiguration(this.fBodyLayer,
				this.tableInstance.getColumns(), new CustomizableLabelProvider(
						this.customizationEngine)));
	}

	/**
	 * Create a configuration for editability
	 *
	 * @param columnLabelAccumulator
	 * @param dataProvider
	 * @return
	 */
	public AbstractRegistryConfiguration editableGridConfiguration(
			final ColumnOverrideLabelAccumulator columnLabelAccumulator,
			final IDataProvider dataProvider) {
		return new AbstractRegistryConfiguration() {
			public void configureRegistry(final IConfigRegistry configRegistry) {
				// not editable if no editing domain was provided
				if (NatTableWidget.this.editingDomain != null) {
					for (int nColumn = 0; nColumn < NatTableWidget.this.tableInstance.getColumns()
							.size(); nColumn++) {
						Column columnDescription = NatTableWidget.this.tableInstance.getColumns()
								.get(nColumn);
						// EEnum is hardcoded
						if (columnDescription instanceof AttributeColumn) {
							AttributeColumn attributeColumn = (AttributeColumn) columnDescription;
							if (attributeColumn.getAttribute().getEType() instanceof EEnum) {
								registerEnumCellEditor(configRegistry, (EEnum) attributeColumn
										.getFeature().getEType(), columnLabelAccumulator, nColumn,
										columnDescription);
							} else {
								registerCellEditorForColumn(columnLabelAccumulator, nColumn,
										columnDescription);
							}
						} else if (columnDescription instanceof FacetAttributeColumn) {
							FacetAttributeColumn attributeColumn = (FacetAttributeColumn) columnDescription;
							if (attributeColumn.getAttribute().getEType() instanceof EEnum) {
								registerEnumCellEditor(configRegistry, (EEnum) attributeColumn
										.getFeature().getEType(), columnLabelAccumulator, nColumn,
										columnDescription);
							} else {
								registerCellEditorForColumn(columnLabelAccumulator, nColumn,
										columnDescription);
							}
						} else {
							registerCellEditorForColumn(columnLabelAccumulator, nColumn,
									columnDescription);
						}
					}
					registerLocalCellEditors(configRegistry);
					List<IModelCellEditorContainer<? extends AbstractModelCellEditor>> allCellEditorContainers = ICellEditorsRegistry.INSTANCE
							.getAllCellEditors();
					for (IModelCellEditorContainer<? extends AbstractModelCellEditor> modelCellEditorContainer : allCellEditorContainers) {
						registerCellEditor(modelCellEditorContainer, configRegistry);
					}
				}
			}
		};
	}

	/**
	 * Register the cell editors which are forced in the tableconfiguration, but
	 * not declared in the plugin.xml
	 */
	private void registerLocalCellEditors(final IConfigRegistry configRegistry) {
		final TableConfiguration tableConfiguration = getTableConfiguration();
		if (tableConfiguration != null) {
			final EList<AbstractModelCellEditor> forcedCellEditors = tableConfiguration
					.getForcedCellEditors();

			for (final AbstractModelCellEditor modelCellEditor : forcedCellEditors) {
				// we register the editor only if it is not in the
				// ICellEditorsRegistry (<=> declared in the plugin.xml file)
				if (!isInCellEditorRegistry(modelCellEditor)) {
					final String bundleName = CellEditorsUtils.getBundleNameFor(modelCellEditor, getTableConfiguration().eResource().getResourceSet());
					ModelCellEditorContainer<?> modelCellEditorContainer = null;
					if (modelCellEditor instanceof BasicCellEditor) {
						BasicCellEditor basicCellEditor = (BasicCellEditor) modelCellEditor;
						modelCellEditorContainer = new ModelCellEditorContainer<BasicCellEditor>();
						modelCellEditorContainer.setBundleName(bundleName);
						modelCellEditorContainer.setModelCellEditor(basicCellEditor);
					} else if (modelCellEditor instanceof UnaryReferenceCellEditor) {
						UnaryReferenceCellEditor unaryReferenceCellEditor = (UnaryReferenceCellEditor) modelCellEditor;
						modelCellEditorContainer = new ModelCellEditorContainer<UnaryReferenceCellEditor>();
						modelCellEditorContainer.setBundleName(bundleName);
						modelCellEditorContainer.setModelCellEditor(unaryReferenceCellEditor);
					} else if (modelCellEditor instanceof NaryFeatureCellEditor) {
						NaryFeatureCellEditor naryFeatureCellEditor = (NaryFeatureCellEditor) modelCellEditor;
						modelCellEditorContainer = new ModelCellEditorContainer<NaryFeatureCellEditor>();
						modelCellEditorContainer.setBundleName(bundleName);
						modelCellEditorContainer.setModelCellEditor(naryFeatureCellEditor);
					} else {
						Logger.logError("Not handled: " //$NON-NLS-1$
								+ modelCellEditor.getClass().getSimpleName(),
								Activator.getDefault());
					}
					registerCellEditor(modelCellEditorContainer, configRegistry);
				}
			}
		}
	}

	/**
	 * 
	 * @param cellEditor
	 *            a cell editor
	 * @return <code>true</code> if the editor is registered in the
	 *         {@link ICellEditorsRegistry} (that is to say that this editor is
	 *         declared in the plugin.xml)
	 */
	private static boolean isInCellEditorRegistry(final AbstractModelCellEditor cellEditor) {
		List<IModelCellEditorContainer<? extends AbstractModelCellEditor>> allCellEditorContainers = ICellEditorsRegistry.INSTANCE
				.getAllCellEditors();
		for (IModelCellEditorContainer<? extends AbstractModelCellEditor> currentEditorContainer : allCellEditorContainers) {
			if (cellEditor.getCellId().equals(
					currentEditorContainer.getModelCellEditor().getCellId())) {
				return true;
			}
		}
		return false;
	}

	private void registerCellEditorForColumn(
			final ColumnOverrideLabelAccumulator columnLabelAccumulator, final int nColumn,
			final Column column) {
		if (column instanceof FeatureColumn) {

			FeatureColumn featureColumn = (FeatureColumn) column;

			EStructuralFeature feature = featureColumn.getFeature();

			if (!feature.isChangeable()) {
				return;
			}

			TableConfiguration tableConfiguration = getTableConfiguration();
			EList<AbstractModelCellEditor> forcedCellEditors = null;
			if (tableConfiguration != null) {
				forcedCellEditors = tableConfiguration.getForcedCellEditors();
			}

			ModelCellEditor selectedCellEditor = ((CellEditorsRegistry) ICellEditorsRegistry.INSTANCE)
					.getCellEditorWrapperFor(feature.getEType(), feature.isMany(),
							forcedCellEditors);

			if (selectedCellEditor != null) {
				// register cell editor for this column
				columnLabelAccumulator.registerColumnOverrides(nColumn,
						selectedCellEditor.getCellID());
			} else {
				Logger.logWarning(
						"No cell editor defined for type: " + EcoreUtil.getURI(feature.getEType()) + " (column " + nColumn + ")", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
						Activator.getDefault());
			}
		}
	}

	/**
	 * Registers a cell editor in the config registry
	 *
	 * @param configRegistry
	 * @param eType
	 * @param columnLabelAccumulator
	 * @param allCellEditorContainers
	 * @param nColumn
	 * @param columnDescription
	 */
	private void registerEnumCellEditor(final IConfigRegistry configRegistry, final EEnum eType,
			final ColumnOverrideLabelAccumulator columnLabelAccumulator, final int nColumn,
			final Column columnDescription) {
		columnLabelAccumulator.registerColumnOverrides(nColumn,
				NatTableWidget.ENUM_CELLEDITOR_LABEL);
		configRegistry.registerConfigAttribute(EditConfigAttributes.CELL_EDITOR,
				new NatTableCellEditorAdapter(new EEnumCellEditor(),
						NatTableWidget.this.tableInstance.getColumns(), this.facetContext), DisplayMode.NORMAL,
				NatTableWidget.ENUM_CELLEDITOR_LABEL);
		configRegistry.registerConfigAttribute(EditConfigAttributes.CELL_EDITABLE_RULE,
				IEditableRule.ALWAYS_EDITABLE, DisplayMode.EDIT,
				NatTableWidget.ENUM_CELLEDITOR_LABEL);
	}

//	@SuppressWarnings({ "unchecked", "rawtypes" }) // type erasure on generic
	private void registerCellEditor(
			final IModelCellEditorContainer<? extends AbstractModelCellEditor> modelCellEditorContainer,
			final IConfigRegistry configRegistry) {

		ModelCellEditor cellEditor = new ModelCellEditor(modelCellEditorContainer.getBundleName(),
				modelCellEditorContainer.getModelCellEditor());
		Object cellEditorImplementation = cellEditor.getCellEditorImplementation();

		if (cellEditorImplementation instanceof IModelCellEditor) {
			IModelCellEditor modelCellEditorImpl = (IModelCellEditor) cellEditorImplementation;

			configRegistry.registerConfigAttribute(EditConfigAttributes.CELL_EDITOR,
					new NatTableCellEditorAdapter(modelCellEditorImpl,
							NatTableWidget.this.tableInstance.getColumns(), this.facetContext), DisplayMode.NORMAL,
					cellEditor.getCellID());
			configRegistry.registerConfigAttribute(EditConfigAttributes.CELL_EDITABLE_RULE,
					IEditableRule.ALWAYS_EDITABLE, DisplayMode.EDIT, cellEditor.getCellID());
		} else if (cellEditorImplementation instanceof INaryFeatureCellEditor) {
			INaryFeatureCellEditor modelCellEditorImpl = (INaryFeatureCellEditor) cellEditorImplementation;

			configRegistry.registerConfigAttribute(EditConfigAttributes.CELL_EDITOR,
					new NatTableNaryReferenceCellEditorAdapter(modelCellEditorImpl,
							NatTableWidget.this.tableInstance.getColumns(), this.editingDomain),
					DisplayMode.NORMAL, cellEditor.getCellID());
			configRegistry.registerConfigAttribute(EditConfigAttributes.CELL_EDITABLE_RULE,
					IEditableRule.ALWAYS_EDITABLE, DisplayMode.EDIT, cellEditor.getCellID());
			// disable automatic commit for INaryReferenceCellEditor
			configRegistry.registerConfigAttribute(EditConfigAttributes.DATA_VALIDATOR,
					IDataValidator.NEVER_VALID, DisplayMode.EDIT, cellEditor.getCellID());
		} else if (cellEditorImplementation instanceof INaryEAttributeCellEditor<?>) {
			INaryEAttributeCellEditor<Object> modelCellEditorImpl = (INaryEAttributeCellEditor<Object>) cellEditorImplementation;
			NatTableNaryEAttributeCellEditorAdapter<Object> cellEditorAdapter = new NatTableNaryEAttributeCellEditorAdapter<Object>(
					modelCellEditorImpl, NatTableWidget.this.tableInstance.getColumns(),
					this.facetContext, this.editingDomain);
			configRegistry.registerConfigAttribute(EditConfigAttributes.CELL_EDITOR,
					cellEditorAdapter, DisplayMode.NORMAL, cellEditor.getCellID());
			configRegistry.registerConfigAttribute(EditConfigAttributes.CELL_EDITABLE_RULE,
					IEditableRule.ALWAYS_EDITABLE, DisplayMode.EDIT, cellEditor.getCellID());
			// disable automatic commit for INaryEReferenceCellEditor
			configRegistry.registerConfigAttribute(EditConfigAttributes.DATA_VALIDATOR,
					IDataValidator.NEVER_VALID, DisplayMode.EDIT, cellEditor.getCellID());

		} else if (cellEditorImplementation instanceof INaryEReferenceCellEditor<?>) {
			INaryEReferenceCellEditor<?> modelCellEditorImpl = (INaryEReferenceCellEditor<?>) cellEditorImplementation;
			configRegistry.registerConfigAttribute(EditConfigAttributes.CELL_EDITOR,
					new NatTableNaryEReferenceCellEditorAdapter(modelCellEditorImpl,
							NatTableWidget.this.tableInstance.getColumns(), this.facetContext, this.editingDomain),
					DisplayMode.NORMAL, cellEditor.getCellID());
			configRegistry.registerConfigAttribute(EditConfigAttributes.CELL_EDITABLE_RULE,
					IEditableRule.ALWAYS_EDITABLE, DisplayMode.EDIT, cellEditor.getCellID());
			// disable automatic commit for INaryEReferenceCellEditor
			configRegistry.registerConfigAttribute(EditConfigAttributes.DATA_VALIDATOR,
					IDataValidator.NEVER_VALID, DisplayMode.EDIT, cellEditor.getCellID());

		} else {
			Logger.logError("Model cell editor implementation must be an instance of " //$NON-NLS-1$
					+ IModelCellEditor.class.getSimpleName() + " or " //$NON-NLS-1$
					+ INaryFeatureCellEditor.class.getSimpleName(), Activator.getDefault());
			return;
		}

		if (cellEditorImplementation instanceof IModelCellEditorValidator) {
			IModelCellEditorValidator modelCellEditorValidator = (IModelCellEditorValidator) cellEditorImplementation;
			configRegistry.registerConfigAttribute(EditConfigAttributes.DATA_VALIDATOR,
					new DataValidatorAdapter(modelCellEditorValidator), DisplayMode.EDIT,
					cellEditor.getCellID());
		}
	}

	public String[] getColumnNames() {
		String[] columnNames = new String[this.tableInstance.getColumns().size()];

		for (int i = 0; i < this.tableInstance.getColumns().size(); i++) {
			Column columnDescription = this.tableInstance.getColumns().get(i);
			String columnName;
			columnName = NatTableWidgetInternalUtils.getColumnName(columnDescription);
			columnNames[i] = columnName;
		}
		return columnNames;
	}

	public class BodyLayerStack extends AbstractLayerTransform {

		private final SelectionLayer selectionLayer;
		private final DataLayer bodyDataLayer;
		private final ViewportLayer viewportLayer;
		private final ColumnHideShowLayer columnHideShowLayer;
		private final ColumnReorderLayer columnReorderLayer;

		public BodyLayerStack(final IDataProvider dataProvider) {
			this.bodyDataLayer = new DataLayer(dataProvider);
			this.columnReorderLayer = new ColumnReorderLayer(this.bodyDataLayer);
			this.columnHideShowLayer = new ColumnHideShowLayer(this.columnReorderLayer);
			this.selectionLayer = new SelectionLayer(this.columnHideShowLayer);
			this.viewportLayer = new ViewportLayer(this.selectionLayer);
			setUnderlyingLayer(this.viewportLayer);
		}

		public SelectionLayer getSelectionLayer() {
			return this.selectionLayer;
		}

		public DataLayer getBodyDataLayer() {
			return this.bodyDataLayer;
		}

		public ViewportLayer getViewportLayer() {
			return this.viewportLayer;
		}

		public ColumnHideShowLayer getColumnHideShowLayer() {
			return this.columnHideShowLayer;
		}

		public ColumnReorderLayer getColumnReorderLayer() {
			return this.columnReorderLayer;
		}
	}

	public class ElementsDescription {
		private boolean containsEObjects;
		private boolean containsPrimitiveTypes;
		private boolean containsContexts;

		public boolean getContainsEObjects() {
			return this.containsEObjects;
		}

		public boolean getContainsPrimitiveTypes() {
			return this.containsPrimitiveTypes;
		}

		public boolean getContainsContexts() {
			return this.containsContexts;
		}

		public void setContainsContexts(final boolean containsContexts) {
			this.containsContexts = containsContexts;
		}

		protected void setContainsEObjects(final boolean containsEObjects) {
			this.containsEObjects = containsEObjects;
		}

		protected void setContainsPrimitiveTypes(final boolean containsPrimitiveTypes) {
			this.containsPrimitiveTypes = containsPrimitiveTypes;
		}
	}

	protected boolean isColumnAlreadyDeclared(final EStructuralFeature eStructuralFeature) {
		if (eStructuralFeature instanceof EReference) {
			for (Column c : this.tableInstance.getColumns()) {
				if (c instanceof ReferenceColumn) {
					if (((ReferenceColumn) c).getReference() == eStructuralFeature) {
						return true;
					}
				} else if (c instanceof FacetReferenceColumn) {
					if (((FacetReferenceColumn) c).getReference() == eStructuralFeature) {
						return true;
					}
				}
			}
		} else if (eStructuralFeature instanceof EAttribute) {
			for (Column c : this.tableInstance.getColumns()) {
				if (c instanceof AttributeColumn) {
					if (((AttributeColumn) c).getAttribute() == eStructuralFeature) {
						return true;
					}
				} else if (c instanceof FacetAttributeColumn) {
					if (((FacetAttributeColumn) c).getAttribute() == eStructuralFeature) {
						return true;
					}
				}
			}
		}

		return false;
	}

	@Override
	public boolean setFocus() {
		return this.natTable.setFocus();
	}

	/**
	 * Sort columns by type of link (attributes first, then references, then
	 * derived references, ...)
	 */
	public void sortColumnsByType() {
		try {
			this.natTable.setRedraw(false);
			this.listenReorderEvent = false;

			int[] order = new int[this.tableInstance.getColumns().size()];
			int i = 0;
			for (int j = 0; j < this.tableInstance.getColumns().size(); j++) {
				Column columnDescription = this.tableInstance.getColumns().get(j);
				if (columnDescription instanceof DefaultLabelColumn) {
					order[i++] = j;
					break;
				}
			}
			for (int j = 0; j < this.tableInstance.getColumns().size(); j++) {
				Column columnDescription = this.tableInstance.getColumns().get(j);
				if (columnDescription instanceof ContextColumn
						|| columnDescription instanceof EContainerColumn
						|| columnDescription instanceof MetaClassColumn) {
					order[i++] = j;
				}
			}

			for (int j = 0; j < this.tableInstance.getColumns().size(); j++) {
				Column columnDescription = this.tableInstance.getColumns().get(j);
				if (columnDescription instanceof AttributeColumn) {
					order[i++] = j;
				}
			}

			// positions of references
			List<Integer> links = new ArrayList<Integer>();
			for (int j = 0; j < this.tableInstance.getColumns().size(); j++) {
				Column columnDescription = this.tableInstance.getColumns().get(j);
				if (columnDescription instanceof ReferenceColumn) {
					links.add(new Integer(j));
				}
			}

			for (Integer linkPos : links) {
				order[i++] = linkPos.intValue();
			}

			for (int j = 0; j < this.tableInstance.getColumns().size(); j++) {
				Column columnDescription = this.tableInstance.getColumns().get(j);
				if (columnDescription instanceof QueryColumn) {
					order[i++] = j;
				}
			}

			CompoundCommand compoundCommand = new CompoundCommand();
			for (int index = 0; index < order.length; index++) {
				int pos = order[index];
				EObject element = NatTableWidget.this.tableInstance.getColumns().get(pos);
				Command command = this.commandFactory.createMoveCommand(
						NatTableWidget.this.editingDomain, NatTableWidget.this.tableInstance,
						TableinstancePackage.eINSTANCE.getTableInstance_Columns(), element, index);
				compoundCommand.append(command);
			}
			NatTableWidget.this.editingDomain.getCommandStack().execute(compoundCommand);
		} catch (Exception e) {
			Logger.logError(e, "Error sorting columns by type", //$NON-NLS-1$
					Activator.getDefault());
		} finally {
			this.natTable.setRedraw(true);
			this.listenReorderEvent = true;
		}
	}

	/** Hide columns which don't contain any elements */
	public void setHideEmptyColumns(final boolean hideEmptyColumns) {
		setWithCommand(NatTableWidget.this.tableInstance,
				TableinstancePackage.eINSTANCE.getTableInstance_HideEmptyColumns(),
				hideEmptyColumns);
		updateHiddenColumns();
	}

	private boolean isEmptyColumn(final Column column) {
		if (column instanceof AttributeColumn) {
			AttributeColumn attributeColumn = (AttributeColumn) column;
			return isEmpty(attributeColumn.getAttribute());
		} else if (column instanceof ReferenceColumn) {
			ReferenceColumn referenceColumn = (ReferenceColumn) column;
			return isEmpty(referenceColumn.getReference());
		}
		return false;
	}

	/**
	 * @author Gregoire Dupe
	 */
	private void updateHiddenColumns() {
		this.customizationEngine.loadCustomizations();
		this.fBodyLayer.getColumnHideShowLayer().showAllColumns();
		List<Integer> columnsToHide = new ArrayList<Integer>();
		List<EObject> tableContents = this.tableInstance.getElements();
		try {
			this.natTable.setRedraw(false);
			int nColumn = 0;
			for (final Column column : this.tableInstance.getColumns()) {
				boolean isHidden = false;
				boolean isEmpty = getTableInstance().isHideEmptyColumns() && isEmptyColumn(column);
				boolean isNonCommon = getTableInstance().isOnlyShowCommonColumns()
						&& isNonCommonColumns(column);

				if ((this.tableInstance instanceof TableInstance2) && column instanceof FeatureColumn && !tableContents.isEmpty()) {

					EStructuralFeature columnFeature = ((FeatureColumn) column)
							.getFeature();
					if (column instanceof FacetAttributeColumn) {
						isHidden = !this.customizationEngine.isAttributeVisibleStatic(
								(EClass) columnFeature.eContainer(), columnFeature.getName());
					} else if (column instanceof FacetReferenceColumn) {
						isHidden = !this.customizationEngine.isReferenceVisibleStatic(
								(EClass) columnFeature.eContainer(), columnFeature.getName());
					} else {
						Set<EClass> featureOwner = EMFUtils.getAllFeatureOwner(tableContents,
								columnFeature);
						isHidden = getHiddenColumnStatus((FeatureColumn) column, featureOwner);
					}
				} else {
					isHidden = column.isIsHidden();
				}

				if (isHidden || isNonCommon || isEmpty) {
					int converted = convertColumnPositionDataToHideShow(nColumn);
					// if not already hidden
					if (converted != -1) {
						columnsToHide.add(new Integer(converted));
					}
				}
				nColumn++;
			}
		} finally {
			this.natTable.setRedraw(true);
		}
		this.fBodyLayer.getColumnHideShowLayer().hideColumnPositions(columnsToHide);
	}

	/**
	 * Determines the status (hidden or visible) for a column
	 *
	 * @param featureColumn
	 *            a Column to show or hide
	 * @param featureOwners
	 *            elements which own the feature
	 * @return the status of the column hidden or visible, using the file
	 *         uiCustom
	 */
private boolean getHiddenColumnStatus(final FeatureColumn c, final Set<EClass> featureOwners) {
		if (c instanceof AttributeColumn) {
			return getHiddenAttributeColumnStatus((AttributeColumn) c, featureOwners);
		} else if (c instanceof ReferenceColumn) {
			return getHiddenReferenceColumnStatus((ReferenceColumn) c, featureOwners);
		} else if (c instanceof FacetAttributeColumn) {
			return !this.customizationEngine.isAttributeVisibleStatic((EClass) c.getFeature()
					.eContainer(), c.getFeature().getName());
		} else if (c instanceof FacetReferenceColumn) {
			return !this.customizationEngine.isReferenceVisibleStatic((EClass) c.getFeature()
					.eContainer(), c.getFeature().getName());
		}
		return false;
	}

	private boolean getHiddenAttributeColumnStatus(final AttributeColumn c,
			final Set<EClass> featureOwners) {
		boolean conflict = false;
		EStructuralFeature feature = c.getFeature();
		String featureName = feature.getName();
		Iterator<EClass> iter = featureOwners.iterator();
		if (iter.hasNext()) {
			boolean visibility = this.customizationEngine.isAttributeVisibleStatic(iter.next(),
					featureName);
			while (iter.hasNext()) {
				if (visibility != this.customizationEngine.isAttributeVisibleStatic(iter.next(),
						featureName)) {
					conflict = true;
					break;
				}
			}
			if (conflict) {
				EClass commonSupertype = EMFUtils.computeLeastCommonSupertype(featureOwners,
						feature);
				visibility = this.customizationEngine.isAttributeVisibleStatic(commonSupertype,
						featureName);
			}
			return !visibility;
		}
		return false;
	}

	private boolean getHiddenReferenceColumnStatus(final ReferenceColumn c,
			final Set<EClass> featureOwner) {
		boolean conflict = false;
		EStructuralFeature feature = c.getFeature();
		String featureName = feature.getName();
		Iterator<EClass> iter = featureOwner.iterator();
		if (iter.hasNext()) {
			boolean visibility = this.customizationEngine.isReferenceVisibleStatic(iter.next(),
					featureName);
			while (iter.hasNext()) {
				if (visibility != this.customizationEngine.isReferenceVisibleStatic(iter.next(),
						featureName)) {
					conflict = true;
					break;
				}
			}
			if (conflict) {
				EClass commonSupertype = EMFUtils
						.computeLeastCommonSupertype(featureOwner, feature);
				visibility = this.customizationEngine.isReferenceVisibleStatic(commonSupertype,
						featureName);
			}
			return !visibility;
		}
		return false;
	}

	/**
	 * Open a selection popup to select which columns to show/hide.
	 */
	public void selectColumnsToHide() {
		openColumnsToHideDialog();
	}

	public IColumnsToHideDialog openColumnsToHideDialog() {
		ILabelProvider labelProvider = new LabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof Column) {
					Column column = (Column) element;
					return NatTableWidgetInternalUtils.getColumnName(column);
				}
				return null;
			}
		};
		final Comparator<Column> columnComparator = new ColumnComparator();
		final SortedColumnContentProvider contentProvider = new SortedColumnContentProvider();
		contentProvider.setComparator(columnComparator);
		contentProvider.setIsSorted(false);
		String mustPutOnTheTop = Activator.getDefault().getPreferenceStore()
				.getString(PreferenceConstants.MUST_PUT_ON_THE_TOP_THE_LOCAL_CUSTOM);
		final boolean localCustomizatinMustBeTheTop = mustPutOnTheTop.equals(NatTablePreferencePage.YES_VALUE);
		final boolean askToPutOnTheTopTheLocalCustomization = !Activator.getDefault().getPreferenceStore()
				.getBoolean(PreferenceConstants.MUST_NOT_ASK_THE_USER_TO_PUT_ON_THE_TOP_THE_LOCAL_CUSTOM);
		final ColumnsToHideDialog dialog = new ColumnsToHideDialog(getShell(), getTableInstance()
				.getColumns(), labelProvider, contentProvider, isOneOfTheLocalCustomizationsNotOnTheTop() && askToPutOnTheTopTheLocalCustomization, localCustomizatinMustBeTheTop) {
			@Override
			protected void okPressed() {
				super.okPressed();
				if (getResult() != null) {
					NatTableWidget.this.showHideColumns(getSelectedColumns(), getPutOnTheTop() && isOneOfTheLocalCustomizationsNotOnTheTop());
					NatTableWidget.setMustAskThUserToPutOnTheTopTheLocalCustomizationNextTime(
							getMustAskTheUserNextTime(), getPutOnTheTop());
				}
			}
			@Override
			public List<Column> getVisibleColumns(final boolean putOnTheTop) {
				return NatTableWidget.this.getVisibleColumns(putOnTheTop);
			}
		};
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				dialog.open();
			}
		});
		return dialog;
	}

	private static void setMustAskThUserToPutOnTheTopTheLocalCustomizationNextTime(
			final boolean mustAskTheUserNextTime, final boolean putOnTheTop) {
		Activator.getDefault().getPreferenceStore()
				.setValue(PreferenceConstants.MUST_NOT_ASK_THE_USER_TO_PUT_ON_THE_TOP_THE_LOCAL_CUSTOM, !mustAskTheUserNextTime);
		if (!mustAskTheUserNextTime) { // we doesn't display the message the next time
			if  (putOnTheTop) {
				Activator
				.getDefault()
				.getPreferenceStore()
				.setValue(PreferenceConstants.MUST_PUT_ON_THE_TOP_THE_LOCAL_CUSTOM,
						NatTablePreferencePage.YES_VALUE);
			} else {
				Activator
						.getDefault()
						.getPreferenceStore()
						.setValue(PreferenceConstants.MUST_PUT_ON_THE_TOP_THE_LOCAL_CUSTOM,
								NatTablePreferencePage.NO_VALUE);
			}
		}
	}

	public void showHideColumns(final List<Column> result, final boolean putOnTheTop) {
		//we build the list of the columns to show/hide
		List<Column> columnsToShow = new ArrayList<Column>();
		List<Column> columnsToHide = new ArrayList<Column>();
		EList<Column> columns = getTableInstance().getColumns();
		for (Column c : columns) {
			boolean mustBeHidden = !result.contains(c);
			if (NatTableWidget.this.tableInstance instanceof TableInstance2) {
				if (mustBeHidden == getVisibleColumns(putOnTheTop).contains(c)) {
					if (mustBeHidden) {
						columnsToHide.add(c);
					} else {
						columnsToShow.add(c);
					}
				}
			} else { // we continue to store the visibility of the
						// column in the file.table!
				if (mustBeHidden != c.isIsHidden()) {
					if (mustBeHidden) {
						columnsToHide.add(c);
					} else {
						columnsToShow.add(c);
					}
				}
			}
		}
		NatTableWidget.this.showHideColumns(columnsToShow, columnsToHide, putOnTheTop);
	}

	public boolean isOneOfTheLocalCustomizationsNotOnTheTop() {
		List<MetamodelView> view = this.tableInstance.getCustomizations();
		List<MetamodelView> locals = getLocalCustomizations();
		if (this.tableInstance instanceof TableInstance2) {
			if (view.size() < locals.size()) {
				throw new RuntimeException("Some local customization files are not loaded"); //$NON-NLS-1$
			}
			for (int i = 0; i < locals.size(); i++) {
					if (!locals.contains(view.get(i))) {
						// one of the local custom is not on the top of the
						// customization
						// engine
						return true;
					}
			}
		}
		return false;
	}

	protected List<MetamodelView> getLocalCustomizations() {
		List<MetamodelView> locals = new ArrayList<MetamodelView>();
		if (this.tableInstance.getLocalCustomization() != null) {
			locals.add(this.tableInstance.getLocalCustomization());
		}
		if (this.tableInstance instanceof TableInstance2) {
			locals.addAll(((TableInstance2) this.tableInstance).getLocalCustomizations());
		}
		return locals;
	}

	public List<Column> getNotHiddenColumns(final int setAtTheTop, final boolean atTheTop) {
		List<Column> notHiddenColumns = new ArrayList<Column>();
		if (this.tableInstance instanceof TableInstance2) {
			// we always show the state of the localCustom
			if (atTheTop) { // we show the current state of the localCustom
				notHiddenColumns = getVisibleColumnsUsingTheLocalCustomizations(
						(TableInstance2) this.tableInstance, false);
			} else if (setAtTheTop == IDialogConstants.NO_ID) {
			// we show the
																// current state
																// of the local
																// custom
				notHiddenColumns = getVisibleColumnsUsingTheLocalCustomizations(
						(TableInstance2) this.tableInstance, false);
			} else if (setAtTheTop == IDialogConstants.YES_ID) {
			// we show the
																// future state
																// of the local
																// custom
				notHiddenColumns = getVisibleColumnsUsingTheLocalCustomizations(
						(TableInstance2) this.tableInstance, true);
			}
		} else {
			// we use the model of the table
			notHiddenColumns = getVisibleColumnsUsingTheModel();
		}
		return notHiddenColumns;
	}

	private void showHideColumns(final List<Column> columnsToShow, final List<Column> columnsToHide,
 final boolean putOnTheTop) {
		Command showHideColumnsCommand = TableInstanceCommandFactory.createShowHideColumnCommand(
				this, columnsToShow, columnsToHide, putOnTheTop);
		if (showHideColumnsCommand.canExecute()) {
			this.editingDomain.getCommandStack().execute(showHideColumnsCommand);
		}
		this.customizationEngine.loadCustomizations();
		updateHiddenColumns();
	}

	public void putLocalCustomizationOnTheTop() {
		this.editingDomain.getCommandStack().execute(
				TableInstanceCommandFactory.createPutLocalCustomizationOnTheTopCommand(this));
		loadCustomizationsInCustomEngine(this.tableInstance.getCustomizations());
	}

	public List<Column> getVisibleColumnsUsingCustomizationEngine(final TableInstance2 table) {
		List<Column> visibleColumns = new ArrayList<Column>();
		List<Column> columns = table.getColumns();
		for (int i = 0; i < columns.size(); i++) {
			Column column = columns.get(i);
			if (column instanceof FeatureColumn) {
				Set<EClass> owners = EMFUtils.getAllFeatureOwner(table.getElements(),
						((FeatureColumn) column).getFeature());
				if (!getHiddenColumnStatus((FeatureColumn) column, owners)) {
					visibleColumns.add(column);
				}
			} else {
				if (!column.isIsHidden()) {
					visibleColumns.add(column);
				}
			}
		}
		return visibleColumns;
	}

	/**
	 *
	 * @param table
	 * a table
	 * @param satAtTheTop
	 * if(<code>true</code>, we calculate the future state of the localCustom
	 * @return
	 *  all the columns which are visible, using the local customization file
	 */
	public List<Column> getVisibleColumnsUsingTheLocalCustomizations(final TableInstance2 table,
			final boolean setToTheTop) {
		List<Column> notHiddenColumns = new ArrayList<Column>();
		List<MetamodelView> locals = getLocalCustomizations();

		// we build a local customization engine
		CustomizationEngine localEngine = new CustomizationEngine();
		if (!setToTheTop) {
			for (MetamodelView current : locals) {
				localEngine.registerMetamodel(current.getMetamodelURI());
				localEngine.registerCustomization(current);
			}
		} else {

			for (MetamodelView current : this.tableInstance.getCustomizations()) {
				localEngine.registerCustomization(current);
				localEngine.registerMetamodel(current.getMetamodelURI());
			}
			// we duplicate the customization engine
//			localEngine.getRegisteredCustomizations().addAll(
//					this.customizationEngine.getRegisteredCustomizations());
//			localEngine.getRegisteredMetamodels().addAll(
//					this.customizationEngine.getRegisteredMetamodels());
//
//			// we set the local customization at the top
//			localEngine.getRegisteredCustomizations().removeAll(getLocalCustomizations());
//			localEngine.getRegisteredCustomizations().addAll(0, getLocalCustomizations());
		}
		localEngine.loadCustomizations();
		EList<Column> allColumns = table.getColumns();
		for (int i = 0; i < allColumns.size(); i++) {
			Column c = allColumns.get(i);
			boolean isVisible = true;
			if (c instanceof FeatureColumn) {
				EStructuralFeature feature = ((FeatureColumn) c).getFeature();
				String featureName = feature.getName();
				if (c instanceof FacetAttributeColumn) {
					isVisible = this.customizationEngine.isAttributeVisibleStatic(
							(EClass) feature.eContainer(), featureName);
				} else if (c instanceof FacetReferenceColumn) {
					isVisible = this.customizationEngine.isReferenceVisibleStatic(
							(EClass) feature.eContainer(), featureName);
				} else {
					Set<EClass> featureOwners = EMFUtils.getAllFeatureOwner(
							this.tableInstance.getElements(), feature);
					EClass commonSupertype = EMFUtils.computeLeastCommonSupertype(featureOwners,
							feature);
					if (c instanceof AttributeColumn) {
						isVisible = localEngine.isAttributeVisibleStatic(commonSupertype,
								featureName);
					} else if (c instanceof ReferenceColumn) {
						isVisible = localEngine.isReferenceVisibleStatic(commonSupertype,
								featureName);
					}
				}
			} else {
				isVisible = !c.isIsHidden();
			}
			if (isVisible) {
				notHiddenColumns.add(c);
			}
		}
		return notHiddenColumns;
	}

	/**
	 *
	 * @param table
	 *            the table
	 * @return returns the visible columns using the model
	 */
	private List<Column> getVisibleColumnsUsingTheModel() {
		List<Column> notHiddenColumns = new ArrayList<Column>();
		EList<Column> allColumns = this.tableInstance.getColumns();
		Iterator<Column> iter = allColumns.iterator();
		while (iter.hasNext()) {
			Column current = iter.next();
			if (!current.isIsHidden()) {
				notHiddenColumns.add(current);
			}
		}
		return notHiddenColumns;
	}

	@Deprecated
	public void hideColumns() {
		selectColumnsToHide();
	}

	/** Whether no element has this reference set */
	private boolean isEmpty(final EStructuralFeature reference) {
		for (EObject eObject : this.tableInstance.getElements()) {
			if (hasStructuralFeature(eObject, reference)) {
				final Object value = getStructuralFeatureValue(eObject, reference);
				if (reference.isMany() && value instanceof List<?>) {
					final List<?> list = (List<?>) value;
					if (!list.isEmpty()) {
						return false;
					}
				} else {
					if (value != null) {
						return false;
					}
				}
			}
		}
		return true;
	}

	/**
	 * Whether to show columns which are specific to a few elements (false), or
	 * only columns which are common to all the elements (true)
	 */
	public void setOnlyShowCommonColumns(final boolean onlyShowCommonColumns) {
		setWithCommand(NatTableWidget.this.tableInstance,
				TableinstancePackage.eINSTANCE.getTableInstance_OnlyShowCommonColumns(),
				onlyShowCommonColumns);
	}

	/**
	 * @return true if there is a cell selected
	 */
	public boolean isCellSelected() {
		return this.fBodyLayer.getSelectionLayer().getSelectedCells().length > 0;
	}

	/**
	 * @return true if there is a column selected
	 */
	public boolean isColumnSelected() {
		return getSelectedColumns().size() > 0;
	}

	/**
	 * remove the selected line from the table
	 */
	public void removeLine() {
		//TODO This method has to be deprecated and replaced by removeLine2
		Command removeLineCommand;
		try {
			removeLineCommand = TableInstanceCommandFactory.createRemoveLineCommand(this);
			this.editingDomain.getCommandStack().execute(removeLineCommand);
		} catch (CoreException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * remove the selected line from the table
	 * @throws CoreException may be caused by query exception in the facet framework
	 */
	public void removeLine2() throws CoreException {
		Command removeLineCommand;
		removeLineCommand = TableInstanceCommandFactory.createRemoveLineCommand(this);
		this.editingDomain.getCommandStack().execute(removeLineCommand);
	}

	/**
	 * Delete the selected elements from their containing resource
	 */
	public void deleteSelection() {
		//TODO This method has to deprecated and replaced by deleteSelection2
		if (isCellsDeletable()) {
			try {
				deleteSelectedElement();
			} catch (CoreException e) {
				throw new RuntimeException(e);
			}
		} else if (isSelectedColumnDeletable()) {
			deleteSelectedColumn();
		}
	}

	/**
	 * Delete the selected elements from their containing resource
	 * @throws CoreException may be cause by query exception in the facet framework
	 */
	public void deleteSelection2() throws CoreException {
		if (isCellsDeletable()) {
			deleteSelectedElement();
		} else if (isSelectedColumnDeletable()) {
			deleteSelectedColumn();
		}
	}

	/**
	 * Delete the selected column from its containing resource
	 */
	private void deleteSelectedColumn() {
		Command deleteSelectedColumnCommand = TableInstanceCommandFactory
				.createDeleteSelectedColumnCommand(this);
		this.editingDomain.getCommandStack().execute(deleteSelectedColumnCommand);
	}

	public List<Column> getSelectedColumns() {
		SelectionLayer selectionLayer = this.fBodyLayer.getSelectionLayer();
		List<Column> listColumn = new ArrayList<Column>();
		List<LayerCell> selectedCells = getSelectedCells();
		for (int colPos : selectionLayer.getSelectedColumns()) {
			if (isColumnSelected(colPos, selectedCells)) {
				int pos = this.fBodyLayer.getSelectionLayer().getColumnIndexByPosition(colPos);
				if (this.tableInstance.getColumns().get(pos) != null) {
					Column localColumn = this.tableInstance.getColumns().get(pos);
					listColumn.add(localColumn);
				}
			} else {
				// It mean that we have at least one column which is not fully
				// selected
				return null;
			}
		}
		return listColumn;
	}

	private boolean isColumnSelected(final int colPos, final List<LayerCell> selectedCells) {
		for (int rowPos = 0; rowPos < this.fBodyLayer.getRowCount(); rowPos++) {
			LayerCell localCell = this.fBodyLayer.getSelectionLayer().getCellByPosition(colPos, rowPos);
			if (localCell != null) {
				if (!selectedCells.contains(localCell)) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * Delete the selected elements from their containing resource
	 * @throws CoreException ay be caused by query exception in the facet framework.
	 */
	private void deleteSelectedElement() throws CoreException {
		Command deleteSelectedElementCommand = TableInstanceCommandFactory
				.createDeleteSelectedElementCommand(this, this.commandFactory);
		this.editingDomain.getCommandStack().execute(deleteSelectedElementCommand);
	}

	/**
	 * Open the query selection dialog Add a column which contains the result of
	 * the query
	 */
	public void addQueryColumn() {
		QuerySelectionDialog query = new QuerySelectionDialog(getShell(), false, null, null);
		query.open();
		if (query.getReturnCode() == Window.OK) {
			CompoundCommand compoundCommand = new CompoundCommand();

			QueryColumn queryColumn = TableinstanceFactory.eINSTANCE.createQueryColumn();
			if (!query.getSelectedQueries().isEmpty()) {
				queryColumn.setQuery(query.getSelectedQueries().get(0));

				Command command = this.commandFactory.createAddCommand(this.editingDomain,
						this.tableInstance,
						TableinstancePackage.eINSTANCE.getTableInstance_Columns(), queryColumn);
				compoundCommand.append(command);
				this.editingDomain.getCommandStack().execute(compoundCommand);
			}
		}
	}

	/**
	 * @return The history of launched instantiation methods.
	 */
	public List<InstantiationMethod> getInstantionMethodHistory() {
		return this.launchedInstMethHistory;
	}

	/**
	 * Open a dialog to choose the default instantiationMethod to perform when
	 * clicking on the "Create" button.
	 */
	public void selectInstantiationMethodToLaunch() {

		ILabelProvider labelProvider = new LabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof InstantiationMethod) {
					InstantiationMethod meth = (InstantiationMethod) element;
					return meth.getName();
				}
				return null;
			}
		};

		IStructuredContentProvider contentProvider = new IStructuredContentProvider() {
			private Object[] objects;

			public void inputChanged(final Viewer viewer, final Object oldInput,
					final Object newInput) {
				if (newInput instanceof List) {
					List<?> list = (List<?>) newInput;
					this.objects = list.toArray();
				}
			}

			public void dispose() {
				// Nothing
			}

			public Object[] getElements(final Object inputElement) {
				return this.objects;
			}
		};

		ListDialog dialog = new ListDialog(getShell());
		dialog.setContentProvider(contentProvider);
		dialog.setLabelProvider(labelProvider);
		dialog.setInput(this.launchedInstMethHistory);
		dialog.setTitle(Messages.NatTableWidget_instantiationMethodSelectionTitle);
		if (dialog.open() != Window.CANCEL) {
			if (dialog.getResult()[0] instanceof InstantiationMethod) {
				InstantiationMethod meth = (InstantiationMethod) dialog.getResult()[0];
				this.launchedInstMethHistory.remove(meth);
				this.launchedInstMethHistory.add(0, meth);
				createNewElement();
			}
		}
	}

	/**
	 * Call the last launched {@link InstantiationMethod} and add the result to
	 * the table if any.
	 */
	public void createNewElement() {
		if (this.launchedInstMethHistory.get(0) != null) {
			createNewElement(this.launchedInstMethHistory.get(0));
		} else {
			selectInstantiationMethodToLaunch();
		}
	}

	public void createNewElement(final InstantiationMethod instantiationMethod) {
		try {
			createNewElement2(instantiationMethod);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Call the {@link InstantiationMethod} and add the result to the table if
	 * any.
	 */
	public void createNewElement2(final InstantiationMethod instantiationMethod) throws Exception {

		ModelQuery query = instantiationMethod.getInstantiationQuery();
		if (query != null) {
				List<ModelQueryParameterValue> parameters = new ArrayList<ModelQueryParameterValue>();

				// Free parameter value
				ModelQueryParameterValue freeParameterValue = RuntimeFactory.eINSTANCE
						.createModelQueryParameterValue();
				freeParameterValue.setParameter(InstantiationMethodParameters.getFreeParemeter());
				freeParameterValue.setValue(getParameter());

				// Add free parameter to the list
				parameters.add(freeParameterValue);
				Object object = null;
				AbstractModelQuery queryImpl = ModelQuerySetCatalog.getSingleton().getModelQueryImpl(query);
			if (queryImpl instanceof AbstractModelQueryWithEditingDomain) {
				AbstractModelQueryWithEditingDomain queryImplWithEditingDomain = (AbstractModelQueryWithEditingDomain) queryImpl;
				if (queryImplWithEditingDomain.getModelQuery() instanceof IJavaModelQueryWithEditingDomain<?, ?>) {
					object = queryImplWithEditingDomain.basicEvaluate(getContext(), parameters,
							this.editingDomain);
				} else {
					ModelQueryParameterValue editingDomainValue = RuntimeFactory.eINSTANCE
							.createModelQueryParameterValue();
					editingDomainValue.setParameter(InstantiationMethodParameters
							.getEditingDomainParameter());
					editingDomainValue.setValue(this.editingDomain);
					parameters.add(editingDomainValue);
					object = queryImplWithEditingDomain.basicEvaluate(getContext(), parameters);
				}
			} else {
				throw new Exception();
			}
				if (object != null) {
					// Update the table
					if (object instanceof EObject) {
						List<EObject> list = new ArrayList<EObject>();
						list.add((EObject) object);
						addRows(list);
					}
				}
				// Store the InstantiationMethod as default
				this.launchedInstMethHistory.remove(instantiationMethod);
				this.launchedInstMethHistory.add(0, instantiationMethod);

		}
	}

	private boolean isNonCommonColumns(final Column columnDescription) {
		boolean common = true;
		if (columnDescription instanceof AttributeColumn) {
			AttributeColumn attributeColumn = (AttributeColumn) columnDescription;
			common = isCommon(attributeColumn.getAttribute());
		} else if (columnDescription instanceof ReferenceColumn) {
			ReferenceColumn referenceColumn = (ReferenceColumn) columnDescription;
			common = isCommon(referenceColumn.getReference());
		}
		return !common;
	}

	/** @return whether the given feature is common to all the given metaclasses */
	private boolean isCommon(final EStructuralFeature feature) {
		for (final EClass metaclass : this.metaclasses) {
			if (!metaclass.getEAllStructuralFeatures().contains(feature)) {
				return false;
			}
		}
		return true;
	}

	/** @return the EObject under the given GridElement, if any. */
	private static EObject getEObject(final GridElement gridElement) {
		if (gridElement == null) {
			return null;
		}
		final EObject mainModelElement;
		if (gridElement.getElement() instanceof EObject) {
			mainModelElement = (EObject) gridElement.getElement();
		} else {
			mainModelElement = null;
		}

		EObject targetObject = null;
		final Column column = gridElement.getColumnDescription();
		final Object value;
		if (column instanceof AttributeColumn) {
			AttributeColumn attributeColumn = (AttributeColumn) column;
			if (mainModelElement != null
					&& hasStructuralFeature(mainModelElement, attributeColumn.getAttribute())) {
				value = getStructuralFeatureValue(mainModelElement, attributeColumn.getAttribute());
				if (value instanceof EObject) {
					targetObject = (EObject) value;
				}
			}
		} else if (column instanceof ReferenceColumn) {
			ReferenceColumn referenceColumn = (ReferenceColumn) column;
			if (mainModelElement != null
					&& hasStructuralFeature(mainModelElement, referenceColumn.getReference())) {
				value = getStructuralFeatureValue(mainModelElement, referenceColumn.getReference());
				if (referenceColumn.getReference().isMany()) {
					@SuppressWarnings("unchecked")
					final EList<EObject> list = (EList<EObject>) value;
					if (list.size() == 1) {
						targetObject = list.get(0);
					}
				} else {
					targetObject = (EObject) value;
				}
			}
		} else if (column instanceof EContainerColumn) {
			if (mainModelElement != null) {
				value = mainModelElement.eContainer();
				if (value instanceof EObject) {
					targetObject = (EObject) value;
				}
			}
		} else if (column instanceof ContextColumn) {
			if (gridElement instanceof GridElementWithContext) {
				GridElementWithContext gridElementWithContext = (GridElementWithContext) gridElement;
				targetObject = gridElementWithContext.getContext();
			}
		} else if (column instanceof DefaultLabelColumn) {
			targetObject = mainModelElement;
		}

		if (targetObject != null) {
			return targetObject;
		}
		return null;
	}

	private ChangeListener fCustomizationEngineChangeListener;

	private void setupContextMenu() {
		if (this.menuMgr != null) {
			Menu menu = this.menuMgr.createContextMenu(this.natTable);
			this.natTable.setMenu(menu);
		}
	}

	protected BodyLayerStack getBodyLayer() {
		return this.fBodyLayer;
	}

	protected GridLayer getGridLayer() {
		return this.gridLayer;
	}

	public TableLabelProvider getTableLabelProvider() {
		return this.tableLabelProvider;
	}

	/** This method perform to usual "copy to clipboard" action. */
	public void copySelectionToClipboard() {
		new CopyToClipboardCommandHandler(this.fBodyLayer.getSelectionLayer(), this)
				.doCommand(new CopyDataToClipboardCommand(new Clipboard(Display.getDefault()),
						"\t", "\n")); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public List<Column> getColumns() {
		return Collections.unmodifiableList(this.tableInstance.getColumns());
	}

	/**
	 * Return the position of a cell in grid coordinates, from a pixel location
	 * relative to the table control.
	 */
	public PositionCoordinate getCellPositionAt(final int x, final int y) {
		int column = this.natTable.getColumnPositionByX(x);
		int row = this.natTable.getRowPositionByY(y);

		PositionCoordinate convertedPosition = LayerCommandUtil.convertPositionToTargetContext(
				new PositionCoordinate(this.gridLayer, column, row),
				this.fBodyLayer.getSelectionLayer());

		return convertedPosition;
	}

	/** Return a cell from grid coordinates */
	public LayerCell getCellAt(final int column, final int row) {
		return this.fBodyLayer.getSelectionLayer().getCellByPosition(column, row);
	}

	public void loadCustomizations(final List<MetamodelView> customizationsToLoad) {
		this.editingDomain.getCommandStack().execute(
				TableInstanceCommandFactory.createLoadCustomizationsCommand(customizationsToLoad, this));
		loadCustomizationsInCustomEngine(customizationsToLoad);
	}

	private void loadCustomizationsInCustomEngine(final List<MetamodelView> customizationsToLoad) {
		List<MetamodelView> locals = getLocalCustomizations();
		if (this.tableInstance.getElements().size() != 0) {
			// if we get an error here, it is the fault of the developper!!!
			// the local customs should always be contained by
			// customizationsToLoad
			// Assert.isTrue(!locals.isEmpty());
			if (locals.isEmpty()) {
				//Here I choosen a logger to avoid to break the editor whereas he can partialy work.
				Logger.logError("The local customs should always be contained by customizationsToLoad", Activator.getDefault()); //$NON-NLS-1$
			}
		}
		if (!locals.isEmpty() && !customizationsToLoad.containsAll(locals)) {
			// if we get an error here, it is the fault of the developper!!!
			// the local customs should always be contained by
			// customizationsToLoad
			// Assert.isTrue(customizationsToLoad.containsAll(locals));
			// Here I choosen a logger to avoid to break the editor whereas he can partialy work.
			Logger.logError("The local customs should always be contained by customizationsToLoad", Activator.getDefault()); //$NON-NLS-1$
		}
		if (this.fCustomizationEngineChangeListener == null) {
			this.fCustomizationEngineChangeListener = new ChangeListener() {
				public void changed() {
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							if (!NatTableWidget.this.natTable.isDisposed()) {
								NatTableWidget.this.natTable.redraw();
							}
						}
					});
				}
			};
			this.customizationEngine.addChangeListener(this.fCustomizationEngineChangeListener);
		}

		this.customizationEngine.clear();
		for (final MetamodelView customizationToLoad : customizationsToLoad) {
			/*
			 * see bug 341222 customization on feature defined in another metamodel
			 * for the following line! we can define that eAnnotation column is hidden for IModelElement
			 * and hide this column with UML element! whitout this line ,it is not possible
			 */
			this.customizationEngine.registerMetamodel(customizationToLoad.getMetamodelURI());
			this.customizationEngine.registerCustomization(customizationToLoad);
		}
		this.customizationEngine.loadCustomizations();
		updateHiddenColumns();
	}

	public CustomizationEngine getCustomizationEngine() {
		return this.customizationEngine;
	}

	private final FacetContext facetContext = new FacetContext();
	private Action loadCustomizationsAction;
	private Action loadFacetsAction;
	private final List<IActionDelegate> actionList = new ArrayList<IActionDelegate>();

	public FacetContext getFacetContext() {
		return this.facetContext;
	}

	public void setFacets(final Collection<Facet> facets) throws CoreException {
		if (NatTableWidget.DEBUG_SET_FACET) {
			System.out.println(this.getClass().getName() + ".setFacets(): " + facets.size()); //$NON-NLS-1$
			for (Facet facet : facets) {
				System.out.println(this.getClass().getName() + ".setFacets(): " //$NON-NLS-1$
						+ facet.getFacetSet().getName() + ":" + facet.getName()); //$NON-NLS-1$
			}
		}
		this.facetContext.clear();
		this.facetContext.addFacets(facets);
		if (this.tableInstance instanceof TableInstance2) {
			Command setFacetsComand = TableInstanceCommandFactory.createSetFacetsCommand(facets,
					this);
			this.editingDomain.getCommandStack().execute(setFacetsComand);
		}
		if (NatTableWidget.DEBUG_SET_FACET) {
			int nbColumns = this.tableInstance.getColumns().size();
			int nbVisibleColumnsUsingTheLayer = getVisibleColumnsUsingTheLayer().size();
			int nbNonFacetColumns = 0;
			for (Column column : this.tableInstance.getColumns()) {
				if (!(column instanceof FacetAttributeColumn || column instanceof FacetReferenceColumn)) {
					nbNonFacetColumns++;
				}
			}
			System.out.println(this.getClass().getName()
					+ ".setFacets(): nbColumns=" + nbColumns); //$NON-NLS-1$
			System.out
					.println(this.getClass().getName()
							+ ".setFacets(): nbVisibleColumnsUsingTheLayer=" + nbVisibleColumnsUsingTheLayer); //$NON-NLS-1$
			System.out.println(this.getClass().getName()
					+ ".setFacets(): nbNonFacetColumns=" + nbNonFacetColumns); //$NON-NLS-1$
		}
	}

	public Collection<Facet> getLoadedFacets() {
		if (this.tableInstance instanceof TableInstance2) {
			return Collections.unmodifiableCollection(((TableInstance2) this.tableInstance)
					.getFacets2());
		}
		ArrayList<Facet> list = new ArrayList<Facet>();
		list.add(this.tableInstance.getFacets());
		return list;
	}

	public void clearFacets() {
		try {
			this.facetContext.clear();
		} catch (Exception e) {
			Logger.logError(e, "Error clearing Facets", Activator.getDefault()); //$NON-NLS-1$
		}
	}

	public Collection<EPackage> getReferencedEPackages() {
		Set<EPackage> ePackages = new HashSet<EPackage>();
		for (EObject eObject : this.tableInstance.getElements()) {
			EClass eClass = eObject.eClass();
			if (eClass != null) {
				EPackage ePackage = eClass.getEPackage();
				if (ePackage != null) {
					ePackages.add(ePackage);
				}
			}
		}
		return ePackages;
	}

	public TableConfiguration getTableConfiguration() {
		return getTableInstance().getTableConfiguration();
	}

	public TableInstance getTableInstance() {
		return this.tableInstance;
	}

	public EObject getContext() {
		return getTableInstance().getContext();
	}

	public Object getParameter() {
		return getTableInstance().getParameter();
	}

	/**
	 * This method takes a list of {@link EObject} and add it to the widget list
	 * if their are not already in it.
	 *
	 * @param newElements
	 *            The list of EObject elements to be added to the table
	 */
	public void addRows(final List<EObject> newElements) {
		CompoundCommand addRowCommand = TableInstanceCommandFactory.createAddRowsCommand(newElements, this);
		if (this.tableInstance instanceof TableInstance2) {
			TableInstance2 tableInstance2 = (TableInstance2) this.tableInstance;
			try {
				this.facetContext.clear();
				this.facetContext.addFacets(tableInstance2.getFacets2());
				Command setFacetsCommand = TableInstanceCommandFactory.createSetFacetsCommand(
						tableInstance2.getFacets2(), newElements, this);
				if (setFacetsCommand != null && setFacetsCommand.canExecute()) {
					addRowCommand.append(setFacetsCommand);
				}
			} catch (CoreException e) {
				throw new RuntimeException(e);
			}
		}
		Command loadCustomizationCommand = TableInstanceCommandFactory
				.createLoadCustomizationsCommand(this.tableInstance.getCustomizations(), this);
		if (loadCustomizationCommand != null && loadCustomizationCommand.canExecute()) {
			addRowCommand.append(loadCustomizationCommand);
		}
		this.editingDomain.getCommandStack().execute(addRowCommand);
	}

	/**
	 * @return True if every selected cells is contained by the
	 *         {@link DefaultLabelColumn} column
	 */
	public boolean isCellsDeletable() {
		if (!isCellSelected()) {
			return false;
		}
		for (LayerCell cell : getSelectedCells()) {
			int position = cell.getOriginColumnPosition();
			int columnIndex = this.fBodyLayer.getColumnHideShowLayer().getColumnIndexByPosition(
					position);
			if (!(getColumns().get(columnIndex) instanceof DefaultLabelColumn)) {
				return false;
			}
		}
		return true;
	}

	public boolean isSelectedColumnDeletable() {
		List<Column> selectedColumn = getSelectedColumns();
		if (selectedColumn == null) {
			return false;
		}
		for (Column column : selectedColumn) {
			if (!(column instanceof QueryColumn)) {
				return false;
			}
		}
		return true;
	}

	public void openLoadCustomizationDialog() {
		if (this.loadCustomizationsAction == null) {
			this.loadCustomizationsAction = new LoadCustomizationsAction(this);
		}
		this.loadCustomizationsAction.run();
		updateHiddenColumns();
	}

	public void openLoadFacetsDialog() {
		if (this.loadFacetsAction == null) {
			this.loadFacetsAction = new LoadFacetsAction(this);
		}
		this.loadFacetsAction.run();
	}

	/**
	 * Open a "Save As" dialog to serialize the table
	 */
	public void saveAs() {
		SaveAsDialog dialog = new SaveAsDialog(getShell());

		dialog.setOriginalName("My.table"); //$NON-NLS-1$
		dialog.open();
		if (dialog.getReturnCode() != Window.CANCEL) {
			URI uri = URI.createPlatformResourceURI(dialog.getResult().toString(), true);
			if (uri != null) {
				this.resource.setURI(uri);
				Map<String, Object> options = new HashMap<String, Object>();
				options.put(XMLResource.OPTION_USE_FILE_BUFFER, new Boolean(true));
				try {
					this.resource.save(options);
				} catch (IOException e) {
					Logger.logError(e, Activator.getDefault());
				}
			}
			if (this.editingDomain != null) {
				((BasicCommandStack) this.editingDomain.getCommandStack()).saveIsDone();
			}
		}
	}

	/**
	 * Performs the usual save action.
	 */
	public void save() {
		final Map<String, Object> options = new HashMap<String, Object>();
		options.put(XMLResource.OPTION_USE_FILE_BUFFER, new Boolean(true));
		options.put(Resource.OPTION_SAVE_ONLY_IF_CHANGED,
				Resource.OPTION_SAVE_ONLY_IF_CHANGED_MEMORY_BUFFER);

		WorkspaceModifyOperation operation = new WorkspaceModifyOperation() {
			@Override
			public void execute(final IProgressMonitor monitor2) {
				// Save the resources to the file system.
				if (NatTableWidget.this.resource == null
						|| (NatTableWidget.this.resource.getURI().lastSegment()
								.equals(NatTableWidget.TMP_FILE_NAME))) {
					saveAs();
				}
				for (Resource resourceElt : NatTableWidget.this.editingDomain.getResourceSet()
						.getResources()) {
					if (isPersisted(resourceElt)) {
						if (!resourceElt.getContents().isEmpty()
								&& !NatTableWidget.this.editingDomain.isReadOnly(resourceElt)) {
							try {
								resourceElt.save(options);

							} catch (Exception e) {
								Logger.logError(e, Activator.getDefault());
							}
						}
					}
				}
			}
		};

		try {
			operation.run(new NullProgressMonitor());
		} catch (InvocationTargetException e) {
			Logger.logError(e, Activator.getDefault());
		} catch (InterruptedException e) {
			Logger.logError(e, Activator.getDefault());
		}

		((BasicCommandStack) this.editingDomain.getCommandStack()).saveIsDone();
	}

	/**
	 * This returns whether something has been persisted to the URI of the
	 * specified resource. The implementation uses the URI converter from the
	 * editor's resource set to try to open an input stream. <!-- begin-user-doc
	 * --> <!-- end-user-doc -->
	 *
	 * This method has been copied from an EMF generated model editor.
	 */
	private boolean isPersisted(final Resource resourceParam) {
		boolean result;
		try {
			InputStream stream = this.editingDomain.getResourceSet().getURIConverter()
					.createInputStream(resourceParam.getURI());
			if (stream != null) {
				result = true;
				stream.close();
			} else {
				result = false;
			}
		} catch (IOException e) {
			result = false;
		}
		return result;
	}

	public boolean isDataEditable() {
		return this.dataEditable;
	}

	protected void drop(final StructuredSelection structuredSelection, final Object element,
			final EStructuralFeature feature) {
		CompoundCommand compoundCommand = TableInstanceCommandFactory.createDropCommand(this,
				structuredSelection, element, feature);
		if (compoundCommand != null && !compoundCommand.isEmpty()) {
			this.editingDomain.getCommandStack().execute(compoundCommand);
		}
	}

	protected boolean canBeDroped(final IStructuredSelection structuredSelection,
			final EStructuralFeature feature, final Object element) {
		CompoundCommand compoundCommand = TableInstanceCommandFactory.createDropCommand(this,
				structuredSelection, element, feature);
		return compoundCommand != null && compoundCommand.canExecute();
	}

	/** Perform an EMF Set Command */
	private void setWithCommand(final Object owner, final Object feature, final Object value) {
		Command cmd = this.commandFactory.createSetCommand(NatTableWidget.this.editingDomain, owner,
				feature, value);
		NatTableWidget.this.editingDomain.getCommandStack().execute(cmd);
	}

	/** Perform the usual "Select All" action on the table's cells */
	public void selectAll() {
		this.natTable.doCommand(new SelectAllCommand());
		this.natTable.redraw();
		tableSelectionChanged();
	}

	/**
	 * Allow the opening of an existing saved .table file within the current
	 * table widget.
	 */
	public void openSavedTable() {
		TableInstance tableInstanceValue = null;
		FileDialog fileDialog = new FileDialog(PlatformUI.getWorkbench().getActiveWorkbenchWindow()
				.getShell(), SWT.OPEN);
		fileDialog.setFilterExtensions(new String[] { "*.table" }); //$NON-NLS-1$
		String path = fileDialog.open();
		if (path != null) {
			Path myPath = new Path(path);
			// find file in workspace
			IFile file = ResourcesPlugin.getWorkspace().getRoot().getFileForLocation(myPath);
			// check if file in workspace
			if (file != null) {
				FileEditorInput fileEditorInput = new FileEditorInput(file);
				URI uri = URI.createFileURI(fileEditorInput.getPath().toOSString());
				if (uri != null) {
					Resource localResource = null;
					if (this.editingDomain != null) {
						localResource = this.editingDomain.loadResource(uri.toString());
					} else {
						ResourceSet rSet = new ResourceSetImpl();
						localResource = rSet.getResource(uri, true);
					}

					for (EObject eObject : localResource.getContents()) {
						if (eObject instanceof TableInstance) {
							tableInstanceValue = (TableInstance) eObject;
							// One instance of tableInstance per .table file
							break;
						}
					}
					if (tableInstanceValue != null) {
						this.tableInstance.getColumns().addAll(tableInstanceValue.getColumns());
						this.tableInstance.setContext(tableInstanceValue.getContext());
						this.tableInstance.getCustomizations().addAll(tableInstanceValue.getCustomizations());
						this.tableInstance.getRows().addAll(tableInstanceValue.getRows());
						this.tableInstance.setDescription(tableInstanceValue.getDescription());
						if (this.tableInstance instanceof TableInstance2 && tableInstanceValue instanceof TableInstance2) {
							((TableInstance2) this.tableInstance).getFacets2().addAll(((TableInstance2) tableInstanceValue).getFacets2());
						}
						try {
							initFacets();
						} catch (CoreException e) {
							Logger.logError(e, Activator.getDefault());
						}
						setInput();
						notifyActions();
					} else {
						throw new RuntimeException(
								"This file is not a valid .table ones: " + path); //$NON-NLS-1$
					}
				}

			} else {
				throw new RuntimeException("Could not load this file: " + path); //$NON-NLS-1$
			}
		}
	}

	public Control getComposite() {
		return this;
	}

	public void registerActionDelegate(final IActionDelegate action) {
		if (!this.actionList.contains(action)) {
			this.actionList.add(action);
		}
	}

	public void unregisterActionDelegate(final IActionDelegate action) {
		this.actionList.remove(action);
	}

	private void notifyActions() {
		for (IActionDelegate action : this.actionList) {
			action.selectionChanged(null, null);
		}
	}

	public Collection<ISelectionChangedListener> getSelectionChangedListeners() {
		return Collections.unmodifiableCollection(this.fSelectionChangedListeners);
	}

	/**
	 * Sort columns by name following this order :
	 * <ul>
	 * <li>the columns beginning by '['</li>
	 * <li>the column "/eContainer"</li>
	 * <li>the other columns, ignoring the '/'</li>
	 * </ul>
	 */
	public void sortColumnByNameAtoZ() {
		try {
			this.natTable.setRedraw(false);
			this.listenReorderEvent = false;
			List<Column> columns = new ArrayList<Column>(this.tableInstance.getColumns());
			Collections.sort(columns, new ColumnComparator());

			Command command = this.commandFactory.createSetCommand(NatTableWidget.this.editingDomain,
					NatTableWidget.this.tableInstance,
					TableinstancePackage.eINSTANCE.getTableInstance_Columns(), columns);
			NatTableWidget.this.editingDomain.getCommandStack().execute(command);
		} catch (Exception e) {
			Logger.logError(e, "Error sorting columns by name From A to Z", //$NON-NLS-1$
					Activator.getDefault());
		} finally {
			this.natTable.setRedraw(true);
			this.listenReorderEvent = true;
		}
	}

	/**
	 * Sort columns by name from Z to A following this order :
	 * <ul>
	 * <li>in first the columns which name doesn't contain '/' 'except
	 * "/eContainer" and the column beginning by '['
	 * <li>the the column "/eContainer"</li>
	 * <li>the column beginning by '['</li>
	 * </ul>
	 */
	public void sortColumnByNameZtoA() {
		try {
			this.natTable.setRedraw(false);
			this.listenReorderEvent = false;
			List<Column> columns = new ArrayList<Column>(this.tableInstance.getColumns());
			ColumnComparator comparator = new ColumnComparator();
			comparator.setInversedOrder(true);
			Collections.sort(columns, comparator);
			// Collections.reverse(columns);

			Command command = this.commandFactory.createSetCommand(NatTableWidget.this.editingDomain,
					NatTableWidget.this.tableInstance,
					TableinstancePackage.eINSTANCE.getTableInstance_Columns(), columns);
			NatTableWidget.this.editingDomain.getCommandStack().execute(command);
		} catch (Exception e) {
			Logger.logError(e, "Error sorting columns by name from Z to A", //$NON-NLS-1$
					Activator.getDefault());
		} finally {
			this.natTable.setRedraw(true);
			this.listenReorderEvent = true;
		}
	}

	public void selectDefaultLabelCell(final EObject eObject) {
		// Check that the given eObject is in the table
		int rowIndex = getRowIndexByEObject(eObject);
		if (rowIndex == -1) {
			return;
		}
		// Look for the DefaultLabel column index
		int columnIndex = -1;
		for (int i = 0; i < this.tableInstance.getColumns().size(); i++) {
			if (this.tableInstance.getColumns().get(i) instanceof DefaultLabelColumn) {
				columnIndex = i;
				break;
			}
		}
		if (columnIndex == -1) {
			return;
		}
		SelectionLayer selectionLayer = this.fBodyLayer.getSelectionLayer();
		selectionLayer.selectCell(columnIndex, rowIndex, false, false);
	}

	public void selectCell(final EObject eObject, final EStructuralFeature eStructuralFeature) {
		// Check that the given eObject is in the table
		int rowIndex = getRowIndexByEObject(eObject);
		if (rowIndex == -1) {
			return;
		}
		// Look for the column with the eStructuralFeature
		int columnIndex = -1;
		for (int i = 0; i < this.tableInstance.getColumns().size(); i++) {
			Column column = this.tableInstance.getColumns().get(i);
			if (column instanceof FeatureColumn) {
				FeatureColumn featureColumn = (FeatureColumn) column;
				if (featureColumn.getFeature() == eStructuralFeature) {
					columnIndex = i;
					break;
				}
			}
		}
		if (columnIndex == -1) {
			return;
		}
		SelectionLayer selectionLayer = this.fBodyLayer.getSelectionLayer();
		selectionLayer.selectCell(columnIndex, rowIndex, false, false);
	}

	private int getRowIndexByEObject(final EObject eObject) {
		int rowIndex = -1;
		EList<Row> rows = this.tableInstance.getRows();
		for (int i = 0; i < rows.size(); i++) {
			if (rows.get(i).getElement().equals(eObject)) {
				rowIndex = i;
				break;
			}
		}
		return rowIndex;
	}

	public void editSelectedCell() {
		if (isSelectedCellEditable()) {
			List<LayerCell> cells = getSelectedCells();
			if (cells != null && !cells.isEmpty()) {
				LayerCell cell = cells.get(0);
				SelectionLayer selectionLayer = this.fBodyLayer.getSelectionLayer();
				selectionLayer.fireLayerEvent(new InlineCellEditEvent(selectionLayer,
						new PositionCoordinate(selectionLayer, cell.getColumnPosition(), cell
								.getRowPosition()), this.natTable, this.natTable.getConfigRegistry(),
						null));
			}
		}
	}

	public List<Column> getVisibleColumns(final boolean setAtTheTop) {
		List<Column> notHiddenColumns = new ArrayList<Column>();
		if (this.tableInstance instanceof TableInstance2) {
			notHiddenColumns = getVisibleColumnsUsingTheLocalCustomizations(
						(TableInstance2) this.tableInstance, setAtTheTop);
		} else {
			// we use the model of the table
			notHiddenColumns = getVisibleColumnsUsingTheModel();
		}
		return notHiddenColumns;
	}

	public List<Column> getVisibleColumnsUsingTheLayer() {
		List<Column> notHiddenColumns = new ArrayList<Column>();
		ColumnHideShowLayer layer = this.fBodyLayer.getColumnHideShowLayer();
		EList<Column> allColumns = this.tableInstance.getColumns();
			for (int i = 0; i < allColumns.size(); i++) {
				if (!layer.isColumnIndexHidden(i)) {
					notHiddenColumns.add(allColumns.get(i));
				}
			}
		return notHiddenColumns;
	}

	public void putLocalCustomizationOnTheBottom() {
		this.editingDomain.getCommandStack().execute(getPutLocalCustomizationOnTheBottomCommand());
		loadCustomizationsInCustomEngine(this.tableInstance.getCustomizations());
	}

	private Command getPutLocalCustomizationOnTheBottomCommand() {
		List<MetamodelView> locals = getLocalCustomizations();
		List<MetamodelView> views = new ArrayList<MetamodelView>();
		views.addAll(this.tableInstance.getCustomizations());
		views.removeAll(locals);
		views.addAll(views.size(), locals);
		return this.commandFactory.createSetCommand(this.editingDomain, this.tableInstance,
				TableinstancePackage.eINSTANCE.getTableInstance_Customizations(), views);
	}

	public Point getCellPositionRelativeToDisplay(final int columnPosition, final int rowPosition) {
		Rectangle rect = this.fBodyLayer.getSelectionLayer().getBoundsByPosition(
				columnPosition, rowPosition);
		return new Point(rect.x, rect.y);
	}

	/**
	 *
	 * @see org.eclipse.emf.facet.widgets.nattable.INatTableWidget2#selectRows(java.util.List)
	 *
	 * @param elementsToSelect
	 * @param clearTheSelection
	 */
	public void selectRows(final List<EObject> elementsToSelect, final boolean clearTheSelection) {
		int columnPosition = 0;
		List<Row> allRows = this.tableInstance.getRows();
		List<Integer> rowsToSelect = new ArrayList<Integer>();
		for (int i = 0; i < elementsToSelect.size(); i++) {
			Row row = findRow(elementsToSelect.get(i));
			if (row != null) {
				rowsToSelect.add(new Integer(allRows.indexOf(row)));
			}
		}
		int[] rowPositions = new int[rowsToSelect.size()];
		for (int i = 0; i < rowsToSelect.size(); i++) {
			rowPositions[i] = rowsToSelect.get(i).intValue();
		}

		SelectionLayer layer = this.fBodyLayer.getSelectionLayer();
		if (clearTheSelection) {
			layer.clear(); // we remove the previous selection
		}
		if (rowPositions.length != 0) {
			this.natTable.doCommand(new SelectRowsCommand(this.fBodyLayer.getSelectionLayer(),
					columnPosition, rowPositions, false, true));
		}
		this.natTable.redraw();
		tableSelectionChanged();
	}


	/**
	 *
	 * @param eObject
	 *            an eObject
	 * @return
	 * the row representing the eObject or <code>null</code> if not found
	 */
	private Row findRow(final EObject eObject) {
		if (this.tableInstance.getElements().contains(eObject)) {
			for (Row row : this.tableInstance.getRows()) {
				if (row.getElement().equals(eObject)) {
					return row;
				}
			}
		}
		return null;
	}

	public boolean isSelectedCellEditable() {
		List<LayerCell> cells = getSelectedCells();
		if (cells.size() > 1) {
			return false;
		}
		LayerCell cell = cells.get(0);
		int columnPos = cell.getOriginColumnPosition();
		int columnIndex = this.fBodyLayer.getColumnHideShowLayer().getColumnIndexByPosition(
				columnPos);
		Column column = getColumns().get(columnIndex);
		if (!(column instanceof FeatureColumn)) {
			return false;
		}

		int rowPos = cell.getOriginRowPosition();
		int rowIndex = this.fBodyLayer.getColumnHideShowLayer().getRowIndexByPosition(rowPos);
		Row row = this.tableInstance.getRows().get(rowIndex);
		EObject eObject = row.getElement();

		if (column instanceof FacetReferenceColumn) {
			try {
				return this.facetContext.isInstance(eObject,
						(Facet) ((FacetReferenceColumn) column).getReference().eContainer());
			} catch (ModelQueryException e) {
				return false;
			}
		}
		if (column instanceof FacetAttributeColumn) {
			try {
				return this.facetContext.isInstance(eObject,
						(Facet) ((FacetAttributeColumn) column).getAttribute().eContainer());
			} catch (ModelQueryException e) {
				return false;
			}
		}

		EStructuralFeature feature = ((FeatureColumn) column).getFeature();
		List<EStructuralFeature> eStructuralFeatures = eObject.eClass().getEAllStructuralFeatures();
		return eStructuralFeatures.contains(feature);
	}

	private void refreshDelayed() {
		if (NatTableWidget.DEBUG_RESOURCE_LISTENER) {
			System.out.println(this.getClass().getName() + ".refreshDelayed()"); //$NON-NLS-1$
		}
		if (this.refreshJob == null) {
			// TODO : should be externalized
			this.refreshJob = new Job("Refresh EMF Facet Table") { //$NON-NLS-1$
				@Override
				protected IStatus run(final IProgressMonitor monitor) {
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							if (!NatTableWidget.this.natTable.isDisposed()) {
								setInput();
							}
						}
					});
					return Status.OK_STATUS;
				}
			};
		} else {
			/*
			 * If refreshTreeDelayed is called again before the job delay
			 * passed, then cancel the job and re-schedule a new one.
			 */
			this.refreshJob.cancel();
		}
		this.refreshJob.setPriority(Job.DECORATE);
		this.refreshJob.schedule(NatTableWidget.REFRESH_JOB_DELAY);
	}

	protected EditingDomain getEditingDomain() {
		return this.editingDomain;
	}

	protected ICommandFactory getCommandFactory() {
		return this.commandFactory;
	}

	public void waitForResfreshJob() {
		boolean wait = true;
		while (wait) {
			try {
				if (this.refreshJob != null) {
					this.refreshJob.join();
				}
			wait = false;
			} catch (InterruptedException e) {
				wait = true;
			}
		}
	}

	/**
	 * If a table element has been deleted from outside the table, then we need
	 * to remove the inconsistent row from its model.
	 */
	private void removeUselessRowsAndColumns() {
		CompoundCommand compoundCommand = new CompoundCommand();
		List<Row> rowsToRemove = new ArrayList<Row>();
		for (Row row : new ArrayList<Row>(this.tableInstance.getRows())) {
			if (row.getElement() == null || row.getElement().eResource() == null) {
				Command deleteCommand = this.commandFactory.createDeleteCommand(this.editingDomain, row);
//				Command removeCommand = this.commandFactory.createRemoveCommand(this.editingDomain,
//						this.tableInstance, TableinstancePackage.eINSTANCE.getTableInstance_Rows(),
//						row);
				rowsToRemove.add(row);
				compoundCommand.append(deleteCommand);
			}
		}
		if (NatTableWidget.DEBUG_REMOVE_USELESS_ROWS_AND_COLUMNS) {
			System.out
					.println("org.eclipse.emf.facet.widgets.nattable.internal.NatTableWidget.removeUselessRowsAndColumns(): " + //$NON-NLS-1$
							"Rows to be removed: " + compoundCommand.getCommandList().size() //$NON-NLS-1$
					);
		}
		if (!rowsToRemove.isEmpty()) {
			try {
				CompoundCommand removeColumnCommand = TableInstanceCommandFactory
						.createRemoveUselessColumnsCommand(this.tableInstance, rowsToRemove,
								this.commandFactory, this.editingDomain);
				if (NatTableWidget.DEBUG_REMOVE_USELESS_ROWS_AND_COLUMNS) {
					int nbCommand;
					if (removeColumnCommand != null) {
						nbCommand = removeColumnCommand.getCommandList().size();
					} else {
						nbCommand = 0;
					}
					System.out
							.println("org.eclipse.emf.facet.widgets.nattable.internal.NatTableWidget.removeUselessRowsAndColumns(): " + //$NON-NLS-1$
									"Columns to be removed: " + nbCommand //$NON-NLS-1$
							);
				}
				if (removeColumnCommand != null) {
					if (!removeColumnCommand.canExecute()) {
						throw new RuntimeException("Command cannot be executed"); //$NON-NLS-1$
					}
					compoundCommand.append(removeColumnCommand);
				}
			} catch (MultiModelQueryException e) {
				// We do not want a table break if the a facet is failling
				Logger.logWarning(e, Activator.getDefault());
			}
			this.editingDomain.getCommandStack().execute(compoundCommand);
		}
	}

	/**
	 * This method has to be used to know if the the tableInstance model is stored in a temporay resource.
	 * This is usefull to implements editor which use the table wigdet.
	 * @return true if the table model is stored in a temporary resource
	 */
	public boolean usesTmpResource() {
		//TODO this method has to be added in the API in the version 0.2.0.
		//Cf. Bug 349797 - NatTableWidget.usesTmpResource() has to be added to the API. https://bugs.eclipse.org/bugs/show_bug.cgi?id=349797
		String uri = URI.createFileURI(NatTableWidget.DEFAULT_RESOURCE_FILE.toString())
				.toFileString();
		String tableInstanceResourceURI = getTableInstance().eResource()
				.getURI().toFileString();
		return uri.equals(tableInstanceResourceURI);
	}
}
