/*******************************************************************************
 * Copyright (c) 2010, 2011 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Nicolas Bros (Mia-Software) - Bug 331203 - table model editor - initial API and implementation
 *   Nicolas Bros (Mia-Software) - Bug 331900 - customizable NatTable
 *   Nicolas Bros (Mia-Software) - Bug 332010 - view Facet customizations on NatTable
 *   Nicolas Bros (Mia-Software) - Bug 332215 - customizable NatTable column headers
 *   Nicolas Guyomar (Mia-Software) - Bug 332924 - To be able to save the table
 *   Nicolas Guyomar (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *   Gregoire Dupe (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *   Nicolas Guyomar (Mia-Software) - Bug 340681 - Facet column implementation 
 *   Nicolas Guyomar (Mia-Software) - Bug 340940 - To be able to view facet attributes and facet references in a table
 *   Nicolas Guyomar (Mia-Software) - Bug 342451 - To be able to edit derived facet attributes and derived facet references in a table
 *   Nicolas Guyomar (Mia-Software) - Bug 346465 - [EMF Facet Table] Remove line does not remove obsolete column
 *   Nicolas Bros (Mia-Software) - Bug 356564 - [Table Editor] customizations on attributes and references don't work
 *******************************************************************************/
package org.eclipse.emf.facet.widgets.nattable.internal.painter;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.sourceforge.nattable.command.LayerCommandUtil;
import net.sourceforge.nattable.config.IConfigRegistry;
import net.sourceforge.nattable.coordinate.ColumnPositionCoordinate;
import net.sourceforge.nattable.layer.cell.LayerCell;
import net.sourceforge.nattable.painter.cell.TextPainter;
import net.sourceforge.nattable.style.CellStyleAttributes;
import net.sourceforge.nattable.style.CellStyleUtil;
import net.sourceforge.nattable.style.DisplayMode;
import net.sourceforge.nattable.style.IStyle;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.facet.infra.browser.uicore.internal.CustomTreePainter;
import org.eclipse.emf.facet.infra.browser.uicore.internal.FacetToPaint;
import org.eclipse.emf.facet.infra.browser.uicore.internal.customization.CustomizationEngine;
import org.eclipse.emf.facet.infra.facet.Facet;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.AttributeColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.Column;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.ContextColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.DefaultLabelColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.EContainerColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FacetAttributeColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FacetReferenceColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.MetaClassColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.QueryColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.ReferenceColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance2.ValueColumn;
import org.eclipse.emf.facet.widgets.nattable.internal.Activator;
import org.eclipse.emf.facet.widgets.nattable.internal.GridElement;
import org.eclipse.emf.facet.widgets.nattable.internal.NatTableWidget;
import org.eclipse.emf.facet.widgets.nattable.internal.NatTableWidget.BodyLayerStack;
import org.eclipse.emf.facet.widgets.nattable.internal.TableLabelProvider;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Display;

public class CustomizedCellPainter extends TextPainter {
	private final BodyLayerStack bodyLayerStack;
	private final TableLabelProvider tableLabelProvider;
	private final CustomizationEngine customizationEngine;
	private final NatTableWidget natTableWidget;
	private final Map<RGB, Color> cachedColors = new HashMap<RGB, Color>();

	public CustomizedCellPainter(final BodyLayerStack bodyLayerStack,
			final TableLabelProvider tableLabelProvider,
			final CustomizationEngine customizationEngine, final NatTableWidget natTableWidget) {
		this.bodyLayerStack = bodyLayerStack;
		this.tableLabelProvider = tableLabelProvider;
		this.customizationEngine = customizationEngine;
		this.natTableWidget = natTableWidget;
	}

	@Override
	protected String convertDataType(final LayerCell cell, final IConfigRegistry configRegistry) {
		ColumnPositionCoordinate converted = LayerCommandUtil.convertColumnPositionToTargetContext(
				new ColumnPositionCoordinate(cell.getLayer(), cell.getColumnPosition()),
				this.bodyLayerStack.getBodyDataLayer());
		return this.tableLabelProvider.getColumnText((GridElement) cell.getDataValue(),
				converted.getColumnPosition());
	}

	@Override
	public void paintCell(final LayerCell cell, final GC gc, final Rectangle rectangle,
			final IConfigRegistry configRegistry) {

		int convertedColumn = this.natTableWidget.convertCellPositionToDataLayer(cell);
		if (convertedColumn < this.natTableWidget.getTableInstance().getColumns().size()) {
		Column columns = this.natTableWidget.getColumns().get(convertedColumn);

		// leave a 2 pixel wide margin
		Rectangle rectangleWithMargin = new Rectangle(rectangle.x + 2, rectangle.y + 2,
				rectangle.width - 2 * 2, rectangle.height - 2 * 2);

		Color backgroundColor = CellStyleUtil.getCellStyle(cell, configRegistry).getAttributeValue(
				CellStyleAttributes.BACKGROUND_COLOR);
		Customization customization = null;

		Object dataValue = cell.getDataValue();
		if (dataValue instanceof GridElement) {
			GridElement gridElement = (GridElement) dataValue;
			Object element = gridElement.getElement();
			if (element instanceof EObject) {
				EObject eObject = (EObject) element;
				customization = getCustomization(eObject, columns, rectangleWithMargin);
			}
		}

		if (customization != null && customization.getBackground() != null) {
			Color customizedColor = customization.getBackground();
			if (cell.getDisplayMode() == DisplayMode.SELECT) {
				// merge the customized color with the selection color
				final double opacity = 0.3d;
				Color mergedColor = mergeColors(customizedColor, backgroundColor, opacity);
				backgroundColor = mergedColor;
			} else {
				backgroundColor = customizedColor;
			}
		}
		Color originalBackground = gc.getBackground();
		gc.setBackground(backgroundColor);
		gc.fillRectangle(rectangle);
		gc.setBackground(originalBackground);

		Rectangle originalClipping = gc.getClipping();
		gc.setClipping(rectangleWithMargin);

		IStyle cellStyle = CellStyleUtil.getCellStyle(cell, configRegistry);
		setupGCFromConfig(gc, cellStyle);
		String text = convertDataType(cell, configRegistry);

		if (customization != null) {
			if (customization.getForeground() != null) {
				gc.setForeground(customization.getForeground());
			}
			if (customization.getLabel() != null) {
				text = customization.getLabel();
			}
			if (customization.getFont() != null) {
				gc.setFont(customization.getFont());
			}
		}

		gc.setClipping(rectangle);

		boolean hasIcon = false;
		if (customization != null && customization.getInstanceIcon() != null) {
			gc.drawImage(customization.getInstanceIcon(), rectangleWithMargin.x,
					rectangleWithMargin.y);
			hasIcon = true;
		} else {
			Image columnImage = this.tableLabelProvider.getColumnImage(cell.getDataValue(),
					convertedColumn);
			if (columnImage != null) {
				gc.drawImage(columnImage, rectangleWithMargin.x, rectangleWithMargin.y);
				hasIcon = true;
			}
		}

		boolean hasOverlay = false;
		if (customization != null) {
			int facetMinX = rectangle.width;
			List<FacetToPaint> facetsToMeasure = customization.getFacetsToPaint();
			for (FacetToPaint facetToPaint : facetsToMeasure) {
				if (!facetToPaint.isOverlay()) {
					facetMinX = Math.min(facetMinX, facetToPaint.getBounds().x);
				} else {
					hasOverlay = true;
				}
			}
			// reduce the clipping, so as to not draw again over stickers
			Rectangle textClipping = new Rectangle(rectangleWithMargin.x, rectangleWithMargin.y,
					facetMinX - 2, rectangleWithMargin.height);
			gc.setClipping(textClipping);
		}

		if (hasIcon) {
			final int totalOffsetAfterIcon;
			final int offsetAfterIcon = 18;
			final int overlaySize = 8;
			if (hasOverlay) {
				totalOffsetAfterIcon = offsetAfterIcon + overlaySize;
			} else {
				totalOffsetAfterIcon = offsetAfterIcon;
			}

			gc.drawText(text, rectangleWithMargin.x + totalOffsetAfterIcon, rectangleWithMargin.y,
					true);
		} else {
			gc.drawText(text, rectangleWithMargin.x, rectangleWithMargin.y, true);
		}

		gc.setClipping(rectangle);

		if (customization != null) {
			final int leftMargin = 0;
			final int rightMargin = 0;

			if (customization.isUnderlined()) {
				final int y = rectangleWithMargin.y + rectangleWithMargin.height - 1;
				gc.drawLine(rectangleWithMargin.x + leftMargin, y, rectangleWithMargin.x
						+ rectangleWithMargin.width - rightMargin, y);
			}

			if (customization.isStruckthrough()) {
				final int y = rectangleWithMargin.y + rectangleWithMargin.height / 2 + 1;
				gc.drawLine(rectangleWithMargin.x + leftMargin, y, rectangleWithMargin.x
						+ rectangleWithMargin.width - rightMargin, y);
			}
		}

		if (customization != null) {
			List<FacetToPaint> facetsToPaint = customization.getFacetsToPaint();
			for (FacetToPaint facetToPaint : facetsToPaint) {
				Image customizedIcon = facetToPaint.getImage();
				if (customizedIcon != null) {
					Rectangle bounds = customizedIcon.getBounds();
					Rectangle target = facetToPaint.getBounds();
					Rectangle stickerRectangle = new Rectangle(rectangle.x + target.x + 1,
							rectangle.y + target.y + 1, target.width, target.height);
					gc.drawImage(customizedIcon, 0, 0, bounds.width, bounds.height,
							stickerRectangle.x, stickerRectangle.y, stickerRectangle.width,
							stickerRectangle.height);
				}
			}
		}

		gc.setClipping(originalClipping);
		}
	}

	private Color mergeColors(final Color color1, final Color color2, final double opacity) {
		final double opacity2 = 1.0d - opacity;
		RGB rgb = new RGB(((int) (color1.getRed() * opacity + color2.getRed() * opacity2)),
				((int) (color1.getGreen() * opacity + color2.getGreen() * opacity2)),
				((int) (color1.getBlue() * opacity + color2.getBlue() * opacity2)));
		Color color = this.cachedColors.get(rgb);
		if (color == null) {
			color = new Color(color1.getDevice(), rgb);
			this.cachedColors.put(rgb, color);
		}
		return color;
	}

	private Customization getCustomization(final EObject eObject, final Column column,
			final Rectangle rectangle) {
		// label column : customization on the EObject, defined on the EClass
		// unary reference : customization on the EObject, defined on the EClass
		// n-ary reference with a single element: customization on the EObject,
		// defined on the EClass
		// n-ary reference (!= 1): no customization
		// attribute : customization on the EObject, defined on the EAttribute

		Customization customization = new Customization();
		EObject customizedEObject = null;

		if (column instanceof MetaClassColumn) {
			return customization;
		} else if (column instanceof AttributeColumn) {
			AttributeColumn attributeColumn = (AttributeColumn) column;
			EAttribute attribute = attributeColumn.getAttribute();
			if (!eObject.eClass().getEAllAttributes().contains(attribute)) {
				// not applicable
				return customization;
			}
			customizedEObject = eObject;
		} else if (column instanceof ReferenceColumn) {
			ReferenceColumn referenceColumn = (ReferenceColumn) column;
			EReference reference = referenceColumn.getReference();
			if (!eObject.eClass().getEAllReferences().contains(reference)) {
				// not applicable
				return customization;
			}
			if (reference.isMany()) {
				@SuppressWarnings("unchecked")
				List<EObject> list = (List<EObject>) eObject.eGet(reference);
				if (list.size() == 1) {
					customizedEObject = list.get(0);
				} else {
					return customization;
				}
			} else {
				Object value = eObject.eGet(reference);
				if (value instanceof EObject) {
					customizedEObject = (EObject) value;
				} else {
					return customization;
				}
			}
		} else if (column instanceof ContextColumn) {
			return customization;
		} else if (column instanceof DefaultLabelColumn) {
			customizedEObject = eObject;
		} else if (column instanceof EContainerColumn) {
			customizedEObject = eObject.eContainer();
		} else if (column instanceof QueryColumn) {
			return null;
		} else if (column instanceof FacetAttributeColumn) {
			return null;
		} else if (column instanceof FacetReferenceColumn) {
			return null;
		} else if (column instanceof ValueColumn) {
			return null;
		} else {
			Logger.logError("unhandled case: " + column.eClass().getName(), //$NON-NLS-1$
					Activator.getDefault());
			return customization;
		}

		if (customizedEObject == null) {
			return customization;
		}
		if (column instanceof AttributeColumn) {
			AttributeColumn attributeColumn = (AttributeColumn) column;
			EAttribute attribute = attributeColumn.getAttribute();
			customization.setLabel(this.customizationEngine.getAttributeLabel(
					(EClass) attribute.eContainer(), attribute.getName(), customizedEObject));
			customization.setInstanceIcon(this.customizationEngine.getAttributeIcon(
					(EClass) attribute.eContainer(), attribute.getName(), customizedEObject));
			customization.setBackground(this.customizationEngine.getAttributeBackgroundColor(
					(EClass) attribute.eContainer(), attribute.getName(), customizedEObject));
			customization.setForeground(this.customizationEngine.getAttributeColor(
					(EClass) attribute.eContainer(), attribute.getName(), customizedEObject));
			customization.setFont(this.customizationEngine.getCustomizedAttributeFont(
					(EClass) attribute.eContainer(), attribute.getName(), Display.getDefault()
							.getSystemFont(), customizedEObject));
			customization.setUnderlined(this.customizationEngine.isAttributeUnderlined(
					(EClass) attribute.eContainer(), attribute.getName(), customizedEObject));
			customization.setStruckthrough(this.customizationEngine.isAttributeStruckthrough(
					(EClass) attribute.eContainer(), attribute.getName(), customizedEObject));
		} else {
			customization.setLabel(this.customizationEngine.getTypeLabel(customizedEObject));

			// Facet icon customization
			Image facetIcon = null;
			for (final Facet facet : this.natTableWidget.getFacetContext().getFacets(
					customizedEObject)) {
				facetIcon = this.customizationEngine.getFacetMainIcon(customizedEObject, facet);
				if (facetIcon != null) {
					// first one wins
					break;
				}
			}
			if (facetIcon != null) {
				customization.setInstanceIcon(facetIcon);
			} else {
				customization.setInstanceIcon(this.customizationEngine.getTypeIcon(
						customizedEObject, customizedEObject.eClass()));
			}
			customization.setBackground(this.customizationEngine
					.getTypeBackgroundColor(customizedEObject));
			customization.setForeground(this.customizationEngine.getTypeColor(customizedEObject));
			customization.setFont(this.customizationEngine.getCustomizedTypeFont(customizedEObject,
					Display.getDefault().getSystemFont()));
			customization.setUnderlined(this.customizationEngine.isTypeUnderlined(
					customizedEObject.eClass(), customizedEObject));
			customization.setStruckthrough(this.customizationEngine.isTypeStruckthrough(
					customizedEObject.eClass(), customizedEObject));

			List<FacetToPaint> facetsToPaint = CustomTreePainter.getFacetsToPaintFor(
					customizedEObject, 0, rectangle.width, 0, rectangle.height, rectangle.width,
					this.customizationEngine, this.natTableWidget.getFacetContext());
			customization.setFacetsToPaint(facetsToPaint);

		}
		return customization;
	}

	@Override
	protected void finalize() throws Throwable {
		for (Color color : this.cachedColors.values()) {
			color.dispose();
		}
		super.finalize();
	}
}