/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *
 *******************************************************************************/

package org.eclipse.emf.facet.infra.browser.custom.core;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EValidator;
import org.eclipse.emf.facet.infra.browser.custom.MetamodelView;
import org.eclipse.emf.facet.infra.browser.custom.emf.UicustomPackage;
import org.eclipse.emf.facet.infra.browser.custom.validation.EValidatorAdapter;
import org.eclipse.emf.facet.infra.common.core.internal.builder.AbstractEmfFacetCatalog;
import org.eclipse.emf.facet.infra.common.core.internal.builder.EcoreCatalog;
import org.eclipse.emf.facet.infra.common.core.internal.resource.IEmfFacetResourceListener;
import org.eclipse.emf.facet.infra.common.core.internal.resource.EmfFacetResourceListenerGroup;
import org.eclipse.emf.facet.infra.common.core.internal.resource.EmfFacetResourceSet;
import org.eclipse.emf.facet.infra.common.core.internal.validation.ValidationJob;
import org.eclipse.emf.facet.infra.facet.FacetSet;
import org.eclipse.emf.facet.infra.facet.core.FacetSetCatalog;
import org.eclipse.emf.facet.infra.query.core.ModelQuerySetCatalog;
import org.eclipse.emf.facet.util.core.Logger;

/**
 * A catalog of customization files found in the Workspace and registered
 * through the extension point
 * <code>org.eclipse.emf.facet.infra.browser.custom.registration</code>. It is
 * updated by the {@link CustomizationsBuilder}, which is responsible for adding
 * and removing customization files in response to changes in the Workspace.
 */
public class CustomizationsCatalog extends AbstractEmfFacetCatalog {

	public static final String REGISTRATION_EXTENSION_POINT_ID = "org.eclipse.emf.facet.infra.browser.custom.core.registration"; //$NON-NLS-1$
	public static final String FILE_EXTENSION = "uiCustom"; //$NON-NLS-1$
	private static CustomizationsCatalog instance = null;

	public static CustomizationsCatalog getInstance() {
		if (CustomizationsCatalog.instance == null) {
			EcoreCatalog.getSingleton();
			ModelQuerySetCatalog.getSingleton();
			FacetSetCatalog.getSingleton();
			CustomizationsCatalog.instance = new CustomizationsCatalog();
			EValidator validator = EValidator.Registry.INSTANCE
					.getEValidator(UicustomPackage.eINSTANCE);
			if (validator == null) {
				// Registers the validator adapter in the EMF validator registry
				EValidator.Registry.INSTANCE
						.put(UicustomPackage.eINSTANCE, new EValidatorAdapter());
			}
		}
		return CustomizationsCatalog.instance;
	}

	public CustomizationsCatalog() {
		super();
	}

	@Override
	protected String getRootObjectName(final EObject rootObject) {
		return ((MetamodelView) rootObject).getName();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getRootObjectNsUri(final EObject rootObject) {
		return "emffacet:/customization/" + ((MetamodelView) rootObject).getName(); //$NON-NLS-1$
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Class<?> getRootClass() {
		return MetamodelView.class;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getRegistryExtensionPoint() {
		return CustomizationsCatalog.REGISTRATION_EXTENSION_POINT_ID;
	}

	public Collection<MetamodelView> getAllCustomizations() {
		Collection<MetamodelView> result = new ArrayList<MetamodelView>();
		for (EObject eObject : getAllRootObjects()) {
			if (eObject instanceof MetamodelView && eObject.eResource() != null) {
				MetamodelView metamodelView = (MetamodelView) eObject;
				metamodelView.setLocation(getURI(metamodelView.getName()).toString());
				result.add(metamodelView);
			}
		}
		return result;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getEmfFacetSubProtocol() {
		return "customization"; //$NON-NLS-1$
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.facet.infra.common.core.builder.AbstractEmfFacetCatalog#
	 * getActivator()
	 */
	@Override
	protected Plugin getActivator() {
		return Activator.getDefault();
	}

	@Override
	public String getFileExtension() {
		return CustomizationsCatalog.FILE_EXTENSION;
	}

	@Override
	protected void preRemove(final IFile declarationFile, final String pathName,
			final EObject oldRootObject) {
		// Empty
	}

	@Override
	protected void postOpenResource(final URI uri, final IFile file, final EObject root) {
		// add a listener to referenced FacetSets, so that when the FacetSet is
		// modified, the customization is revalidated
		try {
			addListenerToReferedResources((MetamodelView) root, new IEmfFacetResourceListener() {
				public void aListenedResourceHasChanged(final URI resourceUri,
						final URI dependingResourceURI) {
					if (root.eResource() != null) {
						ValidationJob.getInstance().validateAsync(root.eResource(), file, null);
					}
				}
			});
		} catch (Exception e) {
			Logger.logError(e, Activator.getDefault());
		}
	}

	private void addListenerToReferedResources(final MetamodelView metamodelView,
			final IEmfFacetResourceListener listener) throws Exception {
		final String metamodelURI = metamodelView.getMetamodelURI();
		if (metamodelURI == null) {
			return;
		}
		final URI resourceURI = metamodelView.eResource().getURI();

		FacetSetCatalog.getSingleton();
		// Add listener from Customization to FacetSet
		AbstractEmfFacetCatalog.whenBuilt(new Runnable() {
			public void run() {
				Collection<FacetSet> allFacetSets = FacetSetCatalog.getSingleton()
						.getAllFacetSets();
				for (FacetSet facetSet : allFacetSets) {
					if (metamodelURI.equals(facetSet.getNsURI())) {
						EmfFacetResourceListenerGroup listenerGroup = EmfFacetResourceSet
								.getResourceSetSingleton().getListenerGroup(
										URI.createURI(metamodelURI));
						listenerGroup.addListener(listener, resourceURI);
					}
				}
			}
		});
	}

	/**
	 * Loads and returns a list of default customizations registered through the
	 * "registry" extension point.
	 */
	public List<MetamodelView> getRegistryDefaultCustomizations() {
		final List<MetamodelView> metamodelViews = new ArrayList<MetamodelView>();
		for (final MetamodelView metamodelView : getAllCustomizations()) {
			if (isDefaultCustomization(metamodelView.getName())) {
				metamodelViews.add(metamodelView);
			}
		}
		return metamodelViews;
	}

	private Set<String> defaultCustomizations = null;

	private boolean isDefaultCustomization(final String name) {
		if (this.defaultCustomizations == null) {
			this.defaultCustomizations = new HashSet<String>();
			try {
				final IConfigurationElement[] configurationElements = Platform
						.getExtensionRegistry().getConfigurationElementsFor(
								CustomizationsCatalog.REGISTRATION_EXTENSION_POINT_ID);

				for (final IConfigurationElement configurationElement : configurationElements) {
					final String fileAttribute = configurationElement.getAttribute("file"); //$NON-NLS-1$
					if (fileAttribute == null) {
						Activator
								.logError("Missing 'file' attribute in customization registration extension"); //$NON-NLS-1$
						continue;
					}
					String loadByDefaultAttribute = configurationElement
							.getAttribute("loadByDefault"); //$NON-NLS-1$
					if (loadByDefaultAttribute != null
							&& loadByDefaultAttribute
									.equalsIgnoreCase(String.valueOf(Boolean.TRUE))) {
						String fileName = new Path(fileAttribute).removeFileExtension()
								.lastSegment();
						this.defaultCustomizations.add(fileName);
					}
				}
			} catch (final Exception e) {
				Activator.logException(e);
			}
		}
		return this.defaultCustomizations.contains(name);
	}

	/** Loads and returns a list of customizations found in the Workspace. */
	public List<MetamodelView> getWorkspaceCustomizations() {
		final List<MetamodelView> metamodelViews = new ArrayList<MetamodelView>();
		for (final MetamodelView metamodelView : getAllCustomizations()) {
			URI uri = URI.createURI(metamodelView.getLocation());
			if (uri.isPlatformResource()) {
				metamodelViews.add(metamodelView);
			}
		}
		return metamodelViews;
	}

	/**
	 * Loads and returns a list of customizations registered through the
	 * "registry" extension point.
	 */
	public List<MetamodelView> getRegistryCustomizations() {
		final List<MetamodelView> metamodelViews = new ArrayList<MetamodelView>();
		for (final MetamodelView metamodelView : getAllCustomizations()) {
			URI uri = URI.createURI(metamodelView.getLocation());
			if (uri.isPlatformPlugin()) {
				metamodelViews.add(metamodelView);
			}
		}
		return metamodelViews;
	}

	/**
	 * This method returns the customization with the given name.
	 * 
	 * @param name
	 *            the name of a resource contained in the catalog
	 * @return return a customization
	 */
	public MetamodelView getCustomization(final String name) {
		return (MetamodelView) getRootObject(name);
	}
}
