/*******************************************************************************
 * Copyright (c) 2009, 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Nicolas Bros (Mia-Software)
 *     Nicolas Guyomar (Mia-Software) - Bug 340681 - Facet column implementation 
 *     Gregoire Dupe (Mia-Software) - Bug 341247 - add a method createLoadCustomizationsDialog() to the AbstractLoadCustomizationAction
 *******************************************************************************/
package org.eclipse.emf.facet.infra.browser.custom.ui.actions;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.facet.infra.browser.custom.MetamodelView;
import org.eclipse.emf.facet.infra.browser.custom.TypeView;
import org.eclipse.emf.facet.infra.browser.custom.ui.Activator;
import org.eclipse.emf.facet.infra.browser.custom.ui.dialogs.LoadCustomizationsDialog;
import org.eclipse.emf.facet.infra.common.core.internal.utils.ModelUtils;
import org.eclipse.emf.facet.infra.facet.Facet;
import org.eclipse.emf.facet.infra.facet.FacetSet;
import org.eclipse.emf.facet.infra.facet.core.FacetSetCatalog;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.window.Window;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Shell;

public abstract class AbstractLoadCustomizationsAction extends Action {

	private static final String ICON_UI_CUSTOM = "icons/uiCustom.gif"; //$NON-NLS-1$

	public AbstractLoadCustomizationsAction() {
		super("Load Customizations", IAction.AS_PUSH_BUTTON);
		setToolTipText("Load Customizations");
		setImageDescriptor(Activator
				.createImageDescriptor(AbstractLoadCustomizationsAction.ICON_UI_CUSTOM));
	}

	@Override
	public void run() {
		final List<MetamodelView> registeredCustomizations = getRegisteredCustomizations();

		final LoadCustomizationsDialog loadCustomizationsDialog = createLoadCustomizationDialog(registeredCustomizations);
		if (loadCustomizationsDialog.open() == Window.OK) {
			try {
				List<MetamodelView> selectedCustomizations = loadCustomizationsDialog
						.getSelectedCustomizations();

				if (loadCustomizationsDialog.isLoadRequiredFacetsSelected()) {
					// load facets corresponding to customizations
					loadFacetsForCustomizations(selectedCustomizations);
				}

				loadCustomizations(selectedCustomizations);
			} catch (final Exception e) {
				Logger.logError(e, Activator.getDefault());
			}
		}
	}

	protected LoadCustomizationsDialog createLoadCustomizationDialog(
			final List<MetamodelView> registeredCustomizations) {
		return new LoadCustomizationsDialog(
				getShell(), registeredCustomizations, getReferencedEPackages());
	}

	protected abstract List<MetamodelView> getRegisteredCustomizations();

	protected abstract void loadCustomizations(List<MetamodelView> selectedCustomizations);

	protected abstract Shell getShell();

	protected abstract Collection<EPackage> getReferencedEPackages();

	private void loadFacetsForCustomizations(final List<MetamodelView> customizations) {
		final Set<Facet> referencedFacets = new HashSet<Facet>();
		final Collection<FacetSet> facetSets = FacetSetCatalog.getSingleton().getAllFacetSets();

		for (MetamodelView customization : customizations) {
			String metamodelURI = customization.getMetamodelURI();
			// find customized FacetSet
			FacetSet customizedFacetSet = null;
			if (metamodelURI != null) {
				for (FacetSet facetSet : facetSets) {
					if (metamodelURI.equals(facetSet.getNsURI())) {
						customizedFacetSet = facetSet;
						break;
					}
				}
			}
			if (customizedFacetSet == null) {
				continue;
			}

			// find customized Facets
			EList<TypeView> types = customization.getTypes();
			for (TypeView typeView : types) {
				String metaclassName = typeView.getMetaclassName();
				Facet facet = findFacetWithFullyQualifiedName(metaclassName, customizedFacetSet);
				if (facet != null) {
					referencedFacets.add(facet);
				} else {
					Logger.logError(NLS.bind(
							"Missing required Facet: {0} in FacetSet {1}, for customization {2}", //$NON-NLS-1$
							new Object[] { metaclassName, customizedFacetSet.getName(),
									customization.getName() }), Activator.getDefault());
				}
			}

			for (Facet referencedFacet : referencedFacets) {
				loadFacet(referencedFacet);
			}
		}

		facetsLoaded(referencedFacets);
	}

	protected void facetsLoaded() {
		//
	}

	protected void facetsLoaded(final Set<Facet> referencedFacets) {
		//
	}

	protected abstract void loadFacet(Facet referencedFacet);

	protected Facet findFacetWithFullyQualifiedName(final String metaclassName,
			final FacetSet customizedFacetSet) {
		EList<Facet> facets = customizedFacetSet.getFacets();
		for (Facet facet : facets) {
			String facetName = ModelUtils.getMetaclassQualifiedName(facet);
			if (metaclassName.equals(facetName)) {
				return facet;
			}
		}
		return null;
	}
}
