/*******************************************************************************
 * Copyright (c) 2010, 2011, 2012 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - Bug 331203 - table model editor - initial API and implementation
 *    Nicolas Bros (Mia-Software) - Bug 332437 - NatTable : pluggable cell editors
 *    Nicolas Guyomar (Mia-Software) - Bug 331442 - To be able to add and remove lines (model elements) from the table
 *    Nicolas Bros (Mia-Software) - Bug 332226 - To be able to create or delete model element from the table
 *    Nicolas Guyomar (Mia-Software) - Bug 332226 - To be able to create or delete model element from the table
 *    Nicolas Bros (Mia-Software) - Bug 332438 - NatTable : table type
 *    Nicolas Bros (Mia-Software) - Bug 332440 - NatTable : force cell editors
 *    Nicolas Bros (Mia-Software) - Bug 331675 - NatTable : copy cells as text 
 *    Nicolas Bros (Mia-Software) - Bug 331900 - customizable NatTable
 *    Nicolas Bros (Mia-Software) - Bug 332010 - view Facet customizations on NatTable
 *    Nicolas Bros (Mia-Software) - Bug 332215 - customizable NatTable column headers
 *    Nicolas Guyomar (Mia-Software) - Bug 332215 - customizable NatTable column headers
 *    Nicolas Guyomar (Mia-Software) - Bug 332924 - To be able to save the table
 *    Nicolas Guyomar (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *    Gregoire Dupe (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *    Gregoire Dupe (Mia-Software) - Bug 333015 - To be able to hide columns
 *    Nicolas Guyomar (Mia-Software) - Bug 333015 - To be able to hide columns
 *    Nicolas Guyomar (Mia-Software) - Bug 333029 - To be able to save the size of the lines and the columns
 *    Nicolas Guyomar (Mia-Software) - Bug 333414 - The user must be able to save the column order
 *    Nicolas Guyomar (Mia-Software) - Bug 335154 - Sort Column By Type : Cannot modify resource set without a write transaction
 *    Nicolas Guyomar (Mia-Software) - Bug 335155 - Hide Empty Column : Cannot modify resource set without a write transaction
 *    Nicolas Guyomar (Mia-Software) - Bug 335156 - Only Show Common column : Cannot modify resource set without a write transaction
 *    Nicolas Guyomar (Mia-Software) - Bug 335020 - Nattable widget should use the Eclipse framework
 *    Nicolas Guyomar (Mia-Software) - Bug 337322 - [TableConfiguration] Customization declared in the file tableconfiguration is not loaded
 *    Nicolas Guyomar (Mia-Software) - Bug 338536 - Problem with the refresh of the table : the scrollbar returns to this initial position
 *    Nicolas Bros (Mia-Software) - Bug 338536 - Problem with the refresh of the table : the scrollbar returns to this initial position
 *    Nicolas Guyomar (Mia-Software) - Bug 337454 - We can't delete a query Column
 *    Nicolas Guyomar (Mia-Software) - Bug 337395 - Unused columns should be destroyed
 *    Nicolas Guyomar (Mia-Software) - Bug 339554 - org.eclipse.emf.facet.widgets.celleditors API cleaning
 *    Nicolas Guyomar (Mia-Software) - Bug 339922 - INatTableWidget method isCellSelected should use the plural
 *    Nicolas Guyomar (Mia-Software) - Bug 340681 - Facet column implementation
 *    Vincent Lorenzo (CEA-LIST) - Bug 337326 - Show/Hide Column : Sort the columns by name 
 *    Nicolas Guyomar (Mia-Software) - Bug 340940 - To be able to view facet attributes and facet references in a table
 *    Vincent Lorenzo (CEA-LIST) - Bug 337408 - Add an action to sort columns by name 
 *    Nicolas Guyomar (Mia-Software) - Bug 336482 - KeyBinding to edit element in Table : F2 
 *    Vincent Lorenzo (CEA-LIST) - Bug 341238 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Gregoire Dupe (Mia-Software) - Bug 341238 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Nicolas Guyomar (Mia-Software) - Bug 342451 - To be able to edit derived facet attributes and derived facet references in a table
 *    Vincent Lorenzo (CEA-LIST) - Bug 341238 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Nicolas Guyomar (Mia-Software) - Bug 343411 - [Table] Create new elements does not support IJavaModelQuery anymore
 *    Gregoire Dupe (Mia-Software) - Bug 343811 - EMF Facet Regression : Created elements in a table are not serialized
 *    Vincent Lorenzo (CEA-LIST) - Bug 344125 - The API should provide a method selectRows(List<EObject> elementsToSelect)
 *    Nicolas Guyomar (Mia-Software) - Bug 344274 - SWT BOT fail on Hudson
 *    Nicolas Guyomar (Mia-Software) - Bug 344475 - To be able to select a cell by EStructuralFeature in the table
 *    Nicolas Guyomar (Mia-Software) - Bug 342028 - Field can be edited even if they are marked as N/A
 *    Nicolas Guyomar (Mia-Software) - Bug 344413 - Facet Columns are never created when we begin with an empty table
 *    Gregoire Dupe (Mia-Software) - Bug 343859 - The local customizations are not applied when we reopen a table
 *    Nicolas Guyomar (Mia-Software) - Bug 344670 - Problems with the columns creation : very slow + scrollbar blinked
 *    Vincent Lorenzo (CEA LIST) - Bug 341238 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Nicolas Guyomar (Mia-Software) - Bug 344925 - Undo/Redo after the action Show Columns
 *    Gregoire Dupe (Mia-Software) - Bug 344925 - Undo/Redo after the action Show Columns - Regression fix
 *    Nicolas Guyomar (Mia-Software) - Bug 345665 - Columns are duplicated when you drop many elements in the same time
 *    Nicolas Guyomar (Mia-Software) - Bug 346465 - [EMF Facet Table] Remove line does not remove obsolete column
 *    Gregoire Dupe (Mia-Software) - Bug 345730 - Deleting an element in the model breaks the table
 *    Gregoire Dupe (Mia-Software) - Bug 354224 - mutually exclusive Facets
 *    Grgoire Dup (Mia-Software) - Bug 356795 - [Unit Test Failure][0.2/4.2][0.2/3.8] org.eclipse.emf.facet.widgets.nattable.tests.Bug344413Test.testBug344413
 *    Gregoire Dupe (Mia-Software) - Bug 366804 - [Restructuring] Table widget upgrade
 *    Gregoire Dupe (Mia-Software) - Bug 367613 - Table widget refactoring
 *    Gregoire Dupe (Mia-Software) - Bug 367700 - [Unit Test Failure][0.2/3.8] org.eclipse.emf.facet.widgets.table.tests.internal.v0_2.swtbot.NatTableUITests.testOpenLoadCustomizationDialog
 *    Gregoire Dupe (Mia-Software) - Bug 369987 - [Restructuring][Table] Switch to the new customization and facet framework
 *    Gregoire Dupe (Mia-Software) - Bug 373078 - API Cleaning
 *    Gregoire Dupe (Mia-Software) - Bug 375087 - [Table] ITableWidget.addColumn(List<ETypedElement>, List<FacetSet>)
 *    Gregoire Dupe (Mia-Software) - Bug 372626 - Aggregates
 *    Gregoire Dupe (Mia-Software) - Bug 376158 - [Table] Unexpected columns when customizations are loaded
 *    Nicolas Bros (Mia-Software) - Bug 378475 - unit test failures after table refactoring
 *    Olivier Remaud (Soft-Maint) - Bug 378499 - optimizing table opening
 *******************************************************************************/
package org.eclipse.emf.facet.widgets.table.ui.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CompoundCommand;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.ETypedElement;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.impl.ResourceImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.facet.custom.metamodel.v0_2_0.custom.CustomFactory;
import org.eclipse.emf.facet.custom.metamodel.v0_2_0.custom.CustomPackage;
import org.eclipse.emf.facet.custom.metamodel.v0_2_0.custom.Customization;
import org.eclipse.emf.facet.custom.metamodel.v0_2_0.custom.EClassCustomization;
import org.eclipse.emf.facet.custom.metamodel.v0_2_0.custom.FacetCustomization;
import org.eclipse.emf.facet.custom.ui.internal.exported.ICustomizationCommandFactory;
import org.eclipse.emf.facet.custom.ui.internal.exported.ICustomizationCommandFactoryFactory;
import org.eclipse.emf.facet.efacet.core.FacetUtils;
import org.eclipse.emf.facet.efacet.core.IFacetManagerFactory;
import org.eclipse.emf.facet.efacet.core.exception.FacetManagerException;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.DerivedTypedElement;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.EFacetPackage;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.Facet;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.FacetOperation;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.FacetSet;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.extensible.Query;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.query.IsOneOfQuery;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.query.QueryFactory;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.query.QueryPackage;
import org.eclipse.emf.facet.util.core.DebugUtils;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.emf.facet.util.emf.core.EmfDebugUtils;
import org.eclipse.emf.facet.util.emf.core.command.ICommandFactoryResult;
import org.eclipse.emf.facet.util.emf.core.command.ICommandFactoryResultFactory;
import org.eclipse.emf.facet.widgets.celleditors.ICommandFactoriesRegistry;
import org.eclipse.emf.facet.widgets.celleditors.ICommandFactory;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.Column;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.FeatureColumn;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.Row;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.SourceColumn;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.Table;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.TableFactory;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.TablePackage;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.IGridElement;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.IPositionCoordinate;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.TableWidgetUtils;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.exception.MultiModelQueryException;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.exception.TableWidgetRuntimeException;
import org.eclipse.jface.viewers.IStructuredSelection;

public final class TableInstanceCommandFactory {
	
	protected static final boolean DEBUG = DebugUtils.getDebugStatus(Activator.getDefault());

	/**
	 * This method create EMF commandes to load a set of customizations.
	 * @param customizationsToLoad an ordred list of customization. The first customization in the list has the highest priority.
	 * @param mustLoadRequiredFacets if true, the facets referred by the customization set will be loaded.
	 * @param controller
	 * @return null if no action has to be performed.
	 */
	public static Command createLoadCustomizationsCommand(
			final List<Customization> customizationsToLoad,
			final boolean mustLoadRequiredFacets,
			final TableWidgetController controller) {
		List<Command> cmds = new ArrayList<Command>();
		if (mustLoadRequiredFacets) {
			List<FacetSet> facets = new ArrayList<FacetSet>(controller
					.getTable().getFacetSets());
			facets.addAll(CustomizationUtils
					.findFacetsCustomizedBy(customizationsToLoad));
			Command facetLoad;
			try {
				facetLoad = createSetFacetsCommand(facets, controller);
			} catch (CoreException e) {
				throw new TableWidgetRuntimeException(e);
			}
			if (facetLoad != null) {
				cmds.add(facetLoad);
			}
		}
		Command loadCustomCmd = createLoadCustomizationsCommand(
				customizationsToLoad, controller);
		if (loadCustomCmd != null) {
			cmds.add(loadCustomCmd);
		}
		CompoundCommand result = null;
		if (!cmds.isEmpty()) {
			result = new CompoundCommand(Messages.TableInstanceCommandFactory_customizationLoad, cmds);
		}
		return result;
	}
	
	public static Command createLoadCustomizationsCommand(
			final List<Customization> customizationsToLoad, final TableWidgetController natTableWidget) {
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		List<Customization> localCustomizations = natTableWidget.getLocalCustomizations();
		if (!localCustomizations.isEmpty()
				&& !customizationsToLoad.containsAll(localCustomizations)) {
			// the locals custom must always be set here
			customizationsToLoad.addAll(0, localCustomizations);
		}
		Command result = null;
		Object loadedCustomizations = natTableWidget.getTable().getCustomizations();
		if (!customizationsToLoad.isEmpty()
				&& !customizationsToLoad.equals(loadedCustomizations)) {
			result = commandFactory.createSetCommand(natTableWidget.getEditingDomain(),
					natTableWidget.getTable(),
					TablePackage.eINSTANCE.getTable_Customizations(),
					customizationsToLoad);
		}
		return result;
	}

	public static Command createDeleteSelectedColumnCommand(final TableWidgetController natTableWidget) {
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		CompoundCommand compoundCommand = new CompoundCommand();

		for (Column column : natTableWidget.getSelectedColumns()) {
			Command removeRowCommand = commandFactory.createRemoveCommand(
					natTableWidget.getEditingDomain(), natTableWidget.getTable(),
					TablePackage.eINSTANCE.getTable_Columns(), column);
			compoundCommand.append(removeRowCommand);
		}
		return compoundCommand;
	}

	/**
	 * 
	 * @param natTableWidget
	 * @return
	 * @throws CoreException can be caused by query error in the facet framework
	 */
	public static Command createDeleteSelectedElementsCommand(final TableWidgetController natTableWidget,
			final ICommandFactory commandFactory) throws CoreException {
		Set<EObject> toDelete = new HashSet<EObject>();
		List<Object> rawSelection = natTableWidget.getRawSelection();
		for (Object selectedElement : rawSelection) {
			if (selectedElement instanceof IGridElement) {
				IGridElement gridElement = (IGridElement) selectedElement;
				try {
					Object value = TableWidgetUtils.getValueOf(gridElement, natTableWidget.getFacetContext());
					if (value instanceof EObject) {
						EObject eObject = (EObject) value;
						toDelete.add(eObject);
					}
				} catch (FacetManagerException e) {
					Logger.logError("Error getting the value of a grid element to delete", Activator.getDefault()); //$NON-NLS-1$
				}
			}
		}
		
		CompoundCommand compoundCommand = new CompoundCommand();
		for (EObject eObjectToDelete : toDelete) {
			Command deleteCommand = commandFactory.createDeleteCommand(natTableWidget.getEditingDomain(), eObjectToDelete);
			compoundCommand.append(deleteCommand);
		}
		
		DebugUtils.debug(TableInstanceCommandFactory.DEBUG, "compoundCommand.canExecute()=" + compoundCommand.canExecute()); //$NON-NLS-1$
		return compoundCommand;
	}

	/**
	 * 
	 * @param tableWidgetController
	 * @return
	 * @throws CoreException can be caused by query error in the facet framework
	 */
	public static Command createRemoveLineCommand(final TableWidgetController tableWidgetController) throws CoreException {
		CompoundCommand compoundCommand = new CompoundCommand();
		ICommandFactory commandFactory = tableWidgetController.getCommandFactory();
		Table tableInstance = tableWidgetController.getTable();
		List<Integer> toBeRemoved = new ArrayList<Integer>();
		IPositionCoordinate[] selectedCells = tableWidgetController.getSelectedCells2();

		for (IPositionCoordinate positionCoordinate : selectedCells) {
			if (!toBeRemoved.contains(positionCoordinate.getRowPosition())) {
				toBeRemoved.add(positionCoordinate.getRowPosition());
			}
		}
		return createRemoveLinesCommand(tableWidgetController, toBeRemoved);
	}

	/**
	 * Create a command that removes the given lines.
	 * 
	 * @param tableWidgetController
	 *            the table widget from which to remove the lines
	 * @param toBeRemoved
	 *            the indices of the lines to remove
	 * @return the Command that removes the lines when executed
	 * @throws CoreException
	 */
	public static Command createRemoveLinesCommand(final TableWidgetController tableWidgetController, final List<Integer> linesToRemove) throws CoreException {
		Table tableInstance = tableWidgetController.getTable();
		ICommandFactory commandFactory = tableWidgetController.getCommandFactory();
		CompoundCommand compoundCommand = new CompoundCommand();
		for (int i = 0; i < tableInstance.getRows().size(); i++) {
			if (linesToRemove.contains(i)) {
				Command removeRowCommand = commandFactory.createRemoveCommand(
						tableWidgetController.getEditingDomain(),
						tableInstance, TablePackage.eINSTANCE.getTable_Rows(),
						tableInstance.getRows().get(i));
				compoundCommand.append(removeRowCommand);
			}
		}
		CompoundCommand updateColumnCommand;
		updateColumnCommand = createRemoveUselessColumnsCommand(tableWidgetController, linesToRemove);
		// An empty command is not executable
		if (updateColumnCommand != null) {
			compoundCommand.append(updateColumnCommand);
		}
		return compoundCommand;
	}
	
	/**
	 * This method generate a command to remove useless columns.
	 * @param natTableWidget
	 * @param toBeRemoved indexes of the lines that are planned to be removed 
	 * @return null if non change is needed.
	 * @throws CoreException
	 */
	public static CompoundCommand createRemoveUselessColumnsCommand(
			final TableWidgetController natTableWidget, final List<Integer> toBeRemoved)
			throws CoreException {
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		Table tableInstance = natTableWidget.getTable();
		EditingDomain editingDomain  = natTableWidget.getEditingDomain();
		List<Row> rowsToRemove = new ArrayList<Row>();
		if (toBeRemoved != null) {
			for (Integer i : toBeRemoved) {
				rowsToRemove.add(tableInstance.getRows().get(i.intValue()));
			}
		}
		CompoundCommand result = createRemoveUselessColumnsCommand(
				tableInstance, rowsToRemove, null, commandFactory,
				editingDomain);
		return result;
	}

	/**
	 * This method generates a command to remove useless columns.
	 * 
	 * @param tableInstance
	 * @param rowsToRemove
	 *            rows that are planned to be removed. Must not be null.
	 * @param nextLoadedFacetSets
	 *            The set of facet which is loaded by sibling command, can be
	 *            null.
	 * @param commandFactory
	 * @param editingDomain
	 * @return null if no change is needed.
	 * @throws MultiModelQueryException
	 */
	public static CompoundCommand createRemoveUselessColumnsCommand(
			final Table tableInstance, final List<Row> rowsToRemove,
			final List<FacetSet> nextLoadedFacetSets,
			final ICommandFactory commandFactory,
			final EditingDomain editingDomain) throws MultiModelQueryException {
		List<Command> commandList = new ArrayList<Command>();
		List<Row> rowsToKeep = new ArrayList<Row>(tableInstance.getRows());
		rowsToKeep.removeAll(rowsToRemove);
		List<FacetSet> facetSets;
		if (nextLoadedFacetSets == null) {
			facetSets = tableInstance.getFacetSets();
		} else {
			facetSets = nextLoadedFacetSets;
		}
		List<FeatureColumn> columns = TableInstanceUtils.columnsToRemove(
				tableInstance, facetSets, rowsToKeep);
		
		for (FeatureColumn column : columns) {
			DebugUtils.debug(TableInstanceCommandFactory.DEBUG,
					"column to remove: " //$NON-NLS-1$
							+ EcoreUtil.getURI(column.getFeature()));
			// rmColumnCmd = remove column command
			Command rmColumnCmd = commandFactory.createRemoveCommand(
					editingDomain, tableInstance,
					TablePackage.eINSTANCE.getTable_Columns(), column);
			commandList.add(rmColumnCmd);
		}
		CompoundCommand result = null;
		if (!commandList.isEmpty()) {
			result = new CompoundCommand(Messages.TableInstanceCommandFactory_removeUselessColumn, commandList);
		}
		if (TableInstanceCommandFactory.DEBUG) {
			int nbCommand;
			if (result != null) {
				nbCommand = result.getCommandList().size();
			} else {
				nbCommand = 0;
			}
			DebugUtils.debug("Columns to be removed: " + nbCommand); //$NON-NLS-1$
		}
		return result;
	}

	//TODO: To be factorized in an utility class in the facet API.
	//TODO: cf. Bug 352774 - getStructuralFeatures(EObject eObject, List<Facet> appliedFacets) in an utility class 
	// This is method is not absolutly requiered to fix the bug 345730 but the
	// use of the a FacetContext instance is not required to answer to this
	// question. The FacetContext is used by the table widget but if is not the
	// master component to maintain the list of applied facet. This list is
	// maintained is the tableInstance model. That why I've retrun a static
	// method method to provide this service.
	/**
	 * 
	 * @param eObject
	 * @param appliedFacets
	 * @return
	 * @throws FacetManagerException
	 * @throws ModelQueryException
	 */
	public static List<EStructuralFeature> getStructuralFeatures(
			final EObject eObject, final List<Facet> appliedFacets)
			throws FacetManagerException {
		EList<EStructuralFeature> result = new BasicEList<EStructuralFeature>();
		for (Facet facet : appliedFacets) {
			if (TableInstanceCommandFactory.isConforming(eObject, facet)) {
				result.addAll(facet.getFacetElements());
			}
		}
		return result;
	}
	
	//TODO: To be factorized in an utility class in the facet API.
	//TODO: cf. Bug 352773 - isInstance(EObject eObject, Facet facet) in an utility class
	// This method is a copy of
	// org.eclipse.emf.facet.infra.facet.core.FacetContext.isInstance(EObject,
	// Facet) but accessible in a static way.
	// This is method is not absolutly requiered to fix the bug 345730 but the
	// use of the a FacetContext instance is not required to answer to this
	// question. The FacetContext is used by the table widget but if is not the
	// master component to maintain the list of applied facet. This list is
	// maintained is the tableInstance model. That why I've retrun a static
	// method method to provide this service.
	private static boolean isConforming(final EObject eObject, final Facet facet)
			throws FacetManagerException {
		return IFacetManagerFactory.DEFAULT.getOrCreateFacetManager(
				new ResourceImpl()).isConforming(eObject, facet);
	}
	
	
	
	public static CompoundCommand createAddRowsCommand(final List<EObject> newElements,
			final TableWidgetController natTableWidget) {
		CompoundCommand compoundCommand = new CompoundCommand();
		// the EPackage for which the MetamodelView has already been created
		Set<EPackage> alreadyDone = new HashSet<EPackage>();
		Table tableInstance = natTableWidget.getTable();
		EditingDomain editingDomain = natTableWidget.getEditingDomain();
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		List<EStructuralFeature> processedEStructuralFeatures = new ArrayList<EStructuralFeature>();
		final List<EObject> tableContent = TableWidgetUtils.getElements(tableInstance);
		for (EObject eObject : newElements) {
			if (!tableContent.contains(eObject)) {
				Row row = TableFactory.eINSTANCE.createRow();
				Command setEobjectToRowCMD = commandFactory.createSetCommand(editingDomain, row,
						TablePackage.eINSTANCE.getRow_Element(), eObject);
				compoundCommand.append(setEobjectToRowCMD);
				Command addRowCMD = commandFactory.createAddCommand(editingDomain, tableInstance,
						TablePackage.eINSTANCE.getTable_Rows(), row);
				compoundCommand.append(addRowCMD);
				for (EStructuralFeature eStructuralFeature : eObject.eClass()
						.getEAllStructuralFeatures()) {
					if (!processedEStructuralFeatures.contains(eStructuralFeature)) {
						processedEStructuralFeatures.add(eStructuralFeature);
						if (!natTableWidget.isColumnAlreadyDeclared(eStructuralFeature)) {
							FeatureColumn referenceColumn = TableFactory.eINSTANCE
									.createFeatureColumn();
							referenceColumn.setFeature(eStructuralFeature);
							Command cmd = commandFactory.createAddCommand(
									editingDomain, tableInstance,
									TablePackage.eINSTANCE.getTable_Columns(),
									referenceColumn);
							compoundCommand.append(cmd);
							// we add the local customization file
								List<Customization> localCustoms = natTableWidget
										.getLocalCustomizations();
								EObject container = eStructuralFeature.eContainer();
								if (container != null) {
									container = container.eContainer();
									if (container instanceof EPackage) {
										if (!alreadyDone.contains(container)) {
											if (CustomizationUtils.findCustomizationExtendingEPackage(localCustoms,
													(EPackage) container) == null) {
											Command cmd2 = createMetamodelViewCommand(
														(EPackage) container,
														natTableWidget);
											if (cmd2.canExecute()) {
												compoundCommand.append(cmd2);
												}
												alreadyDone.add((EPackage) container);
											}
										}
									}
								}
						}
					}
				}
			}
		}
		//If need the 3 non feature column are created.
		if (!newElements.isEmpty()) {
			Command addNonFeatureColumnCommand = createAddNonFeatureColumnIfNeeded(natTableWidget);
			if (addNonFeatureColumnCommand != null) {
				compoundCommand.append(addNonFeatureColumnCommand);
			}
		}
		return compoundCommand;
	}

	/**
	 * This method returns a CompoundCommand that create the 3 non feature columns, if needed.
	 * @param natTableWidget
	 * @return null nothing to do
	 */
	private static CompoundCommand createAddNonFeatureColumnIfNeeded(final TableWidgetController natTableWidget) {
		CompoundCommand result = new CompoundCommand();
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		EditingDomain editingdomain = natTableWidget.getEditingDomain();
		Table tableInstance = natTableWidget.getTable();
		if (result.getCommandList().isEmpty()) {
			result = null;
		}
		return result;
	}

	private static Command createMetamodelViewCommand(final EPackage ePackage,
			final TableWidgetController natTableWidget) {
		CompoundCommand compoundCommand = new CompoundCommand();
		Table tableInstance = natTableWidget.getTable();
		EditingDomain editingDomain = natTableWidget.getEditingDomain();
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
//TODO Remove before to commit
//		Customization newMetamodelView = CustomFactory.eINSTANCE.createCustomization();
//		Command createMetamodelView = commandFactory.createSetCommand(editingDomain,
//				newMetamodelView, CustomPackage.eINSTANCE.getCustomization_CustomizedEPackage(), ePackage);
//		compoundCommand.append(createMetamodelView);
//			Command setMetamodelViewLocalCmd = commandFactory.createAddCommand(editingDomain,
//					tableInstance,
//					TablePackage.eINSTANCE.getTableInstance_LocalCustomizations(),
//					newMetamodelView);
//			compoundCommand.append(setMetamodelViewLocalCmd);
//TODO Remove before to commit (end)
		final ICommandFactoryResult<Customization> cmdFactoryResult = createLocalCustom(tableInstance, editingDomain, commandFactory, ePackage);
		Customization newMetamodelView = cmdFactoryResult.getResult();
		compoundCommand.append(cmdFactoryResult.getCommand());
			List<Customization> views = new ArrayList<Customization>();
			views.addAll(tableInstance.getCustomizations());
			// we look for the index of the local new custom
			List<Customization> localCustom = natTableWidget.getLocalCustomizations();
			int i = 0;
			for (; i < views.size(); i++) {
				if (localCustom.contains(views.get(i))) {
					break;
				}
			}
			views.add(i, newMetamodelView);
			Command setMetamodelViewCmd = commandFactory.createSetCommand(editingDomain,
					tableInstance,
				TablePackage.eINSTANCE.getTable_Customizations(), views);
			compoundCommand.append(setMetamodelViewCmd);
		return compoundCommand;
	}

	public static Command createShowHideColumnCommand(final TableWidgetController widgetController,
			final List<Column> columnsToShow, final List<Column> columnsToHide,
			final boolean putOnTheTop) {
		CompoundCommand compoundCommand = new CompoundCommand("Show/hide column"); //$NON-NLS-1$
		EditingDomain editingDomain = widgetController.getEditingDomain();
		ICommandFactory commandFactory = widgetController.getCommandFactory();
		
			for (Column current : columnsToShow) {
			if (current instanceof SourceColumn) {
					Command cmd = commandFactory.createSetCommand(editingDomain, current,
						TablePackage.eINSTANCE.getSourceColumn_IsHidden(), Boolean.FALSE);
					if (cmd.canExecute()) {
						compoundCommand.append(cmd);
					}
				}
			}
			HashSet<FeatureColumn> fColumnsToHide = new HashSet<FeatureColumn>();
			for (Column current : columnsToHide) {
				if (current instanceof FeatureColumn) {
					fColumnsToHide.add((FeatureColumn) current);
				} else {
				Command cmd = commandFactory
						.createSetCommand(widgetController.getEditingDomain(),
								current, TablePackage.eINSTANCE
										.getSourceColumn_IsHidden(), Boolean.TRUE);
					compoundCommand.append(cmd);
				}
			}
		Command tmp = createHideColumnCommand(widgetController, fColumnsToHide);
			if (tmp != null) {
				compoundCommand.append(tmp);
			}
		if (putOnTheTop) {
			Command cmd = createPutLocalCustomizationOnTheTopCommand(widgetController);
			if (cmd.canExecute()) {
				compoundCommand.append(cmd);
			}
		}
		return compoundCommand;
	}

	public static Command createPutLocalCustomizationOnTheTopCommand(
			final TableWidgetController natTableWidget) {
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		List<Customization> localCustoms = natTableWidget.getLocalCustomizations();
		List<Customization> customizations = new ArrayList<Customization>();
		customizations.addAll(natTableWidget.getTable().getCustomizations());
		customizations.removeAll(localCustoms);
		customizations.addAll(0, localCustoms);
		return commandFactory.createSetCommand(natTableWidget.getEditingDomain(),
				natTableWidget.getTable(),
				TablePackage.eINSTANCE.getTable_Customizations(),
				customizations);
	}

	public static Command createSetFacetsCommand(
			final List<FacetSet> facetSets,
			final TableWidgetController natTableWidget) throws CoreException {
		return createSetFacetsCommand(facetSets, null, natTableWidget);
	}

	public static Command createSetFacetsCommand(
			final List<FacetSet> facetSets,
			final List<EObject> newElements, final TableWidgetController controller)
			throws CoreException {
		List<Command> cmdList = new ArrayList<Command>();
		ICommandFactory commandFactory = controller.getCommandFactory();
		Table tableInstance = controller.getTable();
		// Add new facets
		for (FacetSet facet : facetSets) {
			if (!tableInstance.getFacetSets().contains(facet)) {
				Command command = commandFactory.createAddCommand(
						controller.getEditingDomain(), tableInstance,
						TablePackage.eINSTANCE.getTable_FacetSets(), facet);
				cmdList.add(command);
			}
		}
		List<FacetSet> newFacets = new ArrayList<FacetSet>();
		newFacets.addAll(facetSets);
		// Remove non referenced one
		for (FacetSet facet : tableInstance.getFacetSets()) {
			if (!facetSets.contains(facet)) {
				newFacets.add(facet);
				Command command = commandFactory.createRemoveCommand(
						controller.getEditingDomain(), tableInstance,
						TablePackage.eINSTANCE.getTable_FacetSets(), facet);
				cmdList.add(command);
			}
		}
		// Remove outdated columns
		Command removeColumn = createRemoveUselessColumnsCommand(tableInstance,
				Collections.<Row>emptyList(), facetSets, commandFactory,
				controller.getEditingDomain());
		if (removeColumn != null) {
			cmdList.add(removeColumn);
		}
		CompoundCommand compoundCommand = null;
		if (!cmdList.isEmpty()) {
			compoundCommand = new CompoundCommand(Messages.TableInstanceCommandFactory_setFacet, cmdList);
		}
		debugCommand(compoundCommand);
		return compoundCommand;
	}

	public static CompoundCommand createDropCommand(final TableWidgetController natTableWidget,
			final IStructuredSelection structuredSelection, final Object element,
			final EStructuralFeature feature) {
		EditingDomain editingDomain = natTableWidget.getEditingDomain();
		if (editingDomain == null) {
			return null;
		}
		ICommandFactory commandFactory = natTableWidget.getCommandFactory();
		CompoundCommand compoundCommand = new CompoundCommand();
		if (feature.isMany()) {
			Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				Object object = iterator.next();
				Command addCommand = commandFactory.createAddCommand(editingDomain, element,
						feature, object);
				compoundCommand.append(addCommand);
			}
		} else {
			if (structuredSelection.toList().size() == 1) {
				Object firstElement = structuredSelection.getFirstElement();
				Command addCommand = commandFactory.createSetCommand(editingDomain, element,
						feature, firstElement);
				compoundCommand.append(addCommand);
			}
		}
		return compoundCommand;
	}

	/**
	 * 
	 * @param column
	 *            a column
	 * @param isHidden
	 *            the new feature value for this column
	 * @param customizedFeatureName
	 *            the customized feature name
	 * @return
	 */
	public static Command createHideColumnCommand(final TableWidgetController widgetController,
			final Collection<FeatureColumn> featureColumns) {
		final FacetOperation isVisibleProperty = widgetController
				.getCustomPropertiesHandler().getIsVisible();
		final CompoundCommand resultCmd = new CompoundCommand("Show/hide column"); //$NON-NLS-1$
		final EditingDomain editingDomain = widgetController.getEditingDomain();
		final ICommandFactory commandFactory = ICommandFactoriesRegistry.INSTANCE
				.getCommandFactoryFor(editingDomain);
//		final EStructuralFeature featureOfTheColumn = featureColumn
//				.getFeature();
//		EObject packageLevel = featureOfTheColumn.eContainer().eContainer();
//		if (!(packageLevel instanceof EPackage)) {
//			throw new IllegalStateException(
//					"Unexpected feature container type " + packageLevel + " found. (EPackage is execpected)"); //$NON-NLS-1$ //$NON-NLS-2$
//		}
//		final EPackage customizedEPackage = (EPackage) packageLevel;
		// 1 we look for the corresponding customization
		final List<Customization> localCustomizations = widgetController.getTable()
				.getLocalCustomizations();
		Customization customization = null;
		if (!localCustomizations.isEmpty()) {
			customization = localCustomizations.get(0);
		}
		final ICustomizationCommandFactory customCmdFactory = ICustomizationCommandFactoryFactory.DEFAULT
				.createCustomizationCommandFactory(editingDomain, commandFactory);
		// If the wanted customization does not exist then we create it.
		if (customization == null) {
			ICommandFactoryResult<Customization> cmdFactoryResult = createLocalCustom(
					widgetController.getTable(), editingDomain,
					commandFactory, EcorePackage.eINSTANCE);
			customization = cmdFactoryResult.getResult();
			resultCmd.append(cmdFactoryResult.getCommand());
		}
		Facet typeCustomization = CustomizationUtils
				.findFacetByExtendedMetaclass(
						FacetUtils.getFacets(customization),
						EcorePackage.eINSTANCE.getETypedElement());
		IsOneOfQuery conformanceQuery;
		if (typeCustomization == null) {
			conformanceQuery = QueryFactory.eINSTANCE.createIsOneOfQuery();
			ICommandFactoryResult<EClassCustomization> createEClassCustom = customCmdFactory
					.createEClassCustomization(customization,
							EcorePackage.eINSTANCE.getETypedElement(),
							conformanceQuery);
			typeCustomization = createEClassCustom.getResult();
			resultCmd.append(createEClassCustom.getCommand());
		} else {
			if (!(typeCustomization instanceof EClassCustomization)) {
				throw new TableWidgetRuntimeException(
						"Unexpected type for the variable 'featureContainer'"); //$NON-NLS-1$
			}
			DerivedTypedElement conformanceTE = (DerivedTypedElement) typeCustomization.getConformanceTypedElement();
			conformanceQuery = (IsOneOfQuery) conformanceTE.getQuery();
		}
		final HashSet<ETypedElement> featuresToHide = new HashSet<ETypedElement>();
		for (FeatureColumn column : featureColumns) {
			featuresToHide.add(column.getFeature());
		}
		final Command initQuery = commandFactory.createSetCommand(editingDomain,
				conformanceQuery,
				QueryPackage.eINSTANCE.getIsOneOfQuery_ExpectedEObjects(),
				new ArrayList<ETypedElement>(featuresToHide));
		resultCmd.append(initQuery);
		final Query query = QueryFactory.eINSTANCE.createFalseLiteralQuery();
		final ICommandFactoryResult<Facet> propConfigCmd = customCmdFactory
				.setPropertyConfig(typeCustomization, null, isVisibleProperty,
						query);
		resultCmd.append(propConfigCmd.getCommand());
		return resultCmd;
	}

	private static ICommandFactoryResult<Customization> createLocalCustom(
			final Table tableInstance,
			final EditingDomain editingDomain,
			final ICommandFactory commandFactory, 
			final EPackage customizedEPackage
			) {
		final ICustomizationCommandFactory customCmdFactory = ICustomizationCommandFactoryFactory.DEFAULT
				.createCustomizationCommandFactory(editingDomain,
						commandFactory);
		Customization customization;
		CompoundCommand createCustom = new CompoundCommand("Create a local custom."); //$NON-NLS-1$
		ICommandFactoryResult<Customization> cmdFactoryResult = customCmdFactory.createCustomization(
				"Column hiding customization for "
						+ customizedEPackage.getName());
		customization = cmdFactoryResult.getResult();
		createCustom.append(cmdFactoryResult.getCommand());
		Command addCustomToLocalCmd = commandFactory.createAddCommand(
				editingDomain, tableInstance,
				TablePackage.eINSTANCE
.getTable_LocalCustomizations(),
				customization);
		createCustom.append(addCustomToLocalCmd);
		Command addCustomToAllCmd = commandFactory.createAddCommand(
				editingDomain, tableInstance,
				TablePackage.eINSTANCE
.getTable_Customizations(),
				customization);
		createCustom.append(addCustomToAllCmd);
		return ICommandFactoryResultFactory.DEFAULT.createCommandFactoryResult(createCustom, customization);
	}

	private static Facet createEmptyTypeCustomization(
			final FeatureColumn featureColumn,
			final Map<EClassifier, Facet> typeViewMap,
			final CompoundCommand resultCmd, final EditingDomain editingDomain,
			final ICommandFactory commandFactory,
			final Customization customization, final EClassifier featureContainer) {
		Facet typeCustomization = null;
		if (featureContainer instanceof Facet) {
			Facet facet = (Facet) featureColumn.getFeature().eContainer();
			FacetCustomization facetCustomization = CustomFactory.eINSTANCE
					.createFacetCustomization();
			Command linkACustomToAFacet = commandFactory.createSetCommand(
					editingDomain, facetCustomization, CustomPackage.eINSTANCE
							.getFacetCustomization_CustomizedFacet(), facet);
			resultCmd.append(linkACustomToAFacet);
			Command addFacetCustomToCustom = commandFactory.createAddCommand(
					editingDomain, customization,
					EFacetPackage.eINSTANCE.getFacet_FacetElements(),
					facetCustomization);
			resultCmd.append(addFacetCustomToCustom);
			typeCustomization = facetCustomization;
		} else if (featureContainer instanceof EClass) {
			EClass eClass = (EClass) featureContainer;
			EClassCustomization eClassCustomization = CustomFactory.eINSTANCE
					.createEClassCustomization();
			eClassCustomization.setExtendedMetaclass(eClass);
			Command linkACustomToAEClass = commandFactory.createSetCommand(
					editingDomain, eClassCustomization,
					EFacetPackage.eINSTANCE.getFacet_ExtendedMetaclass(),
					eClass);
			resultCmd.append(linkACustomToAEClass);
			typeCustomization = eClassCustomization;
			Command addEClassCustomToCustom = commandFactory.createAddCommand(
					editingDomain, customization,
					EFacetPackage.eINSTANCE.getFacet_FacetElements(),
					typeCustomization);
			resultCmd.append(addEClassCustomToCustom);
		}
		typeViewMap.put(featureContainer, typeCustomization);
		return typeCustomization;
	}

	private TableInstanceCommandFactory() {
		// Prevents instantiation
	}

	/**
	 * This method retruns an EMF command deleting the a collection of EObject 
	 * @param label This label will be visible in the menu 'Edit'.
	 * @param eObjects
	 * @param controller
	 * @return
	 */
	public static Command delete(final String label, final Collection<? extends EObject> eObjects,
			final TableWidgetController controller) {
		List<Command> cmdList = new ArrayList<Command>();
		for (EObject eObject : eObjects) {
			Command command = controller
					.getCommandFactory()
					.createDeleteCommand(controller.getEditingDomain(), eObject);
			cmdList.add(command);
		}
		Command result = null;
		if (!cmdList.isEmpty()) {
			result = new CompoundCommand(label, cmdList);
		}
		return result;
	}

	private static void debugCommand(final Command command) {
		if (TableInstanceCommandFactory.DEBUG) {
			if (command == null) {
				DebugUtils.debug("Null command.", 1); //$NON-NLS-1$
			} else {
				DebugUtils.debug(EmfDebugUtils.debugCommand(command), 1);
			}
		}

	}

	/**
	 * This method create a command deleting use less row and columns.
	 * @param controller
	 * @return null if no action has to be performed.
	 */
	public static final Command createRemoveUselessRowsAndColumnsCommand(
			final TableWidgetController controller) {
		List<Command> cmdList = new ArrayList<Command>();
		List<Row> rowsToRemove = TableInstanceUtils.findUselessRow(controller
				.getTable());
		if (TableWidgetController.DEBUG_REMOVE_USELESS_ROWS_AND_COLUMNS) {
			DebugUtils.debug("Rows to be removed: " + rowsToRemove.size()); //$NON-NLS-1$
		}
		final Command deleteRows = TableInstanceCommandFactory.delete(
				Messages.TableWidgetController_DeleteUselessRows, rowsToRemove,
				controller);
		if (deleteRows != null) {
			cmdList.add(deleteRows);
		}
		CompoundCommand removeColumnCommand = null;
		try {
			removeColumnCommand = TableInstanceCommandFactory
					.createRemoveUselessColumnsCommand(
							controller.getTable(), rowsToRemove, null,
							controller.getCommandFactory(),
							controller.getEditingDomain());
		} catch (MultiModelQueryException e) {
			// We do not want the table to break if a facet is failing
			Logger.logWarning(e, Activator.getDefault());
		}
		if (removeColumnCommand != null) {
			if (!removeColumnCommand.canExecute()) {
				throw new TableWidgetRuntimeException(
						"Command cannot be executed"); //$NON-NLS-1$
			}
			cmdList.add(removeColumnCommand);
		}
		CompoundCommand command = null;
		if (!cmdList.isEmpty()) {
			command = new CompoundCommand(
					Messages.TableWidgetController_RemoveUselessRowsAndColumns,
					cmdList);
		}
		return command;
	}
	
}
