/*******************************************************************************
 * Copyright (c) 2009-2012 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Nicolas Bros (Mia-Software)
 *     Nicolas Guyomar (Mia-Software) - Bug 339874 - Infra QuerySelectionDialog should have a "querySet" icon
 *     Nicolas Bros (Mia-Software) - Bug 366367 - To be able to change the "CanBePresentedInTheTable" query
 *******************************************************************************/
package org.eclipse.emf.facet.infra.query.ui.dialogs;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.facet.infra.query.ModelQuery;
import org.eclipse.emf.facet.infra.query.ModelQuerySet;
import org.eclipse.emf.facet.infra.query.core.ModelQuerySetCatalog;
import org.eclipse.emf.facet.infra.query.ui.Activator;
import org.eclipse.emf.facet.infra.query.ui.ImageProvider;
import org.eclipse.emf.facet.infra.query.ui.Messages;
import org.eclipse.emf.facet.infra.query.ui.controls.QueryFilteredTree;
import org.eclipse.emf.facet.infra.query.ui.wizards.CreateQueryWizard;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.IOpenListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.OpenEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;

/**
 * A dialog to select a {@link ModelQuery}
 * 
 * @deprecated use the new eFacet2 metamodel in org.eclipse.emf.facet.efacet2.metamodel
 *             and the new APIs in org.eclipse.emf.facet.efacet.ui
 */
@Deprecated
public class QuerySelectionDialog extends Dialog {

	private Collection<ModelQuerySet> fQuerySets;
	private final List<ViewerFilter> viewerFilters;
	private QueryFilteredTree filteredTree;
	private final boolean multiSelection;
	private List<ModelQuery> fSelectedQueries = new ArrayList<ModelQuery>();
	private final List<Runnable> listeners = new ArrayList<Runnable>();
	private final boolean allowEmptySelection;
	private Button checkboxEnableQuery;

	/**
	 * @param parentShell
	 *            the parent shell
	 * @param querySets
	 *            the query sets from which queries will be selectable. If
	 *            <code>null</code>, all query sets will be displayed.
	 * @param viewerFilters
	 *            optional filters for selectable queries (can be
	 *            <code>null</code> if no filtering is required)
	 * @param allowEmptySelection
	 *            whether to allow the user to select no query (i.e : select <code>null</code>)
	 */
	public QuerySelectionDialog(final Shell parentShell, final boolean multiSelection,
			final Collection<ModelQuerySet> querySets, final List<ViewerFilter> viewerFilters, final boolean allowEmptySelection) {
		super(parentShell);
		this.multiSelection = multiSelection;
		if (querySets == null) {
			this.fQuerySets = ModelQuerySetCatalog.getSingleton().getAllModelQuerySets();
		} else {
			this.fQuerySets = querySets;
		}
		this.viewerFilters = viewerFilters;
		this.allowEmptySelection = allowEmptySelection;
	}

	/**
	 * @param parentShell
	 *            the parent shell
	 * @param querySets
	 *            the query sets from which queries will be selectable. If
	 *            <code>null</code>, all query sets will be displayed.
	 * @param viewerFilters
	 *            optional filters for selectable queries (can be
	 *            <code>null</code> if no filtering is required)
	 */
	public QuerySelectionDialog(final Shell parentShell, final boolean multiSelection,
			final Collection<ModelQuerySet> querySets, final List<ViewerFilter> viewerFilters) {
		this(parentShell, multiSelection, querySets, viewerFilters, false);
	}
	
	protected QueryFilteredTree getFilteredTree() {
		return this.filteredTree;
	}
	
	protected Button getCheckboxEnableQuery() {
		return this.checkboxEnableQuery;
	}

	@Override
	protected Control createDialogArea(final Composite parent) {
		Composite composite = (Composite) super.createDialogArea(parent);
		if (this.allowEmptySelection) {
			this.checkboxEnableQuery = new Button(composite, SWT.CHECK);
			this.checkboxEnableQuery.setText(Messages.QuerySelectionDialog_enableQuery);
			this.checkboxEnableQuery.setSelection(false);
			this.checkboxEnableQuery.addSelectionListener(new SelectionAdapter() {
				@Override
				public void widgetSelected(final SelectionEvent e) {
					boolean enabled = getCheckboxEnableQuery().getSelection();
					setQuerySelectionEnabled(enabled);
					if (!enabled) {
						select(null);
					}
				}
			});
		}
		this.filteredTree = new QueryFilteredTree(composite, this.fQuerySets, this.viewerFilters,
				this.multiSelection);

		createContextMenu(this.filteredTree.getViewer());

		// double-click or enter to choose the selected query
		this.filteredTree.getViewer().addOpenListener(new IOpenListener() {
			public void open(final OpenEvent event) {
				QuerySelectionDialog.this.okPressed();
			}
		});
		
		if (this.allowEmptySelection) {
			setQuerySelectionEnabled(false);
		}
		
		return composite;
	}

	protected void setQuerySelectionEnabled(final boolean enabled) {
		getCheckboxEnableQuery().setSelection(enabled);
		this.filteredTree.getViewer().getTree().setEnabled(enabled);
		this.filteredTree.getFilterControl().setEnabled(enabled);
	}

	private class OpenCreateQueryWizardAction extends Action {
		private final ModelQuerySet modelQuerySet;

		public OpenCreateQueryWizardAction(final ModelQuerySet modelQuerySet) {
			this.modelQuerySet = modelQuerySet;
			setText(Messages.QuerySelectionDialog_createQuery);
		}

		@SuppressWarnings("synthetic-access")
		@Override
		public void run() {
			final CreateQueryWizard createQueryWizard = new CreateQueryWizard(this.modelQuerySet,
					null, null, null);
			WizardDialog wizardDialog = new WizardDialog(null, createQueryWizard);
			if (wizardDialog.open() == Window.OK) {
				Job refreshJob = new Job(Messages.QuerySelectionDialog_refreshQuerySetsJob) {
					@Override
					protected IStatus run(final IProgressMonitor monitor) {
						try {
							// rebuild the query catalog
							ResourcesPlugin.getWorkspace().build(
									IncrementalProjectBuilder.FULL_BUILD, monitor);
						} catch (CoreException e) {
							Logger.logError(e, Activator.getDefault());
						}
						// the query sets have been re-created => find by name
						Collection<ModelQuerySet> oldQuerySets = new ArrayList<ModelQuerySet>(
								QuerySelectionDialog.this.fQuerySets);
						final Collection<ModelQuerySet> newQuerySets = new ArrayList<ModelQuerySet>();
						for (ModelQuerySet oldQuerySet : oldQuerySets) {
							ModelQuerySet querySet = ModelQuerySetCatalog.getSingleton()
									.getModelQuerySet(oldQuerySet.getName());
							if (querySet != null) {
								newQuerySets.add(querySet);
							}
						}
						QuerySelectionDialog.this.fQuerySets = newQuerySets;
						Display.getDefault().asyncExec(new Runnable() {
							public void run() {
								QuerySelectionDialog.this.filteredTree.setInput(newQuerySets);
								// select the created query
								ModelQuery createdModelQuery = createQueryWizard
										.getCreatedModelQuery();
								ModelQuerySet querySet = ModelQuerySetCatalog.getSingleton()
										.getModelQuerySet(
												createdModelQuery.getModelQuerySet().getName());
								ModelQuery query = querySet.getQuery(createdModelQuery.getName());
								QuerySelectionDialog.this.filteredTree.getViewer().setSelection(
										new StructuredSelection(query), true);
							}
						});
						return Status.OK_STATUS;
					}
				};
				refreshJob.setPriority(Job.DECORATE);
				refreshJob.schedule();
			}
		}
	}

	private void createContextMenu(final TreeViewer viewer) {
		final MenuManager contextMenu = new MenuManager("#PopUp"); //$NON-NLS-1$
		contextMenu.setRemoveAllWhenShown(true);
		contextMenu.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(final IMenuManager manager) {
				ISelection selection = viewer.getSelection();
				if (selection instanceof IStructuredSelection) {
					IStructuredSelection structuredSelection = (IStructuredSelection) selection;
					if (structuredSelection.size() == 1) {
						Object firstElement = structuredSelection.getFirstElement();
						if (firstElement instanceof ModelQuerySet) {
							final ModelQuerySet modelQuerySet = (ModelQuerySet) firstElement;
							manager.add(new OpenCreateQueryWizardAction(modelQuerySet));
							return;
						}
					}
				}
				manager.add(new OpenCreateQueryWizardAction(null));
			}
		});
		final Menu menu = contextMenu.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
	}

	@Override
	protected void configureShell(final Shell shell) {
		super.configureShell(shell);
		shell.setText(Messages.QuerySelectionDialog_selectQuery);
		shell.setImage(ImageProvider.getInstance().getModelQuerySet());
	}

	@Override
	protected boolean isResizable() {
		return true;
	}

	@Override
	protected IDialogSettings getDialogBoundsSettings() {
		final String sectionName = this.getClass().getName() + ".settings"; //$NON-NLS-1$
		IDialogSettings settings = Activator.getDefault().getDialogSettings();
		IDialogSettings section = settings.getSection(sectionName);
		if (section == null) {
			section = settings.addNewSection(sectionName);
		}
		return section;
	}

	@Override
	protected void okPressed() {
		List<ModelQuery> tempSelectedQueries = new ArrayList<ModelQuery>();
		ISelection selection = this.filteredTree.getViewer().getSelection();
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				Object selectedElement = iterator.next();
				if (selectedElement instanceof ModelQuery) {
					tempSelectedQueries.add((ModelQuery) selectedElement);
				}
			}
		}
		this.fSelectedQueries = tempSelectedQueries;
		super.okPressed();
	}

	public List<ModelQuery> getSelectedQueries() {
		return Collections.unmodifiableList(this.fSelectedQueries);
	}
	
	
	public void pressOk() {
		okPressed();
	}

	public void pressCancel() {
		setReturnCode(Window.CANCEL);
		close();
	}
	
	@Override
	public boolean close() {
		boolean result = super.close();
		notifyListeners();
		return result;
	}

	public ModelQuery getFirstSelectedQuery() {
		ModelQuery selectedQuery = null;
		if (!this.filteredTree.isDisposed()) {
			IStructuredSelection selection = (IStructuredSelection) this.filteredTree.getViewer()
					.getSelection();
			Object firstElement = selection.getFirstElement();
			if (firstElement instanceof ModelQuery) {
				selectedQuery = (ModelQuery) firstElement;
			}
		} else {
			List<ModelQuery> queries = getSelectedQueries();
			if (queries.size() > 0) {
				selectedQuery = queries.get(0);
			}
		}
		return selectedQuery;
	}

	public void select(final ModelQuery query) {
		if (query != null) {
			setQuerySelectionEnabled(true);
			this.filteredTree.getViewer().setSelection(new StructuredSelection(query));
		} else {
			setQuerySelectionEnabled(false);
			this.filteredTree.getViewer().setSelection(new StructuredSelection());
		}
	}
	
	private synchronized void notifyListeners() {
		for (Runnable listener : new ArrayList<Runnable>(this.listeners)) {
			listener.run();
		}
	}

	public synchronized void addListener(final Runnable runnable) {
		this.listeners.add(runnable);
	}
}
