/*******************************************************************************
 * Copyright (c) 2011 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - Bug 343533 - Tool to check non-ICU4J references
 *******************************************************************************/
package org.eclipse.emf.facet.dev.ui.internal;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.search.IJavaSearchConstants;
import org.eclipse.jdt.core.search.IJavaSearchScope;
import org.eclipse.jdt.core.search.SearchEngine;
import org.eclipse.jdt.core.search.SearchMatch;
import org.eclipse.jdt.core.search.SearchParticipant;
import org.eclipse.jdt.core.search.SearchPattern;
import org.eclipse.jdt.core.search.SearchRequestor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.handlers.HandlerUtil;

/**
 * Implements an action to check the workspace for violations to the ICU4J rules to be checked for the simultaneous
 * release (http://wiki.eclipse.org/ICU4J)
 */
public class CheckICU4JHandler extends AbstractHandler {
	public CheckICU4JHandler() {
	}

	private class Match {
		private String type;
		private String location;
	}

	public Object execute(final ExecutionEvent event) throws ExecutionException {
		final IWorkbenchWindow window =
				HandlerUtil.getActiveWorkbenchWindow(event);
		Job job = new Job("Searching forbidden references to non-ICU4J classes") { //$NON-NLS-1$
			@Override
			protected IStatus run(final IProgressMonitor monitor) {
				search(monitor, window);
				return Status.OK_STATUS;
			}
		};
		job.setUser(true);
		job.schedule();
		return null;
	}

	private void search(final IProgressMonitor monitor, final IWorkbenchWindow window) {
		final List<CheckICU4JHandler.Match> matches = new ArrayList<CheckICU4JHandler.Match>();
		final int nTasks = 19;
		monitor.beginTask("Searching for forbidden references", nTasks); //$NON-NLS-1$

		search("java.text.BreakIterator", matches, monitor); //$NON-NLS-1$
		search("java.text.CollationKey", matches, monitor); //$NON-NLS-1$
		search("java.text.Collator", matches, monitor); //$NON-NLS-1$
		search("java.text.DateFormat", matches, monitor); //$NON-NLS-1$
		search("java.text.DateFormatSymbols", matches, monitor); //$NON-NLS-1$
		search("java.text.DecimalFormat", matches, monitor); //$NON-NLS-1$
		search("java.text.DecimalFormatSymbols", matches, monitor); //$NON-NLS-1$
		search("java.text.MessageFormat", matches, monitor); //$NON-NLS-1$
		search("java.text.NumberFormat", matches, monitor); //$NON-NLS-1$
		search("java.text.SimpleDateFormat", matches, monitor); //$NON-NLS-1$
		search("java.util.Calendar", matches, monitor); //$NON-NLS-1$
		search("java.util.TimeZone", matches, monitor); //$NON-NLS-1$

		search("java.util.Currency", matches, monitor); //$NON-NLS-1$
		search("java.util.GregorianCalendar", matches, monitor); //$NON-NLS-1$
		search("java.util.SimpleTimeZone", matches, monitor); //$NON-NLS-1$
		search("java.lang.Character", matches, monitor); //$NON-NLS-1$
		search("java.text.Format", matches, monitor); //$NON-NLS-1$
		search("java.util.Locale", matches, monitor); //$NON-NLS-1$
		search("java.util.ResourceBundle", matches, monitor); //$NON-NLS-1$

		if (monitor.isCanceled()) {
			return;
		}

		final String projectName;
		if (!matches.isEmpty()) {
			String newline = System.getProperty("line.separator"); //$NON-NLS-1$
			StringBuilder builder = new StringBuilder();
			for (Match match : matches) {
				builder.append(match.type + " : " + match.location + newline); //$NON-NLS-1$
			}
			IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject("icu4j_" + System.currentTimeMillis()); //$NON-NLS-1$
			try {
				project.create(monitor);
				project.open(monitor);
				projectName = project.getName();
				IFile file = project.getFile("violations.txt"); //$NON-NLS-1$
				ByteArrayInputStream inputStream = new ByteArrayInputStream(builder.toString().getBytes());
				file.create(inputStream, true, monitor);
			} catch (CoreException e) {
				Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Error creating results project", e)); //$NON-NLS-1$
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						MessageDialog.openError(window.getShell(), "Error", "Error creating results project (see log)"); //$NON-NLS-1$//$NON-NLS-2$
					}
				});
				return;
			}
		} else {
			projectName = null;
		}

		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				displayResult(matches, window, projectName);
			}
		});
	}

	protected void displayResult(final List<CheckICU4JHandler.Match> matches, final IWorkbenchWindow window, final String projectName) {
		if (matches.isEmpty()) {
			MessageDialog.openInformation(
					window.getShell(), "MoDisco Development Tools (Incubation)", //$NON-NLS-1$
					"No ICU4J violation found"); //$NON-NLS-1$
		} else {
			MessageDialog.openWarning(
					window.getShell(), "MoDisco Development Tools (Incubation)", //$NON-NLS-1$
					matches.size() + " violation(s) were found. See project " + projectName); //$NON-NLS-1$
		}
	}

	private void search(final String type, final List<CheckICU4JHandler.Match> matches, final IProgressMonitor monitor) {
		if (monitor.isCanceled()) {
			return;
		}
		monitor.subTask("Searching for occurrences of " + type); //$NON-NLS-1$
		SearchRequestor searchRequestor = new SearchRequestor() {
			@Override
			public void acceptSearchMatch(final SearchMatch searchMatch) throws CoreException {
				Object element = searchMatch.getElement();
				if (element instanceof IMember) {
					IMember member = (IMember) element;
					// ignore references in jars
					if (member.isBinary()) {
						return;
					}
				}

				Match match = new Match();
				match.type = type;
				IResource resource = searchMatch.getResource();
				match.location = resource.getFullPath().toString();
				matches.add(match);
				// System.out.println("{{element class}} : " +
				// searchMatch.getElement().getClass());
				// System.out.println(searchMatch);
			}
		};

		IJavaSearchScope scope = SearchEngine.createWorkspaceScope();
		SearchPattern pattern = SearchPattern.createPattern(
				type, IJavaSearchConstants.TYPE, IJavaSearchConstants.REFERENCES, SearchPattern.R_EXACT_MATCH);
		SearchEngine searchEngine = new SearchEngine();
		try {
			searchEngine.search(pattern, new SearchParticipant[] { SearchEngine.getDefaultSearchParticipant() }, scope, searchRequestor,
					new SubProgressMonitor(monitor, 0));
		} catch (CoreException e) {
			Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Error searching for " + type, e)); //$NON-NLS-1$
			Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					MessageDialog.openError(null, "Error", "Error searching for " + type + " (see log)"); //$NON-NLS-1$//$NON-NLS-2$ //$NON-NLS-3$
				}
			});
		}
		monitor.worked(1);
	}
}
