/*******************************************************************************
 * Copyright (c) 2010, 2011 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - Bug 331203 - table model editor - initial API and implementation
 *    Nicolas Bros (Mia-Software) - Bug 332437 - NatTable : pluggable cell editors
 *    Nicolas Guyomar (Mia-Software) - Bug 331442 - To be able to add and remove lines (model elements) from the table
 *    Nicolas Bros (Mia-Software) - Bug 332226 - To be able to create or delete model element from the table
 *    Nicolas Guyomar (Mia-Software) - Bug 332226 - To be able to create or delete model element from the table
 *    Nicolas Bros (Mia-Software) - Bug 332438 - NatTable : table type
 *    Nicolas Bros (Mia-Software) - Bug 332440 - NatTable : force cell editors
 *    Nicolas Bros (Mia-Software) - Bug 331675 - NatTable : copy cells as text
 *    Nicolas Bros (Mia-Software) - Bug 331900 - customizable NatTable
 *    Nicolas Bros (Mia-Software) - Bug 332010 - view Facet customizations on NatTable
 *    Nicolas Bros (Mia-Software) - Bug 332215 - customizable NatTable column headers
 *    Nicolas Guyomar (Mia-Software) - Bug 332215 - customizable NatTable column headers
 *    Nicolas Guyomar (Mia-Software) - Bug 332924 - To be able to save the table
 *    Nicolas Guyomar (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *    Gregoire Dupe (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *    Gregoire Dupe (Mia-Software) - Bug 333015 - To be able to hide columns
 *    Nicolas Guyomar (Mia-Software) - Bug 333015 - To be able to hide columns
 *    Nicolas Guyomar (Mia-Software) - Bug 333029 - To be able to save the size of the lines and the columns
 *    Nicolas Guyomar (Mia-Software) - Bug 333414 - The user must be able to save the column order
 *    Nicolas Guyomar (Mia-Software) - Bug 335154 - Sort Column By Type : Cannot modify resource set without a write transaction
 *    Nicolas Guyomar (Mia-Software) - Bug 335155 - Hide Empty Column : Cannot modify resource set without a write transaction
 *    Nicolas Guyomar (Mia-Software) - Bug 335156 - Only Show Common column : Cannot modify resource set without a write transaction
 *    Nicolas Guyomar (Mia-Software) - Bug 335020 - Nattable widget should use the Eclipse framework
 *    Nicolas Guyomar (Mia-Software) - Bug 337322 - [TableConfiguration] Customization declared in the file tableconfiguration is not loaded
 *    Nicolas Guyomar (Mia-Software) - Bug 338536 - Problem with the refresh of the table : the scrollbar returns to this initial position
 *    Nicolas Bros (Mia-Software) - Bug 338536 - Problem with the refresh of the table : the scrollbar returns to this initial position
 *    Nicolas Guyomar (Mia-Software) - Bug 337454 - We can't delete a query Column
 *    Nicolas Guyomar (Mia-Software) - Bug 337395 - Unused columns should be destroyed
 *    Nicolas Guyomar (Mia-Software) - Bug 339554 - org.eclipse.emf.facet.widgets.celleditors API cleaning
 *    Nicolas Guyomar (Mia-Software) - Bug 339922 - INatTableWidget method isCellSelected should use the plural
 *    Nicolas Guyomar (Mia-Software) - Bug 340681 - Facet column implementation
 *    Vincent Lorenzo (CEA-LIST) - Bug 337326 - Show/Hide Column : Sort the columns by name
 *    Nicolas Guyomar (Mia-Software) - Bug 340940 - To be able to view facet attributes and facet references in a table
 *    Vincent Lorenzo (CEA-LIST) - Bug 337408 - Add an action to sort columns by name
 *    Nicolas Guyomar (Mia-Software) - Bug 336482 - KeyBinding to edit element in Table : F2
 *    Vincent Lorenzo (CEA-LIST) - Bug 341238 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Gregoire Dupe (Mia-Software) - Bug 341238 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Nicolas Guyomar (Mia-Software) - Bug 342451 - To be able to edit derived facet attributes and derived facet references in a table
 *    Vincent Lorenzo (CEA-LIST) - Bug 341238 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Nicolas Guyomar (Mia-Software) - Bug 343411 - [Table] Create new elements does not support IJavaModelQuery anymore
 *    Gregoire Dupe (Mia-Software) - Bug 343811 - EMF Facet Regression : Created elements in a table are not serialized
 *    Vincent Lorenzo (CEA-LIST) - Bug 344125 - The API should provide a method selectRows(List<EObject> elementsToSelect)
 *    Nicolas Guyomar (Mia-Software) - Bug 344274 - SWT BOT fail on Hudson
 *    Nicolas Guyomar (Mia-Software) - Bug 344475 - To be able to select a cell by EStructuralFeature in the table
 *    Nicolas Guyomar (Mia-Software) - Bug 342028 - Field can be edited even if they are marked as N/A
 *    Nicolas Guyomar (Mia-Software) - Bug 344413 - Facet Columns are never created when we begin with an empty table
 *    Gregoire Dupe (Mia-Software) - Bug 343859 - The local customizations are not applied when we reopen a table
 *    Nicolas Guyomar (Mia-Software) - Bug 344670 - Problems with the columns creation : very slow + scrollbar blinked
 *    Nicolas Guyomar (Mia-Software) - Bug 345554 - SWTBot regression on build 303
 *    Nicolas Guyomar (Mia-Software) - Bug 344925 - Undo/Redo after the action Show Columns
 *    Nicolas Guyomar (Mia-Software) - Bug 345730 - Deleting an element in the model breaks the table
 *    Nicolas Guyomar (Mia-Software) - Bug 344921 - Undo/Redo just after the creation of the table
 *    Nicolas Guyomar (Mia-Software) - Bug 346465 - [EMF Facet Table] Remove line does not remove obsolete column
 *    Gregoire Dupe (Mia-Software) - Bug 345730 - Deleting an element in the model breaks the table
 *    Vincent Lorenzo (CEA-LIST) - Bug 352822 - CellEditor EEnum is not available for FacetAttribute
 *    Nicolas Bros (Mia-Software) - Bug 352822 - CellEditor EEnum is not available for FacetAttribute
 *    Vincent Lorenzo (CEA-LIST) - Bug 351931 - Use local cell editor in table
 *    Vincent Lorenzo (CEA-LIST) - Bug 352580 - Facets are loaded to late
 *    Gregoire Dupe (Mia-Software) - Bug 354224 - mutually exclusive Facets
 *    Nicolas Guyomar (Mia-Software) - Bug 342154 - EMF Facet API should be covered with Unit tests
 *    Nicolas Bros (Mia-Software) - Bug 357593 - [Table] Table take focus after each refresh
 *    Grgoire Dup (Mia-Software) - Bug 356795 - [Unit Test Failure][0.2/4.2][0.2/3.8] org.eclipse.emf.facet.widgets.nattable.tests.Bug344413Test.testBug344413
 *    Gregoire Dupe (Mia-Software) - Bug 366804 - [Restructuring] Table widget upgrade
 *    Gregoire Dupe (Mia-Software) - Bug 367613 - Table widget refactoring
 *    Gregoire Dupe (Mia-Software) - Bug 367700 - [Unit Test Failure][0.2/3.8] org.eclipse.emf.facet.widgets.table.tests.internal.v0_2.swtbot.NatTableUITests.testOpenLoadCustomizationDialog
 *    Gregoire Dupe (Mia-Software) - Bug 369987 - [Restructuring][Table] Switch to the new customization and facet framework
 *    Gregoire Dupe (Mia-Software) - Bug 371367 - Hierarchical FacetSets 
 *    Nicolas Bros (Mia-Software) - Bug 370806 - [table] rewrite the "allowed contents" query selection dialog for v0.2
 *    Vincent Lorenzo (CEA-LIST) -  Bug 372644 - Create Customizable tooltips for the TreeViewer using a CustomizableLabelProvider
 *    Gregoire Dupe (Mia-Software) - Bug 373078 - API Cleaning
 *    Gregoire Dupe (Mia-Software) - Bug 374903 - [Table] ITableWidget.setLoadedFacetSets
 *    Gregoire Dupe (Mia-Software) - Bug 375087 - [Table] ITableWidget.addColumn(List<ETypedElement>, List<FacetSet>)
 *    Gregoire Dupe (Mia-Software) - Bug 372626 - Aggregates
 *    Gregoire Dupe (Mia-Software) - Bug 376158 - [Table] Unexpected columns when customizations are loaded
 *    Nicolas Bros (Mia-Software) - Bug 377614 - [Table] getSelection should return all the elements underlying the selected cells
 *    Nicolas Bros (Mia-Software) - Bug 377754 - [Table] right click deselects elements
 *    Nicolas Bros (Mia-Software) - Bug 377866 - selection customization
 *    Nicolas Bros (Mia-Software) - Bug 377773 - [Table] sorting doesn't work on all columns
 *    Nicolas Bros (Mia-Software) - Bug 378475 - unit test failures after table refactoring
 *    Nicolas Bros (Mia-Software) - Bug 378649 - [Table] Errors with non-applicable features
 *    Gregoire Dupe (Mia-Software) - Bug 378701 - [Unit Test Failure] org.eclipse.emf.facet.widgets.table.tests.internal.v0_2.notuithread.Bug354224Test*
 *    Olivier Remaud (Soft-Maint) - Bug 378499 - optimizing table opening
 *    Nicolas Bros (Mia-Software) - Bug 379436 - [Table] read-only resources should not be editable
 *    Nicolas Bros (Mia-Software) - Bug 379535 - [Table] remove "open table" action
 *    Nicolas Bros (Mia-Software) - Bug 379542 - [Table] cannot add non-Facet columns
 *    Gregoire Dupe (Mia-Software) - Bug 380126 - [Table] Row sort too long
 *    Nicolas Bros (Mia-Software) - Bug 380236 - [Table] table is not properly disposed
 *    Gregoire Dupe (Mia-Software) - Bug 387008 - [Table] Papyrus needs methods provided by TableInstanceCommandFactory
 *    Gregoire Dupe (Mia-Software) - Bug 387005 - [Table] ITableWidget doesn't provide a method to get the selected Rows
 *    Gregoire Dupe (Mia-Software) - Bug 387024 - [Table] Column order is not correctly "saved"
 *    Gregoire Dupe (Mia-Software) - Bug 388422 - [Table] Queries for InstanciationMethod needs to have 2 parameters
 *    Gregoire Dupe (Mia-Software) - Bug 377870 - [EFacet] ETypedElementDialog doesn't show all available ETypedElement (library example problem?)
 *    Gregoire Dupe (Mia-Software) - Bug 391442 - Select ETypedElement Dialog doesn't used the subpackages (subEFacetSet)
 *******************************************************************************/

package org.eclipse.emf.facet.widgets.table.ui.internal;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.command.BasicCommandStack;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CompoundCommand;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.ETypedElement;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.emf.edit.domain.AdapterFactoryEditingDomain;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.domain.IEditingDomainProvider;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.facet.custom.core.ICustomizationCatalogManagerFactory;
import org.eclipse.emf.facet.custom.core.ICustomizationManager;
import org.eclipse.emf.facet.custom.core.ICustomizationManagerFactory;
import org.eclipse.emf.facet.custom.core.exception.CustomizationException;
import org.eclipse.emf.facet.custom.metamodel.v0_2_0.custom.Customization;
import org.eclipse.emf.facet.custom.ui.IContentPropertiesHandler;
import org.eclipse.emf.facet.custom.ui.IContentPropertiesHandlerFactory;
import org.eclipse.emf.facet.custom.ui.ICustomizedLabelProvider;
import org.eclipse.emf.facet.custom.ui.ICustomizedLabelProviderFactory;
import org.eclipse.emf.facet.custom.ui.ISelectionPropertiesHandler;
import org.eclipse.emf.facet.custom.ui.ISelectionPropertiesHandlerFactory;
import org.eclipse.emf.facet.custom.ui.internal.exported.dialog.ILoadCustomizationsDialog;
import org.eclipse.emf.facet.custom.ui.internal.exported.dialog.ILoadCustomizationsDialogFactory;
import org.eclipse.emf.facet.efacet.core.FacetUtils;
import org.eclipse.emf.facet.efacet.core.IFacetManager;
import org.eclipse.emf.facet.efacet.core.IFacetManagerFactory;
import org.eclipse.emf.facet.efacet.core.IFacetSetCatalogManager;
import org.eclipse.emf.facet.efacet.core.IFacetSetCatalogManagerFactory;
import org.eclipse.emf.facet.efacet.core.exception.DerivedTypedElementException;
import org.eclipse.emf.facet.efacet.core.exception.FacetManagerException;
import org.eclipse.emf.facet.efacet.core.internal.exported.IResolverManager;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.DerivedTypedElement;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.Facet;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.FacetElement;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.FacetSet;
import org.eclipse.emf.facet.efacet.ui.internal.exported.dialog.IETypedElementSelectionDialog;
import org.eclipse.emf.facet.efacet.ui.internal.exported.dialog.IETypedElementSelectionDialogFactory;
import org.eclipse.emf.facet.efacet.ui.internal.exported.dialog.IFacetSetSelectionDialog;
import org.eclipse.emf.facet.efacet.ui.internal.exported.dialog.IFacetSetSelectionDialogFactory;
import org.eclipse.emf.facet.util.core.DebugUtils;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.emf.facet.util.emf.core.CommandFactory;
import org.eclipse.emf.facet.util.emf.core.EmfDebugUtils;
import org.eclipse.emf.facet.util.emf.core.ModelUtils;
import org.eclipse.emf.facet.util.emf.core.command.ILockableUndoCommand;
import org.eclipse.emf.facet.util.emf.core.internal.EMFUtils;
import org.eclipse.emf.facet.util.ui.internal.exported.dialog.IDialogCallback;
import org.eclipse.emf.facet.util.ui.internal.exported.dialog.IDialogCallbackWithPreCommit;
import org.eclipse.emf.facet.util.ui.internal.exported.dialog.IQuestionDialog;
import org.eclipse.emf.facet.util.ui.internal.exported.dialog.IQuestionDialogFactory;
import org.eclipse.emf.facet.widgets.celleditors.ICellEditorsRegistry;
import org.eclipse.emf.facet.widgets.celleditors.ICommandFactoriesRegistry;
import org.eclipse.emf.facet.widgets.celleditors.ICommandFactory;
import org.eclipse.emf.facet.widgets.celleditors.IModelCellEditorContainer;
import org.eclipse.emf.facet.widgets.celleditors.modelCellEditor.AbstractModelCellEditor;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.Column;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.FeatureColumn;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.Row;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.SourceColumn;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.Table;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.TableFactory;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.TablePackage;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.tableconfiguration.InstantiationMethod;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.tableconfiguration.TableConfiguration;
import org.eclipse.emf.facet.widgets.table.ui.command.ITableCommandFactory;
import org.eclipse.emf.facet.widgets.table.ui.internal.comparator.ColumnComparator;
import org.eclipse.emf.facet.widgets.table.ui.internal.emf.listener.RowDeleteCommandStackListener;
import org.eclipse.emf.facet.widgets.table.ui.internal.emf.listener.RowDeleteTriggerListener;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.ColumnSortDirection;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.IGridElement;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.IPositionCoordinate;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.ITableWidgetInternal;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.ITableWidgetView;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.ITableWidgetViewFactory;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.TableWidgetInternalUtils;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.TableWidgetUtils;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.dialog.IAllowedContentsETypedElementSelectionDialog;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.dialog.IAllowedContentsETypedElementSelectionDialogCallback;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.dialog.IColumnsToHideDialog;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.dialog.IMustRemoveNonConformingElementsDialog;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.exception.TableWidgetRuntimeException;
import org.eclipse.emf.facet.widgets.table.ui.internal.jface.dialog.AllowedContentsETypedElementSelectionDialog;
import org.eclipse.emf.facet.widgets.table.ui.internal.jface.dialog.ColumnsToHideDialog;
import org.eclipse.emf.facet.widgets.table.ui.internal.jface.dialog.MustRemoveNonConformingElementsPromptDialog;
import org.eclipse.emf.facet.widgets.table.ui.internal.jface.dialog.SynchronizedMustRemoveNonConformingElementsPromptDialog;
import org.eclipse.emf.facet.widgets.table.ui.internal.jface.dialog.SynchronizedPresentabilityETypedElementSelectionDialog;
import org.eclipse.emf.facet.widgets.table.ui.internal.preferences.PreferenceConstants;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionDelegate;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.ListDialog;
import org.eclipse.ui.dialogs.SaveAsDialog;

public class TableWidgetController implements DisposeListener, ITableWidgetInternal {

	//////////////// Debugging constants (begin)//////////////////
	
	protected static final boolean DEBUG_EVENTS = DebugUtils
			.getDebugStatus(Activator.getDefault(), "events"); //$NON-NLS-1$
	protected static final boolean DEBUG_RESOURCE_LISTENER = DebugUtils
			.getDebugStatus(Activator.getDefault(), "resource_listener"); //$NON-NLS-1$
	protected static final boolean DEBUG_REMOVE_USELESS_ROWS_AND_COLUMNS = DebugUtils
			.getDebugStatus(Activator.getDefault(),
					"removeUselessRowsAndColumns"); //$NON-NLS-1$
	private static final boolean DEBUG_SET_FACET = DebugUtils.getDebugStatus(
			Activator.getDefault(), "setFacet"); //$NON-NLS-1$
	protected static final boolean DEBUG_SELECTION = DebugUtils
			.getDebugStatus(Activator.getDefault(), "selection"); //$NON-NLS-1$
	protected static final boolean DEBUG_SORT = DebugUtils.getDebugStatus(
			Activator.getDefault(), "sort"); //$NON-NLS-1$

	////////////////Debugging constants (end) //////////////////
	
	private static final int REFRESH_JOB_DELAY = 100;
	protected static final String TMP_FILE_NAME = Activator.getDefault().getBundle().getSymbolicName() + ".tmp.table"; //$NON-NLS-1$
	public static final File DEFAULT_RESOURCE_FILE = new File(Platform.getStateLocation(Activator.getDefault()
			.getBundle()).toOSString(), TableWidgetController.TMP_FILE_NAME);

	/** The Input */
	private final Table tableInstance;
	/** The label provider */
	private ICustomizedLabelProvider tableLabelProvider;

	private final Collection<ISelectionChangedListener> fSelectionChangedListeners = new ArrayList<ISelectionChangedListener>();

	/** To remove the listeners when disposed */
	private final List<Resource> listenedResources = new ArrayList<Resource>();
	private ICustomizationManager customizationEngine;
	/** All the metaclasses of the elements */
	private Set<EClass> metaclasses;
	private Resource resource;
	private final MenuManager menuMgr;
	/** Whether to enable editability of the table's data */
	private boolean dataEditable;
	private final EditingDomain editingDomain;
	/** The command factory used to create every command for the widget*/
	private final transient ICommandFactory commandFactory;
	private transient List<InstantiationMethod> launchedInstMethHistory;
	private boolean listenReorderEvent = true;
	private final ITableWidgetView view;
	private final IFacetManager facetManager;
	private Action loadCustomizationsAction;
	private Action loadFacetsAction;
	private final List<IActionDelegate> actionList = new ArrayList<IActionDelegate>();
	private IContentPropertiesHandler contentPropertiesHandler;
	private ISelectionPropertiesHandler selectionPropertiesHandler;
	private final TableCommandFactory tableCommandFactory;
	private boolean disposed;
	
	/** Allows optimized refreshes of the table */
	// TODO : should be externalized
	private final Job refreshJob = new Job("Refresh EMF Facet Table") { //$NON-NLS-1$
		@Override
		protected IStatus run(final IProgressMonitor monitor) {
			Display.getDefault().syncExec(new Runnable() {
				public void run() {
					DebugUtils.debug(TableWidgetController.DEBUG_EVENTS, "Refresh job execution."); //$NON-NLS-1$
					try {
						if (!isDisposed()) {
							TableWidgetController.this.reset();
						}
					} catch (CustomizationException e) {
						throw new TableWidgetRuntimeException(e);
					}
				}
			});
			return Status.OK_STATUS;
		}
	};
	
	///////////////////  Listeners (begin) ////////////////////////

	private final Adapter modelChangeAdapter = new AdapterImpl() {
		@Override
		public void notifyChanged(final Notification msg) {
			int eventType = msg.getEventType();
			final boolean redrawNeeded = eventType != Notification.REMOVING_ADAPTER && eventType != Notification.RESOLVE;
			if (TableWidgetController.DEBUG_RESOURCE_LISTENER) {
				DebugUtils.debug("notification=" + msg); //$NON-NLS-1$
				DebugUtils.debug("redrawNeeded=" + redrawNeeded); //$NON-NLS-1$
			}
			if (redrawNeeded) {
				// redraw table when model changes
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						if (TableWidgetController.DEBUG_RESOURCE_LISTENER) {
							DebugUtils.debug("Call of the refresh=" + msg); //$NON-NLS-1$
						}
						refreshDelayed();
					}
				});
			}
		}
	};

	private final Adapter tableInstanceAdapter = new AdapterImpl() {
		@Override
		public synchronized void notifyChanged(final Notification msg) {
			int eventType = msg.getEventType();
			if (eventType != Notification.REMOVING_ADAPTER && eventType != Notification.RESOLVE) {
				DebugUtils.debug(TableWidgetController.DEBUG_EVENTS, "tableInstanceAdapter called"); //$NON-NLS-1$
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						DebugUtils.debug(TableWidgetController.DEBUG_EVENTS, "tableInstanceAdapter asyncExec"); //$NON-NLS-1$
						refreshDelayed();
					}
				});
			}
		}
	};
	
	///////////////////  Listeners (end) ////////////////////////
	
	/**
	 *
	 * @param parent
	 *            This composite parent
	 * @param editingDomainProvider
	 *            An editing provider
	 * @param tableInstanceParam
	 *            The {@link Table} which contains all the needed
	 *            information to open a table
	 * @param menuMgr
	 * @throws InvalidCustomizationException 
	 * @throws CoreException
	 */
	public TableWidgetController(final Composite parent,
			final IEditingDomainProvider editingDomainProvider,
			final Table tableInstanceParam, final MenuManager menuMgr,
			final ITableWidgetViewFactory viewFactory)
			throws CustomizationException {
		this.view = viewFactory.createTableView(parent, this);
		this.menuMgr = menuMgr;
		final ComposedAdapterFactory adapterFactory = new ComposedAdapterFactory(
				ComposedAdapterFactory.Descriptor.Registry.INSTANCE);
		final BasicCommandStack commandStack = new BasicCommandStack();
		if (editingDomainProvider == null || editingDomainProvider.getEditingDomain() == null) {
			this.editingDomain = new AdapterFactoryEditingDomain(adapterFactory, commandStack,
					new HashMap<Resource, Boolean>());
			this.dataEditable = false;
		} else {
			this.editingDomain = editingDomainProvider.getEditingDomain();
			this.dataEditable = true;
		}
		if (tableInstanceParam == null) {
			this.tableInstance = TableFactory.eINSTANCE.createTable();
			this.resource = createDefaultResource();
			this.resource.getContents().add(this.tableInstance);
		} else {
			if (tableInstanceParam.eResource() != null) {
				this.tableInstance = tableInstanceParam;
				this.resource = tableInstanceParam.eResource();
			} else {
				this.tableInstance = tableInstanceParam;
				this.resource = createDefaultResource();
				Command setResourceContentCommand = CommandFactory.createSetResourceContentsCommand(this.resource, this.tableInstance);
				if (setResourceContentCommand instanceof ILockableUndoCommand) {
					ILockableUndoCommand lockableUndoCommand = (ILockableUndoCommand) setResourceContentCommand;
					lockableUndoCommand.enableCanUndo(false);
				}
				this.editingDomain.getCommandStack().execute(setResourceContentCommand);
				this.resource.getContents().add(this.tableInstance);
			}
		}
		initLaunchedHistory();
		this.commandFactory = ICommandFactoriesRegistry.INSTANCE
				.getCommandFactoryFor(this.editingDomain);
		this.facetManager = IFacetManagerFactory.DEFAULT
				.getOrCreateFacetManager(this.resource.getResourceSet());
		this.tableCommandFactory = new TableCommandFactory(this.tableInstance,
				this.editingDomain, this.commandFactory, this.facetManager);
		this.customizationEngine = ICustomizationManagerFactory.DEFAULT
				.createICustomizationManager(this.facetManager);
		this.contentPropertiesHandler = IContentPropertiesHandlerFactory.DEFAULT
				.createIContentPropertiesHandler(this.customizationEngine);
		this.selectionPropertiesHandler = ISelectionPropertiesHandlerFactory.DEFAULT
				.createSelectionPropertiesHandler(this.customizationEngine);
		init();
	}


	private final Resource createDefaultResource() {
		return this.editingDomain.getResourceSet().createResource(
				URI.createFileURI(TableWidgetController.DEFAULT_RESOURCE_FILE.getPath()));
	}

	private void init() throws CustomizationException {
		DebugUtils.debug(TableWidgetController.DEBUG_EVENTS);
		reset();
		this.view.asComposite().setLayout(new FillLayout());
		this.view.asComposite().addDisposeListener(this);
		if (this.tableInstance != null) {
			this.tableInstance.eAdapters().add(this.tableInstanceAdapter);
			this.resource.setTrackingModification(true);
			this.resource.eAdapters().add(this.tableInstanceAdapter);
		}
		//Depending of the editing domain nature, we use a differente model listening strategy.
		if (this.editingDomain instanceof TransactionalEditingDomain) {
			TransactionalEditingDomain transactionalEditingDomain = (TransactionalEditingDomain) this.editingDomain;
			transactionalEditingDomain.addResourceSetListener(new RowDeleteTriggerListener(this.tableInstance));
		} else {
			this.editingDomain.getCommandStack().addCommandStackListener(new RowDeleteCommandStackListener(this.tableInstance, this.editingDomain));
		}
	}

	/**
	 * This method should not be executed twice at the same time
	 * @throws InvalidCustomizationException 
	 */
	private final synchronized void reset() throws CustomizationException {
		DebugUtils.debug(TableWidgetController.DEBUG_EVENTS, "Start"); //$NON-NLS-1$
		updateMetaClassList();
		this.tableLabelProvider = ICustomizedLabelProviderFactory.DEFAULT
				.createCustomizedLabelProvider(this.customizationEngine);
		this.facetManager.getManagedFacetSets().clear();
		this.facetManager.getManagedFacetSets().addAll(
				this.tableInstance
				.getFacetSets());
		List<FacetSet> tmp = new ArrayList<FacetSet>(
				this.facetManager.getManagedFacetSets());
		if (this.getTable().getCustomizations() != null) {
			loadCustomizationsInCustomEngine(this.getTable().getCustomizations());
		}
		// FIXME removeUselessRowsAndColumns() should be called there. It should
		// called only at the table opennning.
		removeUselessRowsAndColumns();
		// FIXME (end)
		fullRedraw();
		DebugUtils.debug(TableWidgetController.DEBUG_EVENTS, "End"); //$NON-NLS-1$
	}
	
	/**
	 * This method maintain the metaclass list up to date, in order not to
	 * evaluate its value every time we need it
	 */
	private final void updateMetaClassList() {
		this.metaclasses = new HashSet<EClass>();
		for (EObject eObject : TableWidgetUtils.getElements(this.tableInstance)) {
			this.metaclasses.add(eObject.eClass());
		}
	}

	private final void initLaunchedHistory() {
		this.launchedInstMethHistory = new ArrayList<InstantiationMethod>();
		if (getTableConfiguration() != null
				&& getTableConfiguration().getInstantiationMethod() != null) {
			for (InstantiationMethod meth : getTableConfiguration().getInstantiationMethod()) {
				this.launchedInstMethHistory.add(meth);
			}
		}
	}

	public final void addChangeListener(final List<Row> elementsValue) {
		Set<Resource> resources = new HashSet<Resource>();
		for (Row row : elementsValue) {
			EObject eObject = row.getElement();
			if (eObject != null) {
				resources.add(eObject.eResource());
			}
		}
		for (Resource referedResource : resources) {
			if (referedResource != null) {
				if (!this.listenedResources.contains(referedResource)) {
					this.listenedResources.add(referedResource);
					referedResource.setTrackingModification(true);
					if (!referedResource.eAdapters().contains(this.modelChangeAdapter)) {
						referedResource.eAdapters().add(this.modelChangeAdapter);
					}
				}
			}
		}
	}

	public final void widgetDisposed(final DisposeEvent event) {
		this.disposed = true;
		this.fSelectionChangedListeners.clear();
		for (Resource resourceElt : this.listenedResources) {
			resourceElt.eAdapters().remove(this.modelChangeAdapter);
		}
		if (this.tableInstance != null) {
			this.tableInstance.eAdapters().remove(this.tableInstanceAdapter);
			this.resource.eAdapters().remove(this.tableInstanceAdapter);
		}
	}

	private static Object getStructuralFeatureValue(final EObject eObject,
			final EStructuralFeature structuralFeature) {
		return eObject.eGet(structuralFeature);
	}

	private static boolean hasStructuralFeature(final EObject eObject,
			final EStructuralFeature structuralFeature) {
		final EClass eClass = eObject.eClass();
		return eClass.getEAllStructuralFeatures().contains(structuralFeature);
	}

	public final void tableSelectionChanged() {
		for (ISelectionChangedListener listener : new ArrayList<ISelectionChangedListener>(
				this.fSelectionChangedListeners)) {
			listener.selectionChanged(new SelectionChangedEvent(this, getSelection()));
		}
	}

	public final void setSelection(final ISelection selection) {
		// The selection can not be set through the API
	}

	/**
	 * Add a {@link ISelectionChangedListener} to the widget.
	 */
	public final void addSelectionChangedListener(final ISelectionChangedListener listener) {
		if (!this.fSelectionChangedListeners.contains(listener)) {
			this.fSelectionChangedListeners.add(listener);
		}
	}

	/**
	 * Remove a {@link ISelectionChangedListener} from the widget.
	 */
	public final void removeSelectionChangedListener(final ISelectionChangedListener listener) {
		this.fSelectionChangedListeners.remove(listener);
	}

	/**
	 *
	 * @param cellEditor
	 *            a cell editor
	 * @return <code>true</code> if the editor is registered in the
	 *         {@link ICellEditorsRegistry} (that is to say that this editor is
	 *         declared in the plugin.xml)
	 */
	private static boolean isInCellEditorRegistry(final AbstractModelCellEditor cellEditor) {
		List<IModelCellEditorContainer<? extends AbstractModelCellEditor>> allCellEditorContainers = ICellEditorsRegistry.INSTANCE
				.getAllCellEditors();
		for (IModelCellEditorContainer<? extends AbstractModelCellEditor> currentEditorContainer : allCellEditorContainers) {
			if (cellEditor.getCellId().equals(
					currentEditorContainer.getModelCellEditor().getCellId())) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Sort columns by type of link (attributes first, then references, then
	 * derived references, ...)
	 */
	public final void sortColumnsByType() {
		try {
			this.view.asComposite().setRedraw(false);
			this.listenReorderEvent = false;

			int[] order = new int[this.tableInstance.getColumns().size()];
			int i = 0;
			for (int j = 0; j < this.tableInstance.getColumns().size(); j++) {
				Column columnDescription = this.tableInstance.getColumns().get(j);
				if (columnDescription instanceof SourceColumn) {
					order[i++] = j;
				}
			}

			for (int j = 0; j < this.tableInstance.getColumns().size(); j++) {
				Column columnDescription = this.tableInstance.getColumns().get(j);
				if (columnDescription instanceof FeatureColumn) {
					FeatureColumn featureColumn = (FeatureColumn) columnDescription;
					if (featureColumn.getFeature() instanceof EAttribute) {
						order[i++] = j;
					}
				}
			}

			// positions of references
			List<Integer> links = new ArrayList<Integer>();
			for (int j = 0; j < this.tableInstance.getColumns().size(); j++) {
				Column columnDescription = this.tableInstance.getColumns().get(j);
				if (columnDescription instanceof FeatureColumn) {
					FeatureColumn featureColumn = (FeatureColumn) columnDescription;
					if (featureColumn.getFeature() instanceof EReference) {
						links.add(Integer.valueOf(j));
					}
				}
			}

			for (Integer linkPos : links) {
				order[i++] = linkPos.intValue();
			}

			CompoundCommand compoundCommand = new CompoundCommand();
			for (int index = 0; index < order.length; index++) {
				int pos = order[index];
				EObject element = TableWidgetController.this.tableInstance.getColumns().get(pos);
				Command command = this.commandFactory.createMoveCommand(
						TableWidgetController.this.editingDomain, TableWidgetController.this.tableInstance,
						TablePackage.eINSTANCE.getTable_Columns(), element,
						index);
				compoundCommand.append(command);
			}
			TableWidgetController.this.editingDomain.getCommandStack().execute(compoundCommand);
		} catch (Exception e) {
			Logger.logError(e, "Error sorting columns by type", //$NON-NLS-1$
					Activator.getDefault());
		} finally {
			this.view.asComposite().setRedraw(true);
			this.listenReorderEvent = true;
		}
	}

	/** Hide columns which don't contain any elements */
	public final void setHideEmptyColumns(final boolean hideEmptyColumns) {
		setWithCommand(TableWidgetController.this.tableInstance,
				TablePackage.eINSTANCE.getTable_HideEmptyColumns(),
				Boolean.valueOf(hideEmptyColumns));
	}

	public final boolean isEmptyColumn(final Column column) {
		boolean result = false;
		if (column instanceof FeatureColumn) {
			FeatureColumn featureColumn = (FeatureColumn) column;
			if (featureColumn.getFeature() instanceof EReference) {
				result = isEmpty((EReference) featureColumn.getFeature());
			}
		}
		return result;
	}

	/**
	 * Determines the status (hidden or visible) for a column
	 *
	 * @param featureColumn
	 *            a Column to show or hide
	 * @param featureOwners
	 *            elements which own the feature
	 * @return the status of the column hidden or visible, using the file
	 *         uiCustom
	 * @throws CustomizationException 
	 */
	private boolean getHiddenColumnStatus(final FeatureColumn c, final Set<EClass> featureOwners) throws CustomizationException {
		return this.customizationEngine.getCustomValueOf(c.getFeature(),
				this.contentPropertiesHandler.getIsVisible(), Boolean.class)
				.booleanValue();
	}

	private boolean getHiddenAttributeColumnStatus(final FeatureColumn c,
			final Set<EClass> featureOwners) throws CustomizationException {
		return this.customizationEngine.getCustomValueOf(c.getFeature(),
				this.contentPropertiesHandler.getIsVisible(), Boolean.class)
				.booleanValue();
	}

	private boolean getHiddenReferenceColumnStatus(final FeatureColumn c,
			final Set<EClass> featureOwner) throws CustomizationException {
		return this.customizationEngine.getCustomValueOf(c.getFeature(),
				this.contentPropertiesHandler.getIsVisible(), Boolean.class)
				.booleanValue();
	}

	/**
	 * Open a selection popup to select which columns to show/hide.
	 */
	public final void selectColumnsToHide() {
		openColumnsToHideDialog();
	}

	public final IColumnsToHideDialog openColumnsToHideDialog() {
		ILabelProvider labelProvider = new LabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof Column) {
					Column column = (Column) element;
					return TableWidgetInternalUtils.getColumnName(column);
				}
				return null;
			}
		};

		final boolean localCustomizatinMustBeTheTop = PreferenceConstants.isMustPutOnTheTopTheLocalCustom();
		final boolean askToPutOnTheTopTheLocalCustomization = PreferenceConstants.isAskToPutOnTheTopTheLocalCustomization();
		final ColumnsToHideDialog dialog = new ColumnsToHideDialog(this.view.asComposite().getShell(),
				getTable().getColumns(), labelProvider,
				isOneOfTheLocalCustomizationsNotOnTheTop()
						&& askToPutOnTheTopTheLocalCustomization,
				localCustomizatinMustBeTheTop) {
			@Override
			protected void okPressed() {
				super.okPressed();
				if (getResult() != null) {
					TableWidgetController.this.showHideColumns(getSelectedColumns(), isPutOnTheTop() && isOneOfTheLocalCustomizationsNotOnTheTop());
					TableWidgetController.setMustAskThUserToPutOnTheTopTheLocalCustomizationNextTime(
							isMustAskTheUserNextTime(), isPutOnTheTop());
				}
			}
			@Override
			public List<Column> getVisibleColumns(final boolean putOnTheTop) {
				return TableWidgetController.this.getVisibleColumns(putOnTheTop);
			}
		};
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				dialog.open();
			}
		});
		return dialog;
	}

	private static void setMustAskThUserToPutOnTheTopTheLocalCustomizationNextTime(
			final boolean mustAskTheUserNextTime, final boolean putOnTheTop) {
		PreferenceConstants.setAskToPutOnTheTopTheLocalCustomization(mustAskTheUserNextTime);
		if (!mustAskTheUserNextTime) { // we doesn't display the message the next time
			PreferenceConstants.setMustPutOnTheTopTheLocalCustom(putOnTheTop);
		}
	}

	public final void showHideColumns(final List<Column> columnToShow, final boolean putOnTheTop) {
		//we build the list of the columns to show/hide
		List<Column> columnsToShow = new ArrayList<Column>();
		List<Column> columnsToHide = new ArrayList<Column>();
		EList<Column> columns = getTable().getColumns();
		for (Column c : columns) {
			boolean mustBeHidden = !columnToShow.contains(c);
			if (mustBeHidden == getVisibleColumns(putOnTheTop).contains(c)) {
				if (mustBeHidden) {
					columnsToHide.add(c);
				} else {
					columnsToShow.add(c);
				}
			}
		}
		showHideColumns(columnsToShow, columnsToHide, putOnTheTop);
	}

	public final boolean isOneOfTheLocalCustomizationsNotOnTheTop() {
		List<Customization> view = this.tableInstance.getCustomizations();
		List<Customization> locals = getLocalCustomizations();
		if (view.size() < locals.size()) {
			throw new TableWidgetRuntimeException("Some local customization files are not loaded"); //$NON-NLS-1$
		}
		for (int i = 0; i < locals.size(); i++) {
				if (!locals.contains(view.get(i))) {
					// one of the local custom is not on the top of the
					// customization
					// engine
					return true;
				}
		}
		return false;
	}

	public final List<Customization> getLocalCustomizations() {
		return this.tableInstance.getLocalCustomizations();
	}

	private final List<Column> getNotHiddenColumns(final int setAtTheTop,
			final boolean atTheTop) throws CustomizationException {
		List<Column> notHiddenColumns = new ArrayList<Column>();
		// we always show the state of the localCustom
		if (atTheTop) { // we show the current state of the localCustom
			notHiddenColumns = getVisibleColumnsUsingTheLocalCustomizations(
					this.tableInstance, false);
		} else if (setAtTheTop == IDialogConstants.NO_ID) {
			// we show the current state of the local custom
			notHiddenColumns = getVisibleColumnsUsingTheLocalCustomizations(
					this.tableInstance, false);
		} else if (setAtTheTop == IDialogConstants.YES_ID) {
			// we show the future state of the local custom
			notHiddenColumns = getVisibleColumnsUsingTheLocalCustomizations(
					this.tableInstance, true);
		}
		return notHiddenColumns;
	}

	
	private void showHideColumns(final List<Column> columnsToShow,
			final List<Column> columnsToHide, final boolean putOnTheTop) {
		Command showHideColumnsCommand = TableInstanceCommandFactory.createShowHideColumnCommand(
				this, columnsToShow, columnsToHide, putOnTheTop);
		executeCommand(showHideColumnsCommand);
	}

	public void putLocalCustomizationOnTheTop() {
		this.editingDomain.getCommandStack().execute(
				TableInstanceCommandFactory.createPutLocalCustomizationOnTheTopCommand(this));
	}

	private final List<Column> getVisibleColumnsUsingCustomizationEngine(final Table table) throws CustomizationException {
		List<Column> visibleColumns = new ArrayList<Column>();
		List<Column> columns = table.getColumns();
		for (int i = 0; i < columns.size(); i++) {
			Column column = columns.get(i);
			if (column instanceof FeatureColumn) {
				FeatureColumn featureColumn = (FeatureColumn) column;
				if (featureColumn.getFeature() instanceof EStructuralFeature) {
					EStructuralFeature structuralFeature = (EStructuralFeature) featureColumn;
					Set<EClass> owners = EMFUtils.getAllFeatureOwner(
							TableWidgetUtils.getElements(table),
							structuralFeature);
					if (!getHiddenColumnStatus(featureColumn, owners)) {
						visibleColumns.add(column);
					}
				}
			} else if (column instanceof SourceColumn) {
				SourceColumn sourceColumn = (SourceColumn) column;
				if (!sourceColumn.isIsHidden()) {
					visibleColumns.add(column);
				}
			} else {
				throw new IllegalStateException("Unexpected column type."); //$NON-NLS-1$
			}
		}
		return visibleColumns;
	}

	/**
	 *
	 * @param table
	 * a table
	 * @param satAtTheTop
	 * if(<code>true</code>, we calculate the future state of the localCustom
	 * @return
	 *  all the columns which are visible, using the local customization file
	 * @throws InvalidCustomizationException 
	 * @throws CustomizationException 
	 */
	public final List<Column> getVisibleColumnsUsingTheLocalCustomizations(
			final Table table, final boolean setToTheTop)
			throws CustomizationException {
		// we build a local customization engine
		ICustomizationManager localEngine = ICustomizationManagerFactory.DEFAULT
				.getOrCreateICustomizationManager(this.resource.getResourceSet());
		if (setToTheTop) {
			for (Customization customization : this.tableInstance.getCustomizations()) {
				localEngine.getManagedCustomizations().add(0, customization);
			}
		} else {
			for (Customization customization : this.tableInstance.getLocalCustomizations()) {
				localEngine.getManagedCustomizations().add(0, customization);
			}
		}
		List<Column> notHiddenColumns = new ArrayList<Column>();
		for (Column c : table.getColumns()) {
			boolean isVisible = true;
			if (c instanceof FeatureColumn) {
				final FeatureColumn featureColumn = (FeatureColumn) c;
				final ETypedElement typedElement = featureColumn.getFeature();
				isVisible = localEngine.getCustomValueOf(typedElement,
						this.contentPropertiesHandler.getIsVisible(),
						Boolean.class).booleanValue();
			} else if (c instanceof SourceColumn) {
				final SourceColumn sourceColumn = (SourceColumn) c;
				isVisible = !sourceColumn.isIsHidden();
			} else {
				throw new IllegalStateException("Unknown column type"); //$NON-NLS-1$
			}
			if (isVisible) {
				notHiddenColumns.add(c);
			}
		}
		return notHiddenColumns;
	}

	@Deprecated
	public final void hideColumns() {
		selectColumnsToHide();
	}

	/** Whether no element has this reference set */
	private boolean isEmpty(final EStructuralFeature reference) {
		for (EObject eObject : TableWidgetUtils.getElements(this.tableInstance)) {
			if (hasStructuralFeature(eObject, reference)) {
				final Object value = getStructuralFeatureValue(eObject, reference);
				if (reference.isMany() && value instanceof List<?>) {
					final List<?> list = (List<?>) value;
					if (!list.isEmpty()) {
						return false;
					}
				} else {
					if (value != null) {
						return false;
					}
				}
			}
		}
		return true;
	}

	/**
	 * Whether to show columns which are specific to a few elements (false), or
	 * only columns which are common to all the elements (true)
	 */
	public final void setOnlyShowCommonColumns(final boolean onlyShowCommonColumns) {
		setWithCommand(TableWidgetController.this.tableInstance,
				TablePackage.eINSTANCE.getTable_OnlyShowCommonColumns(),
				onlyShowCommonColumns);
	}

	/**
	 * @return true if there is at least a column selected
	 */
	public final boolean isColumnSelected() {
		List<Column> columns = getSelectedColumns();
		if (columns != null) {
			return columns.size() > 0;
		}
		return false;
	}

	/**
	 * remove the selected line from the table
	 */
	public final void removeLine() {
		//TODO This method has to be deprecated and replaced by removeLine2
		try {
			removeLine2();
		} catch (CoreException e) {
			throw new TableWidgetRuntimeException(e);
		}
	}

	private Collection<Row> getSelectedRow() {
		final List<Integer> selectedRowIndex = getSelecedRowIndexes();
		final List<Row> selectedEObjects = new ArrayList<Row>();
		for (Integer index : selectedRowIndex) {
			selectedEObjects.add(this.tableInstance.getRows().get(
					index.intValue()));
		}
		return selectedEObjects;
	}

	public List<EObject> getSelectedRowEObjects() {
		final List<Integer> selectedRowIndex = getSelecedRowIndexes();
		final List<EObject> selectedEObjects = new ArrayList<EObject>();
		for (Integer index : selectedRowIndex) {
			selectedEObjects.add(this.tableInstance.getRows().get(
					index.intValue()).getElement());
		}
		return selectedEObjects;
	}

	private List<Integer> getSelecedRowIndexes() {
		final List<Integer> selectedRowIndex = new ArrayList<Integer>();
		final IPositionCoordinate[] selectedCells = this.getSelectedCells2();
		for (IPositionCoordinate positionCoordinate : selectedCells) {
			if (!selectedRowIndex.contains(positionCoordinate.getRowPosition())) {
				selectedRowIndex.add(positionCoordinate.getRowPosition());
			}
		}
		return selectedRowIndex;
	}
	
	/**
	 * remove the selected line from the table
	 * @throws CoreException may be caused by query exception in the facet framework
	 */
	private final void removeLine2() throws CoreException {
		final Collection<Row> rows = getSelectedRow();
		final Command cmd = this.tableCommandFactory.createRemoveRowsCommand(rows);
		executeCommand(cmd);
	}

	/**
	 * Delete the selected elements from their containing resource
	 */
	public final void deleteSelection() {
		try {
			deleteSelectedElements();
		} catch (CoreException e) {
			throw new TableWidgetRuntimeException(e);
		}
	}

	/**
	 * Delete the selected elements from their containing resource
	 * @throws CoreException may be caused by query exception in the facet framework.
	 */
	private void deleteSelectedElements() throws CoreException {
		Command deleteSelectedElementsCommand = TableInstanceCommandFactory
				.createDeleteSelectedElementsCommand(this, this.commandFactory);
		this.editingDomain.getCommandStack().execute(deleteSelectedElementsCommand);
	}

	public void addQueryColumns(final List<ETypedElement> eTypedElements) {
		CompoundCommand compoundCommand = new CompoundCommand();
		for (ETypedElement typedElement : eTypedElements) {
			FeatureColumn queryColumn = null;
			if (typedElement instanceof EStructuralFeature) {
				EAttribute eAttribute = (EAttribute) typedElement;
				FeatureColumn attColumn = TableFactory.eINSTANCE
						.createFeatureColumn();
				attColumn.setFeature(eAttribute);
				queryColumn = attColumn;
			} else if (typedElement instanceof EOperation) {
				EOperation eOperation = (EOperation) typedElement;
				//TODO
				throw new IllegalStateException("Not yet implemented"); //$NON-NLS-1$
			}
			Command command = this.commandFactory.createAddCommand(this.editingDomain,
					this.tableInstance,
					TablePackage.eINSTANCE.getTable_Columns(), queryColumn);
			compoundCommand.append(command);
			this.editingDomain.getCommandStack().execute(compoundCommand);
		}
	}

	/**
	 * @return The history of launched instantiation methods.
	 */
	public final List<InstantiationMethod> getInstantiationMethodHistory() {
		return this.launchedInstMethHistory;
	}

	/**
	 * Open a dialog to choose the default instantiationMethod to perform when
	 * clicking on the "Create" button.
	 */
	public final void selectInstantiationMethodToLaunch() {

		ILabelProvider labelProvider = new LabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof InstantiationMethod) {
					InstantiationMethod meth = (InstantiationMethod) element;
					return meth.getName();
				}
				return null;
			}
		};

		IStructuredContentProvider contentProvider = new IStructuredContentProvider() {
			private Object[] objects;

			public void inputChanged(final Viewer viewer, final Object oldInput,
					final Object newInput) {
				if (newInput instanceof List) {
					List<?> list = (List<?>) newInput;
					this.objects = list.toArray();
				}
			}

			public void dispose() {
				// Nothing
			}

			public Object[] getElements(final Object inputElement) {
				return this.objects;
			}
		};

		ListDialog dialog = new ListDialog(this.view.asComposite().getShell());
		dialog.setContentProvider(contentProvider);
		dialog.setLabelProvider(labelProvider);
		dialog.setInput(this.launchedInstMethHistory);
		dialog.setTitle(Messages.NatTableWidget_instantiationMethodSelectionTitle);
		if (dialog.open() != Window.CANCEL) {
			if (dialog.getResult()[0] instanceof InstantiationMethod) {
				InstantiationMethod meth = (InstantiationMethod) dialog.getResult()[0];
				this.launchedInstMethHistory.remove(meth);
				this.launchedInstMethHistory.add(0, meth);
				createNewElement();
			}
		}
	}

	/**
	 * Call the last launched {@link InstantiationMethod} and add the result to
	 * the table if any.
	 */
	public final void createNewElement() {
		if (this.launchedInstMethHistory.get(0) == null) {
			selectInstantiationMethodToLaunch();
		} else {
			createNewElement(this.launchedInstMethHistory.get(0));
		}
	}

	public final void createNewElement(final InstantiationMethod instantiationMethod) {
		try {
			internalCreateNewElement(instantiationMethod);
		} catch (Exception e) {
			throw new TableWidgetRuntimeException(e);
		}
	}

	/**
	 * Call the {@link InstantiationMethod} and add the result to the table if
	 * any.
	 * 
	 * @throws InvocationTargetException
	 * @throws DerivedTypedElementException
	 * @throws FacetManagerException
	 */
	private final void internalCreateNewElement(
			final InstantiationMethod instantiationMethod)
			throws FacetManagerException {
		EOperation eOperation = instantiationMethod.getInstantiationOperation();
		if (eOperation != null) {
			EObject context = getContext();
			if (context == null) {
				throw new IllegalStateException("The table context must not be null when invoking an instantiation method"); //$NON-NLS-1$
			}
			EObject queryResult = this.facetManager.invoke(context, eOperation, EObject.class, this.editingDomain, this.tableInstance.getParameter(), getEditingDomain());
			List<EObject> list = new ArrayList<EObject>();
			list.add(queryResult);
			addRows(list);
			// Store the InstantiationMethod as default
			this.launchedInstMethHistory.remove(instantiationMethod);
			this.launchedInstMethHistory.add(0, instantiationMethod);
		}
	}

	public final boolean isNonCommonColumns(final Column column) {
		boolean common = true;
		if (column instanceof FeatureColumn) {
			FeatureColumn featureColumn = (FeatureColumn) column;
			if (featureColumn.getFeature() instanceof EStructuralFeature) {
				EStructuralFeature structuralFeature = (EStructuralFeature) featureColumn
						.getFeature();
				common = isCommon(structuralFeature);
			}
		}
		return !common;
	}

	/** @return whether the given feature is common to all the given metaclasses */
	private boolean isCommon(final EStructuralFeature feature) {
		for (final EClass metaclass : this.metaclasses) {
			if (!metaclass.getEAllStructuralFeatures().contains(feature)) {
				return false;
			}
		}
		return true;
	}

	public final List<Column> getColumns() {
		return Collections.unmodifiableList(this.tableInstance.getColumns());
	}

	/**
	 * @deprecated cf. https://bugs.eclipse.org/bugs/show_bug.cgi?id=367943
	 */
	@Deprecated
	public final void loadCustomizations(final List<Customization> customizationsToLoad) {
		loadCustomizations(customizationsToLoad, true);
	}

	private final void loadCustomizationsInCustomEngine(
			final List<Customization> customizationsToLoad)
			throws CustomizationException {
		List<Customization> locals = getLocalCustomizations();
		if (this.tableInstance.getRows().size() != 0) {
			// if we get an error here, it is the fault of the developper!!!
			// the local customs should always be contained by
			// customizationsToLoad
			// Assert.isTrue(!locals.isEmpty());
			if (locals.isEmpty()) {
				//Here I choosen a logger to avoid to break the editor whereas he can partialy work.
				Logger.logError(
						"The local customs should always be contained by customizationsToLoad, please update the model: " + this.tableInstance.eResource().getURI(), Activator.getDefault()); //$NON-NLS-1$
			}
		}
		if (!locals.isEmpty() && !customizationsToLoad.containsAll(locals)) {
			// if we get an error here, it is the fault of the developper!!!
			// the local customs should always be contained by
			// customizationsToLoad
			// Assert.isTrue(customizationsToLoad.containsAll(locals));
			// Here I choosen a logger to avoid to break the editor whereas he can partialy work.
			Logger.logError("The local customs should always be contained by customizationsToLoad", Activator.getDefault()); //$NON-NLS-1$
		}
		this.customizationEngine.getManagedCustomizations().removeAll(
				customizationsToLoad);
		for (final Customization customizationToLoad : customizationsToLoad) {
			/*
			 * see bug 341222 customization on feature defined in another metamodel
			 * for the following line! we can define that eAnnotation column is hidden for IModelElement
			 * and hide this column with UML element! whitout this line ,it is not possible
			 */
			this.customizationEngine.getManagedCustomizations().add(0,
					customizationToLoad);
		}
	}

	public ICustomizationManager getCustomizationEngine() {
		return this.customizationEngine;
	}

	public final IFacetManager getFacetContext() {
		return this.facetManager;
	}

	public final void setFacetSets(final List<? extends FacetSet> facetSets) {
		if (TableWidgetController.DEBUG_SET_FACET) {
			System.out.println(this.getClass().getName() + ".setFacets(): " + facetSets.size()); //$NON-NLS-1$ // NOPMD by gdupe on 13/12/11 15:19 //Regulas Eclipse Debuging
			for (FacetSet facetSet : facetSets) {
				System.out.println(this.getClass().getName() + ".setFacets(): " //$NON-NLS-1$  // NOPMD by gdupe on 13/12/11 15:19 //Regulas Eclipse Debuging
						+ facetSet.getName());
			}
		}
		this.setLoadedFacetSets(new ArrayList<FacetSet>(facetSets));
		if (TableWidgetController.DEBUG_SET_FACET) {
			int nbColumns = this.tableInstance.getColumns().size();
			int nbVisibleColumnsUsingTheLayer = getVisibleColumnsUsingTheLayer().size();
			int nbNonFacetColumns = 0;
			for (Column column : this.tableInstance.getColumns()) {
				if (column instanceof FeatureColumn) {
					FeatureColumn featureColumn = (FeatureColumn) column;
					if (!(featureColumn.getFeature() instanceof FacetElement)) {
						nbNonFacetColumns++;
					}
				} else {
					nbNonFacetColumns++;
				}
			}
			System.out.println(this.getClass().getName()  // NOPMD by gdupe on 13/12/11 15:19 //Regulas Eclipse Debuging
					+ ".setFacets(): nbColumns=" + nbColumns); //$NON-NLS-1$
			System.out  // NOPMD by gdupe on 13/12/11 15:19 //Regulas Eclipse Debuging
					.println(this.getClass().getName()
							+ ".setFacets(): nbVisibleColumnsUsingTheLayer=" + nbVisibleColumnsUsingTheLayer); //$NON-NLS-1$
			System.out.println(this.getClass().getName()  // NOPMD by gdupe on 13/12/11 15:19 //Regulas Eclipse Debuging
					+ ".setFacets(): nbNonFacetColumns=" + nbNonFacetColumns); //$NON-NLS-1$
		}
	}

	public final Collection<FacetSet> getLoadedFacetSets() {
		return Collections.unmodifiableCollection(this.tableInstance
				.getFacetSets());
	}

	public final void clearFacets() {
		try {
			this.facetManager.getManagedFacetSets().clear();
		} catch (Exception e) {
			Logger.logError(e, "Error clearing Facets", Activator.getDefault()); //$NON-NLS-1$
		}
	}

	public final Collection<EPackage> getReferencedEPackages() {
		Set<EPackage> ePackages = new HashSet<EPackage>();
		for (EObject eObject : TableWidgetUtils.getElements(this.tableInstance)) {
			EClass eClass = eObject.eClass();
			if (eClass != null) {
				EPackage ePackage = eClass.getEPackage();
				if (ePackage != null) {
					ePackages.add(ePackage);
				}
			}
		}
		return ePackages;
	}

	public final TableConfiguration getTableConfiguration() {
		return getTable().getTableConfiguration();
	}

	public final Table getTable() {
		return this.tableInstance;
	}

	public final EObject getContext() {
		return getTable().getContext();
	}

	/**
	 * This method takes a list of {@link EObject} and add it to the widget list
	 * if their are not already in it.
	 *
	 * @param newElements
	 *            The list of EObject elements to be added to the table
	 */
	public final void addRows(final List<? extends EObject> newElements) {
		final Command addRowCommand = this.tableCommandFactory
				.createAddRowsCommand(newElements);
		executeCommand(addRowCommand);
	}

	/**
	 * @return True if every selected cells is contained by the default label
	 *         column
	 */
	public final boolean isSelectedColumnDeletable() {
		throw new IllegalStateException(
				"Query column does not exist anymore. So no column can be deleted."); //$NON-NLS-1$
	}

	public final void openLoadCustomizationDialog() {
		Set<Customization> availableCustomsSet = new HashSet<Customization>();
		availableCustomsSet.addAll(ICustomizationCatalogManagerFactory.DEFAULT
				.getOrCreateCustomizationCatalogManager(
						this.resource.getResourceSet())
				.getRegisteredCustomizations());
		availableCustomsSet.addAll(this.tableInstance.getCustomizations());
		List<Customization> availableCustomsList = new ArrayList<Customization>(
				availableCustomsSet);
		final IDialogCallbackWithPreCommit<List<Customization>, Boolean, IQuestionDialog> callback = new IDialogCallbackWithPreCommit<List<Customization>, Boolean, IQuestionDialog>() {

			public void committed(final List<Customization> result,
					final Boolean precommitResult) {
				if (precommitResult.booleanValue()) {
					TableWidgetController.this.loadCustomizations(result, true);
				}
			}

			public IQuestionDialog openPrecommitDialog(
					final List<Customization> result,
					final IDialogCallback<Boolean> precommitCallback) {
				return TableWidgetController.this
						.openLoadCustomizationPrecommitDialog(result,
						precommitCallback);
			}
		};
		final ILoadCustomizationsDialog<IQuestionDialog> dialog = ILoadCustomizationsDialogFactory.DEFAULT
				.createLoadCustomizationDialog(null, availableCustomsList,
						this.tableInstance.getCustomizations(),
						this.tableInstance.getLocalCustomizations(),
						callback,
						"It is used to store the hidden columns."); //$NON-NLS-1$
		dialog.asyncOpen();
	}

	protected IQuestionDialog openLoadCustomizationPrecommitDialog(
			final List<Customization> result,
			final IDialogCallback<Boolean> precommitCallback) {
		final List<FacetSet> removedCustoms = new ArrayList<FacetSet>();
		removedCustoms.addAll(TableWidgetController.this.getTable()
				.getCustomizations());
		removedCustoms.removeAll(result);
		final List<FacetSet> newFacetSets = new ArrayList<FacetSet>();
		newFacetSets.addAll(TableWidgetController.this.getTable()
				.getFacetSets());
		newFacetSets.removeAll(removedCustoms);
		return openFacetSetLoadPrecommitDialog(newFacetSets, precommitCallback);
	}

	public final void openLoadFacetSetsDialog() {
		final HashSet<FacetSet> availableFacetSets = new HashSet<FacetSet>();
		availableFacetSets.addAll(IFacetSetCatalogManagerFactory.DEFAULT
				.getOrCreateFacetSetCatalogManager(
						this.editingDomain.getResourceSet())
				.getRegisteredFacetSets());
		if (this.tableInstance.getTableConfiguration() != null) {
			for (Facet facet : this.tableInstance.getTableConfiguration()
					.getDefaultFacets()) {
				availableFacetSets.add(FacetUtils.getRootFacetSet(facet));
			}
		}
		for (FacetSet facetSet : this.tableInstance.getFacetSets()) {
			availableFacetSets.add(FacetUtils.getRootFacetSet(facetSet));
		}
		final IDialogCallbackWithPreCommit<List<FacetSet>, Boolean, IQuestionDialog> callback = new IDialogCallbackWithPreCommit<List<FacetSet>, Boolean, IQuestionDialog>() {

			public void committed(final List<FacetSet> result,
					final Boolean precommitResult) {
				if (precommitResult.booleanValue()) {
					TableWidgetController.this.setLoadedFacetSets(result);
				}
			}

			public IQuestionDialog openPrecommitDialog(
					final List<FacetSet> result,
					final IDialogCallback<Boolean> precommitCallback) {
				return openFacetSetLoadPrecommitDialog(result,
						precommitCallback);
			}
		};
		
		final IFacetSetSelectionDialog<IQuestionDialog> dialog = IFacetSetSelectionDialogFactory.DEFAULT
				.openFacetSetSelectionDialog(availableFacetSets,
						Integer.MAX_VALUE, true, callback, this.view.getShell());
		dialog.setSelectedFacetSets(this.tableInstance.getFacetSets());
	}

	protected IQuestionDialog openFacetSetLoadPrecommitDialog(
			final List<FacetSet> result,
			final IDialogCallback<Boolean> precommitCallback) {
		IQuestionDialog dialog = null;
		if (!InternalTableUtils.columnsToRemove(
				TableWidgetController.this.getTable(), result,
				TableWidgetController.this.getTable().getRows()).isEmpty()) {
			final IQuestionDialog finalDialog = IQuestionDialogFactory.INSTANCE
					.createQuestionDialog(
							TableWidgetController.this.getShell(),
							Messages.TableWidgetController_Remove_columns_after_facetSet_setting_short,
							Messages.TableWidgetController_Remove_columns_after_facetSet_setting_long,
							precommitCallback);
			Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					finalDialog.open();
				}
			});
			dialog = finalDialog;
		}
		return dialog;
	}

	protected Shell getShell() {
		return this.view.getShell();
	}

	/**
	 * Open a "Save As" dialog to serialize the table
	 */
	public final void saveAs() {
		SaveAsDialog dialog = new SaveAsDialog(this.view.asComposite().getShell());

		dialog.setOriginalName("My.table"); //$NON-NLS-1$
		dialog.open();
		if (dialog.getReturnCode() != Window.CANCEL) {
			URI uri = URI.createPlatformResourceURI(dialog.getResult().toString(), true);
			if (uri != null) {
				this.resource.setURI(uri);
				Map<String, Object> options = new HashMap<String, Object>();
				options.put(XMLResource.OPTION_USE_FILE_BUFFER, Boolean.TRUE);
				try {
					this.resource.save(options);
				} catch (IOException e) {
					Logger.logError(e, Activator.getDefault());
				}
			}
			if (this.editingDomain != null) {
				((BasicCommandStack) this.editingDomain.getCommandStack()).saveIsDone();
			}
		}
	}

	/**
	 * Performs the usual save action.
	 */
	public final void save() {
		final Map<String, Object> options = new HashMap<String, Object>();
		options.put(XMLResource.OPTION_USE_FILE_BUFFER, Boolean.TRUE);
		options.put(Resource.OPTION_SAVE_ONLY_IF_CHANGED,
				Resource.OPTION_SAVE_ONLY_IF_CHANGED_MEMORY_BUFFER);

		WorkspaceModifyOperation operation = new WorkspaceModifyOperation() {
			@Override
			public void execute(final IProgressMonitor monitor2) {
				// Save the resources to the file system.
				if (TableWidgetController.this.resource == null
						|| (TableWidgetController.this.resource.getURI().lastSegment()
								.equals(TableWidgetController.TMP_FILE_NAME))) {
					saveAs();
				}
				for (Resource resourceElt : TableWidgetController.this.editingDomain.getResourceSet()
						.getResources()) {
					if (isPersisted(resourceElt)) {
						if (!resourceElt.getContents().isEmpty()
								&& !TableWidgetController.this.editingDomain.isReadOnly(resourceElt)) {
							try {
								resourceElt.save(options);

							} catch (Exception e) {
								Logger.logError(e, Activator.getDefault());
							}
						}
					}
				}
			}
		};

		try {
			operation.run(new NullProgressMonitor());
		} catch (InvocationTargetException e) {
			Logger.logError(e, Activator.getDefault());
		} catch (InterruptedException e) {
			Logger.logError(e, Activator.getDefault());
		}

		((BasicCommandStack) this.editingDomain.getCommandStack()).saveIsDone();
	}

	/**
	 * This returns whether something has been persisted to the URI of the
	 * specified resource. The implementation uses the URI converter from the
	 * editor's resource set to try to open an input stream. <!-- begin-user-doc
	 * --> <!-- end-user-doc -->
	 *
	 * This method has been copied from an EMF generated model editor.
	 */
	private boolean isPersisted(final Resource resourceParam) {
		boolean result;
		try {
			InputStream stream = this.editingDomain.getResourceSet().getURIConverter()
					.createInputStream(resourceParam.getURI());
			if (stream != null) {
				result = true;
				stream.close();
			} else {
				result = false;
			}
		} catch (IOException e) {
			result = false;
		}
		return result;
	}

	public final boolean isDataEditable() {
		return this.dataEditable;
	}

	public final void drop(final StructuredSelection structuredSelection, final Object element,
			final EStructuralFeature feature) {
		CompoundCommand compoundCommand = TableInstanceCommandFactory.createDropCommand(this,
				structuredSelection, element, feature);
		if (compoundCommand != null && !compoundCommand.isEmpty()) {
			this.editingDomain.getCommandStack().execute(compoundCommand);
		}
	}
	
	public void drop(final StructuredSelection structuredSelection) {
		List<EObject> list = new ArrayList<EObject>();
		final Iterator<?> iterator = structuredSelection.iterator();
		while (iterator.hasNext()) {
			final Object selectedElement = iterator.next();
			if (selectedElement instanceof EObject) {
				EObject selectedEObject = (EObject) selectedElement;
				if (!acceptsElement(selectedEObject)) {
					MessageDialog.openWarning(
							this.view.asComposite().getShell(),
							Messages.wrongEObjectType,
							Messages.wrongEObjectTypeLong
									+ EcoreUtil.getURI(selectedEObject.eClass()));
					return;
				}
				list.add(selectedEObject);
			}
		}
		// Update the Table
		addRows(list);
	}

	public final boolean canBeDropped(final IStructuredSelection structuredSelection,
			final EStructuralFeature feature, final Object element) {
		CompoundCommand compoundCommand = TableInstanceCommandFactory.createDropCommand(this,
				structuredSelection, element, feature);
		return compoundCommand != null && compoundCommand.canExecute();
	}

	/** Perform an EMF Set Command */
	private final void setWithCommand(final Object owner, final Object feature, final Object value) {
		Command cmd = this.commandFactory.createSetCommand(TableWidgetController.this.editingDomain, owner,
				feature, value);
		TableWidgetController.this.editingDomain.getCommandStack().execute(cmd);
	}

	public final Control getComposite() {
		return this.view.asComposite();
	}

	public final void registerActionDelegate(final IActionDelegate action) {
		if (!this.actionList.contains(action)) {
			this.actionList.add(action);
		}
	}

	public final void unregisterActionDelegate(final IActionDelegate action) {
		this.actionList.remove(action);
	}

	public final void notifyActions() {
		for (IActionDelegate action : this.actionList) {
			action.selectionChanged(null, null);
		}
	}

	public final Collection<ISelectionChangedListener> getSelectionChangedListeners() {
		return Collections.unmodifiableCollection(new ArrayList<ISelectionChangedListener>(this.fSelectionChangedListeners));
	}

	/**
	 * Sort columns by name following this order :
	 * <ul>
	 * <li>the columns beginning by '['</li>
	 * <li>the column "/eContainer"</li>
	 * <li>the other columns, ignoring the '/'</li>
	 * </ul>
	 */
	public final void sortColumnByNameAtoZ() {
		try {
			this.view.asComposite().setRedraw(false);
			this.listenReorderEvent = false;
			List<Column> columns = new ArrayList<Column>(this.tableInstance.getColumns());
			Collections.sort(columns, new ColumnComparator());

			Command command = this.commandFactory.createSetCommand(TableWidgetController.this.editingDomain,
					TableWidgetController.this.tableInstance,
					TablePackage.eINSTANCE.getTable_Columns(), columns);
			TableWidgetController.this.editingDomain.getCommandStack().execute(command);
		} catch (Exception e) {
			Logger.logError(e, "Error sorting columns by name From A to Z", //$NON-NLS-1$
					Activator.getDefault());
		} finally {
			this.view.asComposite().setRedraw(true);
			this.listenReorderEvent = true;
		}
	}

	/**
	 * Sort columns by name from Z to A following this order :
	 * <ul>
	 * <li>in first the columns which name doesn't contain '/' 'except
	 * "/eContainer" and the column beginning by '['
	 * <li>the the column "/eContainer"</li>
	 * <li>the column beginning by '['</li>
	 * </ul>
	 */
	public final void sortColumnByNameZtoA() {
		try {
			this.view.asComposite().setRedraw(false);
			this.listenReorderEvent = false;
			List<Column> columns = new ArrayList<Column>(this.tableInstance.getColumns());
			ColumnComparator comparator = new ColumnComparator();
			comparator.setInversedOrder(true);
			Collections.sort(columns, comparator);
			// Collections.reverse(columns);

			Command command = this.commandFactory.createSetCommand(TableWidgetController.this.editingDomain,
					TableWidgetController.this.tableInstance,
					TablePackage.eINSTANCE.getTable_Columns(), columns);
			TableWidgetController.this.editingDomain.getCommandStack().execute(command);
		} catch (Exception e) {
			Logger.logError(e, "Error sorting columns by name from Z to A", //$NON-NLS-1$
					Activator.getDefault());
		} finally {
			this.view.asComposite().setRedraw(true);
			this.listenReorderEvent = true;
		}
	}

	private int getRowIndexByEObject(final EObject eObject) {
		int rowIndex = -1;
		EList<Row> rows = this.tableInstance.getRows();
		for (int i = 0; i < rows.size(); i++) {
			if (rows.get(i).getElement().equals(eObject)) {
				rowIndex = i;
				break;
			}
		}
		return rowIndex;
	}

	public final List<Column> getVisibleColumns(final boolean setAtTheTop) {
		List<Column> notHiddenColumns = new ArrayList<Column>();
		try {
			notHiddenColumns = getVisibleColumnsUsingTheLocalCustomizations(
					this.tableInstance, setAtTheTop);
		} catch (Exception e) {
			throw new TableWidgetRuntimeException(e);
		}
		return notHiddenColumns;
	}

	public final void putLocalCustomizationOnTheBottom() {
		this.editingDomain.getCommandStack().execute(getPutLocalCustomizationOnTheBottomCommand());
	}

	private Command getPutLocalCustomizationOnTheBottomCommand() {
		List<Customization> locals = getLocalCustomizations();
		List<Customization> customizations = new ArrayList<Customization>();
		customizations.addAll(this.tableInstance.getCustomizations());
		customizations.removeAll(locals);
		customizations.addAll(customizations.size(), locals);
		return this.commandFactory.createSetCommand(this.editingDomain, this.tableInstance,
				TablePackage.eINSTANCE.getTable_Customizations(),
				customizations);
	}

	/**
	 *
	 * @param eObject
	 *            an eObject
	 * @return
	 * the row representing the eObject or <code>null</code> if not found
	 */
	public final Row findRow(final EObject eObject) {
		for (Row row : this.tableInstance.getRows()) {
			if (row.getElement().equals(eObject)) {
				return row;
			}
		}
		return null;
	}

	protected final void refreshDelayed() {
		DebugUtils.debug(TableWidgetController.DEBUG_RESOURCE_LISTENER);
		// If refreshTreeDelayed is called again before the job delay passed,
		// then cancel the job and re-schedule a new one.
		this.refreshJob.cancel();
		this.refreshJob.setPriority(Job.DECORATE);
		this.refreshJob.schedule(TableWidgetController.REFRESH_JOB_DELAY);
	}
	
	public final EditingDomain getEditingDomain() {
		return this.editingDomain;
	}

	public final ICommandFactory getCommandFactory() {
		return this.commandFactory;
	}

	public final void waitForRefreshJob() {
		boolean wait = true;
		while (wait) {
			try {
				if (this.refreshJob != null) {
					this.refreshJob.join();
				}
			wait = false;
			} catch (InterruptedException e) {
				wait = true;
			}
		}
	}

	/**
	 * If a table element has been deleted from outside the table, then we need
	 * to remove the inconsistent row from its model.
	 */
	private final void removeUselessRowsAndColumns() {
		Command command = TableInstanceCommandFactory.createRemoveUselessRowsAndColumnsCommand(this);
		executeCommand(command);
	}

	/**
	 * This method has to be used to know if the the tableInstance model is
	 * stored in a temporary resource. This is useful to implement an editor
	 * which uses the table wigdet.
	 *
	 * @return <code>true</code> if the table model is stored in a temporary
	 *         resource
	 */
	public final boolean usesTmpResource() {
		//TODO this method has to be added in the API in the version 0.2.0.
		//Cf. Bug 349797 - AbstractTableWidget.usesTmpResource() has to be added to the API. https://bugs.eclipse.org/bugs/show_bug.cgi?id=349797
		String uri = URI.createFileURI(TableWidgetController.DEFAULT_RESOURCE_FILE.toString())
				.toFileString();
		String tableInstanceResourceURI = getTable().eResource()
				.getURI().toFileString();
		return uri.equals(tableInstanceResourceURI);
	}

	public final MenuManager getMenuManager() {
		return this.menuMgr;
	}

	public ICustomizedLabelProvider getTableLabelProvider() {
		return this.tableLabelProvider;
	}

	public void setColumnWidth(final Column column, final int width) {
		Command cmd = this.getCommandFactory()
				.createSetCommand(this.getEditingDomain(),
						column, TablePackage.eINSTANCE
								.getColumn_Width(), width);
		this.getEditingDomain().getCommandStack().execute(cmd);
	}
	

	public void setRowHeight(final Row row, final int height) {
		Command cmd = this.getCommandFactory().createSetCommand(
				this.getEditingDomain(), row,
				TablePackage.eINSTANCE.getRow_Height(), height);
		this.getEditingDomain().getCommandStack()
				.execute(cmd);
	}
	

	public void moveColumn(final Column element, final int newPosition) {
		Command command = this.getCommandFactory().createMoveCommand(
				this.getEditingDomain(),
				this.getTable(),
				TablePackage.eINSTANCE
.getTable_Columns(), element,
						newPosition);
		this.getEditingDomain().getCommandStack().execute(command);
	}
	

	public List<Column> getVisibleColumnsUsingTheLayer() {
		return this.view.getVisibleColumnsUsingTheLayer();
	}

	public List<Column> getSelectedColumns() {
		return this.view.getSelectedColumns();
	}

	public IPositionCoordinate[] getSelectedCells2() {
		return this.view.getSelectedCells2();
	}

	public List<IPositionCoordinate> getSelectedCellsPositions() {
		return Arrays.asList(getSelectedCells2());
	}


	public void copySelectionToClipboard() {
		this.view.copySelectionToClipboard();
	}

	public void selectAll() {
		this.view.selectAll();
	}

	public boolean isCellSelected() {
		return this.view.isCellSelected();
	}

	public boolean isCellsDeletable() {
		return this.view.isCellsDeletable();
	}

	public void selectDefaultLabelCell(final EObject eObject) {
		this.view.selectDefaultLabelCell(eObject);
	}

	public void selectCell(final EObject eObject, final EStructuralFeature structuralFeature) {
		this.view.selectCell(eObject, structuralFeature);
	}

	public void editSelectedCell() {
		if (isSelectedCellEditable()) {
			this.view.editSelectedCell();
		}
	}

	public boolean isSelectedCellEditable() {
		final List<IGridElement> gridElements = getSelectedGridElements();
		// cannot edit zero or more than one cell
		return gridElements.size() == 1 && isGridElementEditable(gridElements.get(0));
	}

	protected boolean isGridElementEditable(final IGridElement gridElement) {
		boolean editable = false;
		final EObject eObject = gridElement.getRow().getElement();
		final Column column = gridElement.getColumn();
		final URI uri = eObject.eResource().getURI();
		if ((uri.isPlatformResource() || uri.isFile()) && column instanceof FeatureColumn) {
			final FeatureColumn featureColumn = (FeatureColumn) column;
			final ETypedElement feature = featureColumn.getFeature();
			if (feature instanceof EStructuralFeature && ((EStructuralFeature) feature).isChangeable()) {
				if (feature instanceof DerivedTypedElement) {
					try {
						final DerivedTypedElement derivedTypedElement = (DerivedTypedElement) feature;
						editable = getFacetContext().isConforming(eObject, (Facet) derivedTypedElement.eContainer());
					} catch (final Exception e) {
						Logger.logWarning(e, "Failed to check Facet conformance", Activator.getDefault()); //$NON-NLS-1$
					}
				} else {
					final List<EStructuralFeature> eStructuralFeatures = eObject.eClass().getEAllStructuralFeatures();
					editable = eStructuralFeatures.contains(feature);
				}
			}
		}
		return editable;
	}

	public void selectRows(final List<EObject> elementsToSelect,
			final boolean clearTheSelection) {
		this.view.selectRows(elementsToSelect, clearTheSelection);
	}

	@SuppressWarnings("null")
	// @SuppressWarnings("null") : the buffer cannot be null when DEBUG_SELECTION is true
	public ISelection getSelection() {
		checkNotDisposed();
		ArrayList<Object> result = new ArrayList<Object>();
		List<Object> gridSelection = this.view.getGridSelection();
		StringBuffer debugBuffer = null;
		if (TableWidgetController.DEBUG_SELECTION) {
			debugBuffer = new StringBuffer();
		}
		for (Object selectedElement : gridSelection) {
			if (selectedElement instanceof IGridElement) {
				IGridElement gridElement = (IGridElement) selectedElement;
				Collection<? extends Object> resolvedSelection = resolveSelection(gridElement);
				for (Object object : resolvedSelection) {
					if (object != null) {
						result.add(object);
						if (TableWidgetController.DEBUG_SELECTION) {
							debugBuffer.append(System.getProperty("line.separator")).append("    ") //$NON-NLS-1$ //$NON-NLS-2$
									.append(object);
						}
					}
				}
			}
		}
		if (TableWidgetController.DEBUG_SELECTION) {
			DebugUtils.debug(debugBuffer.toString());
		}
		return new StructuredSelection(result);
	}

	// @SuppressWarnings("unchecked") : customization manager is not type safe
	@SuppressWarnings("unchecked")
	private Collection<? extends Object> resolveSelection(final IGridElement gridElement) {
		Collection<? extends Object> result = Collections.emptyList();
		EObject element = gridElement.getRow().getElement();
		if (element != null && TableWidgetUtils.isApplicable(gridElement, getFacetContext())) {
			try {
				result = this.customizationEngine.getCustomValueOf(element,
						((FeatureColumn) gridElement.getColumn()).getFeature(),
						this.selectionPropertiesHandler.getSelectionProperty(),
						List.class);
			} catch (CustomizationException e) {
				Logger.logWarning(e, "Error in table selection customization", Activator.getDefault()); //$NON-NLS-1$
			}
		}
		return result;
	}


	/** This is a sanity check to avoid dealing with a disposed widget */
	private void checkNotDisposed() {
		if (this.disposed) {
			throw new IllegalStateException("the widget is disposed"); //$NON-NLS-1$
		}
	}

	private void fullRedraw() {
		checkNotDisposed();
		this.view.fullRedraw();
	}

	public boolean getListenReorderEvent() {
		return this.listenReorderEvent;
	}

	public Display getDisplay() {
		return this.view.asComposite().getDisplay();
	}

	public final void loadCustomizations(
			final List<Customization> customizationsToLoad,
			final boolean mustLoadAdditionalFacets) {
		Command command = this.tableCommandFactory
				.createSetCustomizationCommand(customizationsToLoad);
		executeCommand(command);
	}


	private void executeCommand(final Command command) {
		if (command != null) {
			DebugUtils.debug(EmfDebugUtils.debugCommand(command));
			if (command.canExecute()) {
				this.editingDomain.getCommandStack().execute(command);
			} else {
				throw new TableWidgetRuntimeException(EmfDebugUtils.debugCommand(command));
			}
		}
	}

	public IContentPropertiesHandler getCustomPropertiesHandler() {
		return this.contentPropertiesHandler;
	}
	


	/**
	 * Set the presentability query. This method is meant to be used by unit
	 * tests.
	 * <ul>
	 * <li>If all elements currently in the table match the given query, then
	 * this method just sets the query and returns <code>null</code>.
	 * <li>If not all elements match the query, then this method opens a dialog
	 * that asks the user whether to remove the non-conforming elements, and
	 * returns the dialog, with a listener set so that the presentability query
	 * will be set once the dialog is closed.
	 * </ul>
	 */
	public IMustRemoveNonConformingElementsDialog setCanBePresentedInTheTable(
			final ETypedElement presentabilityQuery) {
		IMustRemoveNonConformingElementsDialog result;
		if (containsOnlyPresentableElements(presentabilityQuery)) {
			setCanBePresentedInTheTable(presentabilityQuery, false);
			result = null;
		} else {
			final IMustRemoveNonConformingElementsDialog dialog = new MustRemoveNonConformingElementsPromptDialog(
					this.view.getShell());
			dialog.addCloseListener(new Runnable() {
				public void run() {
					try {
						setCanBePresentedInTheTable(presentabilityQuery, dialog
								.getMustRemoveNonConformingElements().booleanValue());
					} catch (Exception e) {
						logAndDisplayException(e);
					}
				}
			});
			dialog.open();
			result = new SynchronizedMustRemoveNonConformingElementsPromptDialog(dialog);
		}
		return result;
	}

	public void setCanBePresentedInTheTable(final ETypedElement presentabilityQuery,
			final boolean removeNonConformingElements) {
		if (presentabilityQuery != null) {
			if (!ModelUtils.isBooleanDataType(presentabilityQuery.getEType())) {
				throw new IllegalArgumentException(
						"The presentability query must return a boolean"); //$NON-NLS-1$
			}
			if (presentabilityQuery.getUpperBound() > 1
					|| presentabilityQuery.getUpperBound() == -1
					|| presentabilityQuery.getLowerBound() > 1) {
				throw new IllegalArgumentException("The presentability query must be single-valued"); //$NON-NLS-1$
			}
			if (presentabilityQuery.eContainer() instanceof Facet) {
				Facet facet = (Facet) presentabilityQuery.eContainer();
				if (facet.getExtendedMetaclass() != EcorePackage.eINSTANCE.getEObject()) {
					throw new IllegalArgumentException(
							"The presentability query must be contained in a Facet extending EObject"); //$NON-NLS-1$
				}
			}
		}
		CompoundCommand compoundCommand = new CompoundCommand();
		// tableInstance.setCanBePresentedInTheTable(presentabilityQuery);
		Command setCommand = this.commandFactory.createSetCommand(this.editingDomain,
				this.tableInstance,
				TablePackage.eINSTANCE.getTable_CanBePresentedInTheTable(),
				presentabilityQuery);
		compoundCommand.append(setCommand);
		if (removeNonConformingElements) {
			Command removeNonConformingElementsCommand;
			removeNonConformingElementsCommand = createRemoveNonConformingElementsCommand(presentabilityQuery);
			if (removeNonConformingElementsCommand != null) {
				compoundCommand.append(removeNonConformingElementsCommand);
			}
		}

		compoundCommand.setLabel(Messages.NatTableWidget_setPresentabilityQuery);
		this.editingDomain.getCommandStack().execute(compoundCommand);
	}

	private Command createRemoveNonConformingElementsCommand(
			final ETypedElement presentabilityQuery) {
		final List<Row> rowsToRemove = new ArrayList<Row>();
		for (Row row : this.tableInstance.getRows()) {
			final EObject element = row.getElement();
			if (!acceptsElement(element, presentabilityQuery)) {
				rowsToRemove.add(row);
			}
		}
		return this.tableCommandFactory.createRemoveRowsCommand(rowsToRemove);
	}

	/**
	 * Returns whether the given table contains elements that are all
	 * presentable according to the given presentability query.
	 */
	private boolean containsOnlyPresentableElements(final ETypedElement presentabilityQuery) {
		boolean allPresentable = true;
		EList<Row> rows = this.tableInstance.getRows();
		for (Row row : rows) {
			EObject element = row.getElement();
			if (!acceptsElement(element, presentabilityQuery)) {
				allPresentable = false;
				break;
			}
		}
		return allPresentable;
	}

	public ETypedElement getCanBePresentedInTheTable() {
		return TableWidgetInternalUtils.getPresentabilityQuery(this.tableInstance);
	}

	public IAllowedContentsETypedElementSelectionDialog openCanBePresentedInTheTableDialog() {
		final IAllowedContentsETypedElementSelectionDialogCallback callBack = new IAllowedContentsETypedElementSelectionDialogCallback() {
			
			public boolean mustAskToRemoveNonAllowedElements(
					final ETypedElement allowedContentsTypedElement) {
				return !containsOnlyPresentableElements(allowedContentsTypedElement);
			}
			
			public void committed(final ETypedElement typedElement,
					final boolean removeNonAllowedContents) {
				setCanBePresentedInTheTable(typedElement, removeNonAllowedContents);
			}
			
			public void canceled() {
				//Nothing to do.
			}
		};
		
		
		final AllowedContentsETypedElementSelectionDialog dialog = new AllowedContentsETypedElementSelectionDialog(
				this.view.getShell(), getAllKnownETypedElements(), callBack,
				this.customizationEngine);
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				dialog.open();
				dialog.select(getCanBePresentedInTheTable());
			}
		});
		return new SynchronizedPresentabilityETypedElementSelectionDialog(dialog, this.view.getShell().getDisplay());
	}
	
	private List<ETypedElement> getAllKnownETypedElements() {
		final Set<EClassifier> classifiers = getAllKnownEClassifiers();
		List<ETypedElement> typedElements = new ArrayList<ETypedElement>();
		for (EClassifier eClassifier : classifiers) {
			typedElements.addAll(getContainedETypedElements(eClassifier));
		}
		return typedElements;
	}

	private static Collection<ETypedElement> getContainedETypedElements(final EObject eObject) {
		List<ETypedElement> typedElements = new ArrayList<ETypedElement>();
		TreeIterator<EObject> eAllContents = eObject.eAllContents();
		while (eAllContents.hasNext()) {
			EObject element = eAllContents.next();
			if (element instanceof ETypedElement) {
				ETypedElement eTypedElement = (ETypedElement) element;
				typedElements.add(eTypedElement);
			}
		}
		return typedElements;
	}

	private Set<EClassifier> getAllKnownEClassifiers() {
		final HashSet<EClassifier> eClassifiers = new HashSet<EClassifier>();
		for (FacetSet facetSet : IFacetSetCatalogManagerFactory.DEFAULT
				.getOrCreateFacetSetCatalogManager(
						this.editingDomain.getResourceSet())
				.getRegisteredFacetSets()) {
			eClassifiers.addAll(FacetUtils.getAllFacet(facetSet));
		}
		for (FacetSet facetSet : this.tableInstance.getFacetSets()) {
			eClassifiers.addAll(FacetUtils.getAllFacet(facetSet));
		}
		for (EObject eObject : TableWidgetUtils.getElements(this.tableInstance)) {
			eClassifiers.add(eObject.eClass());
		}
		return eClassifiers;
	}

	protected final void logAndDisplayException(final Exception e) {
		Logger.logError(e, Activator.getDefault());
		MessageDialog.openError(this.view.getShell(), Messages.NatTableWidget_error, e.getLocalizedMessage());
	}

	public List<ETypedElement> getApplicableCanBePresentedQueries() {
		Collection<FacetSet> knownFacetSets = getAllKnownFacetSets();
		List<ETypedElement> applicableQueries = new ArrayList<ETypedElement>();
		for (FacetSet facetSet : knownFacetSets) {
			for (Facet facet : FacetUtils.getFacets(facetSet)) {
				if (facet.getExtendedMetaclass() == EcorePackage.eINSTANCE.getEObject()) {
					final List<ETypedElement> typeElementsToCheck = new LinkedList<ETypedElement>();
					typeElementsToCheck.addAll(facet.getFacetElements());
					typeElementsToCheck.addAll(facet.getFacetOperations());
					for (ETypedElement eTypeElement : typeElementsToCheck) {
						if (ModelUtils.isBooleanDataType(eTypeElement.getEType())) {
							applicableQueries.add(eTypeElement);
						}
					}
				}
			}
		}
		return Collections.unmodifiableList(applicableQueries);
	}
	
	private Collection<FacetSet> getAllKnownFacetSets() {
		final HashSet<FacetSet> result = new HashSet<FacetSet>();
		result.addAll(IFacetSetCatalogManagerFactory.DEFAULT
				.getOrCreateFacetSetCatalogManager(
						this.editingDomain.getResourceSet())
				.getRegisteredFacetSets());
		result.addAll(this.tableInstance.getFacetSets());
		return result;
	}


	public void undo() {
		this.editingDomain.getCommandStack().undo();
	}
	
	public void redo() {
		this.editingDomain.getCommandStack().redo();
	}
	
	/**
	 * Whether the given element should be accepted in the given table instance.
	 * This is decided by the table instance's presentability query, or the
	 * table configuration's presentability query if the table instance's
	 * presentability query is not set.
	 * 
	 * @param element
	 *            the element to test.
	 * @param tableInstance
	 *            the table instance.
	 * @return whether the given element can be added to the given table
	 */
	private boolean acceptsElement(final Object element, final ETypedElement presentabilityQuery) {
		boolean result;
		if (presentabilityQuery != null) {
			if (element instanceof EObject) {
				EObject eObject = (EObject) element;
				try {
					result = this.facetManager.getOrInvoke(eObject, presentabilityQuery, Boolean.class).booleanValue();
				} catch (Exception e) {
					Logger.logError(e,
							"Error executing presentability query: "+presentabilityQuery.getName(), Activator.getDefault()); //$NON-NLS-1$
					result = false;
				}
			} else {
				result = false;
			}
		} else {
			result = true;
		}
		return result;
	}

	/**
	 * Whether the given element should be accepted in the given table instance.
	 * This is decided by the table instance's presentability query, or the
	 * table configuration's presentability query if the table instance's
	 * presentability query is not set.
	 * 
	 * @param element
	 *            the element to test.
	 * @param tableInstance
	 *            the table instance.
	 * @return whether the given element can be added to the given table
	 */
	public boolean acceptsElement(final Object element) {
		ETypedElement presentabilityQuery = TableWidgetInternalUtils.getPresentabilityQuery(this.tableInstance);
		return acceptsElement(element, presentabilityQuery);
	}

	public void setLoadedFacetSets(final List<FacetSet> facetSets) {
		final Command cmd = this.tableCommandFactory
				.createSetLoadedFacetSetsCommand(facetSets);
		this.executeCommand(cmd);
	}

	public IETypedElementSelectionDialog<Object> openAddColumnsDialog() {
		// In the "add column dilog" we only want to see the typed elements:
		// - owned by the regitred and refered facetSets,
		// - owned by the eClass of the eObjects presented in the table
		final List<FacetSet> knownFacetSets = getKnownFacetSet();
		// Searching eTypedElements in known facet sets
		final Set<ETypedElement> eTypedElements = new HashSet<ETypedElement>();
		for (FacetSet facetSet : knownFacetSets) {
			eTypedElements.addAll(FacetUtils.getETypedElements(facetSet));
		}
		// Searching eTypedElements in eObjects presented in the table
		for (Row row : this.tableInstance.getRows()) {
			final EObject eObject = row.getElement();
			eTypedElements
					.addAll(FacetUtils.getETypedElements(eObject.eClass()));
		}
		IDialogCallbackWithPreCommit<List<ETypedElement>, Object, Object> callback = new IDialogCallbackWithPreCommit<List<ETypedElement>, Object, Object>() {

			public void committed(final List<ETypedElement> result,
					final Object precommitResult) {
				final Set<FacetSet> facetSets = new HashSet<FacetSet>();
				for (ETypedElement eTypedElement : result) {
					final FacetSet facetSet = FacetUtils.getFacetSet(eTypedElement);
					if (facetSet != null) {
						final FacetSet rootFacetSet = FacetUtils.getRootFacetSet(facetSet);
						facetSets.add(rootFacetSet);
					}
				}
				// BEGIN: Section dedicated to the aggregate management
				final List<ETypedElement> resolvedTE = new ArrayList<ETypedElement>(
						result.size());
				for (ETypedElement eTypedElement : result) {
					ETypedElement resolvedEObject = IResolverManager.DEFAULT
							.resolve(eTypedElement, ETypedElement.class);
					if (resolvedEObject == null) {
						resolvedTE.add(eTypedElement);
					} else {
						resolvedTE.add(resolvedEObject);
					}
				}
				// END: Section dedicated to the aggregate management
				TableWidgetController.this.addColumns(resolvedTE,
						new ArrayList<FacetSet>(facetSets));
			}

			public Object openPrecommitDialog(final List<ETypedElement> result,
					final IDialogCallback<Object> precommitCallback) {
				return null;
			}
		};
		final IETypedElementSelectionDialog<Object> dialog = IETypedElementSelectionDialogFactory.DEFAULT
				.openETypedElementSelectionDialog(eTypedElements,
						Integer.MAX_VALUE, false,
						callback, this.getShell(),
						this.getCustomizationEngine(), knownFacetSets);
		return dialog;
	}


	private List<FacetSet> getKnownFacetSet() {
		final List<FacetSet> knownFacetSets = new ArrayList<FacetSet>();
		// Searching facetSet known by the table (will be used to definies the
		// tree roots) (rejecting customizations)
		for (FacetSet facetSet : this.tableInstance.getFacetSets()) {
			if (!(facetSet instanceof Customization)) {
				knownFacetSets.add(facetSet);
			}
		}
		// Searching facetSet known by the facetSet catalog (will be used to
		// definies the tree roots) (rejecting customizations)
		final IFacetSetCatalogManager facetSetCatalogManager = IFacetSetCatalogManagerFactory.DEFAULT
				.getOrCreateFacetSetCatalogManager(this.editingDomain
						.getResourceSet());
		final Collection<FacetSet> registeredFacetSets = facetSetCatalogManager
				.getRegisteredFacetSets();
		for (FacetSet facetSet : registeredFacetSets) {
			if (!(facetSet instanceof Customization)) {
				knownFacetSets.add(facetSet);
			}
		}
		return knownFacetSets;
	}

	public final void addColumns(final List<ETypedElement> eTypedElements, final List<FacetSet> facetSets) {
		Command command = this.tableCommandFactory.createAddColumnCommand(eTypedElements, facetSets);
		executeCommand(command);
	}


	public void sortRows(final Column column,
			final ColumnSortDirection direction) {
		if (column instanceof FeatureColumn) {
			final long begin = System.currentTimeMillis();
			final FeatureColumn featureColumn = (FeatureColumn) column;
			final Command command = this.tableCommandFactory.createSortRowCommand(
					featureColumn,
					direction);
			final long createCmd = System.currentTimeMillis();
			executeCommand(command);
			final long end = System.currentTimeMillis();
			if (TableWidgetController.DEBUG_SORT) {
				DebugUtils.debug("sort and createCmd =" + (createCmd - begin)); //$NON-NLS-1$
				DebugUtils.debug("exec =" + (end - createCmd)); //$NON-NLS-1$
				DebugUtils.debug("total =" + (end - begin)); //$NON-NLS-1$
			}
		}
	}
	
	public List<Object> getRawSelection() {
		return this.view.getGridSelection();
	}
	
	/** @return the selected grid elements */
	protected List<IGridElement> getSelectedGridElements() {
		final List<Object> selection = getRawSelection();
		final List<IGridElement> gridElements = new ArrayList<IGridElement>();
		for (final Object object : selection) {
			if (object instanceof IGridElement) {
				gridElements.add((IGridElement) object);
			}
		}
		return gridElements;
	}	

	public boolean isDisposed() {
		return this.disposed;
	}

	public void removeColumns(final List<ETypedElement> elements) {
		final Command command = this.tableCommandFactory
				.createRemoveColumnsCommand(elements);
		executeCommand(command);
	}


	public ITableCommandFactory getTableCommandFactory() {
		return this.tableCommandFactory;
	}
}
