/**
 * Copyright (c) 2015 Soft-Maint, and Mia-Software.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * 	Jonathan Pepin (Soft-Maint) - Bug 463905 - Support of parsing OCL expressions with Facet references
 * 	Grgoire Dup (Mia-Software) - Bug 463905 - Support of parsing OCL expressions with Facet references
 */
package org.eclipse.emf.facet.util.efacet.ocl.core.internal;

import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.ETypedElement;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.facet.efacet.core.IFacetManager;
import org.eclipse.emf.facet.efacet.core.IFacetManagerFactory;
import org.eclipse.emf.facet.efacet.core.exception.FacetManagerException;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.Facet;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.FacetSet;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.ocl.EvaluationEnvironment;
import org.eclipse.ocl.EvaluationHaltedException;
import org.eclipse.ocl.ecore.EcoreEvaluationEnvironment;

public class EFacetOCLEvaluationEnvironment extends EcoreEvaluationEnvironment {

	public EFacetOCLEvaluationEnvironment() {
		super();
	}

	public EFacetOCLEvaluationEnvironment(
			final EvaluationEnvironment<EClassifier, EOperation, EStructuralFeature, EClass, EObject> parent) {
		super(parent);
	}

	@Override
	public Object navigateProperty(final EStructuralFeature property,
			final List<?> qualifiers, final Object target) {
		Object result = null;
		if (property.getEAnnotation(EFacetOCLEnvironment.ANNOT_ENVIRONMENT) == null) {
			result = super.navigateProperty(property, qualifiers, target);
		} else {
			if (target instanceof EObject) {
				final EObject eObject = (EObject) target;
				final String referenceName = property.getName();
				result = getFacetReferences(eObject, referenceName);
			}
		}
		if (result == null) {
			throw new IllegalArgumentException();
		}
		return result;
	}

	private static Object getFacetReferences(final EObject eObject,
			final String referenceName) {
		Object result = null;
		final Resource eResource = eObject.eResource();
		final ResourceSet resourceSet = eResource.getResourceSet();
		final IFacetManager facetManager = IFacetManagerFactory.DEFAULT
				.getOrCreateDefaultFacetManager(resourceSet);
		ETypedElement query = null;
		final List<FacetSet> managedFacetSets = facetManager.getManagedFacetSets();
		final Iterator<FacetSet> facetSetIter = managedFacetSets.iterator();
		while (query == null && facetSetIter.hasNext()) {
			final FacetSet facetSet = facetSetIter.next();
			final EList<EClassifier> eClassifiers = facetSet.getEClassifiers();
			final Iterator<EClassifier> facetIter = eClassifiers.iterator();
			while (query == null && facetIter.hasNext()) {
				final EClassifier classifier = facetIter.next();
				if (classifier instanceof Facet) {
					final Facet facet = (Facet) classifier;
					query = getTypedElement(facetManager, facet, eObject, referenceName);
				}
			}
		}
		try {
			result = facetManager.getOrInvoke(eObject, query, Object.class);
		} catch (FacetManagerException e) {
			throw new EvaluationHaltedException("FacetManager is not loaded:", e); //$NON-NLS-1$
		}
		return result;
	}

	private static ETypedElement getTypedElement(final IFacetManager facetManager,
			final Facet facet, final EObject eObject, final String referenceName) {
		ETypedElement query = null;
		try {
			if (facetManager.isConforming(eObject, facet)) {
				for (ETypedElement etypeElt : facet.getFacetElements()) {
					if (etypeElt.getName().equals(referenceName)) {
						query = etypeElt;
						break;
					}
				}
			}
		} catch (FacetManagerException e) {
			Logger.logError(e.getMessage(), Activator.getDefault());
		}
		return query;
	}

}
