/*******************************************************************************
 * Copyright (c) 2010 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Guyomar (Mia-Software) - Bug 331442 - To be able to add and remove lines (model elements) from the table - initial API and implementation
 *    Nicolas Bros (Mia-Software) - Bug 332438 - NatTable : table type
 *    Nicolas Bros (Mia-Software) - Bug 331757 - drag & drop EObjects to table cells
 *    Nicolas Guyomar (Mia-Software) - Bug 332924 - To be able to save the table
 *    Nicolas Guyomar (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *    Gregoire Dupe (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *    Nicolas Guyomar (Mia-Software) - Bug 333029 - To be able to save the size of the lines and the columns
 *    Nicolas Guyomar (Mia-Software) - Bug 335020 - Nattable widget should use the Eclipse framework
 *    Nicolas Guyomar (Mia-Software) - Bug 340681 - Facet column implementation
 *    Gregoire Dupe (Mia-Software) - Bug 366804 - [Restructuring] Table widget upgrade
 *    Gregoire Dupe (Mia-Software) - Bug 367613 - Table widget refactoring
 *    Gregoire Dupe (Mia-Software) - Bug 373078 - API Cleaning
 *    Nicolas Bros (Mia-Software) - Bug 377614 - [Table] getSelection should return all the elements underlying the selected cells
 *    Nicolas Bros (Mia-Software) - Bug 377754 - [Table] right click deselects elements
 *******************************************************************************/
package org.eclipse.emf.facet.widgets.table.ui.nattable.internal.nattable;

import java.util.Iterator;
import java.util.List;

import net.sourceforge.nattable.NatTable;
import net.sourceforge.nattable.coordinate.PositionCoordinate;
import net.sourceforge.nattable.layer.cell.LayerCell;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.ETypedElement;
import org.eclipse.emf.edit.ui.dnd.LocalTransfer;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.Column;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.FeatureColumn;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.IGridElement;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.IPositionCoordinate;
import org.eclipse.emf.facet.widgets.table.ui.nattable.internal.Activator;
import org.eclipse.emf.facet.widgets.table.ui.nattable.internal.Messages;
import org.eclipse.emf.facet.widgets.table.ui.nattable.internal.NatTableWidget;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.DropTargetListener;
import org.eclipse.swt.graphics.Point;

/** The class which handle drag and drop events on the table */
public class NatTableDropListener implements DropTargetListener {

	private final NatTable natTable;
	private final NatTableWidget natTableWidget;

	public NatTableDropListener(final NatTable fNatTable, final NatTableWidget natTableWidget) {
		this.natTable = fNatTable;
		this.natTableWidget = natTableWidget;
	}

	public void dragEnter(final DropTargetEvent event) {
		// this.natTable.forceFocus();
	}

	public void dragLeave(final DropTargetEvent event) {
		// Nothing for now

	}

	public void dragOperationChanged(final DropTargetEvent event) {
		// Nothing for now
	}

	public void dragOver(final DropTargetEvent event) {

		LocalTransfer localTransfer = LocalTransfer.getInstance();
		Object data = localTransfer.nativeToJava(event.currentDataType);

		List<PositionCoordinate> selectedCellsPositions = this.natTableWidget.getSelectedCellsPositionCoordinates();
		Point point = this.natTableWidget.toControl(event.x, event.y);
		PositionCoordinate cellPosition = this.natTableWidget.getCellPositionAt(point.x, point.y);
		if (cellPosition == null) {
			if (selectedCellsPositions.size() != 0) {
				this.natTableWidget.getBodyLayer().getSelectionLayer().clear();
				this.natTable.redraw();
			}
			event.detail = DND.DROP_COPY;
			return;
		}

		if (selectedCellsPositions.size() != 1
				|| !selectedCellsPositions.get(0).equals(cellPosition)) {
			this.natTableWidget
					.getBodyLayer()
					.getSelectionLayer()
					.selectCell(cellPosition.getColumnPosition(), cellPosition.getRowPosition(),
							false, false);
			this.natTable.redraw();
		}

		IStructuredSelection structuredSelection = null;
		if (data instanceof IStructuredSelection) {
			structuredSelection = (IStructuredSelection) data;
		}

		IGridElement gridElement = getGridElement(cellPosition);
		if (gridElement == null || !canDropOn(gridElement, structuredSelection)) {
			// don't accept drop
			event.detail = DND.DROP_NONE;
		} else {
			event.detail = DND.DROP_DEFAULT;
		}
	}

	private boolean canDropOn(final IGridElement gridElement,
			final IStructuredSelection structuredSelection) {
		boolean result = false;
		final Column column = gridElement.getColumn();
		if (column instanceof FeatureColumn) {
			final FeatureColumn featureColumn = (FeatureColumn) column;
			final ETypedElement feature = featureColumn.getFeature();
			if (feature instanceof EStructuralFeature) {
				EStructuralFeature structuralFeature = (EStructuralFeature) feature;
				result = canDropOnFeature(gridElement, structuredSelection,
						structuralFeature);
			}
		}
		return result;
	}

	/**
	 * Whether the drop is allowed.
	 * 
	 * @param gridElement
	 * @param structuredSelection
	 *            the selection, or <code>null</code> if not available (some
	 *            platforms don't have selection while dragging : see
	 *            {@link DropTargetListener#dragOver(DropTargetEvent)})
	 * @param feature
	 * @return
	 */
	private boolean canDropOnFeature(final IGridElement gridElement,
			final IStructuredSelection structuredSelection, final EStructuralFeature feature) {
		if (feature.getEType() instanceof EClass) {
			EClass eClass = (EClass) feature.getEType();
			Object element = gridElement.getElement();
			if (element instanceof EObject) {
				EObject eObject = (EObject) element;
				if (!eObject.eClass().getEAllStructuralFeatures().contains(feature)) {
					return false;
				}
				if (structuredSelection != null) {
					int nElements = 0;
					Iterator<?> iterator = structuredSelection.iterator();
					while (iterator.hasNext()) {
						Object object = iterator.next();
						nElements++;
						if (nElements > 1 && !feature.isMany()) {
							return false;
						}
						if (!eClass.isInstance(object)) {
							return false;
						}
					}
					return this.natTableWidget.getController().canBeDropped(structuredSelection, feature, element);
				}
				return true;
			}
		}
		return false;
	}

	public void drop(final DropTargetEvent event) {
		Point point = this.natTableWidget.toControl(event.x, event.y);
		PositionCoordinate cellPosition = this.natTableWidget.getCellPositionAt(point.x, point.y);
		if (event.data instanceof StructuredSelection) {
			StructuredSelection structuredSelection = (StructuredSelection) event.data;
			if (cellPosition == null) {
				this.natTableWidget.getController().drop(structuredSelection);
			} else {
				IGridElement gridElement = getGridElement(cellPosition);
				if (gridElement != null) {
					dropOnGridElement(structuredSelection, gridElement);
				}
			}
		}
	}

	private IGridElement getGridElement(final PositionCoordinate cellPosition) {
		LayerCell cell = this.natTableWidget.getCellAt(cellPosition.getColumnPosition(),
				cellPosition.getRowPosition());
		if (cell != null) {
			Object dataValue = cell.getDataValue();
			if (dataValue instanceof IGridElement) {
				IGridElement gridElement = (IGridElement) dataValue;
				return gridElement;
			}
		}
		return null;
	}

	private void dropOnGridElement(final StructuredSelection structuredSelection,
			final IGridElement gridElement) {
		Column column = gridElement.getColumn();
		if (column instanceof FeatureColumn) {
			FeatureColumn featureColumn = (FeatureColumn) column;
			if (featureColumn.getFeature() instanceof EStructuralFeature) {
				EStructuralFeature structuralFeature = (EStructuralFeature) featureColumn
						.getFeature();
				dropOnFeature(structuredSelection, gridElement.getElement(),
						structuralFeature);
			} else {
				Logger.logError(
						"unhandled case:" + column.eClass().getName(), Activator.getDefault()); //$NON-NLS-1$
			}
		} else {
			Logger.logError(
					"unhandled case:" + column.eClass().getName(), Activator.getDefault()); //$NON-NLS-1$
		}
	}

	private void dropOnFeature(final StructuredSelection structuredSelection, final Object element,
			final EStructuralFeature feature) {
		if (!confirmMove((EObject) element, feature)) {
			return;
		}
		this.natTableWidget.getController().drop(structuredSelection, element, feature);
	}

	private boolean confirmMove(final EObject eObject, final EStructuralFeature feature) {
		if (feature instanceof EReference) {
			EReference reference = (EReference) feature;
			if (reference.isContainment()) {
				Object list = eObject.eGet(feature);
				boolean empty = (list == null || list instanceof List && ((List<?>) list).isEmpty());
				String message;
				if (!empty && !feature.isMany()) {
					message = Messages.NatTableDropListener_valueWillBeMovedAndPreviousLost;
				} else {
					message = Messages.NatTableDropListener_valueWillBeMoved;
				}
				String fullMessage = Messages.NatTableDropListener_compositeFeature + message;

				return MessageDialog.openConfirm(null, Messages.NatTableDropListener_moveElement,
						fullMessage);
			}
		}
		return true;
	}

	public void dropAccept(final DropTargetEvent event) {
		//
	}

}
