/*******************************************************************************
 * Copyright (c) 2015, 2016 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Grégoire Dupé (Mia-Software) - Bug 477657 - The catalog manager should be able to manage workspace model
 *    Thomas Cicognani (Mia-Software) - Bug 483684 - NullPointerException on CatalogManager
 *    Thomas Cicognani (Mia-Software) - Bug 482887 - CatalogManager create files with to long name
 *    Grégoire Dupé (Mia-Software) - Bug 482887 - CatalogManager create files with too long name
 *******************************************************************************/
package org.eclipse.emf.facet.util.emf.core.internal.catalog.v2;

import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.edit.command.AddCommand;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.emf.facet.util.emf.catalog.metamodel.internal.v1_1.catalog.CatalogFactory;
import org.eclipse.emf.facet.util.emf.catalog.metamodel.internal.v1_1.catalog.InstallAndWokspaceCatalog;
import org.eclipse.emf.facet.util.emf.core.catalog.ICatalogManager;
import org.eclipse.emf.facet.util.emf.core.catalog.ICatalogManagerConfiguration;
import org.eclipse.emf.facet.util.emf.core.internal.Activator;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.emf.transaction.util.TransactionUtil;
import org.osgi.framework.Bundle;

final class CatalogManager implements ICatalogManager, Adapter {

	private static final String MODEL_DECL_EXT_PT = "org.eclipse.emf.facet.util.emf.core.modeldeclaration"; //$NON-NLS-1$
	private static final String FILE = "file"; //$NON-NLS-1$
	private final Map<Resource, Bundle> resourceToBundle = new HashMap<Resource, Bundle>();
	private final ICatalogManagerConfiguration catalogMgrconfig;
	private InstallAndWokspaceCatalog catalog;
	private final ResourceSet resourceSet;
	private final String catalogId;

	public CatalogManager(final ICatalogManagerConfiguration catalogMgrconfig,
			final ResourceSet resourceSet, final String catalogId) {
		this.resourceSet = resourceSet;
		this.catalogMgrconfig = catalogMgrconfig;
		this.catalogId = catalogId;
		URI uri;
		if (catalogId == null) {
			final String uriStr = String.format("tmp://%s", //$NON-NLS-1$
					this.getClass().getName());
			uri = URI.createURI(uriStr);
		} else {
			final String uriStr = String.format("platform:/meta/%s/%s.xmi", //$NON-NLS-1$
					Activator.getDefault().getBundle().getSymbolicName(),
					catalogId);
			uri = URI.createURI(uriStr);
		}
		Resource resource;
		if (catalogId == null) {
			resource = this.resourceSet.createResource(uri);
		} else {
			try {
				resource = this.resourceSet.getResource(uri, true);
			} catch (org.eclipse.emf.common.util.WrappedException e) {
				if (!(e.getCause() instanceof FileNotFoundException)) {
					Logger.logError(e, Activator.getDefault());
				}
				resource = this.resourceSet.createResource(uri);
			}
		}
		if ((!resource.getContents().isEmpty())
				&& resource.getContents().get(0) instanceof InstallAndWokspaceCatalog) {
			this.catalog = (InstallAndWokspaceCatalog) resource.getContents().get(0);
		} else {
			this.catalog = CatalogFactory.eINSTANCE.createInstallAndWokspaceCatalog();
			final TransactionalEditingDomain editingDomain =
					TransactionUtil.getEditingDomain(resource);
			if (editingDomain == null) {
				resource.getContents().add(this.catalog);
			} else {
				final Command command = new AddCommand(editingDomain,
						resource.getContents(), this.catalog);
				editingDomain.getCommandStack().execute(command);
			}
		}
		initRegisteredEntries();
		CatalogListenerFactory.adaptResource(this.catalog, resource);
	}

	private void initRegisteredEntries() {
		final IExtensionRegistry registry = Platform.getExtensionRegistry();
		final IExtensionPoint modelDeclExtPoint = registry
				.getExtensionPoint(CatalogManager.MODEL_DECL_EXT_PT);
		if (modelDeclExtPoint != null) {
			for (IExtension ext : modelDeclExtPoint.getExtensions()) {
				for (IConfigurationElement configElt : ext.getConfigurationElements()) {
					final String filePath = configElt.getAttribute(CatalogManager.FILE);
					if (filePath != null) {
						final String pluginName = ext.getNamespaceIdentifier();
						final Bundle bundle = Platform.getBundle(pluginName);
						final URI uri = URI.createPlatformPluginURI(pluginName
								+ "/" + filePath, false); //$NON-NLS-1$
						try {
							final Resource resource = this.resourceSet.getResource(uri, true);
							this.resourceToBundle.put(resource, bundle);
							for (EObject root : resource.getContents()) {
								this.onInstalledModelFound(root);
							}
						} catch (Exception e) {
							Logger.logError(e, Activator.getDefault());
						}
					}
				}
			}
		}
	}

	private void onInstalledModelFound(final EObject root) {
		if (this.catalogMgrconfig.canBeManaged(root) && this.catalogMgrconfig.isValid(root)) {
			this.catalog.getInstalledEntries().add(root);
		}
	}

	public <T> List<T> getEntries(final Class<T> expectedClass) {
		final List<T> result = new ArrayList<T>();
		final List<T> filteredEntries = getInstalledEntries(expectedClass);
		result.addAll(filteredEntries);
		final List<T> wsEntriesFiltered = getWsEntries(expectedClass);
		result.addAll(wsEntriesFiltered);
		return result;
	}

	public <T> List<T> getWsEntries(final Class<T> expectedClass) {
		return getEntriesFilter(expectedClass, this.catalog.getWorkspaceEntries());
	}

	public <T> List<T> getInstalledEntries(final Class<T> expectedClass) {
		final List<T> wsEntriesFiltered = getEntriesFilter(expectedClass, 
				this.catalog.getInstalledEntries());
		return wsEntriesFiltered;
	}

	private static <T> List<T> getEntriesFilter(final Class<T> expectedClass, final List<EObject> entries) {
		final List<T> result = new ArrayList<T>();
		for (EObject eObject : entries) {
			if (expectedClass.isInstance(eObject)) {
				@SuppressWarnings("unchecked")
				/*
				 * @SuppressWarnings("unchecked"): gdupe> Checked by the call of
				 * 'isInstance' in the 'if' statement
				 */
				final T tObject = (T) eObject;
				result.add(tObject);
			}
		}
		return result;
	}

	public boolean addWsEntry(final EObject entry) {
		boolean result = false;
		if (this.catalogMgrconfig.canBeManaged(entry)
				&& this.catalogMgrconfig.isValid(entry)) {
			this.catalog.getWorkspaceEntries().add(entry);
			result = true;
		}
		return result;
		
	}

	public void removeWsEntry(final EObject entry) {
		this.catalog.getWorkspaceEntries().remove(entry);
	}

	public void removeAllWsEntries(final IProject project) {
		final List<EObject> toBeRemoved = new ArrayList<EObject>();
		final List<EObject> wsEntries = getWsEntries(EObject.class);
		for (EObject registered : wsEntries) {
			final Resource eResource = registered.eResource();
			if (eResource == null || registered.eIsProxy()) {
				/*
				 * Case where it is impossible to get the Resource (e.g. the
				 * linked object [registered] was deleted)
				 */
				toBeRemoved.add(registered);
			} else {
				final URI uri = eResource.getURI();
				final String uriStr = uri.toString();
				if (uriStr
						.startsWith("platform:/resource/" + project.getName())) { //$NON-NLS-1$
					toBeRemoved.add(registered);
				}
			}
		}
		this.catalog.getWorkspaceEntries().removeAll(toBeRemoved);
	}

	public String getCatalogId() {
		return this.catalogId;
	}
	
	public void notifyChanged(final Notification notification) {
		// Nothing to do
	}

	public Notifier getTarget() {
		return this.resourceSet;
	}

	public void setTarget(final Notifier newTarget) {
		// Nothing to do
	}

	public boolean isAdapterForType(final Object type) {
		return type == ICatalogManager.class;
	}
}
