/*******************************************************************************
 * Copyright (c) 2010 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Guyomar (Mia-Software) - Bug 331442 - To be able to add and remove lines (model elements) from the table - initial API and implementation
 *    Nicolas Bros (Mia-Software) - Bug 332438 - NatTable : table type
 *    Nicolas Bros (Mia-Software) - Bug 331757 - drag & drop EObjects to table cells
 *    Nicolas Guyomar (Mia-Software) - Bug 332924 - To be able to save the table
 *    Nicolas Guyomar (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *    Gregoire Dupe (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *    Nicolas Guyomar (Mia-Software) - Bug 333029 - To be able to save the size of the lines and the columns
 *    Nicolas Guyomar (Mia-Software) - Bug 335020 - Nattable widget should use the Eclipse framework
 *    Nicolas Guyomar (Mia-Software) - Bug 340681 - Facet column implementation
 *******************************************************************************/
package org.eclipse.emf.facet.widgets.nattable.internal;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import net.sourceforge.nattable.NatTable;
import net.sourceforge.nattable.coordinate.PositionCoordinate;
import net.sourceforge.nattable.layer.cell.LayerCell;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.edit.ui.dnd.LocalTransfer;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.AttributeColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.Column;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.ContextColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.DefaultLabelColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.EContainerColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FeatureColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.MetaClassColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.ReferenceColumn;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.DropTargetListener;
import org.eclipse.swt.graphics.Point;

/** The class which handle drag and drop events on the table */
public class NatTableDropListener implements DropTargetListener {

	private final NatTable natTable;
	private final NatTableWidget natTableWidget;

	public NatTableDropListener(final NatTable fNatTable, final NatTableWidget natTableWidget) {
		this.natTable = fNatTable;
		this.natTableWidget = natTableWidget;
	}

	public void dragEnter(final DropTargetEvent event) {
		// this.natTable.forceFocus();
	}

	public void dragLeave(final DropTargetEvent event) {
		// Nothing for now

	}

	public void dragOperationChanged(final DropTargetEvent event) {
		// Nothing for now
	}

	public void dragOver(final DropTargetEvent event) {

		LocalTransfer localTransfer = LocalTransfer.getInstance();
		Object data = localTransfer.nativeToJava(event.currentDataType);

		List<PositionCoordinate> selectedCellsPositions = this.natTableWidget
				.getSelectedCellsPositions();
		Point point = this.natTableWidget.toControl(event.x, event.y);
		PositionCoordinate cellPosition = this.natTableWidget.getCellPositionAt(point.x, point.y);
		if (cellPosition == null) {
			if (selectedCellsPositions.size() != 0) {
				this.natTableWidget.getBodyLayer().getSelectionLayer().clear();
				this.natTable.redraw();
			}
			event.detail = DND.DROP_COPY;
			return;
		}

		if (selectedCellsPositions.size() != 1
				|| !selectedCellsPositions.get(0).equals(cellPosition)) {
			this.natTableWidget
					.getBodyLayer()
					.getSelectionLayer()
					.selectCell(cellPosition.getColumnPosition(), cellPosition.getRowPosition(),
							false, false);
			this.natTable.redraw();
		}

		IStructuredSelection structuredSelection = null;
		if (data instanceof IStructuredSelection) {
			structuredSelection = (IStructuredSelection) data;
		}

		GridElement gridElement = getGridElement(cellPosition);
		if (gridElement == null || !canDropOn(gridElement, structuredSelection)) {
			// don't accept drop
			event.detail = DND.DROP_NONE;
		} else {
			event.detail = DND.DROP_DEFAULT;
		}
	}

	private boolean canDropOn(final GridElement gridElement,
			final IStructuredSelection structuredSelection) {
		Column columnDescription = gridElement.getColumnDescription();
		if (columnDescription instanceof MetaClassColumn
				|| columnDescription instanceof DefaultLabelColumn
				|| columnDescription instanceof EContainerColumn
				|| columnDescription instanceof ContextColumn) {
			return false;
		}
		if (columnDescription instanceof ReferenceColumn
				|| columnDescription instanceof AttributeColumn) {
			FeatureColumn featureColumn = (FeatureColumn) columnDescription;
			EStructuralFeature feature = featureColumn.getFeature();
			return canDropOnFeature(gridElement, structuredSelection, feature);
		}
		Logger.logError(
				"unhandled case:" + columnDescription.eClass().getName(), Activator.getDefault()); //$NON-NLS-1$

		return false;
	}

	/**
	 * Whether the drop is allowed.
	 * 
	 * @param gridElement
	 * @param structuredSelection
	 *            the selection, or <code>null</code> if not available (some
	 *            platforms don't have selection while dragging : see
	 *            {@link DropTargetListener#dragOver(DropTargetEvent)})
	 * @param feature
	 * @return
	 */
	private boolean canDropOnFeature(final GridElement gridElement,
			final IStructuredSelection structuredSelection, final EStructuralFeature feature) {
		if (feature.getEType() instanceof EClass) {
			EClass eClass = (EClass) feature.getEType();
			Object element = gridElement.getElement();
			if (element instanceof EObject) {
				EObject eObject = (EObject) element;
				if (!eObject.eClass().getEAllStructuralFeatures().contains(feature)) {
					return false;
				}
				if (structuredSelection != null) {
					int nElements = 0;
					Iterator<?> iterator = structuredSelection.iterator();
					while (iterator.hasNext()) {
						Object object = iterator.next();
						nElements++;
						if (nElements > 1 && !feature.isMany()) {
							return false;
						}
						if (!eClass.isInstance(object)) {
							return false;
						}
					}
					return this.natTableWidget.canBeDroped(structuredSelection, feature, element);
				}
				return true;
			}
		}
		return false;
	}

	public void drop(final DropTargetEvent event) {
		Point point = this.natTableWidget.toControl(event.x, event.y);
		PositionCoordinate cellPosition = this.natTableWidget.getCellPositionAt(point.x, point.y);
		if (event.data instanceof StructuredSelection) {
			StructuredSelection structuredSelection = (StructuredSelection) event.data;
			if (cellPosition == null) {
				dropInEmptySpace(structuredSelection);
			} else {
				GridElement gridElement = getGridElement(cellPosition);
				if (gridElement != null) {
					dropOnGridElement(structuredSelection, gridElement);
				}
			}
		}
	}

	private GridElement getGridElement(final PositionCoordinate cellPosition) {
		LayerCell cell = this.natTableWidget.getCellAt(cellPosition.getColumnPosition(),
				cellPosition.getRowPosition());
		if (cell != null) {
			Object dataValue = cell.getDataValue();
			if (dataValue instanceof GridElement) {
				GridElement gridElement = (GridElement) dataValue;
				return gridElement;
			}
		}
		return null;
	}

	private void dropInEmptySpace(final StructuredSelection structuredSelection) {
		List<EObject> list = new ArrayList<EObject>();
		final Iterator<?> iterator = structuredSelection.iterator();
		while (iterator.hasNext()) {
			final Object selectedElement = iterator.next();
			if (selectedElement instanceof EObject) {
				EObject selectedEObject = (EObject) selectedElement;
				if (!NatTableWidgetInternalUtils.acceptsElement(selectedEObject,
						this.natTableWidget.getTableInstance().getTableConfiguration())) {
					MessageDialog.openWarning(
							this.natTableWidget.getShell(),
							Messages.wrongEObjectType,
							Messages.wrongEObjectTypeLong
									+ EcoreUtil.getURI(selectedEObject.eClass()));
					return;
				}
				list.add(selectedEObject);
			}
		}
		// Update the Table
		this.natTableWidget.addRows(list);		
	}

	private void dropOnGridElement(final StructuredSelection structuredSelection,
			final GridElement gridElement) {
		Column columnDescription = gridElement.getColumnDescription();
		if (columnDescription instanceof ReferenceColumn
				|| columnDescription instanceof AttributeColumn) {
			FeatureColumn featureColumn = (FeatureColumn) columnDescription;
			dropOnFeature(structuredSelection, gridElement.getElement(), featureColumn.getFeature());
		} else {
			Logger.logError(
					"unhandled case:" + columnDescription.eClass().getName(), Activator.getDefault()); //$NON-NLS-1$
		}
		return;
	}

	private void dropOnFeature(final StructuredSelection structuredSelection, final Object element,
			final EStructuralFeature feature) {
		if (!confirmMove((EObject) element, feature)) {
			return;
		}
		this.natTableWidget.drop(structuredSelection, element, feature);
	}

	private boolean confirmMove(final EObject eObject, final EStructuralFeature feature) {
		if (feature instanceof EReference) {
			EReference reference = (EReference) feature;
			if (reference.isContainment()) {
				Object list = eObject.eGet(feature);
				boolean empty = (list == null || list instanceof List && ((List<?>) list).isEmpty());
				String message;
				if (!empty && !feature.isMany()) {
					message = Messages.NatTableDropListener_valueWillBeMovedAndPreviousLost;
				} else {
					message = Messages.NatTableDropListener_valueWillBeMoved;
				}
				String fullMessage = Messages.NatTableDropListener_compositeFeature + message;

				return MessageDialog.openConfirm(null, Messages.NatTableDropListener_moveElement,
						fullMessage);
			}
		}
		return true;
	}

	public void dropAccept(final DropTargetEvent event) {
		//
	}

}
