/*****************************************************************************
 * Copyright (c) 2011 CEA LIST.
 *
 *    
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Initial API and implementation
 *  Vincent Lorenzo (CEA-LIST) - Bug 337326 - Show/Hide Column : Sort the columns by name
 *  Nicolas Guyomar (Mia-Software) - Bug 340681 - Facet column implementation
 *****************************************************************************/
package org.eclipse.emf.facet.widgets.nattable.internal;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.facet.infra.query.ModelQuery;
import org.eclipse.emf.facet.infra.query.core.AbstractModelQuery;
import org.eclipse.emf.facet.infra.query.core.ModelQuerySetCatalog;
import org.eclipse.emf.facet.infra.query.core.exception.ModelQueryException;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.AttributeColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.Column;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.ContextColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.DefaultLabelColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.EContainerColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FacetAttributeColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FacetReferenceColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.MetaClassColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.QueryColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.ReferenceColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance2.ValueColumn;
import org.eclipse.emf.facet.widgets.nattable.tableconfiguration.TableConfiguration;

public final class NatTableWidgetInternalUtils {

	private NatTableWidgetInternalUtils() {

	}

	/**
	 * This method returns the name for the column
	 * 
	 * @param columnDescription
	 *            a column
	 * @return the name of the column
	 */
	public static String getColumnName(final Column columnDescription) {
		String columnName;
		if (columnDescription instanceof DefaultLabelColumn) {
			columnName = Messages.NatTableWidget_label;
		} else if (columnDescription instanceof MetaClassColumn) {
			columnName = Messages.NatTableWidget_metaclass;
		} else if (columnDescription instanceof ContextColumn) {
			columnName = Messages.NatTableWidget_queryContext;
		} else if (columnDescription instanceof EContainerColumn) {
			// TODO should be externalized
			columnName = "/eContainer"; //$NON-NLS-1$
		} else if (columnDescription instanceof QueryColumn) {
			QueryColumn queryColumn = (QueryColumn) columnDescription;
			columnName = queryColumn.getQuery().getName();
		} else if (columnDescription instanceof AttributeColumn) {
			AttributeColumn attributeColumn = (AttributeColumn) columnDescription;
			final EAttribute attribute = attributeColumn.getAttribute();
			if (attribute.isDerived()) {
				columnName = "/" + attribute.getName(); //$NON-NLS-1$
			} else {
				columnName = attribute.getName();
			}
		} else if (columnDescription instanceof ReferenceColumn) {
			ReferenceColumn referenceColumn = (ReferenceColumn) columnDescription;
			final EReference reference = referenceColumn.getReference();
			final String multiplicity;
			multiplicity = ""; //$NON-NLS-1$
			if (reference.isDerived()) {
				columnName = "/" + reference.getName() + multiplicity; //$NON-NLS-1$
			} else {
				columnName = reference.getName() + multiplicity;
			}
		} else if (columnDescription instanceof FacetReferenceColumn) {
			FacetReferenceColumn facetReferenceColumn = (FacetReferenceColumn) columnDescription;
			columnName = facetReferenceColumn.getReference().getName();
		} else if (columnDescription instanceof FacetAttributeColumn) {
			FacetAttributeColumn facetAttributeColumn = (FacetAttributeColumn) columnDescription;
			columnName = facetAttributeColumn.getAttribute().getName();
		} else if (columnDescription instanceof ValueColumn) {
			// TODO should be externalized
			columnName = "Value"; //$NON-NLS-1$
		} else {
			Logger.logWarning("Unhandled column description type", Activator.getDefault()); //$NON-NLS-1$
			columnName = ""; //$NON-NLS-1$
		}
		return columnName;
	}
	
	/**
	 * Whether the given element would be accepted in a table with the given
	 * table configuration.
	 * 
	 * @param element  the element to test.
	 * @param tableConfiguration the table configuration.
	 * @return whether the given element can be added to the table
	 */
	public static boolean acceptsElement(final Object element,
			final TableConfiguration tableConfiguration) {
		if (tableConfiguration == null) {
			return true;
		}
		ModelQuery query = tableConfiguration.getCanBePresentedInTheTable();
		if (query != null) {
			if (element instanceof EObject) {
				EObject eObject = (EObject) element;
				try {
					AbstractModelQuery modelQueryImpl = ModelQuerySetCatalog.getSingleton()
							.getModelQueryImpl(query);
					Object result = modelQueryImpl.basicEvaluate(eObject);
					if (result instanceof Boolean) {
						return ((Boolean) result).booleanValue();
					}
				} catch (ModelQueryException e) {
					Logger.logError(e, Activator.getDefault());
				}
			} else {
				return false;
			}
		}
		return true;
	}

}
