/*******************************************************************************
 * Copyright (c) 2011, 2012 CEA LIST.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Guyomar (Mia-Software) - Bug 340738 - Utility method to create a coherent tableInstance
 *    Nicolas Guyomar (Mia-Software) - Bug 340940 - To be able to view facet attributes and facet references in a table
 *    Nicolas Guyomar (Mia-Software) - Bug 340681 - Facet column implementation
 *    Vincent Lorenzo (CEA-LIST) - Bug 341328 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Nicolas Guyomar (Mia-Software) - Bug 344921 - Undo/Redo just after the creation of the table
 *    Gregoire Dupe (Mia-Software) - Bug 366804 - [Restructuring] Table widget upgrade
 *    Gregoire Dupe (Mia-Software) - Bug 369987 - [Restructuring][Table] Switch to the new customization and facet framework
 *    Gregoire Dupe (Mia-Software) - Bug 364325 - [Restructuring] The user must be able to navigate into a model using the Facet.
 *    Gregoire Dupe (Mia-Software) - Bug 373078 - API Cleaning
 *    Nicolas Bros (Mia-Software) - Bug 378475 - unit test failures after table refactoring
 *    Nicolas Bros (Mia-Software) - Bug 378649 - [Table] Errors with non-applicable features
 *    Olivier Remaud (Soft-Maint) - Bug 378499 - optimizing table opening
 *******************************************************************************/
package org.eclipse.emf.facet.widgets.table.ui.internal.exported;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.ETypedElement;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.facet.custom.metamodel.v0_2_0.custom.CustomFactory;
import org.eclipse.emf.facet.custom.metamodel.v0_2_0.custom.Customization;
import org.eclipse.emf.facet.efacet.core.FacetUtils;
import org.eclipse.emf.facet.efacet.core.IFacetManager;
import org.eclipse.emf.facet.efacet.core.IFacetManagerFactory;
import org.eclipse.emf.facet.efacet.core.exception.FacetManagerException;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.DerivedTypedElement;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.Facet;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.FacetSet;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.runtime.ETypedElementResult;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.EObjectQueryRow;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.FeatureColumn;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.NavigationTable;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.PrimitiveTypeQueryRow;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.Row;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.Table;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.TableFactory;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.tableconfiguration.TableConfiguration;
import org.eclipse.emf.facet.widgets.table.ui.internal.Activator;
import org.eclipse.emf.facet.widgets.table.ui.internal.CustomizationUtils;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;

public final class TableWidgetUtils {

	private TableWidgetUtils() {
		// Prevent instantiation
	}

	public static Table createTableInstance(final List<EObject> elements,
			final String description, final TableConfiguration tableConfiguration,
			final EObject context, final Object parameter) {

		Table tableInstance = TableFactory.eINSTANCE.createTable();
		tableInstance.setTableConfiguration(tableConfiguration);
		tableInstance.setContext(context);
		tableInstance.setParameter(parameter);
		final Set<EObject> added = new HashSet<EObject>();
		for (EObject eObject : elements) {
			if (!added.contains(eObject)) {
				final Row row = TableFactory.eINSTANCE.createRow();
				row.setElement(eObject);
				tableInstance.getRows().add(row);
				added.add(eObject);
			}
		}

		if (tableConfiguration != null) {
			for (Facet facet : tableConfiguration.getDefaultFacets()) {
				try {
					//TODO Update the meta-model to use "unique" EList.
					if (!tableInstance.getFacetSets().contains(
							facet.eContainer())) {
						tableInstance.getFacetSets().add(
								(FacetSet) facet.eContainer());
					}
				} catch (Exception e) {
					// TODO should be externalized
					Logger.logError("The referenced facet:" + facet + " could not be loaded", //$NON-NLS-1$ //$NON-NLS-2$
							Activator.getDefault());
				}
			}
			for (Customization custom : tableConfiguration.getDefaultCustomizations()) {
				if (!tableInstance.getCustomizations().contains(custom)) {
					tableInstance.getCustomizations().add(custom);
				}
			}
			for (Customization custom : tableConfiguration.getDefaultLocalCustomizations()) {
				if (!tableInstance.getLocalCustomizations().contains(custom)) {
					tableInstance.getLocalCustomizations().add(custom);
				}
			}
		}

		createColumns(tableInstance);

		tableInstance.setDescription(description);
		// we create the localCustomization if they don't exists
		// we store the nsURI which have been already added in the local
		// customization
		Set<EPackage> alreadyDone = new HashSet<EPackage>();
		List<EStructuralFeature> allFeatures = getTableInstanceEStructuralFeatures(tableInstance);
		for (EStructuralFeature structuralFeature : allFeatures) {
			EObject sfContainer = structuralFeature.eContainer();
			EObject sfContainerContainer = null;
			if (sfContainer != null) {
				sfContainerContainer = sfContainer.eContainer();
			}
			if (sfContainerContainer instanceof EPackage) {
				EPackage ePackage = (EPackage) sfContainerContainer;
				if (!alreadyDone.contains(ePackage)) {
					if (CustomizationUtils.findCustomizationExtendingEPackage(
							tableInstance.getLocalCustomizations(), ePackage) == null) {
						// we create this localCustomization
						Customization customizaion = CustomFactory.eINSTANCE
								.createCustomization();
						tableInstance.getLocalCustomizations().add(customizaion);
						alreadyDone.add(ePackage);
					}
				}
			}
		}
		// we create localCustomization for FacetColumn
		for (FacetSet container : tableInstance.getFacetSets()) {
			if (container instanceof EPackage) {
				EPackage ePackage = container;
				if (!alreadyDone.contains(container)) {
					if (CustomizationUtils.findCustomizationExtendingEPackage(
							tableInstance.getLocalCustomizations(), container) == null) {
						// we create this localCustomization
						Customization metamodelView = CustomFactory.eINSTANCE
								.createCustomization();
						tableInstance.getLocalCustomizations().add(metamodelView);
						alreadyDone.add(container);

					}
				}
			}
		}
		//we add all the local customization to the customization, at the beginning of the list
		tableInstance.getCustomizations().addAll(0, tableInstance.getLocalCustomizations());
		return tableInstance;
	}

	public static NavigationTable createQueryTableInstance(final List<ETypedElementResult> results,
			final TableConfiguration tableConfiguration, final EObject context, final Object parameter) {
		NavigationTable queryTableInstance = TableFactory.eINSTANCE
				.createNavigationTable();
		queryTableInstance.getQueryResults().addAll(results);
		queryTableInstance.setTableConfiguration(tableConfiguration);
		queryTableInstance.setContext(context);
		queryTableInstance.setParameter(parameter);
		for (ETypedElementResult result : results) {
			Object resultValue = FacetUtils.getResultValue(result);
			if (resultValue instanceof Collection<?>) {
				Collection<?> collection = (Collection<?>) resultValue;
				for (Object object : collection) {
					createQueryRow(queryTableInstance, result, object);
				}
			} else {
				createQueryRow(queryTableInstance, result, resultValue);
			}
		}
		createColumns(queryTableInstance);
		return queryTableInstance;
	}

	private static void createQueryRow(final NavigationTable queryTableInstance,
			final ETypedElementResult result, final Object object) {
		if (object instanceof EObject) {
			EObject eObject = (EObject) object;
			if (!TableWidgetUtils.getElements(queryTableInstance).contains(
					eObject)) {
				EObjectQueryRow row = TableFactory.eINSTANCE.createEObjectQueryRow();
				row.setElement(eObject);
				row.setQueryResult(result);
				queryTableInstance.getRows().add(row);
			}
		} else {
			PrimitiveTypeQueryRow row = TableFactory.eINSTANCE
					.createPrimitiveTypeQueryRow();
			row.setElement(result);
			row.setQueryResult(result);
			row.setValue(object);
			queryTableInstance.getRows().add(row);
		}
	}

	private static void createColumns(final Table tableInstance) {
		List<EStructuralFeature> features = getTableInstanceEStructuralFeatures(tableInstance);

		if (tableInstance instanceof NavigationTable) {
			tableInstance.getColumns().add(
					TableFactory.eINSTANCE.createSourceColumn());
			for (Row row : tableInstance.getRows()) {
				if (row instanceof PrimitiveTypeQueryRow) {
					tableInstance.getColumns().add(
							TableFactory.eINSTANCE.createValueColumn());
					break;
				}
			}
		}
		for (final EStructuralFeature feature : features) {
			FeatureColumn referenceColumn = TableFactory.eINSTANCE
					.createFeatureColumn();
			referenceColumn.setFeature(feature);
			tableInstance.getColumns().add(referenceColumn);
		}
		ResourceSet resourceSet = null;
		if (tableInstance.eResource() == null) {
			resourceSet = new ResourceSetImpl();
		} else {
			resourceSet = tableInstance.eResource().getResourceSet();
		}
			final IFacetManager facetContext = IFacetManagerFactory.DEFAULT.getOrCreateFacetManager(resourceSet);
			facetContext.getManagedFacetSets().addAll(0, tableInstance.getFacetSets());
			final Set<EStructuralFeature> structuralFeatures = new HashSet<EStructuralFeature>();
			List<IStatus> statusList = new ArrayList<IStatus>();
		for (EObject eObject : TableWidgetUtils.getElements(tableInstance)) {
				try {
				structuralFeatures.addAll(FacetUtils.getETypedElements(eObject,
						EStructuralFeature.class, facetContext));
				} catch (Exception e) {
					IStatus status = new Status(
							IStatus.ERROR,
							Activator.getDefault().getBundle().getSymbolicName(),
							"An exception has occurred while retrieving structural features of:" + eObject, e); //$NON-NLS-1$
					statusList.add(status);
				}
			}
			if (!statusList.isEmpty()) {
				IStatus globalStatus = new MultiStatus(
						Activator.getDefault().getBundle().getSymbolicName(),
						0,
						"An exception has occurred while retrieving structural features of the eObjects", //$NON-NLS-1$
						new Exception());
				Logger.logError(new CoreException(globalStatus), Activator.getDefault());
				MessageDialog.openError(Display.getCurrent().getActiveShell(),
						"Failed to load facets", //$NON-NLS-1$
						"Some facets failed to load. See Error Log for more details."); //$NON-NLS-1$
				// TODO Those strings have to be externalized
			}
			for (EStructuralFeature structuralFeature : structuralFeatures) {
				FeatureColumn column = TableFactory.eINSTANCE
					.createFeatureColumn();
				column.setFeature(structuralFeature);
				tableInstance.getColumns().add(column);
			}
	}


	private static List<EStructuralFeature> getTableInstanceEStructuralFeatures(final Table tableInstance) {
		List<EStructuralFeature> features = new ArrayList<EStructuralFeature>();
		for (EObject eObject : TableWidgetUtils.getElements(tableInstance)) {
			if (eObject != null
					&& !(eObject instanceof ETypedElementResult && tableInstance instanceof NavigationTable)) {
				for (EStructuralFeature eStructuralFeature : eObject.eClass()
						.getEAllStructuralFeatures()) {
					if (!features.contains(eStructuralFeature)) {
						features.add(eStructuralFeature);
					}
				}
			}
		}
		return features;
	}
	
	public static List<EObject> getElements(final Table tableInstance) {
		List<EObject> result = new LinkedList<EObject>();
		for (Row row : tableInstance.getRows()) {
			result.add(row.getElement());
		}
		return result;
	}
	
	public static Object getValueOf(final IGridElement gridElement, final IFacetManager context) throws FacetManagerException {
		Object value = null;
		if (gridElement.getColumn() instanceof FeatureColumn) {
			final EObject element = gridElement.getRow().getElement();
			if (element != null) {
				final FeatureColumn featureColumn = (FeatureColumn) gridElement.getColumn();
				final ETypedElement eTypedElement = featureColumn.getFeature();
				if (hasStructuralFeature(element, eTypedElement, context)) {
					value = context.getOrInvoke(element, eTypedElement, Object.class);
				} else {
					// not applicable
					return ""; //$NON-NLS-1$
				}
			}
		}
		return value;
	}

	public static boolean hasStructuralFeature(final EObject eObject, final ETypedElement structuralFeature, final IFacetManager context)
			throws FacetManagerException {
		boolean result = false;
		final EClass eClass = eObject.eClass();
		if (eClass.getEAllStructuralFeatures().contains(structuralFeature)) {
			result = true;
		} else {
			if (structuralFeature instanceof DerivedTypedElement) {
				DerivedTypedElement derivedTypedElement = (DerivedTypedElement) structuralFeature;
				EObject eContainer = derivedTypedElement.eContainer();
				if (eContainer instanceof Facet) {
					Facet facet = (Facet) eContainer;
					result = context.isConforming(eObject, facet);
				}
			}
		}
		return result;
	}
	
	public static boolean isApplicable(final IGridElement gridElement, final IFacetManager context) {
		boolean result = false;
		if (gridElement.getColumn() instanceof FeatureColumn) {
			final EObject element = gridElement.getRow().getElement();
			if (element != null) {
				final FeatureColumn featureColumn = (FeatureColumn) gridElement.getColumn();
				final ETypedElement eTypedElement = featureColumn.getFeature();
				try {
					if (hasStructuralFeature(element, eTypedElement, context)) {
						result = true;
					}
				} catch (FacetManagerException e) {
					Logger.logError(e, Activator.getDefault());
				}
			}
		}
		return result;
	}
}
