/*******************************************************************************
 * Copyright (c) 2011 Mia-Software
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Nicolas Bros (Mia-Software) - Bug 355951 - Wizard to create EMF Facet plug-ins
 *******************************************************************************/
package org.eclipse.emf.facet.dev.core.internal;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URL;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;

public class NewPluginCreator implements INewPluginCreator {

	private class Data {
		private final String id;
		private final String name;
		private final String version;
		private final String provider;

		public Data(final String id, final String name, final String version, final String provider) {
			this.id = id;
			this.name = name;
			this.version = version;
			this.provider = provider;
		}
	}

	public void createProject(final String id, final String name, final String version, final String provider, final boolean javaNatureEnabled,
			final IProgressMonitor monitor) {
		try {
			Data data = new Data(id, name, version, provider);
			final int nTasks = 20;
			monitor.beginTask(Messages.NewPluginWizard_jobName, nTasks);

			IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(id);
			project.create(monitor);
			monitor.worked(1);
			project.open(monitor);
			monitor.worked(1);

			IFolder metaInf = project.getFolder("META-INF"); //$NON-NLS-1$
			metaInf.create(true, true, monitor);
			monitor.worked(1);
			if (javaNatureEnabled) {
				createFile(metaInf.getFile("MANIFEST.MF"), getExpandedFileContents("resources/MANIFEST.MF.java.template", data)); //$NON-NLS-1$//$NON-NLS-2$
			} else {
				createFile(metaInf.getFile("MANIFEST.MF"), getExpandedFileContents("resources/MANIFEST.MF.template", data)); //$NON-NLS-1$//$NON-NLS-2$
			}
			monitor.worked(1);

			createFile(project.getFile("about.html"), getExpandedFileContents("resources/about.html.template", data)); //$NON-NLS-1$//$NON-NLS-2$
			monitor.worked(1);
			if (javaNatureEnabled) {
				createFile(project.getFile(".project"), getExpandedFileContents("resources/project.java.template", data)); //$NON-NLS-1$//$NON-NLS-2$
			} else {
				createFile(project.getFile(".project"), getExpandedFileContents("resources/project.template", data)); //$NON-NLS-1$//$NON-NLS-2$
			}
			monitor.worked(1);
			if (javaNatureEnabled) {
				createFile(project.getFile(".classpath"), getExpandedFileContents("resources/classpath.template", data)); //$NON-NLS-1$//$NON-NLS-2$
				monitor.worked(1);
			}
			if (javaNatureEnabled) {
				createFile(project.getFile("build.properties"), getExpandedFileContents("resources/build.properties.java.template", data)); //$NON-NLS-1$//$NON-NLS-2$
			} else {
				createFile(project.getFile("build.properties"), getExpandedFileContents("resources/build.properties.template", data)); //$NON-NLS-1$//$NON-NLS-2$
			}
			monitor.worked(1);
			if (javaNatureEnabled) {
				createFile(project.getFile(".checkstyle"), getExpandedFileContents("resources/checkstyle.template", data)); //$NON-NLS-1$//$NON-NLS-2$
				monitor.worked(1);
			}

			if (javaNatureEnabled) {
				IFolder src = project.getFolder("src"); //$NON-NLS-1$
				src.create(true, true, monitor);
				monitor.worked(1);

				String hierarchy = getHierarchy(id);
				IFolder srcInternal = createFolderHierarchy(src, hierarchy + "/internal", monitor); //$NON-NLS-1$
				monitor.worked(1);

				createFile(srcInternal.getFile("Activator.java"), getExpandedFileContents("resources/Activator.java.template", data)); //$NON-NLS-1$//$NON-NLS-2$
				monitor.worked(1);
			}

			IFolder osgiInf = project.getFolder("OSGI-INF"); //$NON-NLS-1$
			osgiInf.create(true, true, monitor);
			monitor.worked(1);
			IFolder i10n = osgiInf.getFolder("l10n"); //$NON-NLS-1$
			i10n.create(true, true, monitor);
			monitor.worked(1);

			createFile(i10n.getFile("bundle.properties"), getExpandedFileContents("resources/bundle.properties.template", data)); //$NON-NLS-1$//$NON-NLS-2$
			monitor.worked(1);

			IFolder settings = project.getFolder(".settings"); //$NON-NLS-1$
			settings.create(true, true, monitor);
			monitor.worked(1);

			createFile(settings.getFile("org.eclipse.core.resources.prefs"), getExpandedFileContents("resources/org.eclipse.core.resources.prefs.template", data)); //$NON-NLS-1$//$NON-NLS-2$
			monitor.worked(1);

			createFile(settings.getFile("org.eclipse.core.runtime.prefs"), getExpandedFileContents("resources/org.eclipse.core.runtime.prefs.template", data)); //$NON-NLS-1$//$NON-NLS-2$
			monitor.worked(1);

			if (javaNatureEnabled) {
				createFile(settings.getFile("org.eclipse.jdt.core.prefs"), getExpandedFileContents("resources/org.eclipse.jdt.core.prefs.template", data)); //$NON-NLS-1$//$NON-NLS-2$
				monitor.worked(1);
				createFile(settings.getFile("org.eclipse.jdt.ui.prefs"), getExpandedFileContents("resources/org.eclipse.jdt.ui.prefs.template", data)); //$NON-NLS-1$//$NON-NLS-2$
				monitor.worked(1);
			}

			monitor.done();
		} catch (Exception e) {
			Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Error creating new plug-in", e)); //$NON-NLS-1$
		}
	}

	private IFolder createFolderHierarchy(final IFolder folder, final String hierarchy, final IProgressMonitor monitor) throws CoreException {
		String[] parts = hierarchy.split("/"); //$NON-NLS-1$
		IFolder current = folder;
		for (String part : parts) {
			if (part.trim().length() > 0) {
				current = current.getFolder(part);
				current.create(true, true, monitor);
			}
		}
		return current;
	}

	private String getHierarchy(final String id) {
		return id.replaceAll("\\.", "/"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	private void createFile(final IFile newFile, final String contents) throws CoreException {
		InputStream source = new ByteArrayInputStream(contents.getBytes());
		if (newFile.exists()) {
			newFile.delete(true, new NullProgressMonitor());
		}
		newFile.create(source, true, new NullProgressMonitor());
	}

	private String getExpandedFileContents(final String path, final Data data) throws IOException {
		URL url = getFileURL(path);
		if (url == null) {
			Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Template not found: " + path)); //$NON-NLS-1$
			return ""; //$NON-NLS-1$
		}
		InputStream source = url.openStream();
		String contents = readInputStream(source);
		String expanded = contents
				.replaceAll("\\{id\\}", data.id).replaceAll("\\{name\\}", data.name).replaceAll("\\{version\\}", data.version).replaceAll("\\{provider\\}", data.provider); //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$ //$NON-NLS-4$
		return expanded;
	}

	protected URL getFileURL(final String path) {
		URL url = Activator.getDefault()
				.getBundle().getResource(path);
		if (url == null) {
			return null;
		}
		return url;
	}

	private static String readInputStream(final InputStream stream) throws IOException {
		return readInputStream(stream, "UTF-8"); //$NON-NLS-1$
	}

	private static String readInputStream(final InputStream stream, final String charset)
			throws IOException {
		final int bufferSize = 65536;
		final char[] buffer = new char[bufferSize];
		StringBuilder builder = new StringBuilder();
		Reader reader = new InputStreamReader(stream, charset);
		int read;
		do {
			read = reader.read(buffer, 0, buffer.length);
			if (read > 0) {
				builder.append(buffer, 0, read);
			}
		} while (read >= 0);

		reader.close();
		return builder.toString();
	}
}
