/*******************************************************************************
 * Copyright (c) 2012 CEA-LIST
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Gregoire Dupe (Mia-Software) - Bug 374903 - [Table] ITableWidget.setLoadedFacetSets
 *     Gregoire Dupe (Mia-Software) - Bug 375087 - [Table] ITableWidget.addColumn(List<ETypedElement>, List<FacetSet>)
 *     Gregoire Dupe (Mia-Software) - Bug 372626 - Aggregates
 *     Gregoire Dupe (Mia-Software) - Bug 376158 - [Table] Unexpected columns when customizations are loaded
 *     Gregoire Dupe (Mia-Software) - Bug 378701 - [Unit Test Failure] org.eclipse.emf.facet.widgets.table.tests.internal.v0_2.notuithread.Bug354224Test*
 *     Gregoire Dupe (Mia-Software) - Bug 380126 - [Table] Row sort too long 
 *******************************************************************************/
package org.eclipse.emf.facet.widgets.table.ui.internal;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CompoundCommand;
import org.eclipse.emf.ecore.ETypedElement;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.facet.custom.metamodel.v0_2_0.custom.Customization;
import org.eclipse.emf.facet.efacet.core.FacetUtils;
import org.eclipse.emf.facet.efacet.core.IFacetManager;
import org.eclipse.emf.facet.efacet.core.exception.FacetManagerException;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.DerivedTypedElement;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.FacetOperation;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.FacetSet;
import org.eclipse.emf.facet.widgets.celleditors.ICommandFactory;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.Column;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.FeatureColumn;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.Row;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.Table;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.TableFactory;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.TablePackage;
import org.eclipse.emf.facet.widgets.table.ui.internal.command.BrutalListSetCommand;
import org.eclipse.emf.facet.widgets.table.ui.internal.comparator.RowComparator;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.ColumnSortDirection;
import org.eclipse.emf.facet.widgets.table.ui.internal.exported.exception.TableWidgetRuntimeException;
import org.eclipse.osgi.util.NLS;

public class TableCommandFactory implements ITableCommandFactory {

	private final Table table;
	private final EditingDomain editingDomain;
	private final ICommandFactory commandFactory;
	private final IFacetManager facetManager;

	public TableCommandFactory(final Table table,
			final EditingDomain editingDomain,
			final ICommandFactory commandFactory,
			final IFacetManager facetManager) {
		this.table = table;
		this.editingDomain = editingDomain;
		this.commandFactory = commandFactory;
		this.facetManager = facetManager;
	}

	/**
	 * @param facetSets
	 *            must not contains more than one reference to a same FacetSet
	 */
	public Command createSetLoadedFacetSetsCommand(
			final List<FacetSet> facetSets) {
		return createSetLoadedFacetSetsCommand(facetSets, true);
	}

	/**
	 * @param facetSets
	 *            must not contains more than one reference to a same FacetSet
	 * @param manageCustoms
	 */
	public Command createSetLoadedFacetSetsCommand(
			final List<FacetSet> facetSets, final boolean manageCustoms) {
		// It is expected that the column points to the override feature (the
		// signature)
		final List<Command> commandList = new CommandList();
		final List<FeatureColumn> columnsToRemove = TableInstanceUtils
				.columnsToRemove(this.table, facetSets, this.table.getRows());
		for (Column columnToRemove : columnsToRemove) {
			final Command removeColumnCmd = createRemoveColumnCommand(columnToRemove);
			commandList.add(removeColumnCmd);
		}
		if (!this.table.getFacetSets().equals(facetSets)) {
			final Command setCommand = this.commandFactory.createSetCommand(
				this.editingDomain, this.table,
				TablePackage.eINSTANCE.getTable_FacetSets(), facetSets);
			commandList.add(setCommand);
		}
		// If some added facetSet are also customizations they have to be
		// loaded. This
		// is required by this aggregate.
		final List<Customization> addedCustoms = new LinkedList<Customization>();
		for (FacetSet facetSet : facetSets) {
			if (facetSet instanceof Customization) {
				final Customization custom = (Customization) facetSet;
				addedCustoms.add(custom);
			}
		}
		if (manageCustoms) {
			// If some removed facetSet are also customizations they have to be
			// unloaded. This
			// is required by this aggregate.

			final List<FacetSet> removedFacetSet = new ArrayList<FacetSet>();
			removedFacetSet.addAll(this.table.getFacetSets());
			removedFacetSet.removeAll(facetSets);
			final List<Customization> removedCustoms = new LinkedList<Customization>();
			for (FacetSet facetSet : removedFacetSet) {
				if (facetSet instanceof Customization) {
					final Customization custom = (Customization) facetSet;
					removedCustoms.add(custom);
				}
			}
			if (!(addedCustoms.isEmpty() && removedCustoms.isEmpty())) {
				final List<Customization> customizations = new ArrayList<Customization>();
				customizations.addAll(this.table.getCustomizations());
				customizations.removeAll(removedCustoms);
				customizations.addAll(addedCustoms);
				final Command loadCustom = createCoreSetCustomizationsCommand(customizations);
				commandList.add(loadCustom);
			}
		}
		return createResult(commandList, "Set loaded facetSets"); //$NON-NLS-1$
	}

	private static Command createResult(final List<Command> commandList,
			final String label) {
		Command result = null;
		while (commandList.contains(null)) {
			commandList.remove(null);
		}
		if (!commandList.isEmpty()) {
			result = new CompoundCommand(label, commandList);
		}
		return result;
	}

	public Command createRemoveColumnCommand(final Column column) {
		final List<Command> commandList = new ArrayList<Command>();
		if (this.table.getColumns().contains(column)) {
			final Command command = this.commandFactory.createRemoveCommand(
					this.editingDomain, this.table,
					TablePackage.eINSTANCE.getTable_Columns(), column);
			// This compoundCommand is only used to hold debug informations.
			commandList.add(command);
		}
		return createResult(commandList, "Remove column"); //$NON-NLS-1$
	}

	public Command createAddColumnCommand(
			final List<ETypedElement> eTypedElements,
			final List<FacetSet> facetSets) {
		final List<Command> commandList = new CommandList();
		final Set<FacetSet> newFacetSets = new LinkedHashSet<FacetSet>();
		newFacetSets.addAll(facetSets);
		newFacetSets.addAll(this.table.getFacetSets());
		for (ETypedElement eTypedElement : eTypedElements) {
			final FacetSet signatureFS = getSignatureFacetSet(eTypedElement);
			if (signatureFS != null) {
				newFacetSets.add(signatureFS);
			}
		}
		final Command addFacetCommand = createSetLoadedFacetSetsCommand(new ArrayList<FacetSet>(
				newFacetSets));
		commandList.add(addFacetCommand);
		final Command addColumn = createAddColumn(eTypedElements);
		commandList.add(addColumn);
		return createResult(commandList, "Add columns and facetSets"); //$NON-NLS-1$
	}

	private static FacetSet getSignatureFacetSet(
			final ETypedElement eTypedElement) {
		FacetSet result = null;
		if (eTypedElement instanceof DerivedTypedElement) {
			final DerivedTypedElement derivedTE = (DerivedTypedElement) eTypedElement;
			DerivedTypedElement signatureTE;
			try {
				signatureTE = FacetUtils.getTopOverrideFeature(derivedTE);
			} catch (FacetManagerException e) {
				throw new TableWidgetRuntimeException(e);
			}
			if (derivedTE.getOverride() != null && signatureTE != null) {
				final FacetSet signatureFS = FacetUtils
						.getFacetSet(signatureTE);
				result = signatureFS;
			}
		}
		return result;
	}

	private Command createAddColumn(final List<ETypedElement> eTypedElements) {
		final List<Command> commandList = new CommandList();
		for (ETypedElement eTypedElement : eTypedElements) {
			final Command addColumn = createAddColumn(eTypedElement);
			commandList.add(addColumn);
		}
		return createResult(commandList, "Add columns"); //$NON-NLS-1$
	}

	private Command createAddColumn(final ETypedElement eTypedElement) {
		ETypedElement signature = eTypedElement;
		if (eTypedElement instanceof FacetOperation) {
			final FacetOperation facetOperation = (FacetOperation) eTypedElement;
			signature = FacetUtils.getSignature(facetOperation);
		}
		Column existingColumn = null;
		for (Column column : this.table.getColumns()) {
			if (column instanceof FeatureColumn) {
				final FeatureColumn featureColumn = (FeatureColumn) column;
				if (featureColumn.getFeature().equals(signature)) {
					existingColumn = column;
					break;
				}
			}
		}
		final List<Command> commandList = new CommandList();
		if (existingColumn == null) {
			final FeatureColumn column = TableFactory.eINSTANCE
					.createFeatureColumn();
			final Command setColumn = this.commandFactory.createAddCommand(
					this.editingDomain, this.table,
					TablePackage.eINSTANCE.getTable_Columns(), column);
			commandList.add(setColumn);
			final Command setETypedElement = this.commandFactory
					.createSetCommand(this.editingDomain, column,
							TablePackage.eINSTANCE.getFeatureColumn_Feature(),
							signature);
			commandList.add(setETypedElement);
		}
		return createResult(commandList, "Add column"); //$NON-NLS-1$
	}
	
	public Command createSetCustomizationCommand(
			final List<Customization> customizations) {
		final List<Command> commandList = new CommandList();
		final Set<FacetSet> referredFS = CustomizationUtils
				.findFacetsCustomizedBy(customizations);
		final List<FacetSet> alreadyLoaded = this.table.getFacetSets();
		final List<Customization> removedCustoms = new ArrayList<Customization>();
		removedCustoms.addAll(this.table.getCustomizations());
		removedCustoms.removeAll(customizations);
		final List<FacetSet> facetSets = new ArrayList<FacetSet>();
		facetSets.addAll(customizations);
		// This "remove" avoid to have a same object in twice in the list.
		facetSets.removeAll(alreadyLoaded);
		facetSets.addAll(alreadyLoaded);
		// This "remove" avoid to have a same object in twice in the list.
		facetSets.removeAll(referredFS);
		facetSets.addAll(referredFS);
		facetSets.removeAll(removedCustoms);
		final Command facetLoad = createSetLoadedFacetSetsCommand(facetSets,
				false);
		commandList.add(facetLoad);
		final Command customLoad = createCoreSetCustomizationsCommand(customizations);
		commandList.add(customLoad);
		return createResult(commandList, "Load cutomizations and associated facet sets"); //$NON-NLS-1$
	}

	private Command createCoreSetCustomizationsCommand(
			final List<Customization> customizations) {
		final List<Customization> sortedCustoms = new ArrayList<Customization>();
		sortedCustoms.addAll(this.table.getLocalCustomizations());
		for (Customization customization : customizations) {
			if (!sortedCustoms.contains(customization)) {
				sortedCustoms.add(customization);
			}
		}
		final List<Command> commandList = new CommandList();
		if (!this.table.getCustomizations().equals(sortedCustoms)) {
			final Command command = this.commandFactory.createSetCommand(
					this.editingDomain, this.table,
					TablePackage.eINSTANCE.getTable_Customizations(),
					sortedCustoms);
			commandList.add(command);
		}
		return createResult(commandList, "Set loaded cutomizations"); //$NON-NLS-1$
	}

	class CommandList extends LinkedList<Command> {

		private static final long serialVersionUID = 8581553157248060152L;

		@Override
		public boolean add(final Command command) {
			boolean result;
			if (command == null) {
				result = false;
			} else {
				result = super.add(command);
			}
			return result;
		}
	}

	public Command createRemoveColumnsCommand(final List<ETypedElement> elements) {
		final List<Command> commandList = new ArrayList<Command>();
		if (elements != null && !elements.isEmpty()) {
			for (Column column : this.table.getColumns()) {
				if (column instanceof FeatureColumn) {
					final FeatureColumn featureColumn = (FeatureColumn) column;
					if (elements.contains(featureColumn.getFeature())) {
						final Command rmCommand = this.commandFactory
								.createRemoveCommand(this.editingDomain,
										this.table, TablePackage.eINSTANCE
												.getTable_Columns(),
										featureColumn);
						commandList.add(rmCommand);
					}
				}
			}

		}
		return createResult(commandList, NLS.bind(
				"Remove {0} columms", Integer.valueOf(commandList.size()))); //$NON-NLS-1$
	}

	public Command createSortRowCommand(final FeatureColumn featureColumn,
			final ColumnSortDirection direction) {
		final List<Row> rows = new ArrayList<Row>(this.table.getRows());
		final ETypedElement eTypedElement = featureColumn.getFeature();
		final RowComparator comparator = new RowComparator(eTypedElement,
				direction,
				this.facetManager);
		Collections.sort(rows, comparator);
		return new BrutalListSetCommand(this.table,
				TablePackage.eINSTANCE.getTable_Rows(), rows, Messages.TableCommandFactory_row_sort_label,
				NLS.bind(Messages.TableCommandFactory_row_sort_description,
						eTypedElement.getName()));
	}
}
