/**
 * Copyright (c) 2013 RCP Vision (http://www.rcp-vision.com) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * Lorenzo Bettini - initial API and implementation
 */
package org.eclipse.emf.parsley.dsl.ui.contentassist;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.parsley.dsl.model.ModelPackage;
import org.eclipse.emf.parsley.dsl.model.Module;
import org.eclipse.emf.parsley.dsl.typing.EmfParsleyDslTypeSystem;
import org.eclipse.emf.parsley.dsl.ui.contentassist.AbstractEmfParsleyDslProposalProvider;
import org.eclipse.emf.parsley.dsl.util.EmfParsleyDslGuiceModuleHelper;
import org.eclipse.emf.parsley.dsl.util.EmfParsleyDslModelUtil;
import org.eclipse.emf.parsley.dsl.validation.EmfParsleyDslExpectedSuperTypes;
import org.eclipse.jface.text.Region;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.swt.graphics.Image;
import org.eclipse.xtext.Assignment;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmParameterizedTypeReference;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeConstraint;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.JvmWildcardTypeReference;
import org.eclipse.xtext.common.types.TypesPackage;
import org.eclipse.xtext.common.types.access.IJvmTypeProvider;
import org.eclipse.xtext.common.types.xtext.ui.ITypesProposalProvider;
import org.eclipse.xtext.conversion.impl.QualifiedNameValueConverter;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.ui.editor.contentassist.ContentAssistContext;
import org.eclipse.xtext.ui.editor.contentassist.ICompletionProposalAcceptor;
import org.eclipse.xtext.ui.editor.contentassist.PrefixMatcher;
import org.eclipse.xtext.ui.editor.hover.IEObjectHover;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure2;
import org.eclipse.xtext.xbase.typesystem.references.LightweightTypeReference;
import org.eclipse.xtext.xbase.typesystem.references.LightweightTypeReferenceFactory;
import org.eclipse.xtext.xbase.ui.contentassist.ImportOrganizingProposal;
import org.eclipse.xtext.xbase.ui.contentassist.ReplacingAppendable;

/**
 * see http://www.eclipse.org/Xtext/documentation.html#contentAssist on how to customize content assistant
 */
@SuppressWarnings("all")
public class EmfParsleyDslProposalProvider extends AbstractEmfParsleyDslProposalProvider {
  @Inject
  private ITypesProposalProvider typeProposalProvider;
  
  @Inject
  private IJvmTypeProvider.Factory typeProviderFactory;
  
  @Inject
  @Extension
  private EmfParsleyDslExpectedSuperTypes _emfParsleyDslExpectedSuperTypes;
  
  @Inject
  @Extension
  private EmfParsleyDslGuiceModuleHelper _emfParsleyDslGuiceModuleHelper;
  
  @Inject
  private ReplacingAppendable.Factory appendableFactory;
  
  @Inject
  private EmfParsleyDslTypeSystem typeSystem;
  
  @Override
  public void completeViewSpecification_Type(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    Class<?> _expectedSupertype = this._emfParsleyDslExpectedSuperTypes.getExpectedSupertype(ModelPackage.Literals.VIEW_SPECIFICATION);
    this.showOnlySubtypesOf(model, context, acceptor, _expectedSupertype);
  }
  
  @Override
  public void completeFeatureAssociatedExpression_ParameterType(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    this.showSubtypesOfEObjectForEmfFeatureAccess(model, context, acceptor);
  }
  
  @Override
  public void completeFeatureSpecification_ParameterType(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    this.showSubtypesOfEObjectForEmfFeatureAccess(model, context, acceptor);
  }
  
  @Override
  public void completeControlFactorySpecification_ParameterType(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    this.showSubtypesOfEObjectForEmfFeatureAccess(model, context, acceptor);
  }
  
  @Override
  public void completeExtendsClause_SuperType(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    EObject _eContainer = model.eContainer();
    Class<?> _expectedSupertype = this._emfParsleyDslExpectedSuperTypes.getExpectedSupertype(_eContainer);
    this.showOnlySubtypesOf(model, context, acceptor, _expectedSupertype);
  }
  
  protected void showSubtypesOfEObjectForEmfFeatureAccess(final EObject model, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    Class<?> _expectedSupertype = this._emfParsleyDslExpectedSuperTypes.getExpectedSupertype(ModelPackage.Literals.FEATURE_ASSOCIATED_EXPRESSION);
    this.showOnlySubtypesOf(model, context, acceptor, _expectedSupertype);
  }
  
  protected void showOnlySubtypesOf(final EObject model, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor, final Class<?> superType) {
    boolean _equals = Objects.equal(superType, null);
    if (_equals) {
      return;
    }
    Resource _eResource = model.eResource();
    ResourceSet _resourceSet = _eResource.getResourceSet();
    final IJvmTypeProvider jvmTypeProvider = this.typeProviderFactory.createTypeProvider(_resourceSet);
    String _name = superType.getName();
    final JvmType interfaceToImplement = jvmTypeProvider.findTypeByName(_name);
    this.typeProposalProvider.createSubTypeProposals(interfaceToImplement, this, context, 
      TypesPackage.Literals.JVM_PARAMETERIZED_TYPE_REFERENCE__TYPE, acceptor);
  }
  
  @Override
  public void completeBinding_TypeDecl(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    this.createStandardJavaTypesProposals(context, acceptor);
    Module _containingModule = EmfParsleyDslModelUtil.containingModule(model);
    Iterable<JvmOperation> _allGuiceValueBindingsMethodsInSuperclass = this._emfParsleyDslGuiceModuleHelper.getAllGuiceValueBindingsMethodsInSuperclass(_containingModule);
    final Procedure2<ReplacingAppendable, JvmOperation> _function = new Procedure2<ReplacingAppendable, JvmOperation>() {
      @Override
      public void apply(final ReplacingAppendable appendable, final JvmOperation op) {
        JvmTypeReference _returnType = op.getReturnType();
        LightweightTypeReference _lightweightTypeReference = EmfParsleyDslProposalProvider.this.toLightweightTypeReference(_returnType, model);
        appendable.append(_lightweightTypeReference);
        appendable.append(" ");
        String _simpleName = op.getSimpleName();
        int _length = "value".length();
        String _substring = _simpleName.substring(_length);
        appendable.append(_substring);
      }
    };
    this.createBindingProposals(model, _allGuiceValueBindingsMethodsInSuperclass, context, acceptor, _function);
  }
  
  @Override
  public void completeBinding_TypeToBind(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    Module _containingModule = EmfParsleyDslModelUtil.containingModule(model);
    Iterable<JvmOperation> _allGuiceTypeBindingsMethodsInSuperclass = this._emfParsleyDslGuiceModuleHelper.getAllGuiceTypeBindingsMethodsInSuperclass(_containingModule);
    final Function1<JvmOperation, JvmTypeReference> _function = new Function1<JvmOperation, JvmTypeReference>() {
      @Override
      public JvmTypeReference apply(final JvmOperation op) {
        return EmfParsleyDslProposalProvider.this.extractWildcardUpperBound(op);
      }
    };
    this.completeTypeOrProvideBinding(model, _allGuiceTypeBindingsMethodsInSuperclass, _function, assignment, context, acceptor);
  }
  
  @Override
  public void completeBinding_Type(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    Module _containingModule = EmfParsleyDslModelUtil.containingModule(model);
    Iterable<JvmOperation> _allGuiceProviderBindingsMethodsInSuperclass = this._emfParsleyDslGuiceModuleHelper.getAllGuiceProviderBindingsMethodsInSuperclass(_containingModule);
    final Function1<JvmOperation, JvmTypeReference> _function = new Function1<JvmOperation, JvmTypeReference>() {
      @Override
      public JvmTypeReference apply(final JvmOperation op) {
        JvmTypeReference _xblockexpression = null;
        {
          JvmTypeReference _extractWildcardUpperBound = EmfParsleyDslProposalProvider.this.extractWildcardUpperBound(op);
          final JvmParameterizedTypeReference providerType = ((JvmParameterizedTypeReference) _extractWildcardUpperBound);
          EList<JvmTypeReference> _arguments = providerType.getArguments();
          _xblockexpression = IterableExtensions.<JvmTypeReference>head(_arguments);
        }
        return _xblockexpression;
      }
    };
    this.completeTypeOrProvideBinding(model, _allGuiceProviderBindingsMethodsInSuperclass, _function, assignment, context, acceptor);
  }
  
  private JvmTypeReference extractWildcardUpperBound(final JvmOperation op) {
    JvmTypeReference _xblockexpression = null;
    {
      JvmTypeReference _returnType = op.getReturnType();
      final JvmParameterizedTypeReference returnType = ((JvmParameterizedTypeReference) _returnType);
      EList<JvmTypeReference> _arguments = returnType.getArguments();
      JvmTypeReference _head = IterableExtensions.<JvmTypeReference>head(_arguments);
      final JvmWildcardTypeReference argument = ((JvmWildcardTypeReference) _head);
      EList<JvmTypeConstraint> _constraints = argument.getConstraints();
      JvmTypeConstraint _head_1 = IterableExtensions.<JvmTypeConstraint>head(_constraints);
      _xblockexpression = _head_1.getTypeReference();
    }
    return _xblockexpression;
  }
  
  private void completeTypeOrProvideBinding(final EObject model, final Iterable<JvmOperation> superClassValueBindings, final Function1<? super JvmOperation, ? extends JvmTypeReference> typeExtractor, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    this.createStandardJavaTypesProposals(context, acceptor);
    final Procedure2<ReplacingAppendable, JvmOperation> _function = new Procedure2<ReplacingAppendable, JvmOperation>() {
      @Override
      public void apply(final ReplacingAppendable appendable, final JvmOperation op) {
        final JvmTypeReference typeReference = typeExtractor.apply(op);
        LightweightTypeReference _lightweightTypeReference = EmfParsleyDslProposalProvider.this.toLightweightTypeReference(typeReference, model);
        appendable.append(_lightweightTypeReference);
      }
    };
    this.createBindingProposals(model, superClassValueBindings, context, acceptor, _function);
  }
  
  /**
   * show the standard Java type completions
   */
  private void createStandardJavaTypesProposals(final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    QualifiedNameValueConverter _qualifiedNameValueConverter = this.getQualifiedNameValueConverter();
    ITypesProposalProvider.Filter _createVisibilityFilter = this.createVisibilityFilter(context);
    this.completeJavaTypes(context, TypesPackage.Literals.JVM_PARAMETERIZED_TYPE_REFERENCE__TYPE, true, _qualifiedNameValueConverter, _createVisibilityFilter, acceptor);
  }
  
  private void createBindingProposals(final EObject model, final Iterable<JvmOperation> superClassValueBindings, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor, final Procedure2<? super ReplacingAppendable, ? super JvmOperation> proposalTextStrategy) {
    for (final JvmOperation op : superClassValueBindings) {
      this.createProposals(model, op, context, acceptor, proposalTextStrategy);
    }
  }
  
  private LightweightTypeReference toLightweightTypeReference(final JvmTypeReference typeRef, final EObject model) {
    return this.typeSystem.toLightweightTypeReference(typeRef, model);
  }
  
  private void createProposals(final EObject model, final JvmOperation op, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor, final Procedure2<? super ReplacingAppendable, ? super JvmOperation> proposalTextStrategy) {
    final IXtextDocument document = context.getDocument();
    Resource _eResource = model.eResource();
    final XtextResource resource = ((XtextResource) _eResource);
    Region _replaceRegion = context.getReplaceRegion();
    final int offset = _replaceRegion.getOffset();
    Region _replaceRegion_1 = context.getReplaceRegion();
    int _length = _replaceRegion_1.getLength();
    final ReplacingAppendable appendable = this.appendableFactory.create(document, resource, offset, _length);
    proposalTextStrategy.apply(appendable, op);
    final Image image = this.getImage(op);
    XtextResource _resource = context.getResource();
    final LightweightTypeReferenceFactory typeConverter = this.getTypeConverter(_resource);
    String _qualifiedName = op.getQualifiedName();
    String _simpleName = op.getSimpleName();
    final StyledString displayString = this.getStyledDisplayString(op, false, 0, _qualifiedName, _simpleName, typeConverter);
    Region _replaceRegion_2 = context.getReplaceRegion();
    final ImportOrganizingProposal completionProposal = this.createCompletionProposal(appendable, _replaceRegion_2, displayString, image);
    completionProposal.setPriority(1500);
    final PrefixMatcher _function = new PrefixMatcher() {
      @Override
      public boolean isCandidateMatchingPrefix(final String name, final String prefix) {
        boolean _xblockexpression = false;
        {
          final PrefixMatcher delegate = context.getMatcher();
          String _simpleName = op.getSimpleName();
          _xblockexpression = delegate.isCandidateMatchingPrefix(_simpleName, prefix);
        }
        return _xblockexpression;
      }
    };
    completionProposal.setMatcher(_function);
    completionProposal.setAdditionalProposalInfo(op);
    IEObjectHover _hover = this.getHover();
    completionProposal.setHover(_hover);
    acceptor.accept(completionProposal);
  }
  
  private ImportOrganizingProposal createCompletionProposal(final ReplacingAppendable appendable, final Region replaceRegion, final StyledString displayString, final Image image) {
    int _offset = replaceRegion.getOffset();
    int _length = replaceRegion.getLength();
    int _offset_1 = replaceRegion.getOffset();
    return new ImportOrganizingProposal(appendable, _offset, _length, _offset_1, image, displayString);
  }
}
