/**
 * Copyright (c) 2016 RCP Vision (http://www.rcp-vision.com) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Lorenzo Bettini - initial API and implementation
 */
package org.eclipse.emf.parsley.dsl.pluginxml;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.parsley.dsl.pluginxml.PluginXmlUtils;
import org.eclipse.jface.text.Document;
import org.eclipse.pde.core.plugin.IPlugin;
import org.eclipse.pde.core.plugin.IPluginAttribute;
import org.eclipse.pde.core.plugin.IPluginElement;
import org.eclipse.pde.core.plugin.IPluginExtension;
import org.eclipse.pde.core.plugin.IPluginModelFactory;
import org.eclipse.pde.core.plugin.IPluginObject;
import org.eclipse.pde.internal.core.text.DocumentElementNode;
import org.eclipse.pde.internal.core.text.IDocumentAttributeNode;
import org.eclipse.pde.internal.core.text.IDocumentElementNode;
import org.eclipse.pde.internal.core.text.plugin.PluginAttribute;
import org.eclipse.pde.internal.core.text.plugin.PluginElementNode;
import org.eclipse.pde.internal.core.text.plugin.PluginExtensionNode;
import org.eclipse.pde.internal.core.text.plugin.PluginModel;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@SuppressWarnings("all")
public class PluginXmlLoader extends PluginModel {
  private List<PluginExtensionNode> pluginExtensionNodes;
  
  private List<PluginElementNode> pluginExtensionElementNodes;
  
  public PluginXmlLoader(final String source) {
    super(new Document(source), true);
  }
  
  /**
   * The nodes corresponding to &lt;extension&gt; elements in the plugin.xml file
   */
  public List<PluginExtensionNode> getExtensionNodes() {
    boolean _equals = Objects.equal(this.pluginExtensionNodes, null);
    if (_equals) {
      this.initializeExtensionNodes();
    }
    return this.pluginExtensionNodes;
  }
  
  protected List<PluginExtensionNode> initializeExtensionNodes() {
    IPlugin _plugin = this.getPlugin();
    IPluginExtension[] _extensions = _plugin.getExtensions();
    Iterable<PluginExtensionNode> _filter = Iterables.<PluginExtensionNode>filter(((Iterable<?>)Conversions.doWrapArray(_extensions)), PluginExtensionNode.class);
    List<PluginExtensionNode> _list = IterableExtensions.<PluginExtensionNode>toList(_filter);
    return this.pluginExtensionNodes = _list;
  }
  
  /**
   * The nodes corresponding to elements inside &lt;extension&gt;, for example,
   * &lt;view&gt;, &lt;editor&gt;
   */
  public List<PluginElementNode> getExtensionElements() {
    boolean _equals = Objects.equal(this.pluginExtensionElementNodes, null);
    if (_equals) {
      this.initializeExtensionElements();
    }
    return this.pluginExtensionElementNodes;
  }
  
  protected List<PluginElementNode> initializeExtensionElements() {
    List<PluginExtensionNode> _extensionNodes = this.getExtensionNodes();
    final Function1<PluginExtensionNode, Iterable<PluginElementNode>> _function = new Function1<PluginExtensionNode, Iterable<PluginElementNode>>() {
      @Override
      public Iterable<PluginElementNode> apply(final PluginExtensionNode it) {
        return PluginXmlLoader.this.mapToNodes(it);
      }
    };
    List<Iterable<PluginElementNode>> _map = ListExtensions.<PluginExtensionNode, Iterable<PluginElementNode>>map(_extensionNodes, _function);
    Iterable<PluginElementNode> _flatten = Iterables.<PluginElementNode>concat(_map);
    List<PluginElementNode> _list = IterableExtensions.<PluginElementNode>toList(_flatten);
    return this.pluginExtensionElementNodes = _list;
  }
  
  private Iterable<PluginElementNode> mapToNodes(final DocumentElementNode it) {
    IDocumentElementNode[] _childNodes = it.getChildNodes();
    return Iterables.<PluginElementNode>filter(((Iterable<?>)Conversions.doWrapArray(_childNodes)), PluginElementNode.class);
  }
  
  public List<PluginElementNode> getExtensionChildren(final PluginElementNode node) {
    Iterable<PluginElementNode> _mapToNodes = this.mapToNodes(node);
    return IterableExtensions.<PluginElementNode>toList(_mapToNodes);
  }
  
  public Iterable<Map.Entry<String, IDocumentAttributeNode>> getPluginAttributesEntrySet(final PluginElementNode node) {
    return PluginXmlUtils.getPluginAttributesEntrySet(node);
  }
  
  public PluginExtensionNode getExtensionByPoint(final String p) {
    List<PluginExtensionNode> _extensionNodes = this.getExtensionNodes();
    final Function1<PluginExtensionNode, Boolean> _function = new Function1<PluginExtensionNode, Boolean>() {
      @Override
      public Boolean apply(final PluginExtensionNode it) {
        String _point = it.getPoint();
        return Boolean.valueOf(Objects.equal(_point, p));
      }
    };
    return IterableExtensions.<PluginExtensionNode>findFirst(_extensionNodes, _function);
  }
  
  public String getElementExtension(final PluginElementNode node) {
    IPluginObject _parent = node.getParent();
    return ((PluginExtensionNode) _parent).getPoint();
  }
  
  public PluginElementNode getElementByTagAndId(final String xmlTag, final String id) {
    List<PluginElementNode> _extensionElements = this.getExtensionElements();
    final Function1<PluginElementNode, Boolean> _function = new Function1<PluginElementNode, Boolean>() {
      @Override
      public Boolean apply(final PluginElementNode it) {
        String _xMLTagName = it.getXMLTagName();
        return Boolean.valueOf(Objects.equal(_xMLTagName, xmlTag));
      }
    };
    Iterable<PluginElementNode> _filter = IterableExtensions.<PluginElementNode>filter(_extensionElements, _function);
    final Function1<PluginElementNode, Boolean> _function_1 = new Function1<PluginElementNode, Boolean>() {
      @Override
      public Boolean apply(final PluginElementNode it) {
        boolean _xblockexpression = false;
        {
          final IDocumentAttributeNode v = PluginXmlLoader.this.getId(it);
          boolean _xifexpression = false;
          boolean _notEquals = (!Objects.equal(v, null));
          if (_notEquals) {
            String _attributeAsString = PluginXmlLoader.this.getAttributeAsString(v);
            _xifexpression = Objects.equal(_attributeAsString, id);
          } else {
            _xifexpression = false;
          }
          _xblockexpression = _xifexpression;
        }
        return Boolean.valueOf(_xblockexpression);
      }
    };
    return IterableExtensions.<PluginElementNode>findFirst(_filter, _function_1);
  }
  
  public IDocumentAttributeNode getId(final DocumentElementNode node) {
    return PluginXmlUtils.getId(node);
  }
  
  public String getAttributeAsString(final IDocumentAttributeNode a) {
    return ((PluginAttribute) a).getValue();
  }
  
  /**
   * Copies all the extension and extension elements from the source
   * plugin xml into this plugin xml.  Attributes with the same id
   * will be overwritten in this plugin xml.
   */
  public void copyFromPluginXml(final String source) throws CoreException {
    PluginXmlLoader _pluginXmlLoader = new PluginXmlLoader(source);
    List<PluginElementNode> _extensionElements = _pluginXmlLoader.getExtensionElements();
    for (final PluginElementNode e : _extensionElements) {
      this.copy(e);
    }
  }
  
  /**
   * Assumes that the source has an id.  If a corresponding element
   * in this plugin xml file is not found it will be inserted first.
   */
  public void copy(final PluginElementNode source) throws CoreException {
    final String xmlTagName = source.getXMLTagName();
    IDocumentAttributeNode _id = this.getId(source);
    String _attributeAsString = this.getAttributeAsString(_id);
    PluginElementNode target = this.getElementByTagAndId(xmlTagName, _attributeAsString);
    boolean _equals = Objects.equal(target, null);
    if (_equals) {
      String _elementExtension = this.getElementExtension(source);
      PluginElementNode _insertExtensionElement = this.insertExtensionElement(_elementExtension, xmlTagName);
      target = _insertExtensionElement;
    }
    this.copy(source, target);
  }
  
  /**
   * This assumes that both the source and the target are not null
   */
  public void copy(final PluginElementNode source, final PluginElementNode target) throws CoreException {
    final IPluginAttribute[] atts = source.getAttributes();
    for (final IPluginAttribute a : atts) {
      {
        final PluginAttribute att = ((PluginAttribute) a);
        final PluginAttribute copy = new PluginAttribute();
        String _name = att.getName();
        copy.setName(_name);
        String _value = att.getValue();
        copy.setValue(_value);
        TreeMap _nodeAttributesMap = PluginXmlUtils.getNodeAttributesMap(target);
        String _name_1 = att.getName();
        _nodeAttributesMap.put(_name_1, copy);
      }
    }
    final IDocumentElementNode[] children = target.getChildNodes();
    IDocumentElementNode[] _childNodes = source.getChildNodes();
    for (final IDocumentElementNode c : _childNodes) {
      {
        final Function1<IDocumentElementNode, Boolean> _function = new Function1<IDocumentElementNode, Boolean>() {
          @Override
          public Boolean apply(final IDocumentElementNode it) {
            String _xMLTagName = it.getXMLTagName();
            String _xMLTagName_1 = c.getXMLTagName();
            return Boolean.valueOf(Objects.equal(_xMLTagName, _xMLTagName_1));
          }
        };
        IDocumentElementNode myChild = IterableExtensions.<IDocumentElementNode>findFirst(((Iterable<IDocumentElementNode>)Conversions.doWrapArray(children)), _function);
        boolean _equals = Objects.equal(myChild, null);
        if (_equals) {
          IPluginModelFactory _pluginFactory = this.getPluginFactory();
          IPluginElement _createElement = _pluginFactory.createElement(target);
          final PluginElementNode newChild = ((PluginElementNode) _createElement);
          String _xMLTagName = c.getXMLTagName();
          newChild.setXMLTagName(_xMLTagName);
          target.addChildNode(newChild);
          this.copy(((PluginElementNode) c), newChild);
        } else {
          this.copy(((PluginElementNode) c), ((PluginElementNode) myChild));
        }
      }
    }
  }
  
  public PluginExtensionNode insertExtension(final String point) throws CoreException {
    IPluginModelFactory _pluginFactory = this.getPluginFactory();
    final IPluginExtension e = _pluginFactory.createExtension();
    e.setPoint(point);
    IPlugin _plugin = this.getPlugin();
    _plugin.add(e);
    this.initializeExtensionNodes();
    return ((PluginExtensionNode) e);
  }
  
  /**
   * If there is no extension element with the specified point, it
   * will be automatically inserted.
   */
  public PluginElementNode insertExtensionElement(final String point, final String xmlTag) throws CoreException {
    PluginExtensionNode ext = this.getExtensionByPoint(point);
    boolean _equals = Objects.equal(ext, null);
    if (_equals) {
      PluginExtensionNode _insertExtension = this.insertExtension(point);
      ext = _insertExtension;
    }
    IPluginModelFactory _pluginFactory = this.getPluginFactory();
    IPluginElement _createElement = _pluginFactory.createElement(ext);
    final PluginElementNode element = ((PluginElementNode) _createElement);
    ext.addChildNode(element);
    element.setXMLTagName(xmlTag);
    this.initializeExtensionElements();
    return element;
  }
  
  public String getContentsAsString() {
    StringConcatenation _builder = new StringConcatenation();
    IPlugin _plugin = this.getPlugin();
    String _string = _plugin.toString();
    String _replaceFirst = _string.replaceFirst("eclipse version=\"3.0", "eclipse version=\"3.4");
    String _replaceFirst_1 = _replaceFirst.replaceFirst("(<plugin)\\r?\\n(>)", "<plugin>");
    _builder.append(_replaceFirst_1, "");
    _builder.newLineIfNotEmpty();
    return _builder.toString();
  }
}
