/**
 * Copyright (c) 2013 RCP Vision (http://www.rcp-vision.com) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * Lorenzo Bettini - initial API and implementation
 */
package org.eclipse.emf.parsley.dsl.validation;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.collect.ListMultimap;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.parsley.dsl.model.EmfFeatureAccess;
import org.eclipse.emf.parsley.dsl.model.ExtendsClause;
import org.eclipse.emf.parsley.dsl.model.FieldSpecification;
import org.eclipse.emf.parsley.dsl.model.ModelPackage;
import org.eclipse.emf.parsley.dsl.model.Module;
import org.eclipse.emf.parsley.dsl.model.PartSpecification;
import org.eclipse.emf.parsley.dsl.model.PartsSpecifications;
import org.eclipse.emf.parsley.dsl.model.ProviderBinding;
import org.eclipse.emf.parsley.dsl.model.TypeBinding;
import org.eclipse.emf.parsley.dsl.model.ValueBinding;
import org.eclipse.emf.parsley.dsl.model.ViewSpecification;
import org.eclipse.emf.parsley.dsl.model.WithExtendsClause;
import org.eclipse.emf.parsley.dsl.typing.EmfParsleyDslTypeSystem;
import org.eclipse.emf.parsley.dsl.util.EmfParsleyDslGuiceModuleHelper;
import org.eclipse.emf.parsley.dsl.validation.AbstractEmfParsleyDslValidator;
import org.eclipse.emf.parsley.dsl.validation.EmfParsleyDslExpectedSuperTypes;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.resource.IContainer;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.IResourceDescription;
import org.eclipse.xtext.resource.IResourceDescriptions;
import org.eclipse.xtext.resource.impl.ResourceDescriptionsProvider;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.CheckType;
import org.eclipse.xtext.xbase.jvmmodel.IJvmModelAssociations;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.typesystem.override.IResolvedOperation;
import org.eclipse.xtext.xbase.typesystem.override.ResolvedFeatures;
import org.eclipse.xtext.xbase.typesystem.util.Multimaps2;

/**
 * Custom validation rules.
 * 
 * see http://www.eclipse.org/Xtext/documentation.html#validation
 */
@SuppressWarnings("all")
public class EmfParsleyDslValidator extends AbstractEmfParsleyDslValidator {
  public final static String TYPE_MISMATCH = "org.eclipse.emf.parsley.dsl.TypeMismatch";
  
  public final static String CYCLIC_INHERITANCE = "org.eclipse.emf.parsley.dsl.CyclicInheritance";
  
  public final static String FINAL_FIELD_NOT_INITIALIZED = "org.eclipse.emf.parsley.dsl.FinalFieldNotInitialized";
  
  public final static String TOO_LITTLE_TYPE_INFORMATION = "org.eclipse.emf.parsley.dsl.TooLittleTypeInformation";
  
  public final static String DUPLICATE_BINDING = "org.eclipse.emf.parsley.dsl.DuplicateBinding";
  
  public final static String DUPLICATE_ELEMENT = "org.eclipse.emf.parsley.dsl.DuplicateElement";
  
  public final static String NON_COMPLIANT_BINDING = "org.eclipse.emf.parsley.dsl.NonCompliantBinding";
  
  @Inject
  private EmfParsleyDslTypeSystem typeSystem;
  
  @Inject
  @Extension
  private EmfParsleyDslExpectedSuperTypes _emfParsleyDslExpectedSuperTypes;
  
  @Inject
  @Extension
  private EmfParsleyDslGuiceModuleHelper _emfParsleyDslGuiceModuleHelper;
  
  @Inject
  @Extension
  private IJvmModelAssociations _iJvmModelAssociations;
  
  @Inject
  private ResourceDescriptionsProvider rdp;
  
  @Inject
  private IContainer.Manager cm;
  
  private final ModelPackage modelPackage = ModelPackage.eINSTANCE;
  
  @Check(CheckType.NORMAL)
  public void checkDuplicateViewSpecificationAcrossFiles(final ViewSpecification viewSpecification) {
    final Iterable<IEObjectDescription> descriptions = this.getVisibleEObjectDescriptions(viewSpecification, 
      ModelPackage.Literals.VIEW_SPECIFICATION);
    for (final IEObjectDescription desc : descriptions) {
      if (((Objects.equal(desc.getQualifiedName().toString(), viewSpecification.getId()) && 
        (!Objects.equal(desc.getEObjectOrProxy(), viewSpecification))) && 
        (!Objects.equal(desc.getEObjectURI().trimFragment(), viewSpecification.eResource().getURI())))) {
        String _id = viewSpecification.getId();
        String _plus = ("The part id " + _id);
        String _plus_1 = (_plus + " is already defined");
        EAttribute _viewSpecification_Id = this.modelPackage.getViewSpecification_Id();
        this.error(_plus_1, _viewSpecification_Id, 
          EmfParsleyDslValidator.DUPLICATE_ELEMENT);
        return;
      }
    }
  }
  
  private Iterable<IEObjectDescription> getVisibleEObjectDescriptions(final EObject o, final EClass type) {
    List<IContainer> _visibleContainers = this.getVisibleContainers(o);
    final Function1<IContainer, Iterable<IEObjectDescription>> _function = new Function1<IContainer, Iterable<IEObjectDescription>>() {
      @Override
      public Iterable<IEObjectDescription> apply(final IContainer container) {
        return container.getExportedObjectsByType(type);
      }
    };
    List<Iterable<IEObjectDescription>> _map = ListExtensions.<IContainer, Iterable<IEObjectDescription>>map(_visibleContainers, _function);
    return Iterables.<IEObjectDescription>concat(_map);
  }
  
  private List<IContainer> getVisibleContainers(final EObject o) {
    List<IContainer> _xblockexpression = null;
    {
      Resource _eResource = o.eResource();
      final IResourceDescriptions index = this.rdp.getResourceDescriptions(_eResource);
      Resource _eResource_1 = o.eResource();
      URI _uRI = _eResource_1.getURI();
      final IResourceDescription rd = index.getResourceDescription(_uRI);
      _xblockexpression = this.cm.getVisibleContainers(rd, index);
    }
    return _xblockexpression;
  }
  
  @Check
  public void checkViewSpecification(final ViewSpecification viewSpecification) {
    JvmTypeReference _type = viewSpecification.getType();
    Class<?> _expectedSupertype = this._emfParsleyDslExpectedSuperTypes.getExpectedSupertype(viewSpecification);
    this.checkType(viewSpecification, _type, _expectedSupertype, 
      ModelPackage.Literals.VIEW_SPECIFICATION__TYPE);
  }
  
  @Check
  public void checkEmfFeatureAccess(final EmfFeatureAccess emfFeatureAccess) {
    JvmTypeReference _parameterType = emfFeatureAccess.getParameterType();
    Class<?> _expectedSupertype = this._emfParsleyDslExpectedSuperTypes.getExpectedSupertype(emfFeatureAccess);
    this.checkType(emfFeatureAccess, _parameterType, _expectedSupertype, 
      ModelPackage.Literals.EMF_FEATURE_ACCESS__PARAMETER_TYPE);
  }
  
  @Check
  public void checkExtendsClause(final WithExtendsClause withExtendsClause) {
    if (((!Objects.equal(withExtendsClause.getExtendsClause(), null)) && (!this.hasCycleInHierarchy(withExtendsClause)))) {
      ExtendsClause _extendsClause = withExtendsClause.getExtendsClause();
      ExtendsClause _extendsClause_1 = withExtendsClause.getExtendsClause();
      JvmTypeReference _superType = _extendsClause_1.getSuperType();
      Class<?> _expectedSupertype = this._emfParsleyDslExpectedSuperTypes.getExpectedSupertype(withExtendsClause);
      this.checkType(_extendsClause, _superType, _expectedSupertype, 
        ModelPackage.Literals.EXTENDS_CLAUSE__SUPER_TYPE);
    }
  }
  
  @Check
  public void checkFieldInitialization(final FieldSpecification f) {
    if (((!f.isWriteable()) && Objects.equal(f.getRight(), null))) {
      String _name = f.getName();
      String _plus = ("The blank final field " + _name);
      String _plus_1 = (_plus + " may not have been initialized");
      this.error(_plus_1, 
        ModelPackage.Literals.FIELD_SPECIFICATION__NAME, 
        EmfParsleyDslValidator.FINAL_FIELD_NOT_INITIALIZED);
    }
    if ((Objects.equal(f.getType(), null) && Objects.equal(f.getRight(), null))) {
      String _name_1 = f.getName();
      String _plus_2 = ("The field " + _name_1);
      String _plus_3 = (_plus_2 + " needs an explicit type since there is no initialization expression to infer the type from.");
      this.error(_plus_3, f, ModelPackage.Literals.FIELD_SPECIFICATION__NAME, 
        EmfParsleyDslValidator.TOO_LITTLE_TYPE_INFORMATION);
    }
  }
  
  @Check
  public void checkModule(final Module module) {
    final JvmGenericType guiceModuleClass = this._emfParsleyDslGuiceModuleHelper.getModuleInferredType(module);
    boolean _equals = Objects.equal(guiceModuleClass, null);
    if (_equals) {
      return;
    }
    final PartsSpecifications partsSpecifications = module.getPartsSpecifications();
    boolean _notEquals = (!Objects.equal(partsSpecifications, null));
    if (_notEquals) {
      EList<PartSpecification> _parts = partsSpecifications.getParts();
      this.checkDuplicateViewSpecifications(_parts);
    }
    final Iterable<JvmOperation> methods = guiceModuleClass.getDeclaredOperations();
    boolean _isEmpty = IterableExtensions.isEmpty(methods);
    if (_isEmpty) {
      return;
    }
    this.checkDuplicateBindings(methods);
    this.checkCorrectValueBindings(guiceModuleClass, methods, module);
    Iterable<JvmGenericType> _allWithExtendsClauseInferredJavaTypes = this._emfParsleyDslGuiceModuleHelper.getAllWithExtendsClauseInferredJavaTypes(module);
    for (final JvmGenericType t : _allWithExtendsClauseInferredJavaTypes) {
      this.checkDuplicateSpecifications(t);
    }
  }
  
  private void checkDuplicateBindings(final Iterable<JvmOperation> methods) {
    final ListMultimap<String, JvmOperation> map = this.<String, JvmOperation>duplicatesMultimap();
    for (final JvmOperation m : methods) {
      String _simpleName = m.getSimpleName();
      map.put(_simpleName, m);
    }
    final Procedure1<JvmOperation> _function = new Procedure1<JvmOperation>() {
      @Override
      public void apply(final JvmOperation d) {
        Set<EObject> _sourceElements = EmfParsleyDslValidator.this._iJvmModelAssociations.getSourceElements(d);
        final EObject source = IterableExtensions.<EObject>head(_sourceElements);
        String _duplicateBindingMessage = EmfParsleyDslValidator.this.duplicateBindingMessage(source, d);
        EStructuralFeature _duplicateBindingFeature = EmfParsleyDslValidator.this.duplicateBindingFeature(source);
        EmfParsleyDslValidator.this.error(_duplicateBindingMessage, source, _duplicateBindingFeature, 
          EmfParsleyDslValidator.DUPLICATE_BINDING);
      }
    };
    this.<JvmOperation>checkDuplicates(map, _function);
  }
  
  /**
   * Since for fields we generate getter/setter, checking duplicate Java methods
   * will automatically check for duplicate fields as well.
   */
  private void checkDuplicateSpecifications(final JvmGenericType inferredType) {
    final ResolvedFeatures inferredFeatures = this._emfParsleyDslGuiceModuleHelper.getJavaResolvedFeatures(inferredType);
    final List<IResolvedOperation> methods = inferredFeatures.getDeclaredOperations();
    final ListMultimap<String, JvmOperation> map = this.<String, JvmOperation>duplicatesMultimap();
    final HashSet<EObject> errorSourceSeen = CollectionLiterals.<EObject>newHashSet();
    for (final IResolvedOperation m : methods) {
      String _javaMethodResolvedErasedSignature = this._emfParsleyDslGuiceModuleHelper.getJavaMethodResolvedErasedSignature(m);
      JvmOperation _declaration = m.getDeclaration();
      map.put(_javaMethodResolvedErasedSignature, _declaration);
    }
    final Procedure1<JvmOperation> _function = new Procedure1<JvmOperation>() {
      @Override
      public void apply(final JvmOperation d) {
        Set<EObject> _sourceElements = EmfParsleyDslValidator.this._iJvmModelAssociations.getSourceElements(d);
        final EObject source = IterableExtensions.<EObject>head(_sourceElements);
        boolean _add = errorSourceSeen.add(source);
        if (_add) {
          EmfParsleyDslValidator.this.error(
            "Duplicate element", source, 
            null, 
            EmfParsleyDslValidator.DUPLICATE_ELEMENT);
        }
      }
    };
    this.<JvmOperation>checkDuplicates(map, _function);
  }
  
  private void checkDuplicateViewSpecifications(final List<PartSpecification> parts) {
    final ListMultimap<String, EObject> map = this.<String, EObject>duplicatesMultimap();
    Iterable<ViewSpecification> _filter = Iterables.<ViewSpecification>filter(parts, ViewSpecification.class);
    for (final ViewSpecification p : _filter) {
      String _id = p.getId();
      map.put(_id, p);
    }
    final Procedure1<EObject> _function = new Procedure1<EObject>() {
      @Override
      public void apply(final EObject d) {
        EAttribute _viewSpecification_Id = EmfParsleyDslValidator.this.modelPackage.getViewSpecification_Id();
        EmfParsleyDslValidator.this.error(
          "Duplicate element", d, _viewSpecification_Id, 
          EmfParsleyDslValidator.DUPLICATE_ELEMENT);
      }
    };
    this.<EObject>checkDuplicates(map, _function);
  }
  
  private <T extends Object> void checkDuplicates(final ListMultimap<String, T> map, final Procedure1<? super T> errorReporter) {
    Map<String, Collection<T>> _asMap = map.asMap();
    Set<Map.Entry<String, Collection<T>>> _entrySet = _asMap.entrySet();
    for (final Map.Entry<String, Collection<T>> entry : _entrySet) {
      {
        final Collection<T> duplicates = entry.getValue();
        int _size = duplicates.size();
        boolean _greaterThan = (_size > 1);
        if (_greaterThan) {
          for (final T d : duplicates) {
            errorReporter.apply(d);
          }
        }
      }
    }
  }
  
  public void checkCorrectValueBindings(final JvmGenericType guiceModuleClass, final Iterable<JvmOperation> methods, final Module module) {
    final Iterable<JvmOperation> superClassValueBindings = this._emfParsleyDslGuiceModuleHelper.getAllGuiceValueBindingsMethodsInSuperclass(guiceModuleClass);
    for (final JvmOperation superBinding : superClassValueBindings) {
      {
        final Function1<JvmOperation, Boolean> _function = new Function1<JvmOperation, Boolean>() {
          @Override
          public Boolean apply(final JvmOperation it) {
            String _simpleName = it.getSimpleName();
            String _simpleName_1 = superBinding.getSimpleName();
            return Boolean.valueOf(Objects.equal(_simpleName, _simpleName_1));
          }
        };
        final JvmOperation matching = IterableExtensions.<JvmOperation>findFirst(methods, _function);
        if (((!Objects.equal(matching, null)) && (!this.typeSystem.isConformant(module, superBinding.getReturnType(), matching.getReturnType())))) {
          JvmTypeReference _returnType = matching.getReturnType();
          String _simpleName = _returnType.getSimpleName();
          String _plus = ("Incorrect value binding: " + _simpleName);
          String _plus_1 = (_plus + 
            " is not compliant with inherited binding\'s type ");
          JvmTypeReference _returnType_1 = superBinding.getReturnType();
          String _simpleName_1 = _returnType_1.getSimpleName();
          String _plus_2 = (_plus_1 + _simpleName_1);
          Set<EObject> _sourceElements = this._iJvmModelAssociations.getSourceElements(matching);
          EObject _head = IterableExtensions.<EObject>head(_sourceElements);
          EReference _valueBinding_TypeDecl = this.modelPackage.getValueBinding_TypeDecl();
          this.error(_plus_2, _head, _valueBinding_TypeDecl, 
            EmfParsleyDslValidator.NON_COMPLIANT_BINDING);
        }
      }
    }
  }
  
  protected void checkType(final EObject context, final JvmTypeReference actualType, final Class<?> expectedType, final EStructuralFeature feature) {
    boolean _notEquals = (!Objects.equal(actualType, null));
    if (_notEquals) {
      boolean _isConformant = this.typeSystem.isConformant(context, expectedType, actualType);
      boolean _not = (!_isConformant);
      if (_not) {
        String _simpleName = actualType.getSimpleName();
        String _plus = ("Type mismatch: cannot convert from " + _simpleName);
        String _plus_1 = (_plus + 
          " to ");
        String _simpleName_1 = expectedType.getSimpleName();
        String _plus_2 = (_plus_1 + _simpleName_1);
        this.error(_plus_2, context, feature, 
          EmfParsleyDslValidator.TYPE_MISMATCH);
      }
    }
  }
  
  protected boolean hasCycleInHierarchy(final WithExtendsClause withExtendsClause) {
    ExtendsClause _extendsClause = withExtendsClause.getExtendsClause();
    JvmTypeReference _superType = _extendsClause.getSuperType();
    JvmType _type = null;
    if (_superType!=null) {
      _type=_superType.getType();
    }
    final JvmType superType = _type;
    if ((superType instanceof JvmGenericType)) {
      HashSet<JvmGenericType> _newHashSet = CollectionLiterals.<JvmGenericType>newHashSet();
      boolean _hasCycleInHierarchy = this.hasCycleInHierarchy(((JvmGenericType)superType), _newHashSet);
      if (_hasCycleInHierarchy) {
        String _simpleName = ((JvmGenericType)superType).getSimpleName();
        String _plus = ("The inheritance hierarchy of " + _simpleName);
        String _plus_1 = (_plus + " contains cycles");
        ExtendsClause _extendsClause_1 = withExtendsClause.getExtendsClause();
        this.error(_plus_1, _extendsClause_1, 
          ModelPackage.Literals.EXTENDS_CLAUSE__SUPER_TYPE, 
          EmfParsleyDslValidator.CYCLIC_INHERITANCE);
        return true;
      }
    }
    return false;
  }
  
  protected boolean hasCycleInHierarchy(final JvmGenericType type, final Set<JvmGenericType> processedSuperTypes) {
    boolean _contains = processedSuperTypes.contains(type);
    if (_contains) {
      return true;
    }
    processedSuperTypes.add(type);
    EList<JvmTypeReference> _superTypes = type.getSuperTypes();
    for (final JvmTypeReference superTypeRef : _superTypes) {
      JvmType _type = superTypeRef.getType();
      if ((_type instanceof JvmGenericType)) {
        JvmType _type_1 = superTypeRef.getType();
        boolean _hasCycleInHierarchy = this.hasCycleInHierarchy(((JvmGenericType) _type_1), processedSuperTypes);
        if (_hasCycleInHierarchy) {
          return true;
        }
      }
    }
    processedSuperTypes.remove(type);
    return false;
  }
  
  private <K extends Object, T extends Object> ListMultimap<K, T> duplicatesMultimap() {
    return Multimaps2.<K, T>newLinkedHashListMultimap();
  }
  
  private String duplicateBindingMessage(final EObject source, final JvmOperation method) {
    String _switchResult = null;
    boolean _matched = false;
    if (source instanceof TypeBinding) {
      _matched=true;
      JvmTypeReference _returnType = method.getReturnType();
      _switchResult = _returnType.getSimpleName();
    }
    if (!_matched) {
      if (source instanceof ProviderBinding) {
        _matched=true;
        JvmTypeReference _returnType = method.getReturnType();
        _switchResult = _returnType.getSimpleName();
      }
    }
    if (!_matched) {
      if (source instanceof ValueBinding) {
        _matched=true;
        _switchResult = ((ValueBinding)source).getId();
      }
    }
    if (!_matched) {
      JvmTypeReference _returnType = method.getReturnType();
      _switchResult = _returnType.getSimpleName();
    }
    return ("Duplicate binding for: " + _switchResult);
  }
  
  private EStructuralFeature duplicateBindingFeature(final EObject e) {
    EStructuralFeature _switchResult = null;
    boolean _matched = false;
    if (e instanceof TypeBinding) {
      _matched=true;
      _switchResult = this.modelPackage.getTypeBinding_TypeToBind();
    }
    if (!_matched) {
      if (e instanceof ProviderBinding) {
        _matched=true;
        _switchResult = this.modelPackage.getProviderBinding_Type();
      }
    }
    if (!_matched) {
      if (e instanceof ValueBinding) {
        _matched=true;
        _switchResult = this.modelPackage.getValueBinding_Id();
      }
    }
    if (!_matched) {
      _switchResult = null;
    }
    return _switchResult;
  }
}
