//------------------------------------------------------------------------------
// Copyright (c) 2005, 2006 IBM Corporation and others.
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Eclipse Public License v1.0
// which accompanies this distribution, and is available at
// http://www.eclipse.org/legal/epl-v10.html
//
// Contributors:
// IBM Corporation - initial implementation
//------------------------------------------------------------------------------
package org.eclipse.epf.authoring.ui.actions;

import org.eclipse.core.internal.resources.ResourceException;
import org.eclipse.epf.authoring.ui.AuthoringUIPlugin;
import org.eclipse.epf.authoring.ui.AuthoringUIResources;
import org.eclipse.epf.authoring.ui.PerspectiveListUtil;
import org.eclipse.epf.authoring.ui.UIActionDispatcher;
import org.eclipse.epf.authoring.ui.preferences.LibraryLocationData;
import org.eclipse.epf.authoring.ui.views.ViewHelper;
import org.eclipse.epf.authoring.ui.wizards.NewLibraryWizard;
import org.eclipse.epf.library.LibraryService;
import org.eclipse.epf.library.ui.LibraryUIUtil;
import org.eclipse.epf.library.ui.preferences.LibraryUIPreferences;
import org.eclipse.epf.library.xmi.XMILibraryUtil;
import org.eclipse.epf.uma.MethodLibrary;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;

/**
 * Helper classes to create a new Method Library.
 * 
 * @author Bingxue Xu
 * @author Kelvin Low
 * @since 1.0
 */
public class NewLibrary {

	static IStructuredSelection selection;

	protected static boolean debug;

	/**
	 * Creates a new Method Library.
	 */
	public static void create() {
		debug = AuthoringUIPlugin.getDefault().isDebugging();
		LibraryLocationData llData = new LibraryLocationData();
		llData.loadFromPreferenceStore();

		String libPath = null;
		String libName = null;
		String libFileName = null;

		// Instantiate and initialize the wizard.
		NewLibraryWizard wizard = new NewLibraryWizard(llData);
		wizard.init(PlatformUI.getWorkbench(), selection); // probably no need
															// for this line

		// Instantiate the wizard container with the wizard and open it.
		WizardDialog dialog = new WizardDialog(Display.getCurrent()
				.getActiveShell(), wizard);
		dialog.create();
		dialog.open();

		if (!llData.isOkPressed())
			return;

		// Save the new library path to preference store.
		llData.saveToPreferenceStore();

		libPath = llData.getParentFolder();
		libName = llData.getLibName();
		libFileName = "library.xmi"; //$NON-NLS-1$

		if (debug) {
			System.out.println("$$$ libPath = " + libPath); //$NON-NLS-1$
			System.out.println("$$$ libName = " + libName); //$NON-NLS-1$
			System.out.println("$$$ libFileName = " + libFileName); //$NON-NLS-1$
		}

		try {
			// Close all method editors.
			ViewHelper.closeAllEditors();

			// Close library error view.
			ViewHelper.closeMessageView();
			
			// Close the current library.
			LibraryService.getInstance().closeCurrentMethodLibrary();
			
			// Create the new library.
			MethodLibrary library = XMILibraryUtil.createMethodLibrary(libName, libPath);
			
			// Set the current method library.
			LibraryService.getInstance().setCurrentMethodLibrary(library);
			
			LibraryUIPreferences.setSavedLibraryPath(libPath
					+ System.getProperty("file.separator") + libFileName); //$NON-NLS-1$

			LibraryUIUtil.updateShellTitle();
		} catch (Exception e) {
			if (e instanceof ResourceException) {
				ResourceException re = (ResourceException) e;
				if (re.getStatus() != null && re.getStatus().getCode() == 77) {
					String title = AuthoringUIResources.errorDialog_title; //$NON-NLS-1$
					String error = AuthoringUIResources.errorDialog_error; //$NON-NLS-1$
					String message = AuthoringUIResources.bind(AuthoringUIResources.errorDialog_message, error); //$NON-NLS-1$
					AuthoringUIPlugin.getDefault().getMsgDialog().displayError(
							title, message);
				}
			} else {
				e.printStackTrace();
				String title = AuthoringUIResources.errorDialog_title; //$NON-NLS-1$
				String message = AuthoringUIResources.bind(AuthoringUIResources.errorDialog_message, e.toString()); //$NON-NLS-1$
				AuthoringUIPlugin.getDefault().getMsgDialog().displayError(
						title, message);
			}
		}

		// Switch to the Authoring perspective.
		if (!PerspectiveListUtil.isAuthoringPerspective()) {
			UIActionDispatcher.openAuthoringPerspective();
		}
	}

}
